#!/usr/bin/env python3
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from dataclasses import dataclass

import RcfTypeFuture as Rcf
import Tac

@dataclass( frozen=True )
class CodeUnitKey:
   """
   Class that uniquely identifies a code unit across all sources of RCF compiler
   input.
   """
   domain: str
   codeUnitName: str

   def __post_init__( self ):
      assert self.domain in Rcf.Metadata.FunctionDomain.attributes

   @staticmethod
   def unnamedCodeUnitKey():
      """Helper method to get the key for the unnamed user-defined code unit"""
      return CodeUnitKey( Rcf.Metadata.FunctionDomain.USER_DEFINED,
                          Rcf.Config.unnamedCodeUnitName )

   def isUnnamedCodeUnitKey( self ):
      """Helper method to determine if this is the key for the unnamed code unit"""
      return self == self.unnamedCodeUnitKey()

   def isBuiltin( self ):
      return self.domain == Rcf.Metadata.FunctionDomain.BUILTIN

   def isUserDefined( self ):
      return self.domain == Rcf.Metadata.FunctionDomain.USER_DEFINED

   def isPoaWrapper( self ):
      return self.domain == Rcf.Metadata.FunctionDomain.POA_WRAPPER

   def isOpenConfig( self ):
      return self.domain == Rcf.Metadata.FunctionDomain.OPEN_CONFIG

   def toStrepForDiag( self ):
      """Helper method to render the code unit key in compiler error messages"""
      if self.isPoaWrapper():
         # BUG998069 - enhance POA string for internal wrappers
         return "in POA config"
      elif self.isOpenConfig():
         return f"function openconfig {self.codeUnitName}"
      elif self.isBuiltin():
         assert False, "Attempting to render error in builtin functions"
         return "" # Make pylint happy
      elif self.isUnnamedCodeUnitKey():
         return ""
      else:
         return f"unit {self.codeUnitName}"

   def toStrepForDebugCli( self ):
      """Helper method to render the code unit key in 'show bgp debug policy' outputs
      """
      assert self.isUserDefined(), (
         "This should never be called for non user defined code units" )
      if self.isUnnamedCodeUnitKey():
         return "Unnamed code unit"
      else:
         return f"Code unit {self.codeUnitName}"

   def getPoaWrapperFunctionName( self ):
      """Helper method to get the actual POA wrapper function name which is hot
      swapped in at the AST gen phase.
      """
      assert self.isPoaWrapper(), (
         "This should never be called on code units that are not POA wrappers" )
      return Rcf.Poa.RcfPoaConfig.getInternalPoaWrapperFunctionName(
         self.codeUnitName )

class CodeUnitMapping:
   """
   Class that contains a simple mapping from code unit keys to the corresponding
   texts.
   """
   def __init__( self, codeUnitDict ):
      """
      Args:
         codeUnitDict ( dict ): CodeUnitKey -> code unit text (str)
      """
      # Verify that 'codeUnitDict' is formed correctly
      assert( all( isinstance( k, CodeUnitKey ) for k in codeUnitDict ) )
      self.codeUnitDict = codeUnitDict
      # Tabs are expanded before compilation to ensure column offsets in Debug
      # Symbols are consistent with how the text will be rendered.
      # Note this is not done to the RCF text in Sysdb so the customers code is
      # not modified
      self.codeUnitRcfTexts = { codeUnitKey: text.expandtabs( 4 )
                                for codeUnitKey, text in self.codeUnitDict.items() }
      self.codeUnitRcfTextLines_ = {}

   def rcfTextLinesForCodeUnit( self, codeUnitKey ):
      '''
      This function splits an Rcf text on new lines. This is used for the rendering
      logic.
      Args:
         codeUnitKey ( CodeUnitKey ): The code unit to provide a text for
      '''
      if codeUnitKey in self.codeUnitRcfTextLines_:
         return self.codeUnitRcfTextLines_[ codeUnitKey ]

      textLines = self.codeUnitRcfTexts[ codeUnitKey ].split( '\n' )
      self.codeUnitRcfTextLines_[ codeUnitKey ] = textLines
      return textLines

def generateCodeUnitsForDomain( domain, perDomainCodeUnitTexts ):
   # Generate code unit texts for a domain e.g.
   # - OpenConfig functions (that are not contained in code units)
   # - Wrapper functions for arguments at POAs
   #
   # Args:
   #    domain (str): Where these internal code units came from, ex. "OPEN_CONFIG" or
   #       "POA_WRAPPER". This should match Rcf::Metadata::FunctionDomain.
   #    perDomainCodeUnitTexts (dict): dict keyed by some name to rcf text. The name
   #       will be replaced with a unique CodeUnitKey encompassing the domain and the
   #       name so that these code units do not collide with user provided code
   #       units.
   # Returns: a dictionary keyed by CodeUnitKey with the same values. This can
   #    be joined with the user-provided code units and passed to CodeUnitMapping.
   return { CodeUnitKey( domain, key ): value for key, value in
            perDomainCodeUnitTexts.items() }
