# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Agent
import BothTrace
import Cli
import CliPatchSigint
import CliServerLib
import ConfigAgentName
import SysdbUtil
import Tac
import os
import resource
import weakref

bt0 = BothTrace.trace0

agentNameStr = ConfigAgentName.name()

# record the entityManager in a global, so pyClient can access it.
__myEntityManager__ = None

class ConfigAgent( Agent.Agent ):
   def __init__( self, entityManager, agentName=agentNameStr, blocking=False ):
      # Disable profile unloading in MountFileInfo. This gives a small persistent
      # memory hit (~2MB), but prevents the profile data from thrashing every time a
      # Cli command is run. We do this (redundantly) in two places: here and
      # Cli.initCli. The Cli.initCli call ensures that CliPlugin load and future Cli
      # commands do not unload plugins (the main slowdown). We also call it here so
      # that a fully-fledged ConfigAgent disables profile unloading from
      # construction. This ensures that the default CAgent mounts don't unload
      # profiles either. This is a small optimization on top the CliPlugins, but will
      # prevent a few extra unloads
      entityManager.cEm_.mountFileInfoWrap.disableUnDoInit()
      global __myEntityManager__
      __myEntityManager__ = entityManager
      self.entMan_ = entityManager
      self.ssoStandby_ = False
      self.sysname_ = self.entMan_.sysname()
      # blocking_ is to be used in breadth tests only
      self.blocking_ = blocking
      self.agentName_ = agentName
      self.cli_ = None
      self.cliServerThread_ = None
      self.cliInitialized_ = False
      super().__init__( entityManager, agentName=agentName )

   def doInit( self, entityManager ):
      bt0( 'doInit' )

      self.ssoStandby_ = SysdbUtil.onStandbySsoSupervisor()

      def cliInitialized( cli ):
         self.cli_ = cli
         self.cliServerThread_ = CliServerLib.CliServerThread( entityManager,
                                                               self.cli_ )
         self.cliServerThread_.start()
         self.cliInitialized_ = True
         bt0( "Agent is now warm" )

      # Monkey-patch Tac sigint handlers befor loading CliPlugins
      CliPatchSigint.init()
      Cli.initCli( entityManager, callback=cliInitialized,
                   block=False,
                   allowActiveMounts=False,
                   runSessionManager=True,
                   redundancyStatus=self.redundancyStatus(),
                   agent=weakref.proxy( self ) )

   def warm( self ):
      return self.cliInitialized_ and self.cli_ and self.cli_.pluginsComplete()

def main():
   # Opt-out of Pulse counter writing.  CliPlugins can load libraries
   # that contain Pulse counters.  Setting this env-var disables the
   # Pulse subsystem from writing the counters.  Pulse remains enabled
   # for reading, and thus the "show debug counter" commands still work.
   os.environ[ 'NOPULSE' ] = '1'

   os.environ[ 'TACC_HANDLESIGTERM' ] = "1"

   # pylint: disable-next=consider-using-f-string
   qtfile = "{}{}.qt".format( agentNameStr, "-%d" if "QUICKTRACEDIR"
                              not in os.environ else "" )
   BothTrace.initialize( qtfile, "0,8,0,0,0,0,0,0,16,32" )

   resource.setrlimit( resource.RLIMIT_NOFILE, ( 5120, 5120 ) )
   ssoStandby = SysdbUtil.onStandbySsoSupervisor()
   bypass = not ssoStandby
   container = Agent.AgentContainer( [ ConfigAgent ],
                                     rootFlags="",
                                     agentTitle=agentNameStr,
                                     # Immediately fail with EntityNotFound if the
                                     # path does not exist in Sysdb
                                     autoWaitUntilExist=False,
                                     bypassSysdbInit=bypass,
                                     mountProfilePath="ConfigAgent*",
                                     useConfigCounter=True )
   container.runAgents()
