# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSession
import GnmiSetCliSession
import Tac
import Tracing

from GenericReactor import GenericReactor

t0 = Tracing.Handle( "ASConfigAgentPlugin" ).trace0
t9 = Tracing.Handle( "ASConfigAgentPlugin" ).trace9

# React to the dirty path to-be-modified by AirStream and add it to the session root
class DirtyPathSm( Tac.Notifiee ):
   notifierTypeName = "AirStream::GnmiSetSessionState"

   def __init__( self, em ):
      self.em = em
      self.state = Tac.singleton( self.notifierTypeName )
      self.state.reset()
      Tac.Notifiee.__init__( self, self.state )

   def __del__( self ):
      self.state.reset()

   @Tac.handler( "dirtyPath" )
   def handleDirtyPath( self, key ):
      t9( 'handleDirtyPath key ' + str( key ) )
      if key not in self.state.dirtyPath:
         # - Skip the session complete aka singleton reset case
         # - All preCommit SMs will stopped in one shot. So, skip stopping the SM
         #   related to this path as well
         return

      # This is for the tests that never set gnmi session but purely test
      # AirStream on session path (i.e., AirStreamServerTest)
      if not self.state.sessionName:
         t9( 'skip bookkeeping for gNMI set on non-gNMI session' )
         return

      # AirStream collect the absolute path but session root require relative one
      path = GnmiSetCliSession.relativePath( key, self.state.sessionPath )
      t9( 'adding ' + path + ' to session root' )
      CliSession.addSessionRoot( self.em, path, self.state.sessionName )

      # If the path is of interest for any pre-commit SM, add other paths of
      # interest of the SM as session roots and start the SM
      GnmiSetCliSession.SmRunner.runSm( self.em, self.state.sessionName, path )

class AirStreamRunnabilitySm:
   def __init__( self, em, mg, agent ):
      self.em_ = em
      # Plugin is laoded by MainCli which can be instantiated either by ConfigAgent
      # or standalone CLI instance.
      self.agent_ = getattr( agent, 'agent_', agent )
      self.runnabilityDir_ = None
      self.localAirStreamDir_ = None
      self.gnmiConf_ = None
      self.gnmiReactor_ = None
      self.octaConf_ = None
      self.octaReactor_ = None
      self.redundancyReactor_ = None
      self.sm_ = None
      self.asClock_ = None
      self.dirtyPathSm_ = None
      self.doMounts( mg )

   def doMounts( self, mg ):
      self.gnmiConf_ = mg.mount( 'mgmt/gnmi/config', 'Gnmi::Config' )
      self.octaConf_ = mg.mount( 'mgmt/octa/config', 'Octa::Config' )

   def run( self ):
      if not self.agent_:
         # Breadth tests e.g. CliTest.runCapiCli()
         return
      self.redundancyReactor_ = GenericReactor( self.agent_.redundancyStatus(),
            [ 'mode' ], self.handleRedundancyMode, callBackNow=True )

   def handleRedundancyMode( self, notifiee=None ):
      redundancy = self.agent_.redundancyStatus()
      t0( 'protocol:', redundancy.protocol, '- mode:', redundancy.mode )
      if redundancy.protocol == 'sso' and redundancy.mode != 'active':
         t0( 'Dont run on standby supe in SSO' )
         self.cleanup()
         return

      if self.sm_:
         return

      # /<sysname>/airstream needed by PathToType entity that holds collection that
      # informs paths and its types that are exported by AirStream in the agent
      localRoot = self.em_.root().parent
      self.localAirStreamDir_ = localRoot.mkdir( 'airstream' )

      t0( 'Creating AirStreamRunnabilitySm' )
      agentRoot = getattr( self.agent_, 'agentRoot_', None )
      if not agentRoot:
         # In breadth tests, MainCli can be cohabiting
         t0( 'No agent root. Skipping runnability SM' )
         return
      # Create a TaskGroup under which all AirStream-related tasks should be run.
      # ConfigAgent does not currently serve gNMI Subscribe requests. If it were
      # to do so in future, the state machines that it creates for this purpose
      # should use the same TaskGroup to leverage the congestion backoff mechanism.
      priorityNormal = Tac.newInstance( 'Ark::TaskPriority', 'normal' )
      priorityEnforcementAny = \
            Tac.newInstance( 'Ark::TaskGroupPriorityEnforcement', 'allowAny' )
      notifyTaskGroup = Tac.newInstance( 'Ark::TaskGroup', 'ConfigAgent-GnmiNotify',
            priorityNormal, priorityEnforcementAny, self.agent_.agentScheduler() )
      self.runnabilityDir_ = agentRoot.mkdir( "airstream/runnability" )

      providerConfig = Tac.newInstance( 'AirStream::BaseCustomProviderConfig',
            notifyTaskGroup )
      self.sm_ = Tac.newInstance( "AirStream::AirStreamRunnabilitySm",
                             self.runnabilityDir_,
                             self.em_.cEm_,
                             self.em_.sysname(),
                             self.agent_.agentName,
                             providerConfig )

      # We need the AirStream server to start in the main thread. When OpenConfig is
      # configured in a Cli thread, we'd like to defer reacting to the next activity
      # cycle so that the main thread can pick it up and start the AirStream (grpc)
      # server
      self.asClock_ = Tac.ClockNotifiee( self.handleRunnability, Tac.endOfTime )
      self.gnmiReactor_ = GenericReactor( self.gnmiConf_, [ 'enabled' ],
                           self.deferHandler )
      self.octaReactor_ = GenericReactor( self.octaConf_, [ 'enabled' ],
                           self.deferHandler,
                           callBackNow=True )

   def deferHandler( self, notifiee=None ):
      t0( 'deferHandler' )
      self.asClock_.timeMin = Tac.now()

   def handleRunnability( self ):
      t0( 'handleRunnability' )
      assert self.runnabilityDir_
      if self.gnmiConf_.enabled or self.octaConf_.enabled:
         self.dirtyPathSm_ = DirtyPathSm( self.em_ )
         self.runnabilityDir_.createEntity(
               "AirStream::Config", self.agent_.agentName )
      else:
         self.dirtyPathSm_ = None
         self.runnabilityDir_.deleteEntity( self.agent_.agentName )

   def cleanup( self ):
      if self.runnabilityDir_ and self.runnabilityDir_.get( self.agent_.agentName ):
         self.runnabilityDir_.deleteEntity( self.agent_.agentName )
      self.sm_ = None
      self.asClock_ = None
      self.gnmiReactor_ = None
      self.octaReactor_ = None

def Plugin( context ):
   context.registerStateMachine( AirStreamRunnabilitySm )
