# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This plugin is responsible for starting the fabric interface configuration
creator.

This is the system which is responsible for listening on requests coming in from
FRU/CLI/etc. and creating the appropriate interface configuration objects.
'''

from __future__ import absolute_import, division, print_function

import Tac
from TypeFuture import TacLazyType

from EthIntfLib.Types import (
   EthPhyIntfDescPtrDir,
   IntfConfigAggregatedRequests,
)
from FabricIntfLib import (
   Constants,
   IntfConfigCreator,
)

RedundancyMode = TacLazyType( 'Redundancy::RedundancyMode' )

class RedundancyStatusWaiter( Tac.Notifiee ):
   notifierTypeName = 'Redundancy::RedundancyStatus'

   def __init__( self, creatorWarapper, redundancyStatus, *args, **kwargs ):
      super( RedundancyStatusWaiter, self ).__init__( redundancyStatus,
                                                      *args,
                                                      **kwargs )
      self.redundancyStatus_ = redundancyStatus
      self.creatorWarapper_ = creatorWarapper
      self.handleRedundancyMode()

   @Tac.handler( 'mode' )
   def handleRedundancyMode( self ):
      if self.redundancyStatus_.mode == RedundancyMode.active:
         self.creatorWarapper_.startFabricIntfCreator()

class FabricIntfConfigCreatorWrapper( object ):
   def __init__( self, em, mg, agent ):
      self.em_ = em
      self.agent_ = agent
      self.intfConfigDir_ = None
      self.intfConfigRequestRootDir_ = None
      self.intfDescRootDir_ = None
      self.intfDefaultConfigDir_ = None
      self.intfConfigCreator_ = None
      self.redundancyStatus_ = None
      self.redundancyStatusWaiter_ = None
      self.doMounts( mg )

   def doMounts( self, mg ):
      self.intfConfigRequestRootDir_ = mg.mount(
         Constants.intfConfigRequestRootPath(),
         'Tac::Dir',
         'ri' )
      self.intfDescRootDir_ = mg.mount( Constants.intfDescRootPath(),
                                        'Tac::Dir',
                                        'ri' )
      self.intfDefaultConfigDir_ = mg.mount( Constants.intfDefaultConfigPath(),
                                             'Interface::FabricIntfDefaultConfigDir',
                                             'r' )
      self.intfConfigDir_ = mg.mount( Constants.intfConfigPath(),
                                     'Interface::FabricIntfConfigDir',
                                     'w' )
      self.redundancyStatus_ = self.agent_.redundancyStatus()

   def run( self ):
      self.redundancyStatusWaiter_ = RedundancyStatusWaiter( self,
                                                             self.redundancyStatus_ )

   def startFabricIntfCreator( self ):
      if self.intfConfigCreator_:
         return

      self.intfConfigCreator_ = IntfConfigCreator( self.intfConfigRequestRootDir_,
                                                   self.intfDescRootDir_,
                                                   IntfConfigAggregatedRequests(),
                                                   EthPhyIntfDescPtrDir( 'aggr' ),
                                                   self.intfDefaultConfigDir_,
                                                   self.intfConfigDir_ )

def Plugin( context ):
   context.registerStateMachine( FabricIntfConfigCreatorWrapper )
