# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import AirStreamLib

from TypeFuture import TacLazyType
from abc import ABC, abstractmethod
from GenericReactor import GenericReactor
from Toggles.QosOpenConfigToggleLib import (
      toggleOCQosClassifierStatePathEnabled,
      toggleOCQosSchedulerPolicyStatePathEnabled )

t0 = Tracing.trace0

StartupConfigStatus = TacLazyType( 'Sysdb::StartupConfigStatus' )
RedundancyMode = TacLazyType( 'Redundancy::RedundancyMode' )

class StartupConfigStatusSm( Tac.Notifiee ):
   notifierTypeName = 'Sysdb::Status'

   def __init__( self, func, sysdbStatus, *args, **kwargs ):
      super().__init__( sysdbStatus, *args, **kwargs )
      self.func = func
      self.sysdbStatus = sysdbStatus

   @Tac.handler( 'startupConfigStatus' )
   def handleStartupConfigStatus( self ):
      self.func()

class QosHardwareStatusSm( Tac.Notifiee ):
   notifierTypeName = 'Qos::HwStatus'

   def __init__( self, func, qosHwStatus, *args, **kwargs ):
      super().__init__( qosHwStatus, *args, **kwargs )
      self.func = func
      self.qosHwStatus = qosHwStatus

   @Tac.handler( 'hwInitialized' )
   def handleHwInitialized( self ):
      self.func()

class ToExternalSmsWrapperBase( ABC, AirStreamLib.ToExternalSmWrapperBase ):
   def __init__( self, em, mg, agent ):
      self.qosHwStatus = None
      self.qosHwStatusSm = None
      self.sysdbStatus = None
      self.startupConfigStatusSm = None
      super().__init__( em, mg, agent )

   def doMounts( self, mg ):
      self.qosHwStatus = mg.mount( "qos/hardware/status/global",
                                    "Qos::HwStatus", "r" )
      self.sysdbStatus = mg.mount( "Sysdb/status", "Sysdb::Status", "r" )
      super().doMounts( mg )

   @abstractmethod
   def createExternalSm( self ):
      raise NotImplementedError

   def createSm( self ):
      if not self.qosHwStatus.hwInitialized:
         t0( 'creating QosHardwareStatusSm' )
         self.qosHwStatusSm = QosHardwareStatusSm( self.createSm, self.qosHwStatus )
      elif self.sysdbStatus.startupConfigStatus not in (
            StartupConfigStatus.completed, StartupConfigStatus.noScriptSpecified ):
         self.startupConfigStatusSm = StartupConfigStatusSm( self.createSm,
                                                             self.sysdbStatus )
      elif not self.toExternalSm:
         self.createExternalSm()
         if self.qosHwStatusSm or self.startupConfigStatusSm:
            # We deferred creation of toExternalSm, so register it now
            self.registerSm( self.toExternalSm )
            self.qosHwStatus = None
            self.qosHwStatusSm = None
            self.sysdbStatus = None
            self.startupConfigStatusSm = None

class ToExternalForwardingGroupsSmWrapper( ToExternalSmsWrapperBase ):
   nativePathAndType = [ ( 'qos/input/config/cli', 'Qos::Input::Config' ) ]
   externalPathAndType = [ ( 'qos/openconfig/config/forwardingGroups',
                             'QosOc::Config::ForwardingGroupsDir' ),
                           ( 'qos/openconfig/config/queues',
                             'QosOc::Config::QueuesDir' ) ]
   smType = 'QosOc::ForwardingGroupEosToOcSm'

   def createExternalSm( self ):
      t0( 'creating ForwardingGroupEosToOcSm' )
      self.toExternalSm = Tac.newInstance(
         self.smType, self.nativeEntity(),
         self.externalEntity(), self.externalEntity( 1 ) )

class ToExternalQosQueueSmWrapper( ToExternalSmsWrapperBase ):
   nativePathAndType = [ ( 'qos/input/config/cli', 'Qos::Input::Config' ) ]
   externalPathAndType = [ ( 'qos/openconfig/config/queues',
                             'QosOc::Config::QueuesDir' ),
                           ( 'cli/config', 'Cli::Config' ) ]
   smType = 'QosOc::QueueEosToOcSm'

   def createExternalSm( self ):
      t0( 'creating QueueEosToOcSm' )
      self.toExternalSm = Tac.newInstance(
            self.smType, self.nativeEntity(), self.externalEntity(),
            self.externalEntity( 1 ) )

class ToExternalSchedulerPoliciesSmWrapper( ToExternalSmsWrapperBase ):
   nativePathAndType = [ ( 'cli/config', 'Cli::Config' ),
                         ( 'qos/input/config/cli', 'Qos::Input::Config' ),
                         ( 'qos/profile', 'Qos::QosProfileConfigDir' ) ]
   externalPathAndType = [ ( 'qos/openconfig/config/schedulerPolicies',
                             'QosOc::Config::SchedulerPoliciesDir' ),
                           ( 'qos/openconfig/config/interfaces',
                             'QosOc::Config::InterfacesDir' ) ]
   smType = 'QosOc::SchedulerPolicyEosToOCSm'

   def createExternalSm( self ):
      t0( 'creating SchedulerPolicyEosToOCSm' )
      self.toExternalSm = Tac.newInstance( self.smType, self.nativeEntity(),
            self.nativeEntity( 1 ), self.nativeEntity( 2 ), self.externalEntity(),
            self.externalEntity( 1 ) )

class ToExternalClassifierCmapSmWrapper( ToExternalSmsWrapperBase ):
   nativePathAndType = [ ( 'qos/acl/input/cli', 'Qos::Input::AclConfig' ) ]
   externalPathAndType = [ ( 'qos/openconfig/config/classifiers',
                             'QosOc::Config::ClassifiersDir' ) ]
   smType = 'QosOc::ClassMapEosToOcSm'

   def createExternalSm( self ):
      t0( 'creating ClassMapEosToOcSm' )
      self.toExternalSm = Tac.newInstance( self.smType, self.nativeEntity(),
                                           self.externalEntity() )

class ToExternalClassifierCmapActionSmWrapper( ToExternalSmsWrapperBase ):
   nativePathAndType = [ ( 'qos/acl/input/cli', 'Qos::Input::AclConfig' ) ]
   externalPathAndType = [ ( 'qos/openconfig/config/classifiers',
                             'QosOc::Config::ClassifiersDir' ) ]
   smType = 'QosOc::ClassActionEosToOcSm'

   def doMounts( self, mg ):
      self.cliConfig = mg.mount( 'qos/input/config/cli', "Qos::Input::Config", "r" )
      super().doMounts( mg )

   def createExternalSm( self ):
      t0( 'creating ClassMapEosToOcSm' )
      self.toExternalSm = Tac.newInstance(
         self.smType, self.nativeEntity(), self.cliConfig,
         self.externalEntity() )

class ToExternalClassifierAclSmWrapper( ToExternalSmsWrapperBase ):
   nativePathAndType = [ ( 'acl/config/cli', 'Acl::Input::Config' ) ]
   externalPathAndType = [ ( 'qos/openconfig/config/classifiers',
                             'QosOc::Config::ClassifiersDir' ) ]
   smType = 'QosOc::AclEosToOcSm'

   def createExternalSm( self ):
      t0( 'creating AclEosToOcSm' )
      self.toExternalSm = Tac.newInstance( self.smType, self.nativeEntity(),
                                           self.externalEntity() )

class ToExternalQueueManagementProfileSmWrapper( ToExternalSmsWrapperBase ):
   nativePathAndType = [ ( 'qos/input/config/cli', 'Qos::Input::Config' ),
                         ( 'cli/config', 'Cli::Config' ),
                         ( 'qos/profile', 'Qos::QosProfileConfigDir' ) ]
   externalPathAndType = [ ( 'qos/openconfig/config/queueManagementProfiles',
                             'QosOc::Config::QueueManagementProfilesDir' ),
                           ( 'qos/openconfig/config/interfaces',
                             'QosOc::Config::InterfacesDir' ) ]
   smType = 'QosOc::QueueManagementProfileEosToOcSm'

   def createExternalSm( self ):
      t0( 'creating QueueManagementProfileEosToOcSm' )
      self.toExternalSm = Tac.newInstance( self.smType, self.nativeEntity(),
         self.nativeEntity( 1 ), self.nativeEntity( 2 ),
         self.externalEntity(), self.externalEntity( 1 ), self.qosHwStatus )

class ToExternalQosPfcSmWrapper( ToExternalSmsWrapperBase ):
   nativePathAndType = [ ( 'qos/input/config/cli', 'Qos::Input::Config' ) ]
   externalPathAndType = [ ( 'qos/openconfig/config/pfc', 'QosOc::Config::Pfc' ) ]
   smType = 'QosOc::PfcEosToOCSm'

   def createExternalSm( self ):
      t0( 'creating PfcEosToOcSm' )
      self.toExternalSm = Tac.newInstance( self.smType, self.nativeEntity(),
                                           self.externalEntity() )

class ToExternalInterfaceClassifierSmWrapper( ToExternalSmsWrapperBase ):
   nativePathAndType = [ ('qos/input/config/cli', 'Qos::Input::Config' ) ]
   externalPathAndType = [ ( 'qos/openconfig/config/interfaces',
                             'QosOc::Config::InterfacesDir' ) ]
   smType = 'QosOc::InterfaceClassifierEosToOcSm'

   def createExternalSm( self ):
      t0( 'Creating InterfaceClassifierEosToOcSm' )
      self.toExternalSm = Tac.newInstance(
            self.smType, self.nativeEntity(), self.externalEntity() )

class ToExternalQosIntfPfcSmWrapper( ToExternalSmsWrapperBase ):
   nativePathAndType = [ ( 'qos/input/config/cli', 'Qos::Input::Config' ) ]
   externalPathAndType = [ ( 'qos/openconfig/config/interfaces',
                             'QosOc::Config::InterfacesDir' ) ]
   smType = 'QosOc::IntfPfcEosToOCSm'

   def createExternalSm( self ):
      t0( 'creating PfcEosToOcSm' )
      self.toExternalSm = Tac.newInstance( self.smType, self.nativeEntity(),
                                           self.externalEntity() )

class QosOpenConfigStateSms():
   def __init__( self, em, mg, agent ):
      self.qosStatus = None
      self.cliConfig = None
      self.qosCliConfig = None
      self.qosHwStatus = None
      self.queueManagementProfilesDir = None
      self.classifierStateDir = None
      self.interfacesDir = None
      self.agent = agent
      self.sysdbStatus = None

      self.redundancyReactor_ = None
      self.qosHwStatusSm = None
      self.startupConfigStatusSm = None

      self.qmpStateSm = None
      self.spStateSm = None
      self.classifierStateSm = None
      self.intfClassifierStateSm = None

      self.doMounts( mg )

   def doMounts( self, mg ):
      self.qosStatus = mg.mount( 'qos/status', 'Qos::Status', 'r' )
      self.cliConfig = mg.mount( 'cli/config', 'Cli::Config', 'r' )
      self.qosCliConfig = mg.mount( 'qos/input/config/cli',
                                    'Qos::Input::Config', 'r' )
      self.qosAclConfig = mg.mount( 'qos/acl/input/cli',
                                    'Qos::Input::AclConfig', 'r' )
      self.aclCliConfig = mg.mount( 'acl/config/cli', 'Acl::Input::Config', 'r' )
      self.qosProfileConfigDir = mg.mount( 'qos/profile', 'Qos::QosProfileConfigDir',
                                           'r' )
      self.queueManagementProfilesDir = \
         mg.mount( 'qos/openconfig/state/queueManagementProfiles',
                   'QosOc::State::QueueManagementProfilesDir', 'w' )
      self.schedulerPoliciesDir = \
         mg.mount( 'qos/openconfig/state/schedulerPolicies',
                   'QosOc::State::SchedulerPoliciesDir', 'w' )
      self.classifierConfigDir = \
            mg.mount( 'qos/openconfig/config/classifiers',
                      'QosOc::Config::ClassifiersDir', 'r' )
      self.classifierStateDir = \
         mg.mount( 'qos/openconfig/state/classifiers',
                   'QosOc::State::ClassifiersDir', 'w' )
      self.interfacesDir = \
            mg.mount( 'qos/openconfig/state/interfaces',
                      'QosOc::State::InterfacesDir', 'w' )
      self.interfacesDir = mg.mount( 'qos/openconfig/state/interfaces',
                                     'QosOc::State::InterfacesDir', 'w' )
      self.qosHwStatus = mg.mount( "qos/hardware/status/global",
                                    "Qos::HwStatus", "r" )
      self.sysdbStatus = mg.mount( "Sysdb/status", "Sysdb::Status", "r" )

   def createSms( self ):
      if not self.qosHwStatus.hwInitialized:
         t0( 'creating QosHardwareStatusSm' )
         self.qosHwStatusSm = QosHardwareStatusSm( self.createSms, self.qosHwStatus )
      elif self.sysdbStatus.startupConfigStatus not in (
            StartupConfigStatus.completed, StartupConfigStatus.noScriptSpecified ):
         t0( 'creating StartupConfigStatusSm' )
         self.startupConfigStatusSm = StartupConfigStatusSm( self.createSms,
                                                             self.sysdbStatus )
      elif not self.qmpStateSm:
         # create Sms
         t0( 'creating QosOpenConfigStateSms' )
         self.qmpStateSm = Tac.newInstance( 'QosOc::QmpStateSm', self.qosStatus,
                                             self.cliConfig, self.qosCliConfig,
                                             self.qosProfileConfigDir,
                                             self.queueManagementProfilesDir,
                                             self.interfacesDir, self.qosHwStatus )
         if toggleOCQosSchedulerPolicyStatePathEnabled():
            self.spStateSm = Tac.newInstance( 'QosOc::SchedulerPolicyStateSm',
                                                self.qosStatus, self.cliConfig,
                                                self.qosCliConfig,
                                                self.qosProfileConfigDir,
                                                self.schedulerPoliciesDir,
                                                self.interfacesDir )
         if toggleOCQosClassifierStatePathEnabled():
            self.classifierStateSm = Tac.newInstance( 'QosOc::ClassifierStateSm',
                                                      self.qosStatus,
                                                      self.aclCliConfig,
                                                      self.qosCliConfig,
                                                      self.classifierConfigDir,
                                                      self.classifierStateDir )
            self.intfClassifierStateSm = \
                  Tac.newInstance( 'QosOc::IntfClassifierStateSm',
                                   self.qosStatus,
                                   self.interfacesDir )
         self.qosHwStatusSm = None
         self.startupConfigStatusSm = None

   def handleRedundancyMode( self, notifiee=None ):
      if self.agent.redundancyStatus().mode != RedundancyMode.active:
         return

      self.createSms()

   def run( self ):
      self.redundancyReactor_ = GenericReactor(
         self.agent.redundancyStatus(), [ 'mode' ],
         self.handleRedundancyMode, callBackNow=True )

def Plugin( context ):
   context.registerStateMachine( ToExternalQosQueueSmWrapper )
   context.registerStateMachine( ToExternalForwardingGroupsSmWrapper )
   context.registerStateMachine( ToExternalSchedulerPoliciesSmWrapper )
   context.registerStateMachine( ToExternalClassifierAclSmWrapper )
   context.registerStateMachine( ToExternalClassifierCmapSmWrapper )
   context.registerStateMachine( ToExternalClassifierCmapActionSmWrapper )
   context.registerStateMachine( ToExternalQueueManagementProfileSmWrapper )
   context.registerStateMachine( ToExternalQosPfcSmWrapper )
   context.registerStateMachine( ToExternalInterfaceClassifierSmWrapper )
   context.registerStateMachine( ToExternalQosIntfPfcSmWrapper )
   context.registerStateMachine( QosOpenConfigStateSms )
