# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

# Common config session-related code that is used by multiple packages.
import ArPyUtils
import BasicCliSession
import CliCommon
import CliSession
import CliSessionDataStore
import CliSessionOnCommit
import CommonGuards
import FileCliUtil
import GitCliLib
import Logging
import SessionUrlUtil
import Tracing
import Url
import UtmpDump

import os
import sys

t0 = Tracing.t0

guardNoSession = "no active session"

def sessionGuard( mode, token ):
   sessionName = CliSession.currentSession( mode.session_.entityManager_ )
   if sessionName is None:
      return guardNoSession
   return None

def sessionSsoGuard( mode, token ):
   ssoGuardCode = CommonGuards.ssoStandbyGuard( mode, token )
   if ssoGuardCode:
      return ssoGuardCode
   else:
      return sessionGuard( mode, token )

SYS_CONFIG_REPLACE_SUCCESS = Logging.LogHandle(
      "SYS_CONFIG_REPLACE_SUCCESS",
      severity=Logging.logNotice,
      fmt="User %s replaced running configuration with %s successfully on %s (%s)",
      explanation="A network administrator has replaced the running "
                  "configuration of the system successfully.",
      recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_CONFIG_REPLACE_FAILURE = Logging.LogHandle(
      "SYS_CONFIG_REPLACE_FAILURE",
      severity=Logging.logError,
      fmt="Configuration replace operation by %s with %s failed, on %s (%s)",
      explanation="A network administrator has attempted to replace the "
                  "running configuration of the system, but the operation "
                  "failed. The running configuration of the system has not "
                  "changed.",
      recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_CONFIG_SESSION_ENTERED = Logging.LogHandle(
      "SYS_CONFIG_SESSION_ENTERED",
      severity=Logging.logNotice,
      fmt="User %s entered configuration session %s on %s (%s)",
      explanation="A network administrator has entered a configuration "
                  "session.",
      recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_CONFIG_SESSION_EXITED = Logging.LogHandle(
      "SYS_CONFIG_SESSION_EXITED",
      severity=Logging.logNotice,
      fmt="User %s exited configuration session %s on %s (%s)",
      explanation="A network administrator has exited a configuration "
                  "session without committing or aborting, thereby having "
                  "no effect on the running configuration fo the system.",
      recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_CONFIG_SESSION_COMMIT_SUCCESS = Logging.LogHandle(
      "SYS_CONFIG_SESSION_COMMIT_SUCCESS",
      severity=Logging.logNotice,
      fmt="User %s committed configuration session %s%s successfully on %s (%s)",
      explanation="A network administrator has commited a configuration "
                  "session, thereby modifying the running configuration of"
                  " the system.",
      recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_CONFIG_SESSION_COMMIT_TIMER_STARTED = Logging.LogHandle(
      "SYS_CONFIG_SESSION_COMMIT_TIMER_STARTED",
      severity=Logging.logNotice,
      fmt="User %s committed session %s on %s (%s), "
          "with timer %d:%d:%d(hr:min:sec).",
      explanation="A network administrator has committed a"
                  "configuration session with a timer, thereby"
                  " modifying the running configuration of the "
                  "system temporarily.",
      recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_CONFIG_SESSION_COMMIT_TIMER_UPDATED = Logging.LogHandle(
      "SYS_CONFIG_SESSION_COMMIT_TIMER_UPDATED",
      severity=Logging.logNotice,
      fmt="User %s updated commit timer with %d:%d:%d"
          "(hr:min:sec) for session %s on %s (%s).",
      explanation="A network administrator has updated the commit"
                   " timer of a configuration session.",
      recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_CONFIG_SESSION_COMMIT_TIMER_COMPLETED = Logging.LogHandle(
      "SYS_CONFIG_SESSION_COMMIT_TIMER_COMPLETED",
      severity=Logging.logNotice,
      fmt="User %s has confirmed the session %s on %s (%s).",
      explanation="A network administrator has confirmed the"
                  "configuration session.",
      recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_CONFIG_SESSION_COMMIT_TIMER_CANCEL_ERROR = Logging.LogHandle(
   "SYS_CONFIG_SESSION_COMMIT_TIMER_CANCEL_ERROR",
   severity=Logging.logError,
   fmt="Failed to cancel commit timer for configuration session (%s)",
   explanation=( "The system configuration failed to be rolled back to the state "
                 "before the configuration session was committed with a timer. "
                 "It could happen for various reasons, such as configure lock "
                 "is being held by another CLI session."
                ),
   recommendedAction=( "Fix the error and retry by running the command "
                       "'configure session pending-timer abort'" )
   )

SYS_CONFIG_SESSION_COMMIT_FAILURE = Logging.LogHandle(
      "SYS_CONFIG_SESSION_COMMIT_FAILURE",
      severity=Logging.logError,
      fmt="User %s committed configuration session %s with errors on %s (%s)",
      explanation="A network administrator has committed a configuration "
                  "session, but the operation failed. The system may be "
                  " in an inconsistent state.",
      recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_CONFIG_SESSION_ABORTED = Logging.LogHandle(
      "SYS_CONFIG_SESSION_ABORTED",
      severity=Logging.logNotice,
      fmt="User %s aborted configuration session %s on %s (%s)",
      explanation="A network administrator has aborted a configuration "
                  "session, thereby having no effect on the running "
                  "configuration of the system.",
      recommendedAction=Logging.NO_ACTION_REQUIRED )

SYS_CONFIG_SESSION_DELETED = Logging.LogHandle(
      "SYS_CONFIG_SESSION_DELETED",
      severity=Logging.logNotice,
      fmt="User %s deleted configuration session %s on %s (%s)",
      explanation="A network administrator has deleted a configuration "
                  "session, thereby having no effect on the running "
                  "configuration of the system.",
      recommendedAction=Logging.NO_ACTION_REQUIRED )

def doLog( mode, slog, target, time=None, description='', loggingTerminal=False ):
   info = UtmpDump.getUserInfo()
   if slog == SYS_CONFIG_SESSION_COMMIT_TIMER_STARTED:
      logParams = ( slog, info[ 'user' ], target, info[ 'tty' ],
                    info[ 'ipAddr' ] ) + time
   elif slog == SYS_CONFIG_SESSION_COMMIT_TIMER_UPDATED:
      logParams = ( slog, info[ 'user' ] ) + time + \
                  ( target, info[ 'tty' ], info[ 'ipAddr' ] )
   elif slog == SYS_CONFIG_SESSION_COMMIT_SUCCESS:
      if description:
         description = f' with description {description}'
      logParams = ( slog, info[ 'user' ], target, description, info[ 'tty' ],
                    info[ 'ipAddr' ] )
   else:
      logParams = ( slog, info[ 'user' ], target, info[ 'tty' ],
                    info[ 'ipAddr' ] )

   if loggingTerminal:
      mode.addMessage( slog.logMsg.format % logParams[ 1 : ] )
   Logging.log( *logParams )

def getComponentName( url ):
   return os.path.splitext( os.path.basename( url ) )[ 0 ]

# Sets up a temporary config session for config replace.
# Create a temporary config session, rollback to clean config, and apply @surl to
# the session.
# Returns a tuple of the temporary config session name and any error message, so that
# the session can be committed later.
#
# Note we create an unreferenced automatic session, meaning that the session is
# directly deleted as soon as it's committed (or aborted).
def configReplaceSession( mode, surl, ignoreErrors=False, sessionName="",
                          md5=None, replace=True, noCommit=False,
                          ignoreENOENT=False ):
   em = mode.entityManager
   CliSession.exitSession( em )

   target = "cfg"
   if surl:
      target = getComponentName( str( surl ) )
   elif sessionName:
      target = sessionName

   targetSessionName = CliSession.uniqueSessionName( em, prefix="cfg" )
   response = CliSession.enterSession( targetSessionName, em, transient=True,
                                       noCommit=noCommit )
   if response:
      return targetSessionName, response

   try:
      configSet = CliSession.sessionConfig.pendingChangeConfigDir.config
      pcc = configSet.get( targetSessionName )
      # add a description to userString if Cli provides one. TODO
      pcc.userString = "config {} of {}".format( "replace" if replace else "copy",
                                                 target )
      if sessionName:
         response = CliSession.copyFromSession( em, sessionName )
         if response:
            CliSession.abortSession( em )
            return targetSessionName, response
      else:
         if replace:
            # rollback from clean-config
            response = CliSession.rollbackSession( em )
            if response:
               CliSession.abortSession( em )
               return targetSessionName, response
         # load file:
         #
         # Disable AAA for config replace (there is no benefit doing it since
         # we just did rollback)
         sessionConfig = SessionUrlUtil.sessionConfig(
            mode.entityManager, True, mode.session,
            sessionName=targetSessionName )
         sessionConfig.abortOnError = not ignoreErrors
         # Since we are replacing with a new config, no need to validate
         # the new one.
         sessionConfig.skipConfigCheck = True
         sessionConfig.md5 = md5
         error = ""
         try:
            t0( 'Loading config from', surl, 'into session' )
            error = FileCliUtil.copyFile( None, mode, surl, sessionConfig,
                                          commandSource="configure replace",
                                          ignoreENOENT=ignoreENOENT )
         except CliCommon.MD5Error as e: # pylint: disable-msg=E1101
            error = e.msg

         if error != "":
            t0( 'Loading config resulted in error:', error )
            CliSession.abortSession( em )
            CliSession.exitSession( em )
            return targetSessionName, error
   except KeyboardInterrupt:
      CliSession.abortSession( em )
      return targetSessionName, "Keyboard Interrupt"
   else:
      return targetSessionName, ""
   finally:
      # exit the session so it's unreferenced.
      CliSession.exitSession( em )

def commitTimerRollback( mode ):
   # rollback config session that is pending timer
   # this is called inline from abortSession() and deleteSession() inside the
   # same CLI session instead of spawning a CliShell process (as in timeout case).
   session = mode.session
   url = CliSession.sessionStatus.preCommitConfig
   if not url:
      return

   with BasicCliSession.AaaDisabler( session ):
      shouldPrint = session.shouldPrint()

      def _write( msg ):
         if shouldPrint:
            sys.stdout.write( msg )
            sys.stdout.flush()

      _write( "Commit timer cancelled, rolling back configuration..." )

      try:
         with ArPyUtils.StdoutInterceptor():
            session.runCmd( f"configure replace {url} ignore-errors",
                            keepErrors=True )
      except Exception as e: # pylint: disable=broad-except
         _write( "error\n" )
         msg = str( e )
         Logging.log( SYS_CONFIG_SESSION_COMMIT_TIMER_CANCEL_ERROR, msg )
         mode.addErrorAndStop( f"Cannot restore saved config: {msg}" )
      else:
         _write( "done\n" )

def abortSession( mode, sessionName ):
   # This aborts a config session. If the session is pending comit timer,
   # revert to previous config as well.
   t0( 'Aborting Session', sessionName, 'in mode', mode )
   em = mode.session.entityManager
   if not sessionName:
      sessionName = CliSession.currentSession( em )

   if ( sessionName ==
      CliSession.sessionStatus.sessionStateDir.commitTimerSessionName ):
      commitTimerRollback( mode )
   response = CliSession.abortSession( em, sessionName )
   if response:
      mode.addError( response )
   return response

# Invoke on-commit handlers on session. If there is an error, rollback to checkpoint
# and invoke the on-commit handlers again on the rolled back config.
# Returns
#  "OK": if on-commit handlers did not complain
#  "some err msg" to display to user if an on-commit handler complained
# Any error display cannot be done now and has to be delayed to ensure it is not
# buried after a forest of warning during the rollback...
# There are 3 error messages/cases:
#   the commit failed and we rolled back
#   the commit failed and the rollback failed too, because either
#      the checkpoint application failed,
#      the checkpoint was applied but its commit failed

def invokeOnCommitHandlersOrRevert( mode, sessionName, changeId, loggingTerminal ):
   em = mode.entityManager

   # We just committed the session. Now go on to commit registered
   # data store(s) and invoke on commit handlers. If there is an
   # error, revert to saved checkpoints.
   success = True
   t0( 'invokeOnCommitHandlersOrRevert: change ID seen', changeId )
   if changeId:
      # Since we have a checkpoint of the main session to revert to,
      # also save a checkpoint of any registered data stores
      success = CliSessionDataStore.backup( mode, sessionName )

   if success:
      success = ( CliSessionDataStore.commit( mode, sessionName ) and
                  CliSessionOnCommit.invokeSessionOnCommitHandlers( mode,
                                                                    sessionName ) )

   errorPrefix = "ERROR: session commit failed"
   if success:
      ret = None
   elif CliSession.isSessionPendingCommitTimer( sessionName ):
      mode.addError( "Error during session commit, starting rollback" )
      # abortSession() will trigger rollback
      ret = abortSession( mode, sessionName )
      if ret:
         ret = "%s, rollback failed too" % errorPrefix
      else:
         ret = "%s, config rolled back" % errorPrefix
   else:
      checkpointUrl = None
      if changeId:
         commit = GitCliLib.getCommit( mode.entityManager.sysname(), changeId )
         if commit:
            t0( 'commit hash', commit, 'for change ID', changeId )
            context = Url.Context( *Url.urlArgsFromMode( mode ) )
            checkpointUrl = Url.parseUrl( f'checkpoint:/{commit["commitHash"]}',
                  context )
         else:
            t0( 'commit hash not found for change ID', changeId )
      else:
         t0( 'no change ID seen' )
      if not checkpointUrl:
         ret = "%s, no checkpoint to rollback to" % errorPrefix
      else:
         mode.addError( "Error during session commit, starting rollback" )
         # revert data store, which cannot fail
         CliSessionDataStore.restore( mode, sessionName )
         # Revert back to checkpointUrl if there is an error from the onCommit
         # handlers
         rollbackSessName, error = configReplaceSession( mode, checkpointUrl )
         if not error:
            rollbackResponse = CliSession.commitSession( em,
                                             sessionName=rollbackSessName )
            if rollbackResponse:
               ret = "%s, commit of rollback failed too" % errorPrefix
               doLog( mode, SYS_CONFIG_REPLACE_FAILURE, checkpointUrl,
                     loggingTerminal=loggingTerminal )
            else:
               # Invoke the commit handlers again for the rolled back config
               CliSessionOnCommit.invokeSessionOnCommitHandlers( mode,
                  sessionName )
               doLog( mode, SYS_CONFIG_REPLACE_SUCCESS, checkpointUrl )
               ret = "%s, config rolled back" % errorPrefix
         else:
            ret = "%s, rollback failed too" % errorPrefix
            doLog( mode, SYS_CONFIG_REPLACE_FAILURE, checkpointUrl,
                  loggingTerminal=loggingTerminal )
   CliSession.discardSessionOnCommitHandlers( sessionName )
   return ret
