# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import CliSession
import GnmiSetCliSession
import AirStreamLib
import Tracing
import Tac
import struct
from Toggles.StpToggleLib import toggleStpOpenConfigEnabled
from StpCliUtil import parseVidToMstiMap

# pkgdeps: rpm Stp-lib

t0 = Tracing.Handle( "ConfigSessionStp" ).trace0
t9 = Tracing.Handle( "ConfigSessionStp" ).trace9

def Plugin( entMan ):
   CliSession.registerConfigGroup( entMan, "airstream-cmv",
         "stp/openconfig/globalConfig" )
   CliSession.registerConfigGroup( entMan, "airstream-cmv",
         "stp/openconfig/interfaceConfig" )
   CliSession.registerConfigGroup( entMan, "airstream-cmv",
         "stp/openconfig/mstiConfig" )

   def toNativeStpGlobalSyncher( cls, sessionName ):
      externalConfig = AirStreamLib.getSessionEntity( entMan, sessionName,
            "stp/openconfig/globalConfig" )
      nativeConfig = AirStreamLib.getSessionEntity( entMan, sessionName,
            "stp/input/config/cli" )

      t0( "Synching from " + str( externalConfig ) + " to " + str( nativeConfig ) )
      enabledProtocol = externalConfig.config.enabledProtocol

      # sync enabled-protocol/mode
      if not enabledProtocol:
         nativeConfig.forceProtocolVersion = "none"
      elif len( enabledProtocol ) == 1:
         if enabledProtocol[ 0 ] == "MSTP":
            nativeConfig.forceProtocolVersion = "mstp"
         elif enabledProtocol[ 0 ] == "RSTP":
            nativeConfig.forceProtocolVersion = "rstp"
         elif enabledProtocol[ 0 ] == "RAPID_PVST":
            nativeConfig.forceProtocolVersion = "rapidPvstp"
         else:
            raise AirStreamLib.ToNativeSyncherError( sessionName,
                  cls.__name__ + "::toNativeStpGlobalSyncher",
                  "Unknown spanning tree mode" )
      else:
         raise AirStreamLib.ToNativeSyncherError( sessionName,
               cls.__name__ + "::toNativeStpGlobalSyncher",
               "Multiple spanning tree modes not supported" )

      # sync bpdu-guard/edge-port bpduguard default
      nativeConfig.portfastBpduguard = externalConfig.config.bpduGuard

      # sync [no] spanning-tree vlan-id
      syncher = Tac.newInstance( "Stp::OCGlobalVlanSyncher",
                                 externalConfig, nativeConfig )
      syncher.syncDisabledVlan()

   def toNativeStpInterfaceSyncher( cls, sessionName ):
      externalIntfs = AirStreamLib.getSessionEntity( entMan, sessionName,
            "stp/openconfig/interfaceConfig" )
      nativeConfig = AirStreamLib.getSessionEntity( entMan, sessionName,
            "stp/input/config/cli" )

      t0( "Synching from " + str( externalIntfs ) + " to " +
            str( nativeConfig ) )
      for name, value in externalIntfs.interface.items():
         try:
            if name not in nativeConfig.portConfig:
               t9( "Config for interface " + name + " doesn't exist. Create it" )
               nativeConfig.newPortConfig( name )
            t9( "Copying interface config " + name )
            nativePortConfig = nativeConfig.portConfig[ name ]

            # sync edge-port/portfast command
            if value.config.edgePort == "EDGE_AUTO":
               nativePortConfig.autoEdgePort = True
               nativePortConfig.adminEdgePort = False
               nativePortConfig.networkPort = False
            elif value.config.edgePort == "EDGE_ENABLE":
               nativePortConfig.autoEdgePort = False
               nativePortConfig.adminEdgePort = True
               nativePortConfig.networkPort = False
            elif value.config.edgePort == "EDGE_DISABLE":
               nativePortConfig.autoEdgePort = False
               nativePortConfig.adminEdgePort = False
               nativePortConfig.networkPort = True
            else: # edgePort is empty string
               nativePortConfig.autoEdgePort = False
               nativePortConfig.adminEdgePort = False
               nativePortConfig.networkPort = False

            # sync bpdu-guard/bpduguard (enable|disable) command
            if value.config.bpduGuard is True:
               nativePortConfig.bpduguard = "bpduguardEnabled"
            elif value.config.bpduGuard is False:
               nativePortConfig.bpduguard = "bpduguardDisabled"
            else:  # bpduGuard is None
               nativePortConfig.bpduguard = "bpduguardDefault"

            if value.config.linkType == "P2P":
               nativePortConfig.adminPointToPointMac = "forceTrue"
            elif value.config.linkType == "SHARED":
               nativePortConfig.adminPointToPointMac = "forceFalse"
            else:
               nativePortConfig.adminPointToPointMac = \
                  nativePortConfig.defaultAdminPointToPoint

            # In yang, "name" is a leafref to "config/name";
            # in Tac, the two need to be synced manually, so this block does that.
            # The Nominal config gets instantiated with config.name == '' by
            # default, so replace value.config with a name config with the same
            # attributes but value.config.name = value.name
            config = Tac.newInstance( "Stp::InterfaceConfig" )
            config.name = name
            config.linkType = value.config.linkType
            config.bpduGuard = value.config.bpduGuard
            config.edgePort = value.config.edgePort
            value.config = config

         except Exception as e: # pylint: disable-msg=broad-except
            raise AirStreamLib.ToNativeSyncherError( sessionName,
                  cls.__name__ + '::toNativeStpInterfaceSyncher', str( e ) )

   def toNativeMstSyncher( cls, sessionName ):
      externalMstInstances = AirStreamLib.getSessionEntity( entMan, sessionName,
            "stp/openconfig/mstiConfig" )
      nativeConfig = AirStreamLib.getSessionEntity( entMan, sessionName,
            "stp/input/config/cli" )

      if "Mst" not in nativeConfig.stpiConfig:
         nativeConfig.newStpiConfig( "Mst" )
      mst = nativeConfig.stpiConfig[ "Mst" ]

      t0( "Synching from " + str( externalMstInstances ) + " to " +
            str( nativeConfig ) )

      # get the current vlan to msti mapping
      vidToMstiDict = parseVidToMstiMap( nativeConfig.mstConfigSpec.vidToMstiMap )

      # new vlan to msti map, used for comparison against current
      newVidToMstiDict = {}

      # this map omits vlans mapped to msti 0 (they are considered mapped to msti 0
      # implicitly)
      # add explicit mapping to 0
      for v in range( 1, 4095 ):
         if v not in vidToMstiDict:
            vidToMstiDict[ v ] = 0

      for mstId, extMstInst in externalMstInstances.mstInstance.items():
         # 1. sync bridge priority
         try:
            name = "Cist" if mstId == 0 else f"Mst{mstId}"

            if ( extMstInst.config.bridgePriority % 4096 ) != 0:
               raise AirStreamLib.ToNativeSyncherError( sessionName,
                     cls.__name__ + "::toNativeMstSyncher",
                     "\nBridge Priority must be in increments of 4096. " +
                     "Allowed values are:\n0     4096  8192  12288 16384 20480 " +
                     "24576 28672\n32768 36864 40960 45056 49152 53248 57344 61440" )

            if name not in mst.mstiConfig:
               t9( "Config for Mst instance " + name + " doesn't exist. Create it" )
               mst.newMstiConfig( name, mstId )
            mst.mstiConfig[ name ].bridgePriority = extMstInst.config.bridgePriority

            # In yang, "mst-id" is a leafref to "config/mst-id";
            # in Tac, the two need to be synced manually, so this block does that.
            # The Nominal config gets instantiated with config.mstId == 0 by
            # default, so replace extMstInst.config with a config with the same
            # attributes but extMstInst.config.mstId = extMstInst.mstId
            config = Tac.newInstance( "Stp::OCMstiConfig" )
            config.mstId = mstId
            config.bridgePriority = extMstInst.config.bridgePriority

            for vlanIdOrRange in extMstInst.config.vlan.values():
               config.vlan.push( vlanIdOrRange )

            extMstInst.config = config

         except Exception as e: # pylint: disable-msg=broad-except
            raise AirStreamLib.ToNativeSyncherError( sessionName,
                  cls.__name__ + '::toNativeMstSyncher', str( e ) )

         # 2. store information about the msti to vlan mapping, to be applied after
         # iterating through all external mst instances
         for vidOrRange in extMstInst.config.vlan.values():
            if vidOrRange.containsSingle():
               # if vid not in newVidToMstiDict yet, add an empty list for it, then
               # populate it with this mstId
               newVidToMstiDict.setdefault( vidOrRange.single, [] ).append( mstId )
            else:
               # vidOrRange is a range
               lowVid = vidOrRange.range.lowVid
               highVid = vidOrRange.range.highVid

               for vid in range( lowVid, highVid + 1 ):
                  newVidToMstiDict.setdefault( vid, [] ).append( mstId )

      newVidToMstiMapStr = b""
      for vlan in range( 1, 4095 ):
         msti = newVidToMstiDict.get( vlan, [ 0 ] )

         # if more than 1 msti, the vlan is being moved from one msti to another;
         # remove the msti from the current active mapping and only keep the new one
         if len( msti ) > 1:
            msti.remove( vidToMstiDict[ vlan ] )

         # bad set request mapping the same vlan to multiple mst instances at once
         if len( msti ) > 1:
            raise AirStreamLib.ToNativeSyncherError( sessionName,
                  cls.__name__ + "::toNativeMstSyncher",
                  f"VLAN {vlan} mapped to multiple Mst instances {msti[0]},"
                  f"{msti[1]}" )

         # only add msti mapping if vlan mapped to non-default msti
         if msti[ 0 ] != 0:
            newVidToMstiMapStr += struct.pack( "!HH", vlan, msti[ 0 ] )

      # commit to native mstConfigSpec
      mstConfigSpec = nativeConfig.mstConfigSpec
      newMstConfigSpec = Tac.Value( "Stp::MstConfigSpec",
                                    regionId=mstConfigSpec.regionId,
                                    configRevision=mstConfigSpec.configRevision,
                                    vidToMstiMap=newVidToMstiMapStr )
      try:
         nativeConfig.mstConfigSpec = newMstConfigSpec
      except Exception as e: # pylint: disable-msg=broad-except
         raise AirStreamLib.ToNativeSyncherError( sessionName,
               cls.__name__ + '::toNativeStpMstiSyncher', str( e ) )

      mstiToVlanMap = {}
      for vlan in range( 1, 4095 ):
         mstId = newVidToMstiDict.get( vlan, [ 0 ] )[ 0 ]

         if mstId not in mstiToVlanMap:
            mstiToVlanMap[ mstId ] = [ vlan ]
            continue

         last = mstiToVlanMap[ mstId ][ -1 ]

         if isinstance( last, int ):
            lowVid, highVid = last, last
         else:
            lowVid, highVid = map( int, last.split( ".." ) )

         if vlan - highVid == 1:
            mstiToVlanMap[ mstId ].pop()
            mstiToVlanMap[ mstId ].append( f"{lowVid}..{vlan}" )
         else:
            mstiToVlanMap[ mstId ].append( vlan )

      # use local map to update the vlan collection in external mst instances
      for mstId, vlanList in mstiToVlanMap.items():
         if mstId not in externalMstInstances.mstInstance:
            externalMstInstances.newMstInstance( mstId )

         extMstInst = externalMstInstances.mstInstance[ mstId ]

         # create new config container with same attributes but updated vlan
         config = Tac.Value( "Stp::OCMstiConfig" )
         config.mstId = extMstInst.config.mstId
         config.bridgePriority = extMstInst.config.bridgePriority
         for vlanIdOrRange in vlanList:
            if isinstance( vlanIdOrRange, int ):
               config.vlan.push( Tac.Value( "Stp::VlanIdOrRange",
                                            single=vlanIdOrRange ) )
            else:
               lowVid, highVid = map( int, vlanIdOrRange.split( ".." ) )
               config.vlan.push( Tac.Value( "Stp::VlanIdOrRange",
                  range=Tac.Value( "Bridging::VlanIdRange", lowVid=lowVid,
                                                            highVid=highVid ) ) )

         extMstInst.config = config

   # Register one-time-sync callback to be called before commit
   class ToNativeStpSyncher( GnmiSetCliSession.PreCommitHandler ):
      externalPathList = [ "stp/openconfig/globalConfig",
                           "stp/openconfig/interfaceConfig",
                           "stp/openconfig/mstiConfig" ]
      nativePathList = [ "stp/input/config/cli" ]

      @classmethod
      def run( cls, sessionName ):
         toNativeStpGlobalSyncher( cls, sessionName )
         toNativeStpInterfaceSyncher( cls, sessionName )
         toNativeMstSyncher( cls, sessionName )

   if toggleStpOpenConfigEnabled():
      GnmiSetCliSession.registerPreCommitHandler( ToNativeStpSyncher )
      AirStreamLib.registerCopyHandler( entMan, "StpGlobal",
                                        typeName="Stp::OCGlobalConfig" )
      AirStreamLib.registerCopyHandler( entMan, "StpInterface",
                                        typeName="Stp::OCInterfaceConfigDir" )
      AirStreamLib.registerCopyHandler( entMan, "StpMsti",
                                        typeName="Stp::OCMstiConfigDir" )
