# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from Tracing import defaultTraceHandle
from CliPlugin.TrafficPolicyCliLib import ( ClassPriorityConstant,
                                            ReservedClassMapNames,
                                            matchIpAccessGroup,
                                            matchIpv6AccessGroup,
                                            matchMacAccessGroup )
from Toggles.TrafficPolicyToggleLib import (
      toggleTrafficPolicyAegisOpenConfigEnabled,
      toggleTrafficPolicyDefaultMacMatchEnabled )
import Tac
import AirStreamLib
import CliSession
import GnmiSetCliSession

th = defaultTraceHandle()
t0 = th.t0
t2 = th.t2

EXT_PATH = 'acl/openconfig/aclSets'
NATIVE_PATH = 'trafficPolicies/input/cli'
ipv4 = Tac.Type( "Arnet::AddressFamily" ).ipv4
ipv6 = Tac.Type( "Arnet::AddressFamily" ).ipv6
PRIO_STEP = ClassPriorityConstant.classPriorityInc
MAX_PRIO = ClassPriorityConstant.classPriorityMax
MAX_RULES_SUPPORTED = MAX_PRIO // PRIO_STEP

def Plugin( entMan ):
   if toggleTrafficPolicyAegisOpenConfigEnabled():
      CliSession.registerConfigGroup( entMan, "airstream-cmv", EXT_PATH )
      ocAclTypeToYangStr = {
         'aclL2': 'ACL_L2',
         'aclIpv4': 'ACL_IPV4',
         'aclIpv6': 'ACL_IPV6',
         'aclMpls': 'ACL_MPLS',
         'aclMixed': 'ACL_MIXED'
      }

      class OCAclSm( GnmiSetCliSession.GnmiSetCliSessionSm ):
         entityPaths = [ EXT_PATH, NATIVE_PATH ]
         sm = None

         def run( self ):
            t0( "Instantiating TrafficPolicyToNativeSm" )
            extEnt = self.sessionEntities[ EXT_PATH ]
            smType = 'TrafficPolicy::OpenConfig::OCTrafficPolicyUpdateSm'
            self.sm = Tac.newInstance( smType, extEnt )

      class OcAclPreCommitHandler( GnmiSetCliSession.PreCommitHandler ):
         externalPathList = [ EXT_PATH ]
         nativePathList = [ NATIVE_PATH ]
         UniqueId = Tac.Type( "Ark::UniqueId" )
         entityManager = entMan
         sessionName = None
         defaultPrios = []

         @classmethod
         def createMatchRule( cls, pmapSubCfg, cmapName, sequenceId, prio,
                              matchOption ):
            t2( f'Creating match rule policyName {pmapSubCfg.policyName} '
                f'prio {prio} cmapName {cmapName}' )
            if cmapName in pmapSubCfg.rawClassMap:
               msg = ( f'acl-set:{pmapSubCfg.policyName}; acl-entry:{sequenceId}; '
                       f'an entry may configure one of the L2, IPv4, or IPv6 '
                        'containers' )
               raise AirStreamLib.ToNativeSyncherError(
                        sessionName=cls.sessionName,
                        syncherName=cls.__name__, userMsg=msg )

            cmap = pmapSubCfg.rawClassMap.newMember( cmapName, cls.UniqueId() )
            pmapSubCfg.classPrio[ prio ] = cmapName
            pmapSubCfg.classAction.newMember( cmapName )
            cmapMatch = cmap.match.newMember( matchOption )
            cmapMatch.structuredFilter = ( "", )

         @classmethod
         def createPmapSubCfg( cls, policyName, pmapCfg ):
            t2( f'Creating new subconfig for policy {policyName}' )
            pmapSubCfg = pmapCfg.subconfig.newMember( policyName, cls.UniqueId() )
            # create default rules
            cls.createMatchRule( pmapSubCfg, ReservedClassMapNames.classV4Default,
                                 ClassPriorityConstant.classV4DefaultPriority,
                                 ClassPriorityConstant.classV4DefaultPriority,
                                 matchIpAccessGroup )
            cls.createMatchRule( pmapSubCfg, ReservedClassMapNames.classV6Default,
                                 ClassPriorityConstant.classV6DefaultPriority,
                                 ClassPriorityConstant.classV6DefaultPriority,
                                 matchIpv6AccessGroup )
            if toggleTrafficPolicyDefaultMacMatchEnabled():
               cls.createMatchRule( pmapSubCfg,
                                    ReservedClassMapNames.classMacDefault,
                                    ClassPriorityConstant.classMacDefaultPriority,
                                    ClassPriorityConstant.classMacDefaultPriority,
                                    matchMacAccessGroup )
            return pmapSubCfg

         @classmethod
         def getPmapCfg( cls, policyName, native ):
            pmaps = native.pmapType.pmap
            pmapCfg = pmaps.get( policyName )
            # if new policy, create sub config
            if not pmapCfg:
               t2( f'Creating Pmap input for policy {policyName}' )
               pmapCfg = native.pmapType.newPmapInput( policyName,
                                                       native.pmapType.type )
               pmaps.addMember( pmapCfg )
            return pmapCfg

         @classmethod
         def syncl2( cls, pmapSubCfg, oldPmapCfg, ocAclType, prio, cmapName,
                     config ):
            return False

         @classmethod
         def syncIp( cls, pmapSubCfg, oldPmapCfg, ipType, ocAclType, cmapName, prio,
                     ipConfig, icmpConfig ):
            return False

         @classmethod
         def syncTransport( cls, pmapSubCfg, oldPmapCfg, ocAclType, cmapName, prio,
                            config ):
            return False

         @classmethod
         def syncActions( cls, native, pmapSubCfg, oldPmapCfg, policyName, cmapName,
                          prio, config ):
            return False

         @classmethod
         def run( cls, sessionName ):
            t0( 'Running Pre-commit handler' )
            cls.sessionName = sessionName
            cls.defaultPrios = [ ClassPriorityConstant.classV4DefaultPriority,
                                 ClassPriorityConstant.classV6DefaultPriority ]
            if toggleTrafficPolicyDefaultMacMatchEnabled():
               cls.defaultPrios.append(
                       ClassPriorityConstant.classMacDefaultPriority )

            external = AirStreamLib.getSessionEntity( cls.entityManager,
                                                      sessionName,
                                                      EXT_PATH )
            native = AirStreamLib.getSessionEntity( cls.entityManager,
                                                    sessionName,
                                                    NATIVE_PATH )
            numMaxEntries = MAX_RULES_SUPPORTED - len( cls.defaultPrios )
            for ocAclKey in external.policiesToSync:
               policyName = ocAclKey.name
               ocAclType = ocAclKey.type
               t0( f'syncing policy {policyName} type {ocAclType}' )
               if ocAclType != 'aclMixed':
                  msg = ( f'acl-set:{policyName}; type '
                          f'{ocAclTypeToYangStr[ ocAclType ]} is not supported' )
                  raise AirStreamLib.ToNativeSyncherError(
                                sessionName=cls.sessionName,
                                syncherName=cls.__name__, userMsg=msg )
               if ocAclKey not in external.aclSet:
                  t2( f'policy {policyName} type {ocAclType} was deleted' )
                  del native.pmapType.pmap[ policyName ]
                  continue

               aclEntries = sorted( external.aclSet[ ocAclKey ].aclEntries.aclEntry )
               numAclEntries = len( aclEntries )
               t2( f'numAclEntries: {numAclEntries} numMaxEntries: {numMaxEntries}' )
               if numAclEntries > numMaxEntries:
                  msg = ( f'acl-set:{policyName}; number of acl-entries '
                          f'{numAclEntries} greater than max supported '
                          f'({numMaxEntries})' )
                  raise AirStreamLib.ToNativeSyncherError(
                                sessionName=cls.sessionName,
                                syncherName=cls.__name__, userMsg=msg )
               # if policy is new, we do want to update
               isUpdated = policyName not in native.pmapType.pmap
               pmapCfg = cls.getPmapCfg( policyName, native )
               oldPmapCfg = pmapCfg.currCfg
               # create new sub config
               pmapSubCfg = cls.createPmapSubCfg( policyName, pmapCfg )
               for i, sequenceId in enumerate( aclEntries, start=1 ):
                  aclEntry = external.aclSet[ ocAclKey ].aclEntries.\
                          aclEntry[ sequenceId ]
                  cmapName = str( sequenceId )
                  prio = i * PRIO_STEP
                  assert prio not in cls.defaultPrios
                  isUpdated |= cls.syncl2( pmapSubCfg, oldPmapCfg, ocAclType,
                                           cmapName, prio, aclEntry.l2.config )
                  isUpdated |= cls.syncIp( pmapSubCfg, oldPmapCfg, ipv4, ocAclType,
                                           cmapName, prio, aclEntry.ipv4.config,
                                           aclEntry.ipv4.icmpv4.config )
                  isUpdated |= cls.syncIp( pmapSubCfg, oldPmapCfg, ipv6, ocAclType,
                                           cmapName, prio, aclEntry.ipv6.config,
                                           aclEntry.ipv6.icmpv6.config )
                  isUpdated |= cls.syncTransport( pmapSubCfg, oldPmapCfg, ocAclType,
                                                  cmapName, prio,
                                                  aclEntry.transport.config )
                  isUpdated |= cls.syncActions( native, pmapSubCfg, oldPmapCfg,
                                                policyName, cmapName, prio,
                                                aclEntry.actions.config )
               if isUpdated:
                  pmapCfg.currCfg = pmapSubCfg
                  if oldPmapCfg:
                     del pmapCfg.subconfig[ oldPmapCfg.pmapVersion ]
               else:
                  del pmapCfg.subconfig[ pmapSubCfg.pmapVersion ]
            external.policiesToSync.clear()

      t0( 'registering OCAclSm and OcAclPreCommitHandler' )
      GnmiSetCliSession.registerPreCommitSm( OCAclSm )
      GnmiSetCliSession.registerPreCommitHandler( OcAclPreCommitHandler )

      # Register special copy handler for external entities
      AirStreamLib.registerCopyHandler( entMan, "acl",
                         typeName="Acl::OpenConfig::OCAclSets" )
