# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Logging
import QuickTrace
import Tac
import Tracing
import Url

traceHandle = Tracing.Handle( "ContainerMgrImageLoadSm" )

t0 = traceHandle.trace0 # function calls
t1 = traceHandle.trace1 # error/exception
t2 = traceHandle.trace2 # login/logout from a registry
t3 = traceHandle.trace3 # container info traces

qv = QuickTrace.Var
qt0 = QuickTrace.trace0 # Important function calls
qt1 = QuickTrace.trace1 # error/exception
qt2 = QuickTrace.trace2 # Other important info

CONTAINERMGR_IMAGE_LOAD_FAILED = None

CONTAINERMGR_IMAGE_LOAD_FAILED = Logging.LogHandle(
      "CONTAINERMGR_IMAGE_LOAD_FAILED",
      severity=Logging.logError,
      fmt='Image load from %s failed, '
             'container-runtime reported "%s".',
      explanation="Configured container image wasn't loaded successfully.",
      recommendedAction="Please check the configured image." )

class ImageLoadNotifiee( Tac.Notifiee ):
   notifierTypeName = "ContainerMgr::ImageLoadConfigDir"

   def __init__( self, configDir, statusDir, master ):
      self.configDir = configDir
      self.statusDir = statusDir
      self.master_ = master

      self.handleInitialized()

      Tac.Notifiee.__init__( self, configDir )

   def loadImage( self, pathUrl ):
      traceMsg = "loadImage %s called" % pathUrl
      t1( traceMsg )
      qt2( qv( traceMsg ) )

      path = Url.parseUrl( pathUrl, None ).localFilename()
      status = self.statusDir.status[ pathUrl ]
      try:
         output = Tac.run( [ 'docker', 'load', '--input', path ],
                           stdout=Tac.CAPTURE, stderr=Tac.CAPTURE )
         outputLines = output.splitlines()
         for line in outputLines:
            prefix = 'Loaded image: '
            if line.startswith( prefix ):
               imageName = line[ len( prefix ) : ]
               status.tag.add( imageName )
      except Tac.SystemCommandError as e:
         errMsg = e.output
         msg = f"docker load for {pathUrl} failed with error {errMsg}"
         t1( msg )
         qt1( msg )
         Logging.log( CONTAINERMGR_IMAGE_LOAD_FAILED,
                      pathUrl,
                      errMsg )

   def removeImage( self, pathUrl ):
      traceMsg = "removeImage %s called" % pathUrl
      t1( traceMsg )
      qt2( qv( traceMsg ) )

      status = self.statusDir.status[ pathUrl ]
      for tag in status.tag:
         try:
            Tac.run( [ 'docker', 'rmi', tag ],
                     stdout=Tac.DISCARD, stderr=Tac.CAPTURE )
         except Tac.SystemCommandError as e:
            errMsg = e.output
            msg = f"docker rmi {tag} failed with error {errMsg}"
            t1( msg )
            qt1( msg )

   @Tac.handler( 'imagePathUrl' )
   def handleImagePathUrl( self, imagePathUrl ):
      t1( "handleImagePathUrl %s called" % imagePathUrl )
      qt2( "handleImagePathUrl", qv( imagePathUrl ), "called" )

      if not self.master_.daemonStarted():
         # Do nothing
         return

      if imagePathUrl in self.configDir.imagePathUrl:
         self.statusDir.newStatus( imagePathUrl )
         self.loadImage( imagePathUrl )
      else:
         if imagePathUrl in self.statusDir.status:
            self.removeImage( imagePathUrl )
            del self.statusDir.status[ imagePathUrl ]

   def cleanup( self ):
      for imagePathUrl in self.statusDir.status:
         self.removeImage( imagePathUrl )
         del self.statusDir.status[ imagePathUrl ]

   def handleInitialized( self ):
      t1( "ImageNotifiee handleInitialized called" )
      qt2( "ImageNotifiee handleInitialized called" )
      # Remove all stale images
      for imagePathUrl in self.statusDir.status:
         if imagePathUrl not in self.configDir.imagePathUrl:
            self.handleImagePathUrl( imagePathUrl )

      # Now load all configured images
      for imagePathUrl in self.configDir.imagePathUrl:
         self.handleImagePathUrl( imagePathUrl )
