# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import abc
import collections

import urllib.parse

import Tracing

__defaultTraceHandle__ = Tracing.Handle( 'ContainerTracerLib' )
debug = Tracing.trace8

PodInfo = collections.namedtuple( 'PodInfo', 'name phase nodeName' )

class ClientError( Exception ):
   pass

class BaseClient( metaclass=abc.ABCMeta ):
   @abc.abstractmethod
   def getNodes( self, nodeName=None):
      """Return list of node names for a cluster.

         If nodeName is provided the results will be filtered for exact matches.
      """

   @abc.abstractmethod
   def getPods( self, podName=None, nodeName=None):
      """Return a list of PodInfo tuples.

      The optional parameters will cause the query to return only exact matches
      for the provided parameters.
      """

class K8sThinClient( BaseClient ):
   def __init__( self, url, token, timeout=None ):
      # avoids importing requests and deps until it's really needed
      import requests # pylint: disable=import-outside-toplevel
      import urllib3 # pylint: disable=import-outside-toplevel
      urllib3.disable_warnings(urllib3.exceptions.InsecureRequestWarning)
      self.session = requests.Session()
      self.session.verify = False
      self.session.headers[ 'Authorization' ] = 'Bearer %s' % token
      self.session.headers[ 'Accept' ] = 'application/json'
      self.timeout = timeout
      self.baseURL = url + ( '' if url[ -1 : ] == '/' else '/' )

   def _get( self, path, kind, fieldSelectors=None ):
      # avoids importing requests and deps until it's really needed
      import requests # pylint: disable=import-outside-toplevel
      url = urllib.parse.urljoin( self.baseURL, path )
      params = {}
      if fieldSelectors:
         params[ 'fieldSelector' ] = ','.join(
            '%s=%s' % i for i in fieldSelectors.items()
         )
      try:
         resp = self.session.get( url, params=params, timeout=self.timeout )
         resp.raise_for_status()
         retval = resp.json()
         if retval[ 'apiVersion' ] == 'v1' and retval[ 'kind' ] == kind:
            return retval[ 'items' ]
      except requests.exceptions.ConnectionError as e:
         raise ClientError(
            f'Unable to connect to cluster at {self.baseURL}' ) from e
      except KeyError as e:
         debug( 'Response missing expected key: %s' % e )
      except Exception as e:  # pylint: disable=broad-except
         debug( 'Exception during client call: %s' % e )
      raise ClientError( 'Error processing request to %s ' % self.baseURL )


   def getNodes( self, nodeName=None, ):
      path = '/api/v1/nodes'
      f = {}
      if nodeName:
         f[ 'metadata.name' ] = nodeName

      return [
         n[ 'metadata' ][ 'name' ]
         for n in self._get( path, 'NodeList', f )
      ]

   def getPods( self, podName=None, nodeName=None ):
      path = '/api/v1/pods'
      f = {}
      if podName:
         f[ 'metadata.name' ] = podName
      if nodeName:
         f[ 'spec.nodeName' ] = nodeName

      return [
         PodInfo(
            pi[ 'metadata' ][ 'name' ],
            pi[ 'status' ][ 'phase' ],
            pi.get( 'spec', {} ).get( 'nodeName', '' )
         ) for pi in self._get( path, 'PodList', f )
      ]

def getClient( config ):
   return K8sThinClient(
      config.url,
      config.authSecret.getClearText(),
      timeout=config.requestTimeout
   )
