#!/usr/bin/env arista-python
# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re

import Assert
import DesiredTracing
import Tracing

TEXT = "plain"
REGX = "regex"
RLST = "list-regex"
SKIP = "skip"
TLST = "list-text"

th = Tracing.defaultTraceHandle()
t0 = th.t0

DesiredTracing.desiredTracingIs( th.name )
DesiredTracing.applyDesiredTracing()

def _printDebugInfo( expectLines, outLines, matchedOutLines, skippedOutLines,
                     stopIndex ):
   t0( "Expected Lines" )
   for index, ( lineType, value ) in enumerate( expectLines ):
      t0( f"{index:<3} {lineType:<14} |{value}" )
   t0( "Output Lines" )
   for index, value in enumerate( outLines ):
      matchLine = matchedOutLines.get( index )
      skipLine = skippedOutLines.get( index )
      if matchLine is not None:
         matchStr = f"{matchLine:<2} MATCHED"
      elif skipLine is not None:
         matchStr = f"{skipLine:<2} (skipped)"
      elif stopIndex is not None and index >= stopIndex:
         matchStr = ""
      else:
         matchStr = "(skipped)"
      t0( f"{index:<3} {matchStr:<14} |{value}" )

def extractSectionIter( lines, marker, indentWidth=2 ):
   """Returns an iterator for the dumpState section starting at `marker`

   The end of the section is determined by the end of the output, or the next line at
   the same, or lower, indent level than `marker`.

   Assume a basic indent of 2 spaces since that is the "standard" used by most of the
   existing dumpState implementations.
   """
   markerIndent = len( marker ) - len( marker.lstrip() )
   start = lines.index( marker )
   assert markerIndent % indentWidth == 0

   yield lines[ start ]

   for line in lines[ start + 1 : ]:
      lineIndent = len( line ) - len( line.lstrip() )
      assert lineIndent % indentWidth == 0
      if lineIndent <= markerIndent:
         return

      yield line

def extractSection( lines, marker, indentWidth=2 ):
   """Returns a section of the dumpState output starting at `marker`

   The end of the section is determined by the end of the output, or the next line at
   the same, or lower, indent level than `marker`.

   Assume a basic indent of 2 spaces since that is the "standard" used by most of the
   existing dumpState implementations.
   """
   return list( extractSectionIter( lines, marker, indentWidth=indentWidth ) )

def assertDumpState( expectLines, output ):
   """
   Ensure that expectLines matches the output provided

   Lines in expectLines can be marked with TEXT, REGX, or SKIP.
    TEXT: Plain text match.
    REGX: Regular expression match.
    SKIP: Skip all lines starting with this level of indentation until the
         first line (useful for ignoring a section).
   """
   if isinstance( output, list ):
      outLines = output
   else:
      outLines = output.splitlines()
   outIdx = 0
   matchedOutLines = {}
   skippedOutLines = {}
   skipLines = None
   expectRes = [] # match all of the lines in no particular order
   values = [] # original values to match corresponsing to expectRes

   def _isListEntry( index ):
      return ( index < len( expectLines ) and
               expectLines[ index ][ 0 ] in ( RLST, TLST ) )

   # pylint: disable=too-many-nested-blocks
   for index, ( lineType, value ) in enumerate( expectLines ):
      if lineType is REGX:
         expectRe = re.compile( '^' + value + '$' )
      elif lineType is RLST:
         expectRes.append( re.compile( '^' + value + '$' ) )
         values.append( value )
         if _isListEntry( index + 1 ):
            continue # collect all the list entries first before matching them
      elif lineType is TEXT:
         expectRe = re.compile( '^' + re.escape( value ) + '$' )
      elif lineType is TLST:
         expectRes.append( re.compile( '^' + re.escape( value ) + '$' ) )
         values.append( value )
         if _isListEntry( index + 1 ):
            continue # collect all the list entries first before matching them
      elif lineType is SKIP:
         skipLines = len( value ) - len( value.lstrip() )
         assert skipLines, "SKIP only supported for indented lines"
      else:
         assert False, f"Unknown lineType {lineType}"

      matched = False
      ignored = False
      while outIdx < len( outLines ):
         outValue = outLines[ outIdx ]
         leadingSpace = len( outValue ) - len( outValue.lstrip() )
         if skipLines:
            if skipLines <= leadingSpace: # pylint: disable=no-else-continue
               skippedOutLines[ outIdx ] = index
               outIdx += 1
               matched = True
               continue
            else:
               ignored = True
               skipLines = None
               break

         try:
            if expectRes:
               for i, expectRe in enumerate( expectRes ):
                  if expectRe.match( outValue ):
                     matchedOutLines[ outIdx ] = index
                     outIdx += 1
                     expectRes.pop( i ) # matched this line -> remove it
                     matched = True
                     break
               # pylint: disable-next=consider-using-f-string
               assert matched, "Expected {}: {} not found in {}".format(
                  index, repr( outValue ), repr( values ) )
               if not expectRes:
                  # all expected lines have been matched
                  values = []
                  break
               matched = False # not all lines from the list have matched yet
            else:
               # pylint: disable-next=deprecated-method
               Assert.assertRegexpMatches( outValue, expectRe )
               matchedOutLines[ outIdx ] = index
               outIdx += 1
               matched = True
               break
         except AssertionError:
            _printDebugInfo( expectLines, outLines, matchedOutLines,
                             skippedOutLines, outIdx )
            raise

      if ignored:
         continue

      if not matched and outIdx >= len( outLines ):
         _printDebugInfo( expectLines, outLines, matchedOutLines,
                          skippedOutLines, outIdx )
         # pylint: disable-next=consider-using-f-string
         assert not expectRes, "Expected {}: not all of {} matched in output".format(
            index, repr( values ) )
         # pylint: disable-next=consider-using-f-string
         assert False, "Expected {}: {} not matched in output".format(
            index, repr( value ) )

   if outIdx != len( outLines ):
      _printDebugInfo( expectLines, outLines, matchedOutLines,
                       skippedOutLines, outIdx )
      Assert.assertEqual( [], outLines[ outIdx : ] ) # pylint: disable=no-member

   _printDebugInfo( expectLines, outLines, matchedOutLines,
                    skippedOutLines, outIdx )

def dumpStateLines( capfd, obj ):
   capfd.readouterr()
   obj.dumpState( None, 0 )
   stdout, _ = capfd.readouterr()
   print( "Output:\n" + stdout )
   return stdout.splitlines()

def dumpStateVerify( capfd, obj, expected ):
   capfd.readouterr()
   obj.dumpState( None, 0 )
   stdout, _ = capfd.readouterr()
   print( "Output:\n" + stdout )
   assertDumpState( expected, stdout )

class DumpStateEntity:
   def __init__( self, entity, capfd ):
      self._entity = entity
      self._capfd = capfd

   @property
   def entity( self ):
      return self._entity

   def section( self, marker ):
      lines = self.dumpStateString().splitlines()
      return extractSection( lines, marker )

   def dumpStateString( self ):
      self.entity.dumpState( None, 0 )
      stdout, _ = self._capfd.readouterr()
      print( "Output:\n" + stdout )
      return stdout

   def dumpStateVerify( self, expected ):
      dumpStateVerify( self._capfd, self._entity, expected )
