#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=W0611
# pylint: disable=W0123
# pylint: disable=W0212

import argparse
import CliCommon
from collections import OrderedDict, namedtuple
from datetime import datetime, timedelta
from EapiClientLib import EapiClient, EapiException
from email.mime.multipart import MIMEMultipart
from email.mime.text import MIMEText
import inspect
import json
import os
import pydoc
import pexpect
import re
import readline
import smtplib
import subprocess
import sys
import Tac
import time
from traceback import print_exc

TRACE_NONE = 0
TRACE_FATAL = 1
TRACE_ERROR = 2
TRACE_WARN = 3
TRACE_INFO = 4
TRACE_DEBUG = 5
TRACE_TRACE = 6

traceLevel = 0
traceFile = None

defaultCliUser = "admin"
defaultCliPasswd = None
defaultRootUser = "root"
defaultRootPasswd = "arastra"

###################################################################
# print msg to stdout/stderr according to traceLevel
###################################################################
def traceLevelIs( level ):
   global traceLevel
   traceLevel = level

def traceFileIs( path, append=False ):
   global traceFile
   # pylint: disable-next=R1732
   traceFile = open( path, "w+" if append else "w" )

def traceMsg( level, msg ):
   '''
   Trace levels:
   1 = Fatal: One ore more key business functionalities are not working and
              the whole system doesn't fulfill the business functionalities
   2 = Error: One ore more functionalities are not working, preventing some
              functionalities from working correctly
   3 = Warn: Unexpected behavior happened inside the application, but it is
             continuing its work and the key business features are operating
             as expected
   4 = Info: An event happened, the event is purely informative and can be
             igonred during normal operations
   5 = Debug: A log level used for events considered to be useful during software
              debugging when more granular information is needed
   6 = Trace: A log level describing events showing step by setp execution of
              your code that can be ignored during the standar operation, but
              may be useful during extended debugging sessions
   '''
   if level > traceLevel:
      return
   if traceFile is not None:
      # Write msg to trace file.
      print( f"{datetime.now()} -{level}- {msg}", file=traceFile )
      traceFile.flush()
   else:
      # Write msg to stdout or stderr.
      outFile = sys.stdout if level > TRACE_ERROR else sys.stderr
      print( msg, file=outFile )
      outFile.flush()

def errorMsg( msg ):
   # Write error message to trace file or stderr.
   traceMsg( TRACE_ERROR, msg )

###################################################################
# find default grabbed dut
###################################################################
def grabbedDut( ):
   # Find grabbed dut
   dut = os.getenv( "ARTEST_DUT" )
   if dut is None:
      arosTestDutFile = "/tmp/ArosTest/.ArosTest-duts"
      if os.path.isfile( arosTestDutFile ):
         lines = None
         with open( arosTestDutFile ) as f:
            lines = f.readlines()
         if lines:
            dutUrl = lines[ 0 ]
            match = re.search( r"rdam.*://([^']+)", dutUrl )
            if match:
               dut = match.group( 1 )
   return dut

###################################################################
# execute local shell command
###################################################################
def shellCmd( cmd, echoCmd=False ):
   if echoCmd:
      print( cmd )
   # pylint: disable-next=R1732
   p = subprocess.Popen( cmd, shell=True, stdout=subprocess.PIPE,
                         stderr=subprocess.PIPE )
   outbuf, errbuf = p.communicate()
   assert p.returncode == 0, errbuf
   return outbuf.decode( 'utf-8' )

###################################################################
# SSH session
###################################################################
class SshSession:
   def __init__( self, hostname, timeout=30, trace=False, username=None,
                 password=None, passwordPrompt=None, shellPrompt=None,
                 sshCmd=None ):
      self.p = None
      self.hostname = hostname
      self.username = username
      self.timeout = timeout
      self.trace = trace
      self.traceOut = trace
      self.password = password
      self.passwordPrompt = passwordPrompt
      self.shellPrompt = shellPrompt
      if self.shellPrompt is None:
         self.shellPrompt = self.hostname + ".sjc> "
      self.sshCmd = sshCmd
      self.latencyCheck = False
      self.cmdLatency = 0
      # start ssh connection
      self.login()

   def login( self ):
      # Start ssh session and wait for shell prompt
      if self.sshCmd is None:
         dest = self.username + "@" + self.hostname if self.username else \
                  self.hostname
         self.sshCmd = "ssh " + dest
      self.p = pexpect.spawn( self.sshCmd, encoding="us-ascii" )
      if self.password:
         if self.passwordPrompt is None:
            self.passwordPrompt = "Password:"
         self.p.expect( self.passwordPrompt, timeout=self.timeout )
         self.p.sendline( self.password )
      self.p.expect( self.shellPrompt, timeout=self.timeout )

   def logout( self ):
      self.sendCmd( "exit", waitForPrompt=False )
      self.p = None

   def __del__( self ):
      if self.p:
         self.logout()

   def removeNonAsciiAndCR( self, text ):
      return str( ''.join( i for i in text if ord( i ) < 128 and i != '\r' ) )

   def sendCmd( self, cmd, waitForPrompt=True, timeout=None, newPrompt=None,
                skipFirstLineOut=True, skipLastLineOut=True, showOutput=False,
                echo=False ):
      if self.trace:
         print( self.hostname, "cmd:", cmd )
      if newPrompt is not None:
         self.shellPrompt = newPrompt
      if self.latencyCheck:
         startTime = time.time()
      if echo or self.traceOut:
         self.p.logfile = sys.stdout
         showOutput = False
      self.p.sendline( cmd )
      if waitForPrompt:
         if timeout is None:
            timeout = self.timeout
         self.p.expect( self.shellPrompt, timeout=timeout )
      if self.latencyCheck:
         endTime = time.time()
         self.cmdLatency = endTime - startTime
      # Get command output
      outbuf = self.removeNonAsciiAndCR( self.p.before )
      lines = outbuf.split( '\n' )
      if skipFirstLineOut:
         lines = lines[ 1 : ]  # skip cmd echo
      if skipLastLineOut:
         lines = lines[ : -1 ] # skip prompt
      if echo or self.traceOut:
         self.p.logfile = None
      # Return output
      if showOutput:
         print( '\n'.join( lines ) )
      return lines

###################################################################
# SCP session
###################################################################
class ScpSession:
   def __init__( self, hostname, timeout=60, trace=False, username=None,
         password=None, passwordPrompt="Password: ", shellPrompt=None ):
      self.p = None
      self.hostname = hostname
      self.username = username
      self.timeout = timeout
      self.trace = trace
      self.password = password
      self.passwordPrompt = passwordPrompt if passwordPrompt else "password: "
      self.shellPrompt = shellPrompt if shellPrompt else "% "
      self.latencyCheck = False
      self.remotePrefix = self.hostname + ":"
      if self.username:
         self.remotePrefix = self.username + "@" + self.remotePrefix

   def doCopy( self, src, dst ):
      prompts = [ pexpect.EOF ]
      if self.password:
         prompts.append( self.passwordPrompt )
      cmd = f"/usr/bin/a4 scp -q {src} {dst}"
      self.p = pexpect.spawn( cmd, timeout=self.timeout )
      i = self.p.expect( prompts )
      if i == 1:
         self.p.sendline( self.password )
         self.p.expect( pexpect.EOF )
      self.p.close()
      if self.p.exitstatus != 0:
         print( "FAILED:", cmd )
      return self.p.exitstatus

   def get( self, localPath, remotePath ):
      return self.doCopy( self.remotePrefix + remotePath, localPath )

   def put( self, localPath, remotePath ):
      return self.doCopy( localPath, self.remotePrefix + remotePath )

###################################################################
# get RDAM DUT info
###################################################################
class DutInfo:
   def __init__( self, dutName ):
      self.dutName = dutName
      self.dutProperties = {}
      self.connectedIntfMap = {}

   def isMlag( self ):
      return "MLAG peers" in self.dutProperties

   def parseProperty( self, line, sep=':' ):
      prop, val = line.split( sep, 1 )
      prop = prop.strip()
      val = val.strip().split()
      if len( val ) == 1:
         val = val[ 0 ]
      self.dutProperties[ prop ] = val

   def getArtInfo( self ):
      tmpFile = f"/mnt/flash/ArtInfo-{self.dutName}"
      if not os.path.isfile( tmpFile ):
         return
      with open( tmpFile ) as f:
         dutInfo = f.readlines()
         for line in dutInfo:
            # Get dut properties from Art info
            line = line.strip()
            if line.startswith( "MLAG" ):
               self.parseProperty( line )
            elif line.startswith( "namespace" ):
               self.parseProperty( line, sep=None )
            elif line.startswith( "Ethernet" ):
               if "(disabled)" in line:
                  continue
               ethInfo = line.split()
               if len( ethInfo ) >= 3:
                  self.connectedIntfMap[ ethInfo[ 0 ] ] = \
                        ( ethInfo[ 1 ], ethInfo[ 2 ] )

   def connectedIntfs( self, includeSnake=False ):
      if not self.connectedIntfMap:
         self.getArtInfo()
      intfs = [ localIntf
                for localIntf, ( rmtIntf, _ )
                     in list( self.connectedIntfMap.items() )
                if includeSnake or rmtIntf != "<snake>" ]
      return sorted( intfs )

   def getProperty( self, propName ):
      if not self.dutProperties:
         self.getArtInfo()
      return self.dutProperties.get( propName )

###################################################################
# DUT CLI exception
###################################################################
class DutCliException( Exception ):
   pass

###################################################################
# DUT CLI session using EAPI
###################################################################
class DutCli:
   def __init__( self, timeout=30, trace=False, ignoreConfigErrors=False ):
      # Create CLI client.
      self.cliReqVersion = 1
      self.cliClient = EapiClient( disableAaa=True, privLevel=15 )
      self.configFile = None
      self.timeout = timeout
      self.trace = trace
      self.ignoreConfigErrors = ignoreConfigErrors

   def ignoreConfigErrorsIs( self, ignoreConfigErrors ):
      self.ignoreConfigErrors = ignoreConfigErrors

   def openConfigFile( self, fileName, mode="wt" ):
      # pylint: disable-next=R1732
      filePath = "/mnt/flash/" + fileName
      # pylint: disable-next=R1732
      self.configFile = open( filePath, mode )

   def closeConfigFile( self, apply=False ):
      if self.configFile:
         fileName = os.path.basename( self.configFile.name )
         self.configFile.close()
         if apply:
            self.applyConfigFile( fileName )
         self.configFile = None

   def applyConfigFile( self, fileName ):
      print( f"applying config file: {fileName}..." )
      sys.stdout.flush()
      self.runCmds( f"copy flash:{fileName} running-config" )
      print( "done" )
      sys.stdout.flush()

   def runCmds( self, cmds, initCmds=None, timeout=None, ignoreErrors=False,
                showOutput=False, jsonOutput=False,
                skipFirstLineOut=False, skipLastLineOut=True,
                skipBlankLines=False ):
      # convert commands to a list
      if not isinstance( cmds, list ):
         cmds = [ x.strip() for x in cmds.split( ";" ) ]
      if initCmds:
         if not isinstance( cmds, list ):
            initCmds = [ x.strip() for x in initCmds.split( ";" ) ]
         cmds = initCmds + cmds
      # execute commands
      if timeout is None:
         timeout = self.timeout
      if self.trace:
         print( '; '.join( cmds ) )
      cmd_format = CliCommon.ResponseFormats.JSON if jsonOutput else \
                   CliCommon.ResponseFormats.TEXT
      try:
         result = self.cliClient.runCmds( version=self.cliReqVersion, cmds=cmds,
                                          format=cmd_format, autoComplete=True,
                                          requestTimeout=timeout )
      except EapiException as e:
         if not ignoreErrors:
            raise e
         return None
      # parse result
      result = result[ "result" ][ 0 ]
      if jsonOutput:
         if showOutput:
            print( result )
      else:
         assert "output" in result
         result = result[ "output" ]
         if showOutput:
            print( result )
         result = result.split( '\n' )
         if skipFirstLineOut:
            result = result[ 1 : ]
         if skipLastLineOut:
            result = result[ : -1 ]
         if skipBlankLines:
            result = [ line for line in result if line ]
      return result

   def showCmd( self, cmd, **kwargs ):
      return self.runCmds( cmd, **kwargs )

   def privCmd( self, cmd, **kwargs ):
      return self.runCmds( cmd, **kwargs )

   def configCmd( self, cmd, cont=False, **kwargs ):
      if isinstance( cmd, str ):
         cmd = [ x.strip() for x in cmd.split( ";" ) ]
      if self.configFile is not None:
         # Write command(s) to config file.
         self.configFile.write( '\n'.join( cmd ) + '\n' )
         return None
      else:
         # Start with config command if not a continuation.
         initCmds = None if cont else [ "config" ]
         return self.runCmds( cmd, initCmds=initCmds,
                              ignoreErrors=self.ignoreConfigErrors, **kwargs )

   def endConfig( self, **kwargs ):
      if self.configFile:
         self.closeConfigFile( apply=True )
      else:
         self.runCmds( "end", **kwargs )

   def bashCmd( self, cmd, timeout=None, skipFirstLineOut=True,
                skipLastLineOut=True, showOutput=False, echo=False ):
      result = shellCmd( cmd, echoCmd=( self.trace or echo ) )
      if showOutput:
         print( result )
      result = result.split( '\n' )
      if skipLastLineOut:
         result = result[ : - 1 ]
      return result

###################################################################
# Base class for all EDT Plugin classes
###################################################################
class EDTPlugin:
   def __init__( self ):
      self.dutName = None
      self.dutInfo = None
      self.dutNetNs = None
      self.dutCli = None
      self.debugCtx = None
      self.debugServer = None

   def resetDebugCtx( self ):
      self.debugCtx.clear()

   def setupPlugin( self, dutName, dutInfo, dutCli, debugCtx, debugServer ):
      # Setup plugin state.
      self.dutName = dutName
      self.dutInfo = dutInfo
      if dutInfo:
         self.dutNetNs = dutInfo.dutProperties.get( "namespace" )
      self.dutCli = dutCli
      self.debugCtx = debugCtx
      self.debugServer = debugServer

   def cleanupPlugin( self ):
      # Cleanup plugin state.
      pass

###################################################################
# DUT Accessor base class
#   - contains methods which can be utilized on all EOS platforms
###################################################################
class DutAccessorException( Exception ):
   pass

class DutAccessorBase( EDTPlugin ):
   def allocConnectedIntf( self ):
      connIntfs = self.debugCtx.get( "connectedIntfs" )
      if connIntfs is None:
         # Cache connected interfaces from dut info.
         connIntfs = self.dutInfo.connectedIntfs()
         if connIntfs:
            self.debugCtx[ "connectedIntfs" ] = connIntfs
      # Get a free connected interface.
      assert connIntfs, "no available connected interface found"
      allocIntf = connIntfs.pop()
      self.debugCtx[ "connectedIntfs" ] = connIntfs
      return allocIntf

   def namedIntf( self, name, allocIntf=True ):
      # Get a cononically-named interface.
      key = name + "Intf"
      intf = self.debugCtx.get( key )
      if not intf and allocIntf:
         # Allocate a new interface and store in debugCtx.
         intf = self.allocConnectedIntf()
         if intf:
            self.debugCtx[ key ] = intf
      assert intf, f"{key} not found in debugCtx"
      return intf

   def namedIntfIs( self, name, intf ):
      # Set cononically-named interface.
      key = name + "Intf"
      self.debugCtx[ key ] = intf

   def allocRxIntf( self ):
      # Allocate a connected interface for Rx.
      rxIntf = self.allocConnectedIntf()
      self.debugCtx[ "rxIntf" ] = rxIntf
      return rxIntf

   def allocTxIntf( self ):
      # Allocated a connected interface for Tx.
      txIntf = self.allocConnectedIntf()
      self.debugCtx[ "txIntf" ] = txIntf
      return txIntf

   def rxOrTxIntf( self, direction ):
      assert direction in [ "rx", "tx" ]
      return self.namedIntf( direction, allocIntf=False )

   def isNamespaceDut( self ):
      # Determine if this is a namespace dut
      return self.dutNetNs() is not None

   def agentPid( self, agentName ):
      # Return PID of dut agent process.
      out = self.dutCli.bashCmd( f'''ps aux | grep 'agenttitle={agentName}' |
            grep -v grep''' )
      assert len( out ) == 1
      pid = int( out[ 0 ].split()[ 1 ] )
      # print "%s pid = %d" % ( agentName, pid )
      return pid

   def copyFileFromDut( self, dutFile, localFile,
                        username=defaultRootUser, password=defaultRootPasswd ):
      scp = ScpSession( self.dutName, username=username, password=password )
      return scp.get( localFile, dutFile )

   def copyFileToDut( self, localFile, dutFile,
                      username=defaultRootUser, password=defaultRootPasswd ):
      scp = ScpSession( self.dutName, username=username, password=password )
      return scp.put( localFile, dutFile )

   def parseTraceLevels( self, levels ):
      # Convert levels to list of integers 0-9
      levelList = []
      if not levels:
         pass
      elif isinstance( levels, int ):
         levelList = [ levels ]
      elif isinstance( levels, list ):
         for item in levels:
            levelList.extend( self.parseTraceLevels( item ) )
      elif isinstance( levels, str ):
         if '-' in levels:
            i = levels[ 1 : -1 ].find( '-' ) + 1
            assert i > 0, f"malformed trace levels: {levels}"
            # pylint: disable=R1721
            low = int( levels[ i - 1 ] )
            high = int( levels[ i + 1 ] )
            levelList += [ j for j in range( low, high + 1 ) ]
            if ( i - 2 ) >= 0:
               levelList += self.parseTraceLevels( levels[ : i - 2 ] )
            if ( i + 2 ) <= len( levels ) - 1:
               levelList += self.parseTraceLevels( levels[ i + 2 : ] )
         else:
            levelList = [ int( x ) for x in list( levels ) ]
      else:
         assert False, f"malformed trace levels: {levels}"
      return levelList

   def filterTraceLog( self, logFile, levels ):
      # Filter trace message file to keep only lines which have desired trace level.
      levelList = self.parseTraceLevels( levels )
      tmpFile = logFile + ".tmp"
      with open( tmpFile, "w" ) as fout:
         with open( logFile ) as fin:
            for line in fin:
               comp = line.split()
               if len( comp ) >= 5 and comp[ 2 ].isdigit() and comp[ 4 ].isdigit():
                  level = int( comp[ 4 ] )
                  if level not in levelList:
                     continue
               fout.write( line )
      os.rename( tmpFile, logFile )

   def copyAgentQtLog( self, agentName, levels=None ):
      # Use qtcat to write messages to temp log file.
      tmpLogFile = f"/tmp/{agentName}.qtlog"
      qtopt = f"-l {levels}" if levels else ""
      cmd = f"qtcat {qtopt} /var/log/qt/{agentName}.qt > {tmpLogFile}"
      self.dutCli.bashCmd( cmd )

      # Use SCP to upload qtrace log file.
      localFile = f"./{self.dutName}-{agentName}.qtlog"
      print( "copying agent qtrace file to", localFile )
      self.copyFileFromDut( tmpLogFile, localFile )
      self.dutCli.bashCmd( "rm " + tmpLogFile )
      return localFile

   def copyAgentLog( self, agentName, levels=None ):
      ''' Upload agent log file from dut to current directory on this host '''
      # Concatenate latest and compressed log files for agent to temp log file.
      # used in a f string not detected by pylint
      # pylint: disable-next=W0612
      pid = self.agentPid( agentName )
      agentProcess = "{agentName}-{pid}"
      tmpLogFile = f"/tmp/{agentProcess}-all.log"
      self.dutCli.bashCmd( f'''cd /var/log/agents; cat <(gunzip -c
            {agentProcess}*.gz) {agentProcess} > {tmpLogFile}''' )

      # Use SCP to upload full agent log file.
      localFile = f"./{self.dutName}-{agentName}.log"
      print( "copying agent log file to", localFile )
      self.copyFileFromDut( tmpLogFile, localFile )
      self.dutCli.bashCmd( "rm " + tmpLogFile )
      if levels is not None:
         # Filter levels in log file.
         self.filterTraceLog( localFile, levels )
      return localFile

   def showSysLog( self, numLines=30 ):
      ''' Show output from dut system log '''
      self.dutCli.privCmd( "show logging all | tail -{numLines}",
                           showOutput=True )

   def traceAgent( self, agentName, traceLevels, enable=True ):
      cmd = f"trace {agentName} setting "
      if enable:
         cmd += ",".join( traceLevels )
      else:
         cmd = "no " + cmd
      self.dutCli.configCmd( cmd )

   def restartAgent( self, agentName ):
      ''' Restart agent and remove previous log files '''
      print( f"restarting agent {agentName}..." )
      self.dutCli.configCmd( f"agent {agentName} shut", showOutput=True )
      time.sleep( 2 )
      self.dutCli.bashCmd( f"sudo rm -f /var/log/agents/{agentName}-*" )
      self.dutCli.configCmd( f"no agent {agentName} shut", showOutput=True )

   def shutdownAgent( self, agentName, enableShut=True ):
      ''' Shutdown or enable agent '''
      no = "no " if not enableShut else ""
      self.dutCli.configCmd( no + f"agent {agentName} shut" )

   def showVersion( self ):
      ''' Show DUT version info '''
      self.dutCli.showCmd( "show version", showOutput=True )

   def systemMacAddr( self ):
      sysInfo = self.dutCli.showCmd( "show version", jsonOutput=True ) or {}
      return sysInfo.get( "systemMacAddress" )

   def intfMacAddr( self, dutIntf ):
      cmd = f"show interface {dutIntf}"
      intfInfo = self.dutCli.showCmd( cmd, jsonOutput=True ) or {}
      return intfInfo.get( "physicalAddress" )

   def intfNameLinux( self, intfName, useDutPrefix=False ):
      if intfName is None:
         return intfName
      intfName = intfName.lower()
      if re.match( r"ethernet\d", intfName ):
         intfName = intfName.replace( "ethernet", "et" )
      elif re.match( r"eth\d", intfName ):
         intfName = intfName.replace( "eth", "et" )
      elif re.match( r"et\d", intfName ):
         pass
      else:
         assert False, f"invalid linux intf: {intfName}"
      intfName = intfName.replace( "/", "_" )
      if useDutPrefix:
         intfName = self.dutName + "-" + intfName
      return intfName

   def localIntfName( self, dutIntf ):
      return self.dutName + "-" + \
         dutIntf.replace( "Ethernet", "et" ).replace( '/', '_' )

   def intfNameLong( self, intfName ):
      if intfName is None:
         return intfName
      intfName = intfName.capitalize()
      if re.match( r"Et\d", intfName ):
         intfName = intfName.replace( "Et", "Ethernet" )
      elif re.match( r"Po\d", intfName ):
         intfName = intfName.replace( "Po", "Port-channel" )
      elif re.match( r"Tu\d", intfName ):
         intfName = intfName.replace( "Tu", "Tunnel" )
      return intfName

   def intfNameShort( self, intfName ):
      if intfName is None:
         return intfName
      intfName = intfName.capitalize()
      if re.match( r"Ethernet\d", intfName ):
         intfName = intfName.replace( "Ethernet", "Et" )
      elif re.match( r"Port-channel\d", intfName ):
         intfName = intfName.replace( "Port-channel", "Po" )
      return intfName

   def getDutMpcDevIntf( self, dutIntf ):
      '''return connected MPC's device name and interface name given a DUT'''
      if not self.dutInfo.connectedIntfMap:
         self.dutInfo.getArtInfo()
      mpcDevIntfTuple = self.dutInfo.connectedIntfMap.get( dutIntf )
      if mpcDevIntfTuple is None:
         return( None, None )
      else:
         # Convert MPC intf name to linux netdev name.
         return ( mpcDevIntfTuple[ 0 ], self.intfNameLinux( mpcDevIntfTuple[ 1 ] ) )

   def resetDutConfig( self, config="startup-config" ):
      # Save existing EDTClient CLI alias.
      edtCliAlias = self.debugCtx.get( "edtCliAlias" )
      # Replace config with startup-config.
      self.dutCli.privCmd( f"config replace {config}" )
      # Clear debug context.
      self.debugCtx.clear()
      # Restore EDTClient CLI aliases.
      self.debugServer.setEdtCliAlias( edtCliAlias )

   def saveDutConfig( self, config="startup-config" ):
      self.dutCli.privCmd( f"copy running-config flash:{config}" )

   #######################################################################
   # Attribute formatting
   #######################################################################
   def splitLabelValue( self, valStr ):
      if ':' in valStr:
         label, val = valStr.split( ':', 1 )
         label = label.strip()
         val = val.strip()
      else:
         label = val = valStr.strip()
      return ( label, val )

   def splitLabelName( self, valStr ):
      return self.splitLabelValue( valStr )

   def fmtAttrValues( self, attrVals, brief=False, sortNames=False, useHex=False,
                      labelSep=": ", labelFmt=None, attrSep=None, attrValFmt=None,
                      prefix='' ):
      # Format attribute names with values, either brief form or long form.
      labels = list( attrVals.keys() )
      if sortNames:
         labels = sorted( labels )
      if labelFmt is None:
         labelFmt = "%s" if brief else \
               f"%-{max( [ len( str( label ) ) for label in labels ] ) }s"
      if attrSep is None:
         attrSep = " " if brief else "\n "

      def _fmtLabelVal( _label, _val ):
         labelStr = labelFmt % _label
         valStr = None
         if attrValFmt is not None:
            for lpat, fmtFunc in attrValFmt.items():
               if lpat.match( _label ):
                  valStr = fmtFunc( _val )
                  break
         if valStr is None:
            if useHex and isinstance( _val, int ):
               valStr = hex( _val )
            else:
               valStr = str( _val )
         return labelStr + labelSep + valStr

      return ( prefix + attrSep +
               attrSep.join( [ _fmtLabelVal( label, attrVals[ label ] )
                               for label in labels ] ) )

   #######################################################################
   # Object rendering (format object data for pretty printing)
   #######################################################################
   def render_default_type( self, value, detail=None ):
      return str( value )

   def render( self, value, detail=None, variant=None ):
      '''
      Render a value to a string based on its type.  Use render_<type>_<variant>
      function if it exists.
      '''
      # Render a value as a string, possibly using other render methods.
      renderFunc = None
      funcName = "render_" + type( value ).__name__.replace( "::", "_" )
      if variant:
         renderFunc = getattr( self, funcName + "_" + variant )
      if not renderFunc:
         renderFunc = getattr( self, funcName, self.render_default_type )
      return renderFunc( value, detail )

   def showRenderedValue( self, val, detail=None, variant=None ):
      ''' Render a single value '''
      print( self.render( val, detail=detail, variant=variant ) )

   def showRenderedMap( self, myMap, detail=None, variant=None,
                        keys=None, sortKey=None, sortReverse=False, kvSep=" => ",
                        label=None ):
      ''' Render entries in a map '''
      if keys is None:
         keys = myMap.keys()
      if sortKey or sortReverse:
         keys = sorted( keys, key=sortKey, reverse=sortReverse )
      if label and keys:
         print( label )
      for k in keys:
         v = myMap[ k ]
         print( kvSep.join( ( self.render( k, detail, variant ),
                              self.render( v, detail, variant ) ) ) )

   def showRenderedMapPair( self, map1, map2, detail=None, variant=None,
                            keys=None, sortKey=None, sortReverse=False, kvSep=" => ",
                            label=None ):
      ''' Render entries from two maps sharing common key '''
      if keys is None:
         keys = set( map1.keys() ) | set( map2.keys() )
      if sortKey or sortReverse:
         keys = sorted( keys, key=sortKey, reverse=sortReverse )
      if label and keys:
         print( label )
      for k in keys:
         v1 = map1.get( k )
         v2 = map2.get( k )
         print( kvSep.join( ( self.render( k, detail, variant ),
                              self.render( v1, detail, variant ),
                              self.render( v2, detail, variant ) ) ) )

   def showRenderedList( self, myList, detail=None, variant=None,
                         sortKey=None, sortReverse=False, label=None ):
      ''' Renter entries in a list '''
      if sortKey or sortReverse:
         vals = sorted( myList, key=sortKey, reverse=sortReverse )
      else:
         vals = myList
      if label and vals:
         print( label )
      for v in vals:
         print( self.render( v, detail, variant ) )

   def getTacValueAttrs( self, value, labelNames=None ):
      # Convert Tac value to map of labels to values.  Extract Tac attribute matching
      # name and replace name with label in output map.
      labelVals = {}
      for labelName in labelNames:
         label, name = self.splitLabelValue( labelName )
         v = value
         for cname in name.split( '.' ):
            v = getattr( v, cname )
            assert v is not None, f"missing attribute name: '{name}'"
         labelVals[ label ] = v
      return labelVals

   #######################################################################
   # Wait for condition to be true before continuing.
   #######################################################################
   def waitFor( self, cond, description=None, timeout=None, minDelay=2.0 ):
      ''' Poll condition and wait for it to be true '''
      if description is None:
         description = str( cond )
      if timeout is None:
         timeout = 120.
      endTime = datetime.now() + timedelta( seconds=timeout )
      waiting = False
      done = False
      while True:
         if cond():
            done = True
            break
         if datetime.now() > endTime:
            break
         if not waiting:
            waiting = True
            sys.stdout.write( "waiting for " + description + ".." )
            sys.stdout.write( '.' )
            sys.stdout.flush()
            time.sleep( minDelay )
      # if waiting:
      #    print
      if not done:
         raise DutAccessorException( f"timeout waiting for {description}" )

   #######################################################################
   # Bit field manipulation
   #######################################################################
   def extractBits( self, val, start, cnt ):
      # Extract bits from int value.
      return ( val >> start ) & ( ( 1 << cnt ) - 1 )

   def extractFields( self, data, fieldDescrList, useDict=False ):
      fieldVals = []
      for fieldDescr in fieldDescrList:
         name, startBit, numBits = fieldDescr
         val = self.extractBits( data, startBit, numBits )
         fieldVals.append( ( name, val ) if useDict else val )
      if useDict:
         fieldVals = dict( fieldVals )
      return fieldVals

   def modifyBits( self, val, start, cnt, field ):
      # Modify bits in value.
      mask = ( ( 1 << cnt ) - 1 ) << start
      val &= ~mask
      val |= ( ( field << start ) & mask )
      return val

   def parseBitFields( self, val, fieldsMeta, offset=0 ):
      # Convert entry value to fields using meta info.
      fields = {}
      for name, finfo in fieldsMeta.items():
         assert 'offset' in finfo, f"no offset for field '{name}'"
         assert 'width' in finfo, f"no width for field '{name}'"
         start = finfo[ 'offset' ] + offset
         cnt = finfo[ 'width' ]
         fields[ name ] = self.extractBits( val, start, cnt )
      return fields

   def mergeBitFields( self, fields, fieldsMeta, init=0, offset=0 ):
      # Convert entry fields to single value using meta info.
      val = init
      for name, finfo in fieldsMeta.items():
         if name in fields:
            fval = fields[ name ]
         else:
            fval = finfo.get( 'default', 0 )
         assert 'offset' in finfo, f"no offset for field '{name}'"
         assert 'width' in finfo, f"no width for field '{name}'"
         start = finfo[ 'offset' ] + offset
         cnt = finfo[ 'width' ]
         val = self.modifyBits( val, start, cnt, fval )
      return val

###################################################################
# Plugin for EDT Server
###################################################################
# EDT command function decorator
def EDTCmd( func, *args, **kwargs ):
   def decorated_func( *args, **kwargs ):
      return func( *args, **kwargs )
   # add is_cool property to function so that we can check for its existence later
   decorated_func._is_EDTCmd = True
   decorated_func._original = func
   return decorated_func

###################################################################
# Perform commands on remote dut and transfer files to/from dut
#   - used by EDTClient and EDTXfer
###################################################################
class DutCmdSession:
   '''
   Create a session for executing commands or copying files to/from remote dut.
   '''
   def __init__( self, hostname, timeout=60, trace=False, username=None,
                 password=None, passwordPrompt="Password: ", sshCmd="a ssh" ):
      self.hostname = hostname
      self.timeout = timeout
      self.trace = trace
      self.username = username
      self.password = password
      self.passwordPrompt = passwordPrompt
      self.sshCmd = sshCmd

      self.p = None
      self.nextCmdPrompt = None
      self.remotePrefix = self.hostname + ":"
      if self.username:
         self.remotePrefix = self.username + "@" + self.remotePrefix

      self.cliUnprivPrompt = re.compile( fr'{self.hostname}.*>$' )
      self.cliPrivPrompt = re.compile( fr'{self.hostname}.*#$' )
      self.cliConfigPrompt = re.compile( fr'{self.hostname}.*config.*#$' )

   def waitCmdDone( self ):
      prompts = [ pexpect.EOF, self.passwordPrompt ]
      if self.nextCmdPrompt:
         prompts.append( self.nextCmdPrompt )
      token = self.p.expect( prompts, timeout=self.timeout )

      if token == 1 and self.passwordPrompt:
         # prompted for password
         self.p.sendline( self.password )
         token = self.p.expect( prompts, timeout=self.timeout )
         if token == 1:
            # error - repeated prompt for password
            self.p.close()
            self.p.before = "Incorrect username or password".encode( 'utf-8' )
            self.p.exitstatus = 1
      return prompts[ token ]

   def open( self, newPrompt=None ):
      if newPrompt:
         self.nextCmdPrompt = newPrompt
      # Open SSH session.
      sshCmd = f"{self.sshCmd} {self.username}@{self.hostname}"
      self.p = pexpect.spawn( sshCmd, encoding='utf-8', timeout=self.timeout )
      return self.waitCmdDone() == self.nextCmdPrompt

   def close( self ):
      # Close SSH session.
      if self.p:
         self.p.close()
         self.p = None

   def sendCmd( self, cmd, newPrompt=None ):
      if newPrompt:
         self.nextCmdPrompt = newPrompt
      self.p.sendline( cmd )
      prompt = self.waitCmdDone()
      return ( self.p.before, prompt == self.nextCmdPrompt )

   def doCliCmd( self, cmd, newPrompt=None, configMode=False ):
      if not self.p:
         # Open CLI session in privileged mode.
         assert self.open( self.cliUnprivPrompt )
         outbuf, status = self.sendCmd( "enable", self.cliPrivPrompt )
         assert status, "CLI enable failed: " + outbuf

      if configMode:
         # Switch to config mode.
         outbuf, status = self.sendCmd( "config", self.cliConfigPrompt )
         assert status

      # Execute CLI command(s).
      if newPrompt:
         self.nextCmdPrompt = newPrompt
      outAll = ""
      cmds = cmd.split( ';' )
      for line in cmds:
         outbuf, status = self.sendCmd( line )
         outAll += outbuf
         if not status:
            return outAll, status

      if configMode:
         # Switch to priviledged mode.
         self.sendCmd( "end", self.cliPrivPrompt )

      # Return all output and status.
      return ( outAll, status )

   def execLocalShellCmd( self, cmd, prompt=None, close=True ):
      # Execute single command and return output and exit status.
      self.p = pexpect.spawn( cmd, timeout=self.timeout )
      self.waitCmdDone()
      outbuf = self.p.before.decode()
      status = self.p.exitstatus
      self.close()
      return ( outbuf, status )

   def execRemoteShellCmd( self, cmd ):
      localCmd = f"{self.sshCmd} {self.username}@{self.hostname} {cmd}"
      return self.execLocalShellCmd( localCmd )

   def doRsyncCmd( self, src, dst ):
      rsyncCmd = f"rsync -vzP --update -e '{self.sshCmd}' {src} {dst}"
      msg, status = self.execLocalShellCmd( rsyncCmd )
      if status:
         errorMsg( f"rsync FAIL: {src} => {dst}\n{msg}" )
      else:
         print( f"rsync SUCCESS: {src} => {dst}" )
      return status

   def getFile( self, localPath, remotePath ):
      # Copy file from remote dut to local host.
      return self.doRsyncCmd( self.remotePrefix + remotePath, localPath )

   def putFile( self, localPath, remotePath ):
      # Copy file from local host to remote dut.
      return self.doRsyncCmd( localPath, self.remotePrefix + remotePath )

   def copyFile( self, localPath, remotePath, upload=False ):
      # Copy file to/from remote dut.
      xferFunc = self.getFile if upload else self.putFile
      return xferFunc( localPath, remotePath )
