# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
# pylint: disable-msg=E0401
import time
from EDTAccess import DutAccessorBase, EDTCmd

class BasePlugin( DutAccessorBase ):

   @EDTCmd
   def resetConfig( self ):
      '''Reset dut to startup config and reset debugCtx'''
      self.resetDutConfig()

   @EDTCmd
   def replaceConfig( self, config ):
      '''Replace dut running-config with config from file'''
      self.resetDutConfig( config )

   @EDTCmd
   def saveConfig( self, config="startup-config" ):
      '''Save dut running-config to config file'''
      self.saveDutConfig( config )

   @EDTCmd
   def reloadDut( self ):
      ''' Reload dut '''
      self.dutCli.privCmd( "reload force" )

   @EDTCmd
   def rxIntfIs( self, intfName ):
      '''Set debug Rx interface'''
      self.debugCtx[ "rxIntf" ] = self.intfNameLong( intfName )

   @EDTCmd
   def txIntfIs( self, intfName ):
      '''Set debug Tx interface'''
      self.debugCtx[ "txIntf" ] = self.intfNameLong( intfName )

   @EDTCmd
   def echoCliCmd( self, enable ):
      '''Enable/disable echo of CLI commands'''
      self.dutCli.trace = enable

   @EDTCmd
   def showConnIntfs( self ):
      '''Show connected interfaces'''
      print( "\n".join( self.dutInfo.connectedIntfs() ) )

   @EDTCmd
   def showRxTxIntf( self ):
      '''Show Rx/Tx interface'''
      rxIntf = self.debugCtx.get( "rxIntf" )
      rxPartner = self.getDutMpcDevIntf( rxIntf )
      txIntf = self.debugCtx.get( "txIntf" )
      txPartner = self.getDutMpcDevIntf( txIntf )
      print( f"Rx:{rxIntf} Partner:{rxPartner}\nTx:{txIntf} Partner:{txPartner}" )

   @EDTCmd
   def flapIntf( self, intf, cnt=1, downTime=1.0, upTime=1.0 ):
      '''Flap interface admin state down/up'''
      for i in range( cnt ):
         self.dutCli.configCmd( f"int {intf}; shut" )
         time.sleep( downTime )
         self.dutCli.configCmd( f"int {intf}; no shut" )
         if i < ( cnt - 1 ):
            time.sleep( upTime )

   @EDTCmd
   def showIntfCntr( self, intf=None, excludeIntf="Ma1", nonZero=True, clear=True ):
      '''Show/clear interface counters'''
      # Show interface counters.
      if isinstance( excludeIntf, str ):
         excludeIntf = excludeIntf.split( ';' )
      cmd = f"show int {intf} counters" if intf else "show int counters"
      if nonZero:
         cmd += " | nz"
      out = self.dutCli.showCmd( cmd )
      hdr = None
      for line in out:
         if line.startswith( 'Port ' ):
            hdr = line
            continue
         comps = line.split()
         if len( comps ) >= 2 and comps[ 0 ] not in excludeIntf:
            if hdr:
               print( '\n' + hdr )
               hdr = None
            print( line )
      if clear:
         # Clear counters.
         self.dutCli.privCmd( "clear counters" )
