#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
# Ignore too-broad exceptions for now.
# pylint: disable-msg=W0703
import argparse
import datetime
import pickle
from datetime import datetime
import os
import sys
from sysconfig import get_path
from shutil import copy
import stat
import atexit
import pydoc
import importlib.util
import inspect
import signal
import socket
from traceback import print_exc
import threading
from EDTAccess import ( EDTPlugin, DutInfo, DutCli,
                        traceMsg, traceFileIs, traceLevelIs,
                        TRACE_FATAL, TRACE_ERROR, TRACE_WARN, TRACE_INFO )

#########################################################
# Utility to redirect output to socket in threads
#########################################################

class SysRedirect():
   ''' This class allows streams, such as stdout or stderr, to be
   redirected to a per-thread output file.
   threadingOutFile maps thread ID to output file.
   When the thread finishes, call "deregister" to remove the output
   file for that thread..'''

   def __init__( self, default_file ):
      self.default_file = default_file
      # A dictionary of file pointers for file logging
      self.threadOutputFile = {}

   def register( self, file ):
      # To start redirecting to file
      ident = threading.currentThread().ident
      # Get thread ident
      self.threadOutputFile[ ident ] = file
      # Creating a new file  pointed associated with thread id

   def deregister( self ):
      # Remove key value pair on thread closing
      ident = threading.currentThread().ident
      if ident in self.threadOutputFile:
         del self.threadOutputFile[ ident ]

   def write( self, message ):
      ident = threading.currentThread().ident
      outFile = self.threadOutputFile.get( ident, self.default_file )
      outFile.write( message )

   def flush( self ):
      ident = threading.currentThread().ident
      outFile = self.threadOutputFile.get( ident, self.default_file )
      outFile.flush()

####################################################################################
# Debug Server for EOS
####################################################################################
class EDTServerInterrupt( Exception ):
   pass

class EDTServer:

   def __init__( self ):
      super().__init__()
      # Command Args
      self.args = None
      # Server info: server address for UDS, Host and Port for TCP
      self.serverAddress = '/tmp/edt_socket'
      self.serverHost = 'localhost'
      self.serverPort = None
      self.serverSock = None
      self.serverReset = False
      self.cmdToServerFunc = {
            # built-in commands mapped to EDTserver functions
            "clearEdtCliAlias" : self.clearEdtCliAlias,
            "setEdtCliAlias" : self.setEdtCliAlias,
            "resetDebugCtx" : self.resetDebugCtx,
            "showDebugCtx" : self.showDebugCtx,
            "loadPlugin" : self.loadPlugin,
            "unloadPlugin" : self.unloadPlugin,
            "quitDebug" : self.quitDebug,
            "help" : self.showCmdHelp,
            }
      self.wantToQuit = False
      # Plugin info
      self.pluginDir = "/etc/EDTPlugin"
      self.archiveDir = get_path( "purelib" ) + "/EDTPlugin/"
      self.pluginModTime = {}
      self.pluginObjs = {}
      self.cmdToPluginFunc = {}
      self.pluginFuncToCmd = {}
      sys.path.append( self.pluginDir )
      # Daemon info
      self.fstdin = "/dev/null"
      self.fstdout = "/dev/null"
      self.fstderr = "/dev/null"
      self.pidfile = "/tmp/EDTServer.pid"
      self.traceFilePath = "/tmp/EDTServer.out"
      # Initialize DutAccessorBase info (shared with all plugin classes).
      self.dutName = socket.gethostname()
      self.dutCli = DutCli()
      self.dutInfo = DutInfo( self.dutName )
      self.debugCtx = {}
      self.debugCtxFile = "/tmp/EDTServer.ctx"
      # Client info
      self.clientExecPrefix = "bash edtc"

   def sigUsr1Handler( self, signum, frame ):
      # Handle SIGUSR1 signal sent by "EDTServer.py --reset".
      traceMsg( TRACE_INFO, "received SIGUSR1" )
      self.serverReset = True
      raise EDTServerInterrupt( "SIGUSR1 detected" )

   def sigIntHandler( self, signum, frame ):
      # Handle SIGINT signal (CTRL-C).
      traceMsg( TRACE_INFO, "received SIGINT" )
      raise EDTServerInterrupt( "SIGINT detected" )

   def sendSignal( self, pid, signum ):
      # Send signal to process and return true if successful.
      traceMsg( TRACE_INFO, f"sending signal {signum} to pid {pid}" )
      try:
         os.kill( pid, signum )
      except OSError as e:
         if "No such process" not in e.strerror:
            raise e
         return False
      else:
         return True

   def getModuleFilePath( self, module ):
      # Return file path to plugin module.
      return self.pluginDir + "/" + module + ".py"

   def getPluginModules( self ):
      # Return list of installed modules and modification times in plugin directory.
      if not os.path.isdir( self.pluginDir ):
         traceMsg( TRACE_ERROR, "no plugin directory exists" )
         return {}

      def _moduleName( f ):
         return os.path.basename( f ).split( '.' )[ 0 ]

      def _moduleTime( f ):
         path = os.path.join( self.pluginDir, f )
         return datetime.fromtimestamp( os.path.getmtime( path ) )
      return { _moduleName( file ) : _moduleTime( file )
               for file in os.listdir( self.pluginDir )
               if file.endswith( ".py" ) }

   def getPluginClasses( self, module ):
      # Return list of plugin classes within module.
      return [ cls
               for _, cls in inspect.getmembers( sys.modules[ module ],
                                                 inspect.isclass )
               if cls.__module__ == module and issubclass( cls, EDTPlugin ) ]

   def getPluginCmdFuncs( self, pluginObj ):
      # Return list of bound functions for plugin commands.
      # Must have EDTCmd decorator.
      pluginCls = pluginObj.__class__
      # ignore protected member _original
      # pylint: disable-next=W0212
      return [ func._original.__get__( pluginObj, pluginCls )
               for func in pluginCls.__dict__.values()
               if hasattr( func, "_is_EDTCmd" ) ]

   def getPluginCmdName( self, pluginObj, pluginFunc ):
      # get plugin command.
      cmd = pluginFunc.__name__
      if cmd in self.cmdToPluginFunc or cmd in self.cmdToServerFunc:
         cmd = pluginObj.__class__.__name__ + "." + cmd
      return cmd

   def registerPluginObj( self, module, pluginObj ):
      traceMsg( TRACE_INFO, f'''register plugin class
                {pluginObj.__class__.__name__} in module {module}''' )
      edtCliAlias = self.debugCtx.get( "edtCliAlias" )
      cmds = []
      # Setup plugin object.
      pluginObj.setupPlugin( self.dutName, self.dutInfo, self.dutCli, self.debugCtx,
                             self )
      # Register command functions within plugin object.
      for func in self.getPluginCmdFuncs( pluginObj ):
         # Generate command name.
         cmd = self.getPluginCmdName( pluginObj, func )
         if cmd in self.cmdToPluginFunc:
            traceMsg( TRACE_WARN, f"command '{cmd}' already exists" )
            continue
         # Register command.
         self.cmdToPluginFunc[ cmd ] = func
         self.pluginFuncToCmd[ func ] = cmd
         if not edtCliAlias:
            cmds.append( cmd )
      if cmds:
         # Add CLI aliases associated with commands in this plugin.
         self.dutCli.configCmd( [ f"alias {cmd} {self.clientExecPrefix} {cmd}"
                                  for cmd in cmds ] )

   def unregisterPluginObj( self, module, pluginObj ):
      traceMsg( TRACE_INFO, f'''unregister plugin class
                {pluginObj.__class__.__name__} in module {module}''' )
      edtCliAlias = self.debugCtx.get( "edtCliAlias" )
      cmds = []
      # Cleanup plugin object.
      pluginObj.cleanupPlugin()
      # Unregister command functions within plugin object.
      for func in self.getPluginCmdFuncs( pluginObj ):
         cmd = self.pluginFuncToCmd.get( func )
         if cmd in self.cmdToPluginFunc:
            del self.cmdToPluginFunc[ cmd ]
            if not edtCliAlias:
               cmds.append( cmd )
         if func in self.pluginFuncToCmd:
            del self.pluginFuncToCmd[ func ]
      if cmds:
         # Remove CLI aliases associated with commands in this plugin.
         self.dutCli.configCmd( [ f"no alias {cmd}" for cmd in cmds ] )

   def loadPluginModule( self, module ):
      traceMsg( TRACE_INFO, f"loading plugins from '{module}'" )
      # Load module.
      filePath = self.getModuleFilePath( module )
      spec = importlib.util.spec_from_file_location( module, filePath )
      mymod = importlib.util.module_from_spec( spec )
      sys.modules[ module ] = mymod
      spec.loader.exec_module( mymod )
      # Create/register plugins within this module.
      if module not in self.pluginObjs:
         self.pluginObjs[ module ] = []
      pluginClasses = self.getPluginClasses( module )
      for cls in pluginClasses:
         pluginObj = cls()
         self.pluginObjs[ module ].append( pluginObj )
         self.registerPluginObj( module, pluginObj )

   def unloadPluginModule( self, module ):
      traceMsg( TRACE_INFO, f"unloading plugins from '{module}'" )
      # Unregister/delete plugins.
      for pluginObj in self.pluginObjs.get( module, [] ):
         self.unregisterPluginObj( module, pluginObj )
      if module in self.pluginObjs:
         del self.pluginObjs[ module ]
      # Unload module?
      if module in sys.modules:
         del sys.modules[ module ]

   def syncPlugins( self ):
      # Get currently installed plugins and modification times.
      curModTime = self.getPluginModules()
      if curModTime == self.pluginModTime:
         return

      # Unload/load new modules.
      for name, modTime in curModTime.items():
         prevTime = self.pluginModTime.get( name )
         if prevTime == modTime:
            continue
         if prevTime is not None:
            self.unloadPluginModule( name )
         self.loadPluginModule( name )

      # Unload any remaining stale modules.
      for name in self.pluginModTime:
         if name not in curModTime:
            self.unloadPluginModule( name )

      # Save plugin modification times.
      self.pluginModTime = curModTime

   def showCmdHelp( self, cmd=None ):
      '''Show help on EDT commands'''
      # Merge commands from server and plugins first.
      cmdToFunc = self.cmdToPluginFunc | self.cmdToServerFunc
      if cmd and cmd in cmdToFunc:
         # Show detailed pydoc on this command function.
         func = cmdToFunc[ cmd ]
         if func.__doc__ is not None:
            print( pydoc.render_doc( func ) )
            return
      # Show help summary for commands beginning with prefix.  Summary is the first
      # line of docstring.
      cmd = cmd or ''
      if cmd.endswith( '*' ):
         cmd = cmd[ : -1 ]
      names = [ name for name in cmdToFunc if name.startswith( cmd ) ]
      if not names:
         print( "No EDT commands available" )
         return
      names = sorted( names )
      width = max( len( name ) for name in names )

      def _summary( name ):
         func = cmdToFunc[ name ]
         descr = func.__doc__.split( '\n', 1 )[ 0 ] if func.__doc__ else "No help"
         clsName = func.__self__.__class__.__name__
         return f"{name.ljust(width)}  {descr} [{clsName}]"
      print( "\n" + "\n".join( _summary( name ) for name in names ) )

   def clearEdtCliAlias( self ):
      '''Remove all CLI aliases to EDT commands'''
      # Clear aliases to EDT commands.
      out = self.dutCli.showCmd( f"show alias | grep '{self.clientExecPrefix}'" )
      cmds = [ line.split()[ 0 ] for line in out if self.clientExecPrefix in line ]
      self.dutCli.configCmd( [ f"no alias {cmd}" for cmd in cmds ] )

   def setEdtCliAlias( self, alias ):
      '''Set CLI alias used as a prefix for EDT commands'''
      # Set CLI alias to prefix EDTClient commands.  We use this function to avoid
      # depending on any plugins being installed.
      # Clear any existing CLI aliases which use EDTClient.
      self.clearEdtCliAlias()
      if alias:
         print( f"set EDT prefix CLI alias to '{alias}'" )
         # Save alias in debugCtx so we can restore after replacing config.
         self.debugCtx[ "edtCliAlias" ] = alias
         # Add single CLI alias for EDT command prefix.  Prepend this alias to EDT
         # command when invoking debug commands from CLI session.
         self.dutCli.configCmd( f"alias {alias} {self.clientExecPrefix}" )
      else:
         print( "set CLI aliases for all EDT commands" )
         # Remove alias from debugCtx.
         if "edtCliAlias" in self.debugCtx:
            del self.debugCtx[ "edtCliAlias" ]
         # Add CLI aliases for all built-in and plugin commands.
         cmds = self.cmdToPluginFunc | self.cmdToServerFunc
         self.dutCli.configCmd( [ f"alias {cmd} {self.clientExecPrefix} {cmd}"
                                  for cmd in cmds ] )

   def resetDebugCtx( self ):
      '''Reset debug context (scratchpad)'''
      edtCliAlias = self.debugCtx.get( "edtCliAlias" )
      self.debugCtx.clear()
      if edtCliAlias:
         self.debugCtx[ "edtCliAlias" ] = edtCliAlias

   def showDebugCtx( self, key=None, showAll=False ):
      '''Show keys/values in debug context (scratchpad)'''
      if key:
         if key in self.debugCtx:
            print( f"{key}: {self.debugCtx[ key ]}" )
         else:
            print( f"{key} not found" )
      else:
         for k in sorted( list( self.debugCtx.keys() ) ):
            print( f"{k}: {self.debugCtx[ k ]}" if showAll else k )

   def loadPlugin( self, plugin ):
      '''Load (copy) a plugin from the plugin archive directory'''
      if not os.path.isdir( self.pluginDir ):
         os.mkdir( self.pluginDir )
      if not os.path.isdir( self.archiveDir ):
         print( f"Archive dir at '{self.archiveDir}' is invalid" )
         traceMsg( TRACE_ERROR, f"Archive dir at '{self.archiveDir}' is invalid" )
         return
      plugpath = self.archiveDir + plugin
      if not plugin.endswith( ".py" ):
         plugpath += ".py"
      if os.path.exists( plugpath ):
         copy( plugpath, self.pluginDir )
         self.syncPlugins()
         print( f"Plugin {plugin} has been loaded" )
      else:
         print( f"Plugin {plugin} does not exist" )
         traceMsg( TRACE_ERROR, f"Plugin path {plugpath} does not exist" )

   def unloadPlugin( self, plugin ):
      '''Unload (remove) a plugin from the active plugin directory'''
      if not os.path.isdir( self.pluginDir ):
         print( "No active plugin directory found" )
         traceMsg( TRACE_ERROR, "No active plugin directory found" )
         return
      plugpath = self.pluginDir + "/" + plugin
      if not plugin.endswith( ".py" ):
         plugpath += ".py"
      # Make sure the user isn't trying to remove a path outside the plugin dir
      plugpath = os.path.abspath( plugpath )
      if not plugpath.startswith( self.pluginDir ):
         print( f"Plugin {plugin} not found" )
         traceMsg( TRACE_ERROR, f"Plugin path {plugpath} is invalid" )
         return
      if os.path.exists( plugpath ):
         os.remove( plugpath )
         self.syncPlugins()
         print( f"Plugin {plugin} has been removed" )
      else:
         print( f"Plugin {plugin} does not exist" )
         traceMsg( TRACE_ERROR, f"Plugin {plugin} does not exist" )

   def quitDebug( self, clearAlias=False, resetDebugCtx=False ):
      '''Quit EDT Server'''
      if clearAlias:
         # Remove all EDT client aliases.
         self.clearEdtCliAlias()
      if resetDebugCtx:
         # Reset debug context.
         self.resetDebugCtx()
      # Indicate we want to stop server process by setting flag, so that it doesn't
      # actually stop until all commands in current client request are completed.
      traceMsg( TRACE_WARN, "requesting to stop EDTServer" )
      print( "stopping EDTServer" )
      self.wantToQuit = True

   def execCmd( self, cmdStr ):
      # Execute server or plugin command function.
      cmdStr, argStr = cmdStr.split( '(', 1 )
      argStr = '(' + argStr
      try:
         if cmdStr in self.cmdToServerFunc:
            # Build execStr using built-in server command function
            execStr = f"self.cmdToServerFunc['{cmdStr}']{argStr}"
            exec( execStr )
         elif cmdStr in self.cmdToPluginFunc:
            # Build evalStr using plugin command function
            execStr = f"self.cmdToPluginFunc['{cmdStr}']{argStr}"
            exec( execStr )
         else:
            print( "no command found: " + cmdStr )
      except Exception:
         print_exc()
      return False

   def daemonize( self ):
      # Do first fork.
      try:
         pid = os.fork()
         if pid > 0:
            # exit on parent
            sys.exit( 0 )
      except OSError as e:
         sys.stderr.write( f"fork #1 failed: {e.strerror}\n" )
         sys.exit( 1 )
      # Decouple from parent environment.
      os.chdir( "/" )
      os.setsid()
      os.umask( 0 )
      # Do second fork.
      try:
         pid = os.fork()
         if pid > 0:
            # exit on parent
            sys.exit( 0 )
      except OSError as e:
         sys.stderr.write( f"fork #2 failed: {e.strerror}\n" )
         sys.exit( 1 )
      # Redirect std file descriptors.
      sys.stdout.flush()
      sys.stderr.flush()
      # pylint: disable=R1732
      # With-style ops only useful when file handle used more than once
      si = open( self.fstdin )
      so = open( self.fstdout, 'a+' )
      se = open( self.fstderr, 'a+' )
      os.dup2( si.fileno(), sys.stdin.fileno() )
      os.dup2( so.fileno(), sys.stdout.fileno() )
      os.dup2( se.fileno(), sys.stderr.fileno() )
      # Register exit handler to cleanup.
      atexit.register( self.deletePidFile )

   def createPidFile( self ):
      pid = os.getpid()
      with open( self.pidfile, 'w' ) as f:
         f.write( str( pid ) + "\n" )

   def deletePidFile( self ):
      if os.path.exists( self.pidfile ):
         os.unlink( self.pidfile )

   def getPid( self ):
      # Get server process pid from pidfile.
      pid = None
      try:
         with open( self.pidfile ) as f:
            pid = int( f.read().strip() )
      except OSError:
         pass
      if pid is not None and not self.sendSignal( pid, 0 ):
         # Pid file exist but no process with pid is running.
         traceMsg( TRACE_FATAL, "pid file exist but no running process" )
         pid = None
      return pid

   def loadDebugCtxFile( self ):
      # Load debug context from file, if exists.
      try:
         with open( self.debugCtxFile, 'rb' ) as f:
            self.debugCtx = pickle.load( f, encoding="UTF-8" )
            traceMsg( TRACE_INFO, f"loaded debug context from {self.debugCtxFile}" )
      except OSError:
         traceMsg( TRACE_WARN,
                   f"failed to load debug context to {self.debugCtxFile}" )

   def saveDebugCtxFile( self ):
      # Save debug context to file.
      try:
         with open( self.debugCtxFile, 'wb' ) as f:
            pickle.dump( self.debugCtx, f, protocol=pickle.HIGHEST_PROTOCOL )
            traceMsg( TRACE_INFO, f"saved debug context to {self.debugCtxFile}" )
      except OSError:
         traceMsg( TRACE_WARN,
                   f"failed to save debug context to {self.debugCtxFile}" )

   def removeDebugCtxFile( self ):
      # Delete debug context file if it exists.
      try:
         os.unlink( self.serverAddress )
      except OSError:
         pass

   def openSocket( self ):
      # Make sure the socket does not already exist.
      self.removeSocketFile()
      if not self.args.port:
         # Use Unix local socket file.
         traceMsg( TRACE_INFO, f"creating unix socket: {self.serverAddress}" )
         sock = socket.socket( socket.AF_UNIX, socket.SOCK_STREAM )
         sock.setsockopt( socket.SOL_SOCKET, socket.SO_REUSEADDR, 1 )
         sock.bind( self.serverAddress )
         os.chmod( self.serverAddress, stat.S_IRWXU | stat.S_IRWXG | stat.S_IRWXO )
         sock.listen( 5 )
      else:
         # Use TCP socket with port specified by user
         traceMsg( TRACE_INFO, "creating TCP socket:"
                   f"{self.serverHost}:{self.serverPort}" )
         self.serverPort = int( self.args.port[ 0 ] )
         sock = socket.socket( socket.AF_INET, socket.SOCK_STREAM )
         sock.setsockopt( socket.SOL_SOCKET, socket.SO_REUSEADDR, 1 )
         sock.bind( ( self.serverHost, self.serverPort ) )
         sock.listen( 5 )
      return sock

   def removeSocketFile( self ):
      traceMsg( TRACE_INFO, f"removing socket file {self.serverAddress}" )
      # Delete server socket file if it exists.
      try:
         os.unlink( self.serverAddress )
      except OSError:
         traceMsg( TRACE_WARN, f"socket file not removed: {self.serverAddress}" )

   def terminate( self, exitCode=0 ):
      # Terminate server process.
      # Remove socket file so local clients can't make any further requests.
      self.removeSocketFile()
      # Close server socket.
      if self.serverSock is not None:
         self.serverSock.close()
      # Remove PID file.
      self.deletePidFile()
      if self.serverReset:
         # Remove debug context file on reset request.
         self.removeDebugCtxFile()
      else:
         # Save debug context file.
         self.saveDebugCtxFile()

      # Exit server process.
      traceMsg( TRACE_INFO, "exiting" )
      # ignore protected member _exit
      # pylint: disable-next=W0212
      os._exit( exitCode )

   def stop( self, reset=False ):
      # Remove socket file first so clients can't make any further requests
      # until the server restarts.
      self.removeSocketFile()
      # Kill server process if it exists.
      pid = self.getPid()
      if pid is not None:
         # Send SIGUSR1 to reset/terminate server or SIGINT to just terminate.
         self.sendSignal( pid, signal.SIGUSR1 if reset else signal.SIGINT )
      else:
         print( "PID file not found" )

   def start( self ):
      pid = self.getPid()
      if pid is not None:
         traceMsg( TRACE_WARN, "already running?" )
         sys.exit( 1 )
      # Daemonize server process.
      self.daemonize()
      # Create trace file.
      traceFileIs( self.traceFilePath )
      # Run server main thread.
      self.run()

   def restart( self ):
      self.stop()
      self.start()

   def clientThread( self, conn, addr ):
      if not self.args.pdb:
         # Start redirect of stdout/stderr within this thread to client connection.
         f = conn.makefile( 'w' )
         for sysOut in [ sys.stdout, sys.stderr ]:
            sysOut.register( f )

      # Receive request.
      request = conn.recv( 4096 )

      if request:
         # Execute all commands in request.
         request = request.decode()
         cmds = request.split( '|||' )
         for cmd in cmds:
            self.execCmd( cmd )

      # Flush stdout/stderr for this thread.
      sys.stdout.flush()
      sys.stderr.flush()

      if not self.args.pdb:
         # Stop redirect of stdout/stderr within this thread.
         for sysOut in [ sys.stdout, sys.stderr ]:
            sysOut.deregister()

      # Shut down client connection socket.
      traceMsg( TRACE_INFO, "closing client connection" )
      conn.shutdown( socket.SHUT_RDWR )
      conn.close()

      if self.wantToQuit:
         # Send SIGINT to terminate server process.  We do it this way because we
         # shouldn't call terminate from a child thread.
         self.sendSignal( os.getpid(), signal.SIGINT )

   def run( self ):
      # Write pid to file.
      self.createPidFile()
      # Load debug context from file, if exists.
      self.loadDebugCtxFile()
      # Open the service socket:
      self.serverSock = self.openSocket()
      # Create file to allow redirection per thread.
      sys.stdout = SysRedirect( sys.stdout )
      sys.stderr = SysRedirect( sys.stderr )
      # Main loop.
      while True:
         try:
            # Accept client connection.
            conn, addr = self.serverSock.accept()
            traceMsg( TRACE_INFO, f"accepted client connection from '{addr}'" )

            # Sync all plugins before processing client request.
            try:
               self.syncPlugins()
            except Exception as e:
               traceMsg( TRACE_ERROR, str( e ) )
               connOutFile = conn.makefile( 'w' )
               print_exc( file=connOutFile )
               connOutFile.flush()
               conn.shutdown( socket.SHUT_RDWR )
               conn.close()
               self.terminate( exitCode=1 )

            # Handle each client request in a separate thread.
            # This will also shutdown/close the client connection.
            threading.Thread( target=self.clientThread,
                              args=( conn, addr ) ).start()

         except EDTServerInterrupt:
            # Terminate process when server interrupt signal received.
            self.terminate( exitCode=1 )

   def processCmdOpts( self ):
      # Use argparse to get options.
      parser = argparse.ArgumentParser()
      parser.add_argument( '--start', action='store_true' )
      parser.add_argument( '--stop', action='store_true' )
      parser.add_argument( '--restart', action='store_true' )
      parser.add_argument( '--port', action='append' )
      parser.add_argument( '--reset', action='store_true' )
      parser.add_argument( '--pdb', action='store_true' )
      parser.add_argument( '-v', '--verbose', type=int, default=TRACE_WARN )
      self.args = parser.parse_args()
      traceLevelIs( self.args.verbose )
      if self.args.start:
         self.start()
      elif self.args.stop:
         self.stop()
      elif self.args.restart:
         self.restart()
      elif self.args.reset:
         self.stop( reset=True )
      else:
         # Run server in the foreground.
         if self.args.pdb:
            # pylint: disable=C0415
            # pylint: disable=W1515
            # importing pdb only in debug mode and adding a breakpoint
            import pdb
            pdb.set_trace() # A4NOCHECK
         self.run()

if __name__ == "__main__":
   # Create server.
   myServer = EDTServer()
   # Install signal handlers
   signal.signal( signal.SIGUSR1, myServer.sigUsr1Handler )
   signal.signal( signal.SIGINT, myServer.sigIntHandler )
   # Process command options.
   myServer.processCmdOpts()
