#!/usr/bin/env python3
# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import re
import sys
import argparse
from collections import OrderedDict
from configparser import ConfigParser
from EDTAccess import ( defaultCliUser, defaultCliPasswd,
                        defaultRootUser, defaultRootPasswd,
                        DutCmdSession, shellCmd, errorMsg, grabbedDut )

class ConfigParserMultiValues( OrderedDict ):
   def __setitem__( self, key, value ):
      if key in self and isinstance( value, list ):
         self[ key ].extend( value )
      else:
         # Now, section in configparse won't work because of this super() call
         super().__setitem__( key, value )

   @staticmethod
   def getlist( value ):
      return value.splitlines()

class EDTXfer:
   '''
   Create a EDTXfer session to transfer files between Workspace and Dut
   '''
   def __init__( self ):
      self.exitStatus = 0
      self.dutCmd = None
      self.args = None
      self.srcDir = None
      self.pkgPluginDir = None
      self.initPluginFiles = []
      self.tgtBinDir = None
      self.tgtLibDir = None
      self.tgtPluginDir = None
      self.tgtDutInfoDir = None
      self.dutRootUser = None
      self.dutRootPasswd = None
      self.dutCliUser = None
      self.dutCliPasswd = None
      self.libFiles = None

   def parseCmdLineArgs( self ):
      parser = argparse.ArgumentParser( prog=sys.argv[ 0 ],
            usage='%(prog)s [options]',
            description="Tranfer EOS debug tool components to switch" )
      parser.add_argument( "-a", "--all", action='store_true',
            help="xfer all EDT tool/lib files" )
      parser.add_argument( "-c", "--cli-alias", dest="cliAlias",
            help="create CLI alias for edtc" )
      parser.add_argument( "-d", "--dut",
            help="select dut, if no dut selected, use default grabbed dut" )
      parser.add_argument( "-P", "--package", action="append",
            help="package name containing plugins (can be more than one)" )
      parser.add_argument( "-p", "--plugin", action="append",
            help="file containing EDT plugin (can be more than one)" )
      parser.add_argument( "--upload", action='store_true',
            help="upload modified plugins from dut back to host" )
      parser.add_argument( "--username", default=self.dutRootUser,
            help="username for root bash session on a dut" )
      parser.add_argument( "--password", default=self.dutRootPasswd,
            help="password for root bash session on a dut" )
      self.args = parser.parse_args()

   def parseConfigFile( self ):
      # Get source/target dirs and files from config file or defaults.
      # Config file overrides defaults.
      defaults = { "src_dir" : "/src/Debug/scripts",
                   "pkg_plugin_dir" : "/src/{}/EDTPlugin",
                   "plugin_file" : "",
                   "lib_file" : "/src/Debug/scripts/EDTAccess.py",
                   "tgt_bin_dir" : "/usr/bin",
                   "tgt_lib_dir" : "/usr/lib/python{}/site-packages",
                   "tgt_plugin_dir" : "/etc/EDTPlugin",
                   "tgt_dut_info_dir" : "/mnt/flash",
                   "dut_root_user" : defaultRootUser,
                   "dut_root_passwd" : defaultRootPasswd,
                   "dut_admin_user" : defaultCliUser,
                   "dut_admin_passwd" : defaultCliPasswd,
                   }
      config = ConfigParser( defaults=defaults,
                  strict=False, empty_lines_in_values=False, allow_no_value=True,
                  dict_type=ConfigParserMultiValues,
                  converters={ "list" : ConfigParserMultiValues.getlist } )
      cfgFile = "edt.ini"
      section = "EDTXfer"
      if not os.path.isfile( cfgFile ):
         cfgFile = os.path.join( os.environ.get( "HOME", "./" ), cfgFile )
         if not os.path.isfile( cfgFile ):
            section = "DEFAULT"
      config.read( [ cfgFile ] )
      self.srcDir = str( config.get( section, "src_dir" ) )
      self.libFiles = str( config.get( section, "lib_file" ) )
      self.libFiles = \
            [ fName for fName in self.libFiles.split( '\n' ) if fName ]
      self.pkgPluginDir = str( config.get( section, "pkg_plugin_dir" ) )
      self.initPluginFiles = str( config.get( section, "plugin_file" ) )
      self.initPluginFiles = \
            [ fName for fName in self.initPluginFiles.split( '\n' ) if fName ]
      self.tgtBinDir = str( config.get( section, "tgt_bin_dir" ) )
      self.tgtLibDir = str( config.get( section, "tgt_lib_dir" ) )
      self.tgtPluginDir = str( config.get( section, "tgt_plugin_dir" ) )
      self.tgtDutInfoDir = str( config.get( section, "tgt_dut_info_dir" ) )
      self.dutRootUser = str( config.get( section, "dut_root_user" ) )
      self.dutRootPasswd = str( config.get( section, "dut_root_passwd" ) )
      self.dutCliUser = str( config.get( section, "dut_admin_user" ) )
      self.dutCliPasswd = str( config.get( section, "dut_admin_passwd" ) )

   def isPluginFile( self, fileName ):
      # Check if file is an EDT plugin.
      return os.path.isfile( fileName ) and fileName.endswith( ".py" )

   def getLocalPluginFiles( self ):
      # Get a list of EDT plugin files from packages or other files.
      pluginFiles = []
      if self.args.plugin:
         pluginFiles += self.args.plugin
      if self.args.package:
         for pkg in self.args.package:
            pluginDir = self.pkgPluginDir.format( pkg )
            if not os.path.isdir( pluginDir ):
               errorMsg( pluginDir + " does not exist or is not directory" )
               continue
            for f in os.listdir( pluginDir ):
               localPath = os.path.join( pluginDir, f )
               if self.isPluginFile( localPath ):
                  pluginFiles.append( localPath )
      if not self.args.plugin and not self.args.package:
         pluginFiles = list( self.initPluginFiles )
      if not pluginFiles:
         errorMsg( "no local plugin files found" )
      return list( set( pluginFiles ) )

   def getRemotePluginFiles( self ):
      # Get a list of EDT plugin files on remote dut.
      outbuf, _ = self.dutCmd.execRemoteShellCmd( "ls " + self.tgtPluginDir )
      outbuf = str( outbuf )
      pluginFiles = [ os.path.join( self.tgtPluginDir, fname )
                      for fname in re.findall( r'\n([^|\r]+)', outbuf ) ]
      if not pluginFiles:
         errorMsg( "no remote plugin files found" )
      return pluginFiles

   def downloadPlugins( self ):
      # Copy plugin files from local host to remote dut.
      self.dutCmd.execRemoteShellCmd( "mkdir -p " + self.tgtPluginDir )
      for fileName in self.getLocalPluginFiles():
         localPath = fileName
         remotePath = os.path.join( self.tgtPluginDir, os.path.basename( fileName ) )
         if self.dutCmd.copyFile( localPath, remotePath ):
            self.exitStatus = 1

   def uploadPlugins( self ):
      # Copy plugin files from remote dut to local host.
      localPluginMap = { os.path.basename( fileName ) : fileName
                         for fileName in self.getLocalPluginFiles() }
      for fileName in self.getRemotePluginFiles():
         baseName = os.path.basename( fileName )
         localPath = localPluginMap.get( baseName )
         if not localPath:
            continue
         remotePath = fileName
         if self.dutCmd.copyFile( localPath, remotePath, upload=True ):
            self.exitStatus = 1

   def syncEDT( self ):
      # Update target library dir.
      pyVersion, exitCode = self.dutCmd.execRemoteShellCmd( "python3 -V" )
      assert exitCode in [ 0, None ], f"check python3 on dut failed: {pyVersion}"
      pyVersion = pyVersion.replace( '\r', '' ).replace( '\n', '' )
      match = re.match( r"Python (3\.\d+)", pyVersion )
      assert match, "python3 not found on dut"
      version = match.group( 1 )
      self.tgtLibDir = self.tgtLibDir.format( version )
      # Transfer EDT library modules
      for fname in self.libFiles:
         localPath = fname
         remotePath = os.path.join( self.tgtLibDir, os.path.basename( fname ) )
         if self.dutCmd.copyFile( localPath, remotePath, upload=self.args.upload ):
            self.exitStatus = 1
      # Transfer EDTClient.py
      localPath = os.path.join( self.srcDir, "EDTClient.py" )
      remotePath = os.path.join( self.tgtBinDir, "edtc" )
      if self.dutCmd.copyFile( localPath, remotePath, upload=self.args.upload ):
         self.exitStatus = 1
      # Transfer EDTServer.py
      localPath = os.path.join( self.srcDir, "EDTServer.py" )
      remotePath = os.path.join( self.tgtBinDir, "edts" )
      if self.dutCmd.copyFile( localPath, remotePath, upload=self.args.upload ):
         self.exitStatus = 1

   def getDutInfo( self, dutName ):
      localPath = "/tmp/ArtInfo-" + dutName
      if not os.path.isfile( localPath ):
         print( "generating ArtInfo for", dutName )
         shellCmd( f"Art info -i {dutName} > {localPath}" )
      return localPath

   def syncDutInfo( self, dutName ):
      localPath = self.getDutInfo( dutName )
      remotePath = "/mnt/flash/" + os.path.basename( localPath )
      if self.dutCmd.copyFile( localPath, remotePath, upload=self.args.upload ):
         self.exitStatus = 1

   def syncPlugins( self ):
      if self.args.upload:
         self.uploadPlugins()
      else:
         self.downloadPlugins()

   def createCliAlias( self, dutName ):
      # Create CLI alias for edtc.
      cli = DutCmdSession( dutName, username=self.dutCliUser,
                           password=self.dutCliPasswd )
      aliasCmd = f"bash edtc setEdtCliAlias {self.args.cliAlias}"
      print( cli.doCliCmd( aliasCmd )[ 0 ] )

   def run( self ):
      # Parse configuration file for defaults.
      self.parseConfigFile()
      # Parse command arguments/options (overrides/extends defaults).
      self.parseCmdLineArgs()

      # Get dut for transfer.
      dutName = self.args.dut or grabbedDut()
      if not dutName:
         errorMsg( "no dut specified or grabbed" )
         sys.exit( 1 )

      # Create DUT command session.
      self.dutCmd = DutCmdSession( dutName, username=self.args.username,
                                   password=self.args.password )

      if self.args.all:
         # Sync EDTAccess, EDTClient, EDTServer files.
         self.syncEDT()
         if not self.args.upload:
            # Sync ArtInfo file.
            self.syncDutInfo( dutName )

      # Sync EDTplugin modules.
      self.syncPlugins()

      if self.args.all or self.args.cliAlias:
         # Create Cli aliases.
         self.createCliAlias( dutName )

if __name__ == "__main__":
   edtx = EDTXfer()
   edtx.run()
   sys.exit( edtx.exitStatus )
