# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import json
import copy
import errno
import io
import os
import select

import CliCommon
from CliShellLib import EapiCliConnector
import FastServUtil
import Tracing

traceHandle = Tracing.Handle( 'EapiClientLib' )
log = traceHandle.trace0
warn = traceHandle.trace1
info = traceHandle.trace2
trace = traceHandle.trace3
debug = traceHandle.trace4

# External api to invoke eAPI going directly to the CLI process. This is similiar
# other eAPI examples using jsonrpclib
#    from EapiClientLib import EapiClient
#    eapiClient = EapiClient()
#    print( eapiClient.runCmds( 1, [ "show version" ] ) )

REQUEST_CONTENT = { 'jsonrpc': '2.0',
                    'method': None,
                    'params': None,
                    'id': 'FastCliSocketLib',
                  }

REMOTE_SOCK_READ_SIZE = 65536
SELECT_TIMEOUT = 5

# pylint: disable-msg=protected-access

class EapiException( Exception ):
   def __init__( self, msg=None, code=None, data=None ):
      Exception.__init__( self, msg )
      self.msg = msg
      self.code = code
      self.data = data

   def __str__( self ):
      msg = f'EapiClient error {self.code}: \'{self.msg}\''
      if self.data:
         msg += f' Data: {self.data}'
      return msg

class EapiClient:
   def __init__( self, *args, **kwargs ):
      self.eapiClientCtx_ = _EapiClientCtx( *args, **kwargs )

   def setPrivLevel( self, privLevel ):
      self.self.eapiClientCtx_.setPrivLevel( privLevel )

   def __enter__( self ):
      # when using a context the client is stateful
      self.connect( stateless=False )
      return self

   def __exit__( self, errType, value, tb ):
      self.close()

   def connect( self, *args, **kwargs ):
      self.eapiClientCtx_.connect( *args, **kwargs )

   def close( self, *args, **kwargs ):
      self.eapiClientCtx_.close( *args, **kwargs )

   def __getattr__( self, name ):
      return self.eapiClientCtx_.__getattr__( name )

class _Method:
   def __init__( self, name, eapiClientCtx ):
      self.name_ = name
      self.eapiClientCtx_ = eapiClientCtx

   def _getJsonRpcRequest( self, args, kwargs ):
      request = copy.deepcopy( REQUEST_CONTENT )
      request[ 'method' ] = self.name_
      if kwargs:
         request[ 'params' ] = kwargs
      else:
         request[ 'params' ] = args
      return json.dumps( request )

   def _jsonRpcCall( self, stateless, args, kwargs ):
      jsonRpcRequest = self._getJsonRpcRequest( args, kwargs )
      try:
         self.eapiClientCtx_._sendRequest( jsonRpcRequest )
         responseBuffer = io.BytesIO()
         for i in self.eapiClientCtx_._readStreamedResponse( stateless ):
            responseBuffer.write( i )
         self.eapiClientCtx_._readStatistics()
         response = json.loads( responseBuffer.getvalue() )
         responseBuffer.close()
      except EapiException as e:
         raise e
      except Exception as e: # pylint: disable-msg=W0703
         # pylint: disable-next=raise-missing-from
         raise EapiException( str( e ), CliCommon.JsonRpcErrorCodes.INTERNAL_ERROR )

      if 'error' in response:
         raise EapiException( response[ 'error' ][ 'message' ],
                              response[ 'error' ][ 'code' ],
                              response[ 'error' ].get( 'data', None ) )
      return response

   def _sendRpcRequest( self, stateless, args, kwargs ):
      jsonRpcRequest = None
      if args:
         jsonRpcRequest = args[ 0 ]
      if kwargs:
         jsonRpcRequest = kwargs[ 'jsonRpcRequest' ]
      if jsonRpcRequest is None:
         raise EapiException( '\'jsonRpcRequest\' not specified',
                              CliCommon.JsonRpcErrorCodes.INTERNAL_ERROR )

      try:
         self.eapiClientCtx_._sendRequest( jsonRpcRequest )
      except Exception as e: # pylint: disable-msg=W0703
         # pylint: disable-next=raise-missing-from
         raise EapiException( str( e ), CliCommon.JsonRpcErrorCodes.INTERNAL_ERROR )

      return ( self.eapiClientCtx_._readStreamedResponse( stateless ),
               self.eapiClientCtx_._readStatistics )

   def __call__( self, *args, **kwargs ):
      keepConnection = self.eapiClientCtx_.connected_
      stateless = self.eapiClientCtx_.stateless_
      if not self.eapiClientCtx_.connected_:
         assert self.name_ != 'sendRpcRequest', 'must be connected'
         self.eapiClientCtx_.connect( stateless=True )
      try:
         if kwargs and args:
            raise EapiException( 'JSON-RPC does not support both '
                                 'positional and keyword arguments.',
                                 CliCommon.JsonRpcErrorCodes.INVALID_PARAMS )
         if self.name_ == 'sendRpcRequest':
            return self._sendRpcRequest( stateless, args, kwargs )
         else:
            return self._jsonRpcCall( stateless, args, kwargs )
      finally:
         if not keepConnection:
            self.eapiClientCtx_.close()

class _EapiClientCtx:
   def __init__( self, sysname='ar', disableAaa=False, disableGuards=False,
                 privLevel=1, uid=os.getuid(), gid=os.getgid(), aaaAuthnId=None,
                 realTty=None, logName=None, sshConnection=None ):
      self.sysname_ = sysname
      self.disableAaa_ = disableAaa
      self.disableGuards_ = disableGuards
      self.privLevel_ = privLevel
      self.uid_ = uid
      self.gid_ = gid
      self.env_ = {}
      if aaaAuthnId is not None:
         self.env_[ 'AAA_AUTHN_ID' ] = aaaAuthnId
      if realTty is not None:
         self.env_[ 'REALTTY' ] = realTty
      if logName is not None:
         self.env_[ 'LOGNAME' ] = logName
      if sshConnection is not None:
         self.env_[ 'SSH_CONNECTION' ] = sshConnection

      self.signalSock_ = None
      self.responseSock_ = None
      self.requestSock_ = None
      self.statisticsSock_ = None
      self.connected_ = False
      # this is a stateless or a non-statless client.
      # stateless: Only 1 request/respose is expected and sockets will be closed
      # non-stateless: Multiple requests/responses can be sent, and sockets will be
      #                closed once the client is done sending requests.
      self.stateless_ = None

   def setPrivLevel( self, privLevel ):
      assert not self.connected_, 'Cant change privLevel while connected'
      self.privLevel_ = privLevel

   def connect( self, stateless=False ):
      assert not self.connected_
      assert self.stateless_ is None
      self.connected_ = True
      self.stateless_ = stateless
      cliConnector = EapiCliConnector( stateless=stateless )
      # pylint: disable-next=consider-using-f-string
      argv = [ '-p=%s' % self.privLevel_, '-s=%s' % self.sysname_ ]
      if self.disableAaa_:
         argv.append( '-A' )
      if self.disableGuards_:
         argv.append( '-G' )
      signalSock, responseSock, requestSock, statisticsSock = \
            cliConnector.connectToBackend( self.sysname_, argv, self.env_,
                                           self.uid_, self.gid_ )
      self.signalSock_ = signalSock
      self.responseSock_ = responseSock
      self.requestSock_ = requestSock
      self.statisticsSock_ = statisticsSock

   def close( self ):
      assert self.connected_
      assert self.stateless_ is not None
      self.connected_ = False
      self.stateless_ = None

      # we are going to block until the front-end has disconnected
      self.requestSock_.close()
      self.requestSock_ = None
      self.responseSock_.close()
      self.responseSock_ = None
      self.statisticsSock_.close()
      self.statisticsSock_ = None
      self.signalSock_.close()
      self.signalSock_ = None

   def __getattr__( self, name ):
      return _Method( name, self )

   def _sendRequest( self, jsonRpcRequest ):
      if isinstance( jsonRpcRequest, str ):
         jsonRpcRequest = jsonRpcRequest.encode()
      FastServUtil.writeBytes( self.requestSock_, jsonRpcRequest )

   def _socketReadable( self, sock ):
      while True:
         try:
            filesReadyToRead, _, _ = select.select( [ sock ], [], [], 0 )
            return sock in filesReadyToRead
         except OSError as e:
            if e.args[ 0 ] in ( errno.EINTR, errno.EAGAIN ):
               continue
            raise

   def _readStreamedResponse( self, stateless ):
      try:
         assert self.statisticsSock_.fileno() > 0
         assert self.responseSock_.fileno() > 0
         assert self.signalSock_.fileno() > 0
         socksToRead = [ self.statisticsSock_, self.responseSock_, self.signalSock_ ]
         if stateless:
            # if stateless don't monitor the statisticsSock
            socksToRead.remove( self.statisticsSock_ )
         while True:
            try:
               filesReadyToRead, _, _ = select.select( socksToRead,
                                                       [], [],
                                                       SELECT_TIMEOUT )
            except OSError as e:
               if e.args[ 0 ] in ( errno.EINTR, errno.EAGAIN ):
                  # If we get interrupted we just go back to where we were before
                  continue
               raise

            if stateless:
               # in the stateless case read from the response socket until it is
               # closed, and the signal socket has been written to
               if self.signalSock_ in filesReadyToRead:
                  # the response on the signal socket doesn't matter. As long as it's
                  # readable is all that really matters to say that the frontend
                  # has written and closed the signal socket.
                  self.signalSock_.recv( 1 )
                  socksToRead.remove( self.signalSock_ )
               if self.responseSock_ in filesReadyToRead:
                  response = self.responseSock_.recv( REMOTE_SOCK_READ_SIZE )
                  if not response:
                     socksToRead.remove( self.responseSock_ )
                  yield response
               if not socksToRead:
                  # responseSock has been read and closed, and the signalSock
                  # has been read and closed as well. So function is done
                  return
            else:
               if self.responseSock_ in filesReadyToRead:
                  trace( 'responseSock ready to read' )
                  response = self.responseSock_.recv( REMOTE_SOCK_READ_SIZE )
                  if not response:
                     trace( 'responseSock is empty, breaking' )
                     return
                  yield response
               elif ( self.statisticsSock_ in filesReadyToRead or
                      self.signalSock_ in filesReadyToRead ):
                  trace( 'signalSock or statisticsSock ready to read' )
                  # this means that the backend is done writting either because it
                  # wrote to the statisticsSock or because it closed statisticsSock
                  # or statisticsSock. This means that the backend is done writting
                  # in either case

                  # In the non-stateless sessions the self.responseSock_
                  # doesn't close, so it might still be readable. This is a very
                  # rare condition.
                  # Drain any data from the socket that is immediately readable.
                  while self._socketReadable( self.responseSock_ ):
                     response = self.responseSock_.recv( REMOTE_SOCK_READ_SIZE )
                     if not response:
                        trace( 'responseSock is empty, breaking' )
                        return
                     yield response
                  return
      except Exception as e: # pylint: disable-msg=W0703
         # pylint: disable-next=raise-missing-from
         trace( '_readStreamedResponse exception seen', e )
         raise EapiException( str( e ), CliCommon.JsonRpcErrorCodes.INTERNAL_ERROR )

   def _readStatistics( self ):
      try:
         requestCount = FastServUtil.readInteger( self.statisticsSock_ )
         commandCount = FastServUtil.readInteger( self.statisticsSock_ )
         return requestCount, commandCount
      except Exception as e: # pylint: disable-msg=W0703
         # pylint: disable-next=raise-missing-from
         raise EapiException( str( e ), CliCommon.JsonRpcErrorCodes.INTERNAL_ERROR )
