# Copyright (c) 2019 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import json

class EapiWebsocketException( Exception ):
   def __init__( self, msg, code, data=None ):
      Exception.__init__( self, msg )
      self.code = code
      self.data = data

   def __str__( self ):
      # pylint: disable-next=no-member
      msg = f'EapiWebsocket error {self.code}: {self.message!r}'
      if self.data:
         msg += ' Data: %s' % self.data # pylint: disable=consider-using-f-string
      return msg

class _Method:
   def __init__( self, name, ws ):
      self.name_ = name
      self.ws_ = ws

   def __call__( self, *args, **kwargs ):
      if kwargs and args:
         raise EapiWebsocketException( msg='JSON-RPC does not support both '
                                       'positional and keyword arguments.',
                                       code=-32602 )
      request = { 'jsonrpc': '2.0' }
      request[ 'id' ] = 'EapiWebsocketClient'
      request[ 'method' ] = self.name_
      request[ 'params' ] = kwargs if kwargs else args
      try:
         # Request Format: { 'request': jsonrpcRequest }
         # Response Format: { 'response': jsonrpcResponse }
         self.ws_.send( json.dumps( { 'request': request } ) )
         response = self.ws_.recv()
         response = json.loads( response )
         response = response[ 'response' ]
      except Exception as e: # pylint: disable-msg=bare-except
         # pylint: disable-next=raise-missing-from
         raise EapiWebsocketException( msg=str( e ), code=1001 )

      if 'error' in response:
         raise EapiWebsocketException( msg=response[ 'error' ][ 'message' ],
                                       code=response[ 'error' ][ 'code' ],
                                       data=response[ 'error' ].get( 'data' ) )
      return response

class _MethodObj:
   def __init__( self, ws ):
      self.ws_ = ws

   def __getattr__( self, name ):
      return _Method( name, self.ws_ )

class EapiWebsocketClient:
   def __init__( self, ws, host, username, password, secure=True ):
      self.ws_ = ws
      self.username_ = username
      self.password_ = password
      self.connected_ = False
      protocol = 'wss' if secure else 'ws'
      self.url_ = f'{protocol}://{host}/command-api-websocket'

   def connect( self ):
      if self.connected_:
         raise EapiWebsocketException( code=1000, msg='Client is already connected' )
      self.ws_.connect( self.url_ )
      negotiationInfo = { 'connectionType': 'jsonrpc',
                          'authentication': { 'username': self.username_,
                                              'password': self.password_ } }
      self.ws_.send( json.dumps( negotiationInfo ) )
      authResult = json.loads( self.ws_.recv() )
      if authResult[ 'authentication' ][ 'status' ] != 'success':
         self.ws_.close()
         raise EapiWebsocketException(
                  msg=authResult[ 'authentication' ][ 'message' ],
                  code=1005 )
      self.connected_ = True

   def disconnect( self ):
      if not self.connected_:
         raise EapiWebsocketException( code=1000, msg='Client is not connected' )
      self.ws_.close()
      self.connected_ = False

   def __enter__( self ):
      self.connect()
      return _MethodObj( self.ws_ )

   def __exit__( self, errType, value, tb ):
      self.disconnect()

   def __getattr__( self, name ):
      return _Method( name, self.ws_ )
