# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from ForwardingHelper import (
      forwardingHelperFactory,
      RouteOrFecChangeReactor,
   )
from IpLibConsts import DEFAULT_VRF
from FibUtils import (
      forwardingStatusInfo,
      routeStatusInfo,
   )
import Cell
import SmashLazyMount
import SharedMem
import Smash
import Tac
import Tracing
from TypeFuture import TacLazyType
from EbraTestBridgePythonImpl import getPythonIntoCppShimContext

# Following import is needed because we mount arp/... in sysdb
# pkgdeps: rpm Arp-lib

handle = Tracing.Handle( 'EbraTestBridgeFwd' )
t1 = handle.trace1
t2 = handle.trace2

TunnelProgrammingStatus = TacLazyType( "Tunnel::Hardware::TunnelProgrammingStatus" )
TunnelStatus = Tac.Type( 'Tunnel::Hardware::TunnelStatus' )
clientMountName = 'etba'
TunnelTableIdentifier = TacLazyType( "Tunnel::TunnelTable::TunnelTableIdentifier" )
TunnelTableMounter = TacLazyType( "Tunnel::TunnelTable::TunnelTableMounter" )
TunnelType = TacLazyType( "Tunnel::TunnelTable::TunnelType" )
TunnelViaStatus = TacLazyType( 'Tunnel::Hardware::TunnelViaStatus' )
NexthopGroupSm = Tac.Type( 'Ira::NexthopGroupSm' )

class AllVrfStatusLocalReactor( Tac.Notifiee ):
   notifierTypeName = 'Ip::AllVrfStatusLocal'

   def __init__( self, bridge, avsl ):
      self._bridge = bridge
      self._vrfReactor = {}
      Tac.Notifiee.__init__( self, avsl )
      for vrfName in avsl.vrf:
         self.handleVrf( vrfName )

   def bridge( self ):
      return self._bridge

   @Tac.handler( 'vrf' )
   def handleVrf( self, vrfName ):
      func = 'AllVrfStatusLocalReactor.handleVrf'
      avsl = self.notifier_
      create = vrfName in avsl.vrf
      t1( func, 'vrfName', vrfName, 'create', create )
      # mount routingStatus for the vrf
      bridge = self.bridge()
      if create and vrfName != DEFAULT_VRF:
         # mount the smash paths
         shmemEm = SharedMem.entityManager( sysdbEm=bridge.em() )
         routeInfo = routeStatusInfo( 'keyshadow' )
         shmemEm.doMount( "routing/vrf/status/" + vrfName,
                          "Smash::Fib::RouteStatus", routeInfo )
         shmemEm.doMount( "routing6/vrf/status/" + vrfName,
                          "Smash::Fib6::RouteStatus", routeInfo )

         sEm = bridge.sEm()
         routingStatus = sEm.getTacEntity( 'routing/vrf/status/' + vrfName )
         routing6Status = sEm.getTacEntity( 'routing6/vrf/status/' + vrfName )

         bridge.vrfRoutingStatus_[ vrfName ] = routingStatus
         bridge.vrfRouting6Status_[ vrfName ] = routing6Status
         # create the tries
         trie = Tac.root.newEntity( "Routing::Trie", "trie" + vrfName )
         v6trie = Tac.root.newEntity( "Routing6::Trie", "v6trie" + vrfName )
         # pylint: disable=unused-variable
         trieBuilder = Tac.newInstance( "Routing::TrieBuilder", routingStatus, trie )
         v6trieBuilder = Tac.newInstance( "Routing6::TrieBuilder",
                                          routing6Status, v6trie )
         bridge.vrfTrie_[ vrfName ] = trie
         bridge.vrfTrie6_[ vrfName ] = v6trie
         bridge.vrfTrieBuilder_[ vrfName ] = trieBuilder
         bridge.vrfTrie6Builder_[ vrfName ] = v6trieBuilder
         bridge.vrfRouteOrFecChangeReactor[ vrfName ] = \
               RouteOrFecChangeReactor( trie,
                                        v6trie, bridge.forwardingStatus_,
                                        bridge.forwarding6Status_,
                                        bridge.forwardingGenStatus_,
                                        None )
      elif not create and vrfName != DEFAULT_VRF:
         del bridge.vrfTrie_[ vrfName ]
         del bridge.vrfTrie6_[ vrfName ]
         del bridge.vrfTrieBuilder_[ vrfName ]
         del bridge.vrfTrie6Builder_[ vrfName ]
         del bridge.vrfRouteOrFecChangeReactor[ vrfName ]
         del bridge.vrfRoutingStatus_[ vrfName ]
         del bridge.vrfRouting6Status_[ vrfName ]
         # delete the tries
         trie = Tac.root.deleteEntity( "trie" + vrfName )
         v6trie = Tac.root.deleteEntity( "v6trie" + vrfName )

def forwardingBridgeInit( bridge, clientName=clientMountName ):
   t2( 'forwardingBridgeInit' )

   em = bridge.em()

   shim = getPythonIntoCppShimContext()
   if shim:
      # Other python plugins use this
      bridge.fecModeStatus = shim.arfaRoot_.arfaPlugins.plugin[
         "CoreRouting" ].fecModeStatus
   else:
      # in btest
      l3Config = em.entity( 'l3/config' )
      bridge.fecModeStatus = Tac.newInstance( 'Smash::Fib::FecModeStatus', 'fms' )
      # fecModeSm is instantiated to generate the correct fecMode in fecModeStatus
      _ = Tac.newInstance( 'Ira::FecModeSm', l3Config, bridge.fecModeStatus )

   bridge.routingVrfInfo_ = em.entity( 'routing/vrf/routingInfo/status' )
   bridge.routing6VrfInfo_ = em.entity( 'routing6/vrf/routingInfo/status' )
   # pylint: disable-next=consider-using-f-string
   bridge.nhrConfig = em.entity( 'l3/nexthop-resolver/config/%s' %
                                 clientName )
   # pylint: disable-next=consider-using-f-string
   bridge.nhrStatus = em.entity( 'l3/nexthop-resolver/status/%s' %
                                 clientName )
   sEm = bridge.sEm()
   bridge.tunnelRib = sEm.doMount(
      'tunnel/tunnelRibs/status/0', 'Tunnel::TunnelTable::TunnelRib',
      Smash.mountInfo( 'keyshadow' ) )
   bridge.intfConfigDir_ = em.entity( 'l3/intf/config' )
   bridge.ethIntfStatusDir_ = em.entity( 'interface/status/eth/intf' )

   bridge.tunnelTables = dict() # pylint: disable=use-dict-literal
   for tableId in TunnelTableIdentifier.attributes:
      if tableId != TunnelTableIdentifier.invalidTunnelTable:
         tableInfo = TunnelTableMounter.getMountInfo( tableId ).tableInfo
         tunnelTable = SmashLazyMount.mount(
            em, tableInfo.mountPath, tableInfo.tableType,
            SmashLazyMount.mountInfo( "keyshadow" ) )
         bridge.tunnelTables.setdefault( tunnelTable.tableType, [] ).append(
            tunnelTable )

   bridge.srTeSegmentListTunnelTable = (
      bridge.tunnelTables[ TunnelType.srTeSegmentListTunnel ][ 0 ] )

   bridge.tunnelFib_ = sEm.getTacEntity( 'tunnel/tunnelFib' )
   bridge.routingStatus_ = sEm.getTacEntity( 'routing/status' )
   bridge.routing6Status_ = sEm.getTacEntity( 'routing6/status' )
   if bridge.fecModeStatus.fecMode == 'fecModeUnified':
      bridge.forwardingStatus_ = sEm.getTacEntity( 'forwarding/unifiedStatus' )
      bridge.forwarding6Status_ = sEm.getTacEntity( 'forwarding6/unifiedStatus' )
      bridge.forwardingGenStatus_ = sEm.getTacEntity( 'forwardingGen/unifiedStatus' )
      bridge.vrfNameStatus_ = em.entity( Cell.path( 'vrf/vrfNameStatus' ) )
   else:
      bridge.forwardingStatus_ = sEm.getTacEntity( 'forwarding/status' )
      bridge.forwarding6Status_ = sEm.getTacEntity( 'forwarding6/status' )
      bridge.forwardingGenStatus_ = sEm.getTacEntity( 'forwardingGen/status' )
      bridge.vrfNameStatus_ = None
   bridge.srTeForwardingStatus_ = sEm.getTacEntity( 'forwarding/srte/status' )
   bridge.nhgEntryStatus_ = sEm.getTacEntity(
      'routing/nexthopgroup/entrystatus' )
   bridge.arpSmash_ = sEm.getTacEntity( 'arp/status' )

   bridge.trie = Tac.root.newEntity( "Routing::Trie", "trie" )
   bridge.trieBuilder = Tac.newInstance( "Routing::TrieBuilder",
                bridge.routingStatus_, bridge.trie )
   bridge.v6trie = Tac.root.newEntity( "Routing6::Trie", "v6trie" )
   bridge.v6trieBuilder = Tac.newInstance( "Routing6::TrieBuilder",
                bridge.routing6Status_, bridge.v6trie )
   bridge.brStatus = bridge.brStatus_
   bridge.brConfig = em.entity( 'bridging/config' )
   bridge.pwRouteIndex = {}
   bridge.routeOrFecChangeReactor_ = RouteOrFecChangeReactor( bridge.trie,
                           bridge.v6trie, bridge.forwardingStatus_,
                           bridge.forwarding6Status_,
                           bridge.forwardingGenStatus_,
                           None )
   bridge.vrfRoutingStatus_ = {}
   bridge.vrfRouting6Status_ = {}
   bridge.vrfTrie_ = {}
   bridge.vrfTrie6_ = {}
   bridge.vrfTrieBuilder_ = {}
   bridge.vrfTrie6Builder_ = {}
   bridge.vrfRouteOrFecChangeReactor = {}
   bridge.vrfRoutingStatus_[ DEFAULT_VRF ] = bridge.routingStatus_
   bridge.vrfRouting6Status_[ DEFAULT_VRF ] = bridge.routing6Status_
   bridge.vrfTrie_[ DEFAULT_VRF ] = bridge.trie
   bridge.vrfTrie6_[ DEFAULT_VRF ] = bridge.v6trie
   bridge.vrfTrieBuilder_[ DEFAULT_VRF ] = bridge.trieBuilder
   bridge.vrfTrie6Builder_[ DEFAULT_VRF ] = bridge.v6trieBuilder

   bridge.vrfRouteOrFecChangeReactor[ DEFAULT_VRF ] = \
            RouteOrFecChangeReactor( bridge.trie,
                                     bridge.v6trie, bridge.forwardingStatus_,
                                     bridge.forwarding6Status_,
                                     bridge.forwardingGenStatus_,
                                     None )

   bridge.fwdHelper = forwardingHelperFactory( bridge )


   avsl = bridge.em().entity( Cell.path( 'ip/vrf/status/local' ) )
   bridge.vrfRouteStatusSm = AllVrfStatusLocalReactor( bridge, avsl )

def mountNexthopResolver( mounter, clientName=clientMountName ):
   # pylint: disable-next=consider-using-f-string
   mounter.mount( 'l3/nexthop-resolver/config/%s' % clientName,
                  'Routing::NexthopConfig', 'w' )
   # pylint: disable-next=consider-using-f-string
   mounter.mount( 'l3/nexthop-resolver/status/%s' % clientName,
                  'Routing::NexthopStatus', 'r' )

def forwardingAgentInit( em ):
   t2( 'forwardingAgentInit' )

   sEm = SharedMem.entityManager( sysdbEm=em )
   routeInfo = routeStatusInfo( 'keyshadow' )
   fwdInfo = forwardingStatusInfo( 'keyshadow' )
   sEm.doMount( "routing/status", "Smash::Fib::RouteStatus", routeInfo )
   sEm.doMount( "routing6/status", "Smash::Fib6::RouteStatus", routeInfo )
   sEm.doMount( 'forwarding/unifiedStatus', "Smash::Fib::ForwardingStatus", fwdInfo )
   sEm.doMount( 'forwarding6/unifiedStatus', "Smash::Fib6::ForwardingStatus",
                fwdInfo )
   sEm.doMount( 'forwardingGen/unifiedStatus', "Smash::FibGen::ForwardingStatus",
                fwdInfo )
   sEm.doMount( 'forwarding/status', "Smash::Fib::ForwardingStatus", fwdInfo )
   sEm.doMount( 'forwarding6/status', "Smash::Fib6::ForwardingStatus", fwdInfo )
   sEm.doMount( 'forwardingGen/status', "Smash::FibGen::ForwardingStatus", fwdInfo )
   sEm.doMount( 'forwarding/srte/status',
                'Smash::Fib::ForwardingStatus',
                Smash.mountInfo( 'reader' ) )
   sEm.doMount( 'arp/status', "Arp::Table::Status", Smash.mountInfo( 'shadow' ) )
   sEm.doMount( 'tunnel/tunnelFib', 'Tunnel::TunnelFib::TunnelFib',
                Smash.mountInfo( 'keyshadow' ) )
   NexthopGroupSm.smashMountNhgStatus( em.cEntityManager(), True )

   tunProgWriterInfo = Smash.mountInfo( 'writer', [ ( "tunnelStatus", 1024 ) ] )
   sEm.doMount( TunnelProgrammingStatus.mountPath,
                "Tunnel::Hardware::TunnelProgrammingStatus",
                tunProgWriterInfo )

   em.mount( 'l3/intf/config', 'L3::Intf::ConfigDir', 'r' )
   em.mount( 'interface/status/eth/intf', 'Interface::EthIntfStatusDir', 'r' )
   em.mount( Cell.path( 'vrf/vrfNameStatus' ),
             'Vrf::VrfIdMap::NameToIdMapWrapper', 'r' )

   em.mount( 'l3/config', 'L3::Config', 'r' )

   em.mount( 'routing/hardware/nexthopgroup/status',
             'Routing::Hardware::NexthopGroupStatus', 'r' )
   em.mount( 'routing/vrf/routingInfo/status', 'Tac::Dir', 'r' )
   em.mount( 'routing6/vrf/routingInfo/status', 'Tac::Dir', 'r' )

   # Ira L3 Resolver etba config and status
   mountNexthopResolver( em )

def Plugin( ctx ):
   # In breadth tests, the ctx will be None, ignore it in this case
   if ctx is None:
      return

   # forwarding handlers
   ctx.registerBridgeInitHandler( forwardingBridgeInit )
   ctx.registerAgentInitHandler( forwardingAgentInit )
