#!/usr/bin/env python3
# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable-msg=ungrouped-imports
from Arnet import PktParserTestLib
from EbraTestBridgeLib import (
   applyVlanChanges,
   PKTEVENT_ACTION_ADD,
   PKTEVENT_ACTION_REMOVE,
)
from ForwardingHelper import (
      Af,
      noMatchNexthopInfo,
   )
from EbraTestBridgePlugin import FwdIntfEtba
from EbraTestBridgePlugin import MplsL3EvpnEtbaDecap
from EbraTestBridgePlugin import MplsL3EvpnEtbaEncap
from EbraTestBridgePlugin import MplsMldpEtbaDecap
from EbraTestBridgePlugin.EtbaPacketTracer import packetTracerSmFactory
from EbraTestBridgePlugin.FwdIntfEtba import FwdIntfDevice
from EbraTestBridgePlugin.MplsEtbaNhgHardwareStatusSm import (
      mplsNhgHardwareStatusSmFactory,
   )
from Arnet.PktParserTestLib import parseDataPacket
from IpLibConsts import DEFAULT_VRF
from MplsEtbaForwardingHelper import mplsEtbaForwardingHelperFactory
from MplsEtbaLib import (
      computeFlowOrEntropyLabel,
      ecmpHash,
      ELI,
      getHashInfo,
      getIntfVlan,
      getTunnelEntry,
      removeImpNullFromLabelStack,
      BUD_ROLE_SPL_PKT,
   )
from Arnet.MplsLib import (
      tcToMplsCosStatic,
      constructMplsHeader,
   )
from TypeFuture import TacLazyType
import copy
from MplsPktHelper import isMpls # pylint: disable=no-name-in-module
import QuickTrace
import re
import SharedMem
import Smash
import Tac
import Toggles.MplsEtbaToggleLib
import Tracing
import FibUtils

# Following import is needed because we mount arp/... in sysdb
# pkgdeps: rpm Arp-lib

handle = Tracing.Handle( 'EbraTestBridgeMpls' )
t0 = handle.trace0
t1 = handle.trace1
t2 = handle.trace2
t5 = handle.trace5
t8 = handle.trace8
t9 = handle.trace9

qtInitialized = False
inArfaMode = None

def qTrace():
   '''Initializes QuickTrace once, return the module itself'''
   if inArfaMode:
      # Fast etba enables this itself, and when we call this here the 80 char limit
      # we had set gets reset to 24
      return QuickTrace
   global qtInitialized
   if not qtInitialized:
      QuickTrace.initialize( 'etba-routingHandlers.qt' )
      qtInitialized = True
   return QuickTrace

def qv( arg ):
   return qTrace().Var( arg )

def qt0( *args ):
   # Use tracef with frameOffset=0 (second argument) instead of trace0 so that
   # QuickTrace finds the correct stack frame for the message (the caller of qt0)
   # and allocates a message id per message.
   # Otherwise it thinks that qt0 is the logger and only allocates a single message
   # id for all messages, resulting a corrupted qt file.
   qTrace().tracef( 0, 0, args )

decapHandler = 'mplsPwDecapHandler:'
pushHandler = 'mplsPushRoutingHandler:'
pwHandler = 'mplsPwRoutingHandler:'
swapHandler = 'mplsSwapPopRoutingHandler:'

ethHdrLen = 14

AdjacencyType = TacLazyType( 'Mpls::AdjacencyType' )
DynamicTunnelIntfId = TacLazyType( 'Arnet::DynamicTunnelIntfId' )
EthAddr = TacLazyType( 'Arnet::EthAddr' )
IntfEncap = TacLazyType( 'Ebra::IntfEncap' )
LfibViaSetType = TacLazyType( 'Mpls::LfibViaSetType' )
LfibViaType = TacLazyType( 'Mpls::LfibViaType' )
MplsLabel = TacLazyType( 'Arnet::MplsLabel' )
MplsLabelAction = TacLazyType( 'Arnet::MplsLabelAction' )
MplsRouteKeyAndMetric = TacLazyType( 'Mpls::RouteKeyAndMetric' )
MplsVia = TacLazyType( 'Tunnel::TunnelTable::MplsVia' )
PseudowireDot1qTagOperation = TacLazyType( 'Pseudowire::Dot1qTagOperation' )
PseudowireIntfId = TacLazyType( 'Arnet::PseudowireIntfId' )
PseudowireType = TacLazyType( 'Pseudowire::PseudowireType' )
RouteKey = TacLazyType( 'Mpls::RouteKey' )
RsvpLerTunnelTableEntry = TacLazyType(
   'Tunnel::TunnelTable::RsvpLerTunnelTableEntry' )
SubIntfId = TacLazyType( 'Arnet::SubIntfId' )
TunnelId = TacLazyType( 'Tunnel::TunnelTable::TunnelId' )
VlanId = TacLazyType( 'Bridging::VlanId' )
VlanTag = TacLazyType( 'Ebra::VlanTag' )
VlanTagEncap = TacLazyType( 'Ebra::VlanTagEncap' )

ethType_ = None

def EthType():
   global ethType_
   if ethType_ is None:
      ethType_ = Tac.Type( 'Arnet::EthType' )
   return ethType_

def intfIdForPseudowire( parentIntf, vlanIdOuter, vlanIdInner=None ):
   if vlanIdInner is None:
      intfIdValue = PseudowireIntfId.intfIdFromIntfIdVlan( parentIntf, vlanIdOuter )
   else:
      intfIdValue = PseudowireIntfId.intfIdFromIntfIdVlans( parentIntf, vlanIdOuter,
                                                            vlanIdInner )
   return Tac.Value( 'Arnet::IntfId', intfIdValue )

impNull_ = None

def impNull():
   global impNull_
   if impNull_ is None:
      impNull_ = MplsLabel.implicitNull
   return impNull_

class TtlMode:
   pipe = 'pipe'
   uniform = 'uniform'
   undefinedTtlMode = 'undefinedTtlMode'

class PayloadType:
   autoDecide = 'autoDecide'
   ipv4 = 'ipv4'
   ipv6 = 'ipv6'
   mpls = 'mpls'
   undefinedPayload = 'undefinedPayload'

def resolveMplsPushNexthop( bridge, dstIp, intf=None ):
   func = 'resolveMplsPushNexthop:'

   t8( func, 'resolving dstIp', dstIp )
   helper = bridge.fwdHelper
   nexthopInfo = helper.getResolvedNexthopInfo( dstIp, intf )

   # If no labelStack, route is not mpls push and should be handled by kernel instead
   if nexthopInfo.labelStack is None:
      t8( func, 'invalid via: no label configured' )
      return noMatchNexthopInfo

   return nexthopInfo

def lookupPkt( bridge, topLabel ):
   ''' Return viaSetType and Mpls::Hardware::ViaList if there is a route for
    the topLabel, or None, None.'''
   func = 'lookupPkt'
   hwStatus = bridge.mplsHwStatus_
   routeKey = RouteKey.fromLabel( topLabel )
   route = hwStatus.route.get( routeKey )
   if not route:
      t8( func, 'no hwStatus route for routeKey', routeKey )
      return None, None
   adj = hwStatus.adjacencyBase( route.adjBaseKey )
   if adj is None:
      t8( func, 'no adjacency for route key', route.adjBaseKey )
      return None, None
   t8( func, 'adjacency type is', adj.viaSetType )
   if adj.usingBackup:
      viaList = list( adj.backupVia.keys() )
   else:
      viaList = list( adj.via.keys() )
   if not viaList:
      t8( func, 'no vias for adjacency', route.adjBaseKey )
      return None, None

   if viaList[ 0 ].viaType == LfibViaType.viaTypeExternalFec:
      # Pass Ext FECs into their own handler to filter away invalid labelActions
      t8( 'using ext FEC for adjacency', route.adjBaseKey )
      lfibRoute = bridge.lfib_.lfibRoute.get( routeKey )
      if not lfibRoute:
         t8( 'ext route exists, but route not found in the lfib', route.routeKey )
         return None, None

      viaSet = bridge.lfib_.viaSet.get( lfibRoute.viaSetKey )
      if not viaSet:
         t8( 'Route exists in the lfib, but viaSet not found. RouteKey:',
             lfibRoute.key, 'viaSetKey', lfibRoute.viaSetKey )
         return None, None

      extInfo = bridge.lfib_.extFecVia.get( viaSet.viaKey[ 0 ] )
      if not extInfo:
         t8( 'route exists, but no adj for adjacency', route.adjBaseKey )
         return None, None

      return adj.viaSetType, bridge.fwdHelper.extFecViaList( extInfo )

   return adj.viaSetType, viaList

def _newEthType( payloadType, pkt, mplsHdr ):
   func = '_newEthType:'
   # Pop all MPLS labels, output frame is IPv4/IPv6
   if payloadType == PayloadType.ipv4:
      return 'ethTypeIp'
   elif payloadType == PayloadType.ipv6:
      return 'ethTypeIp6'
   elif payloadType == PayloadType.autoDecide:
      verByte = pkt.stringValue[ mplsHdr.offset + 4 ] & 0xf0
      if verByte == 0x40:
         return 'ethTypeIp'
      elif verByte == 0x60:
         return 'ethTypeIp6'
      else:
         t8( func, 'unknown version nibble', hex( verByte ) )
         qt0( swapHandler, 'unknown version nibble', qv( hex( verByte ) ) )
         return None
   else:
      t8( func, 'wrong payload type', payloadType )
      qt0( swapHandler, 'wrong payload type', qv( payloadType ) )
      return None

   assert False, "unreachable code, payloadType={}, pkt={}".format(
         payloadType, repr( pkt ) )

class PktUpdateType:
   none = 'none'
   mpls = 'mpls'
   ip = 'ip'

def generateOutFrameForMplsPkt( inPkt, ethHdr, ethDot1QHdr, mplsHdr, dstMac,
                                srcMac, labelStack, labelAction,
                                payloadType=PayloadType.undefinedPayload,
                                ttlMode=TtlMode.uniform,
                                dscpMode=TtlMode.pipe ):
   '''
   From an MPLS input frame in <inPkt> with associated pre-parsed *Hdr, generate
   a new MPLS output frame using <srcMac, <dstMac>, and <labelStack>, after popping
   1 label.

   Acronyms (http://go/acronyms):
    - COS = Class of Service
    - BOS = Bottom of Stack
    - DSCP = Differentiated Services Code Point
    - EEDB = Egress Encapsulation Database
    - PMF = Programmable Mapping and Filtering

   Conforming with the behavior of the Sand platform (http://go/sandmpls):

   Swap/Forward:
      Nothing changes to payload, but COS and TTL-1 are copied from the top header
      to the new header(s)

   Pop:
      1) If TTL mode is uniform, perform a PMF pop:
         a) If BOS: Copy over TTL-1 from top header to inner IP header
         b) If not BOS: Copy over TTL-1 and COS from top header to next MPLS header
      2) If TTL mode is pipe, perform an EEDB pop:
         - The inner IP/IPv6 payload is preserved (TTL and DSCP are not touched)

   Pop + Push:
      First perform one of the pop actions above. Then:
      a) If payload is MPLS: Copy over TTL-1 and COS from top header
      b) If payload is IP: Copy over TTL-1 from top header and derive the new
         COS value from the inner IP header's DSCP bits
   '''
   func = 'generateOutFrameForMplsPkt:'

   if mplsHdr.ttl <= 1:
      t8( func, 'drop frame, MPLS ttl', mplsHdr.ttl )
      qt0( swapHandler, 'drop frame, MPLS ttl', qv( mplsHdr.ttl ) )
      return None
   newTtl = mplsHdr.ttl - 1

   ethHdr.dst = dstMac
   ethHdr.src = srcMac
   removeImpNullFromLabelStack( labelStack )

   mplsHdr2 = None
   mplsHdr3 = None
   # Set the PktUpdateType to determine what action to perform on the inner payload
   if labelAction == MplsLabelAction.pop:
      if not mplsHdr.bos:
         # Payload is MPLS
         mplsHdrStr = b''
         mplsHdr2 = Tac.newInstance( "Arnet::MplsHdrWrapper", inPkt,
                                     mplsHdr.offset + 4 )
         if mplsHdr2.bos is False:
            mplsHdr3 = Tac.newInstance( "Arnet::MplsHdrWrapper", inPkt,
                                        mplsHdr.offset + 8 )

         if mplsHdr2.label == MplsLabel.entropyLabelIndicator:
            if mplsHdr2.bos is True:
               t0( 'ELI cannot be BOS' )
               return None
            elif mplsHdr3.bos:
               newEthType = _newEthType( payloadType, inPkt, mplsHdr3 )
               if newEthType is None:
                  return None
               if ethDot1QHdr:
                  ethDot1QHdr.ethType = newEthType
               else:
                  ethHdr.ethType = newEthType

               pktUpdate = PktUpdateType.ip
            else:
               pktUpdate = PktUpdateType.mpls
         else:
            pktUpdate = PktUpdateType.mpls
      else:
         # Payload is IP, so update the ethType of the EthHdr
         mplsHdrStr = b''
         newEthType = _newEthType( payloadType, inPkt, mplsHdr )
         if newEthType is None:
            return None
         if ethDot1QHdr:
            ethDot1QHdr.ethType = newEthType
         else:
            ethHdr.ethType = newEthType
         pktUpdate = PktUpdateType.ip
   else:
      assert labelStack, "New label stack must exist for swap and fwd actions"
      mplsHdrStr = constructMplsHeader( labelStack, mplsTtl=newTtl,
                                        mplsCos=mplsHdr.cos,
                                        setBos=bool( mplsHdr.bos ) )
      pktUpdate = PktUpdateType.none

   # Pop one label (4 bytes) and add the new MPLS header(s) if action was swap/fwd.
   # For pop + push, the new headers will be added below, after the popping, and
   # updating the payload
   newData = inPkt.stringValue[ : mplsHdr.offset ]
   newData += mplsHdrStr
   if ( labelAction == MplsLabelAction.pop and not mplsHdr.bos and
        mplsHdr2 and mplsHdr2.label == MplsLabel.entropyLabelIndicator ):
      newData += inPkt.stringValue[ mplsHdr.offset + 12 : ]
   else:
      newData += inPkt.stringValue[ mplsHdr.offset + 4 : ]

   # Update the inner MPLS or IP payload's TTL/COS value depending on the conditions
   if pktUpdate == PktUpdateType.none:
      pass
   elif pktUpdate == PktUpdateType.mpls:
      # Copy decremented TTL and CoS fields to the next label in the stack
      pkt, headers, _ = PktParserTestLib.parsePktStr( newData )
      newMplsHdr = PktParserTestLib.findHeader( headers, "MplsHdr" )
      if not newMplsHdr:
         # Top label was not BoS so there must be at least 1 more MPLS header
         t8( func, 'invalid frame, expected MPLS label after non-BOS label',
             repr( inPkt ) )
         qt0( swapHandler, 'invalid frame, expected MPLS label after non-BOS label',
              qv( repr( inPkt ) ) )
         return None
      newMplsHdr.ttl = newTtl
      newMplsHdr.cos = mplsHdr.cos
      newData = pkt.stringValue
   elif pktUpdate == PktUpdateType.ip:
      # Update the TTL of the IP header depending on the TTL mode
      pkt, headers, _ = PktParserTestLib.parsePktStr( newData )
      ipHdr = PktParserTestLib.findHeader( headers, "IpHdr" )
      ip6Hdr = PktParserTestLib.findHeader( headers, "Ip6Hdr" )
      if not ( ipHdr or ip6Hdr ):
         t8( func, 'invalid frame, expected IP/IPv6 frame after BOS label',
             repr( inPkt ) )
         qt0( swapHandler, 'invalid frame, expected IP/IPv6 frame after BOS label',
              qv( repr( inPkt ) ) )
         return None
      if ttlMode == TtlMode.uniform:
         if ipHdr:
            ipHdr.ttl = newTtl
            ipHdr.checksum = 0
            ipHdr.checksum = ipHdr.computedChecksum
         else: # ip6Hdr, see ipHdr or ip6Hdr check above
            ip6Hdr.hopLimit = newTtl
      # TODO - Code needed here to handle dscpMode when it is not 'pipe'
      # ...
      newData = pkt.stringValue

   # Pop + push:
   # Now that the pop action is complete and the payload has been updated, we can
   # construct the new MPLS header(s) required for the push action
   if labelAction == MplsLabelAction.pop and labelStack:
      pkt, headers, _ = PktParserTestLib.parsePktStr( newData )

      # Update the ethType of the ethernet header
      ethHdr = PktParserTestLib.findHeader( headers, "EthHdr" )
      ethDot1QHdr = PktParserTestLib.findHeader( headers, "EthDot1QHdr" )
      if ethDot1QHdr:
         ethDot1QHdr.ethType = 'ethTypeMpls'
      else:
         ethHdr.ethType = 'ethTypeMpls'

      # If the payload type is IP, we need to derive the new MPLS COS value
      # from the inner IP header. Else, just copy it from the top-label header
      if pktUpdate == PktUpdateType.mpls:
         mplsCos = mplsHdr.cos
      elif pktUpdate == PktUpdateType.ip:
         ipHdr = ( PktParserTestLib.findHeader( headers, "IpHdr" ) or
                   PktParserTestLib.findHeader( headers, "Ip6Hdr" ) )
         isV6 = isinstance( ipHdr, Tac.Type( 'Arnet::Ip6HdrWrapper' ) )
         tos = ipHdr.trafficClass if isV6 else ipHdr.tos
         # BUG363148
         mplsCos = ( tos >> 5 ) & 0b111
      else:
         assert False, "PktUpdateType must be MPLS or IP"

      # Create the new MPLS header(s) and insert into the original frame
      mplsHdrStr = constructMplsHeader( labelStack, mplsTtl=newTtl,
                                        mplsCos=mplsCos, setBos=mplsHdr.bos )
      newData = ( pkt.stringValue[ : mplsHdr.offset ] + mplsHdrStr +
                  pkt.stringValue[ mplsHdr.offset : ] )

   return newData

def mplsHandleSwapPop( bridge, pkt, ethHdr, ethDot1QHdr, mplsHdr, ipHdr ):
   noMatch = [ ( None, None ) ]
   func = mplsHandleSwapPop.__name__ + ':'

   t8( func, 'mplsHandleSwapPop: mplsLabel:', mplsHdr.label )

   if mplsHdr.ttl <= 1:
      t8( func, 'drop mplsHdr label:', mplsHdr.label, 'on ttl:', mplsHdr.ttl )
      qt0( swapHandler, 'drop mplsHdr label:', qv( mplsHdr.label ),
           'on ttl:', qv( mplsHdr.ttl ) )
      return noMatch

   adjType, origViaList = lookupPkt( bridge, mplsHdr.label )

   if not origViaList:
      t8( func, 'LFIB lookup failed for label', mplsHdr.label )
      qt0( swapHandler, 'LFIB lookup failed for label', qv( mplsHdr.label ) )
      return noMatch

   if Toggles.MplsEtbaToggleLib.toggleMplsEtbaHashingEnabled():
      hashInfo = getHashInfo( pkt, ethHdr, mplsHdr, ipHdr )
      viaIndices = range( len( origViaList ) )
      index = ecmpHash( viaIndices, *hashInfo )
      t8( func, 'Hashing', hashInfo, 'to index', index )
   else:
      index = 0
      t8( func, 'Hashing disabled - choosing index', index )

   viaList = origViaList
   if adjType != LfibViaSetType.multicast:
      t8( func, 'Retain only 1 via for non-multicast adjs. origViaList length:',
          len( origViaList ) )
      viaList = [ origViaList[ index ] ]

   newDataAndIntfsList = []
   newViaList = []
   for via in viaList:
      # Mpls multicast via list can have both MplsVia and MulticastIpLookup/IpLookup
      # Skip MulticastIpLookup/IpLookup for swap/pop
      # The use case for this is MLDP (static and MVPN entries).
      if via.viaType in ( 'viaTypeMulticastIpLookup',
                          'viaTypeIpLookup' ):
         continue
      newViaList.append( via )
      ttlMode = via.l2Via.ttlMode
      if ttlMode == TtlMode.undefinedTtlMode:
         ttlMode = TtlMode.uniform
      dscpMode = via.l2Via.dscpMode
      if dscpMode == TtlMode.undefinedTtlMode:
         dscpMode = TtlMode.pipe
      payloadType = via.l2Via.payloadType
      if payloadType == PayloadType.undefinedPayload:
         payloadType = PayloadType.autoDecide
      labelAction = via.l2Via.labelAction
      if labelAction == MplsLabelAction.swap:
         labelStack = [ via.l2Via.outLabel ]
      elif labelAction == MplsLabelAction.forward:
         # For multicast viasets, we should not forward
         if adjType == LfibViaSetType.multicast:
            t8( func, 'drop, multicast viaset cannot have forward action and'
               ' via label', via )
            qt0( swapHandler, 'drop,  multicast viaset cannot have forward'
                 ' action and via label', qv( via ) )
            return noMatch
         labelStack = [ mplsHdr.label ]
         if via.l2Via.getRawAttribute( 'outLabel' ).isValid():
            t8( func, 'drop, invalid via, forward action and via label', via )
            qt0( swapHandler, 'drop, invalid via, forward action and via label',
                 qv( via ) )
            return noMatch
      else:
         # pop case
         labelStack = []

      if via.l3Intf and via.l2Via.getRawAttribute( 'macAddr' ):
         intf = via.l3Intf
         dstMac = via.l2Via.macAddr
      else:
         info = bridge.fwdHelper.resolveHierarchical( allowEncap=True,
                                                      hop=via.resolvedNextHop,
                                                      l3IntfId=via.l3Intf )
         if info[ 0 ] == noMatchNexthopInfo:
            t8( func, 'unable to resolve via L2 nexthop' )
            qt0( swapHandler, 'unable to resolve via L2 nexthop' )
            return noMatch
         labelStack = info[ 0 ].labelStack + labelStack
         intf = info[ 0 ].intf
         dstMac = info[ 0 ].dstMac

      viaList = newViaList
      t8( func, 'label:', mplsHdr.label, 'action:', labelAction, 'outLabel:',
          via.l2Via.outLabel, 'payloadType:', payloadType,
          'ttlMode:', ttlMode, 'dscpMode:', dscpMode,
          'intf:', intf, 'dstMac:', dstMac, 'labelStack:', labelStack )

      newData = generateOutFrameForMplsPkt( pkt, ethHdr, ethDot1QHdr, mplsHdr,
                                            dstMac=dstMac,
                                            srcMac=bridge.bridgeMac_,
                                            labelStack=labelStack,
                                            payloadType=payloadType,
                                            ttlMode=ttlMode,
                                            dscpMode=dscpMode,
                                            labelAction=labelAction )
      t8( func, 'Generating newData' )
      newDataAndIntfsList.append( ( newData, intf ) )

   t8( func, 'viaList len is', len( viaList ), 'newDataAndIntfsList len is',
       len( newDataAndIntfsList ) )

   # For mldp (multicast) routes, if we have multiple vias but we somehow
   # don't succeed generating packet for all, we should not send packets
   # for any via. So we make sure that the length of the generated pktList
   # and viaList match. All multicast vias must be resolvable.
   if len( viaList ) != len( newDataAndIntfsList ):
      t8( func, 'viaList and newDataAndIntfsList lengths do not match' )
      return noMatch

   return newDataAndIntfsList

def mplsHandlePush( bridge, pkt, ethHdr, ethDot1QHdr, ipHdr ):
   noMatch = ( None, None )
   func = 'mplsHandlePush:'
   isV6 = isinstance( ipHdr, Tac.Type( 'Arnet::Ip6HdrWrapper' ) )
   if isV6:
      routingInfo = bridge.routing6VrfInfo_.get( DEFAULT_VRF )
   else:
      routingInfo = bridge.routingVrfInfo_.get( DEFAULT_VRF )

   if not routingInfo or not routingInfo.routing:
      af = 'ipv6' if isV6 else 'ipv4'
      t8( func, af, 'routing is disabled' )
      qt0( pushHandler, qv( af ), 'routing is disabled' )
      return noMatch

   nexthopInfo = resolveMplsPushNexthop( bridge, str( ipHdr.dst ) )
   if not nexthopInfo.resolved:
      t8( func, 'cannot resolve', ipHdr.dst )
      qt0( pushHandler, 'cannot resolve', qv( ipHdr.dst ) )
      return noMatch

   labelStack = copy.copy( nexthopInfo.labelStack )
   removeImpNullFromLabelStack( labelStack )

   if ELI in labelStack:
      entropyLabel = computeFlowOrEntropyLabel( pkt, ethHdr, None, ipHdr )
      eliIndexes = [ i + 1 for i, x in enumerate( labelStack ) if x == ELI ]
      for idx in eliIndexes:
         labelStack[ idx ] = entropyLabel

   # Check TTL value
   # mplsHandlePush must not decrement the TTL; the TTL in the frame at this point
   # is already the oTtl.  The frames are delivered to mplsHandlePush via the
   # fwd0 interface, and so the oTtl calculation has already been performed by
   # the kernel during the kernel routing action.  This also means that TTL=1
   # is valid at this point in the packet handling code.
   if isV6:
      if ipHdr.hopLimit < 1:
         t8( func, 'Dropping packet with ttl of', ipHdr.hopLimit )
         qt0( pushHandler, 'Dropping packet with ttl of', qv( ipHdr.hopLimit ) )
         return noMatch
      oTtl = ipHdr.hopLimit
      ipHdr.hopLimit = oTtl
      ipEthType = 'ethTypeIp6'
   else:
      if ipHdr.ttl < 1:
         t8( func, 'dropping packet with ttl of', ipHdr.ttl )
         qt0( pushHandler, 'dropping packet with ttl of', qv( ipHdr.ttl ) )
         return noMatch
      oTtl = ipHdr.ttl
      ipHdr.ttl = oTtl
      ipEthType = 'ethTypeIp'

   # recompute the ip header checksum
   if not isV6:
      ipHdr.checksum = 0
      ipHdr.checksum = ipHdr.computedChecksum

   if labelStack:
      # Copy over the top 3 bits from the DSCP field in the ip hdr
      # The DSCP field is the top 6 bits of the tos field
      tos = ipHdr.trafficClass if isV6 else ipHdr.tos
      mplsCosValue = int( bin( tos )[ 2 : ][ : 3 ], 2 )
      mplsHdr = constructMplsHeader( labelStack, mplsTtl=oTtl, mplsCos=mplsCosValue )
      newEthType = 'ethTypeMpls'
   else:
      mplsHdr = b''
      newEthType = ipEthType

   ethHdr.dst = nexthopInfo.dstMac
   ethHdr.src = bridge.bridgeMac_
   if ethDot1QHdr:
      ethDot1QHdr.ethType = newEthType
   else:
      ethHdr.ethType = newEthType

   tmpData = pkt.stringValue
   newData = tmpData[ : ipHdr.offset ] + mplsHdr + tmpData[ ipHdr.offset : ]

   t1( "Returning new data from fwd0" )
   return( newData, nexthopInfo.intf )

def mplsSwapPopRoutingHandler( bridge, vlanId, destMac, data, fromFwdIntf=False ):
   t9( swapHandler, 'Processing pkt' )
   noMatch = [ ( None, {} ) ]

   # Check mpls routing is enabled
   if not bridge.mplsRoutingInfo_.mplsRouting:
      t8( swapHandler, 'mplsRouting is disabled ' )
      qt0( swapHandler, 'mplsRouting is disabled' )
      return noMatch

   isBudPkt = False
   pmsiVlan = None
   if not isMpls( data ):
      t8( "Non-MPLS frame (MplsPktHelper), skip" )
      qt0( swapHandler, "Non-MPLS frame (MplsPktHelper), skip" )
      # Check if packetContext has a BUD Role Packet inside it.
      if bridge.packetContext.get( BUD_ROLE_SPL_PKT ):
         t8( swapHandler, "Found BUD_ROLE related un-poped MPLS Frame" )
         destMac, _, data, pmsiVlan = bridge.packetContext[ BUD_ROLE_SPL_PKT ]
         isBudPkt = True
         del bridge.packetContext[ BUD_ROLE_SPL_PKT ]
      else:
         return noMatch

   # Retrieve and validate packet info
   hashingEnabled = Toggles.MplsEtbaToggleLib.toggleMplsEtbaHashingEnabled()
   pktInfo = parseDataPacket( data, parseBeyondMpls=hashingEnabled )
   if not pktInfo.ethHdr:
      t8( swapHandler, 'No L2 Hdr in frame, drop' )
      qt0( swapHandler, 'No L2 Hdr in frame, drop' )
      return noMatch

   if not pktInfo.mplsHdr:
      t8( swapHandler, 'Non-MPLS frame, drop' )
      qt0( swapHandler, 'Non-MPLS frame, drop' )
      return noMatch

   # Ensure the destMac passed in by EbraTestBridge matches the bridgeMac.
   # Drop if not, since this should be handled by a different routingHandler
   if destMac != bridge.bridgeMac_:
      t8( swapHandler, 'destMac', destMac, 'is not bridgeMac',
          bridge.bridgeMac_, 'drop' )
      qt0( swapHandler, 'destMac', qv( destMac ), 'is not bridgeMac',
           qv( bridge.bridgeMac_ ), ', drop' )
      return noMatch

   # Ensure that the ethHdr.dst matches the bridgeMac
   if pktInfo.ethHdr.dst != bridge.bridgeMac_:
      t8( swapHandler, 'ethHdr.dst(Mac)', pktInfo.ethHdr.dst,
          'is not bridgeMac', bridge.bridgeMac_, ', drop' )
      qt0( swapHandler, 'ethHdr.dst(Mac)', qv( pktInfo.ethHdr.dst ),
           'is not bridgeMac', qv( bridge.bridgeMac_ ), ', drop' )
      return noMatch

   # There is MPLS header, swap/pop action
   newDataAndIntfsList = mplsHandleSwapPop( bridge, pktInfo.pkt, pktInfo.ethHdr,
                                            pktInfo.ethDot1QHdr,
                                            pktInfo.mplsHdr,
                                          ( pktInfo.ipHdr or pktInfo.ip6Hdr ) )
   if newDataAndIntfsList == [ ( None, None ) ]:
      return noMatch

   if fromFwdIntf:
      return newDataAndIntfsList

   newDataVlanTupList = []

   for newDataIntfTuple in newDataAndIntfsList:
      newData = newDataIntfTuple[ 0 ]
      if isBudPkt:
         # Decapped mpls packet in MVPN bud role is always tagged and ingress bridged
         # in the PMSI vlan of the associated vrf. As a result, processFrame() of
         # EbraTestBridgePythonImpl thinks the received packet to be vlan tagged and
         # tries to replace the vlan tag with the destination intf's vlan for the
         # routed packets. So, explicitly tag the swapped or popped data with the
         # PMSI vlan so that the tag replace logic in processFrame() doesn't break.
         newData = applyVlanChanges( newData, 0, pmsiVlan, PKTEVENT_ACTION_ADD )
      intf = newDataIntfTuple[ 1 ]
      # Get the vlan Id for outgoing interface
      vlan = getIntfVlan( bridge.brConfig, intf )
      if vlan is None:
         t8( swapHandler, 'Cannot find vlan for', intf )
         qt0( swapHandler, 'Cannot find vlan for', qv( intf ) )
         return noMatch

      t9( swapHandler, 'Finished processing pkt' )
      newDataVlanTupList.append( ( newData, { vlan } ) )

   return newDataVlanTupList

def mplsFwdIntfSwapPopRoutingHandler( self, data ):
   t9( swapHandler, 'Processing pkt on fwdIntf' )

   pktInfo = parseDataPacket( data )
   destMac = pktInfo.ethHdr.dst
   routeMatch = mplsSwapPopRoutingHandler( self.bridge, None, destMac, data, True )

   # The FwdIntfDevice expects its routing handlers' return type to be of
   # the form ( newData, newIntfId )
   noMatch = ( None, None )
   if ( isinstance( routeMatch, list ) and routeMatch
        and routeMatch != [ ( None, {} ) ] ):
      t8( swapHandler, 'routeMatch', routeMatch[ 0 ] )
      return routeMatch[ 0 ]
   else:
      t8( swapHandler, 'no routeMatch on fwdIntf' )
      qt0( swapHandler, 'no routeMatch on fwdIntf' )
      return noMatch

def mplsPushRoutingHandler( self, data ):
   'MPLS push route handler to be registered by the fwd0 device'
   t9( pushHandler, 'Processing pkt' )
   routeMatch = ( None, None )
   noMatch = ( None, None )

   # Check mpls routing is enabled
   if not self.bridge.mplsRoutingInfo_.mplsRouting:
      t8( pushHandler, 'mplsRouting is disabled' )
      qt0( pushHandler, 'mplsRouting is disabled' )
      return noMatch

   # Ensure frame is for default VRF
   if self.vrfName != DEFAULT_VRF:
      t8( pushHandler, 'frame is from ND VRF, drop' )
      qt0( pushHandler, 'frame is from ND VRF, drop' )
      return noMatch

   # Retrieve and validate packet info
   pktInfo = parseDataPacket( data )
   if not pktInfo.ethHdr:
      t8( pushHandler, 'No L2 Hdr in frame, drop' )
      qt0( pushHandler, 'No L2 Hdr in frame, drop' )
      return noMatch

   if pktInfo.mplsHdr:
      t8( pushHandler, 'MPLS frame, drop' )
      qt0( pushHandler, 'MPLS frame, drop' )
      return noMatch

   ipHdr = pktInfo.ipHdr or pktInfo.ip6Hdr
   if not ipHdr:
      t8( pushHandler, 'Non-IP frame, drop' )
      qt0( pushHandler, 'Non-IP frame, drop' )
      return noMatch

   routeMatch = mplsHandlePush( self.bridge,
                                pktInfo.pkt,
                                pktInfo.ethHdr,
                                pktInfo.ethDot1QHdr,
                                ipHdr )
   return routeMatch

def pwMsg( *args, **kwargs ):
   tf = kwargs.get( 'tf', t8 )
   tf( 'mplsPwRoutingHandler:', *args )

def resolvePwTunnel( bridge, tunnelId, ethHdr ):
   noMatch = None
   if not tunnelId.isValid():
      pwMsg( 'encap tunnel not valid, id:', tunnelId.value )
      qt0( pwHandler, 'encap tunnel not valid, id:', qv( tunnelId.value ) )
      return noMatch
   pwMsg( 'using encap tunnel', tunnelId.toStrep(), 'id:', tunnelId.value )
   tunEnt = getTunnelEntry( bridge.tunnelTables, bridge.pfs, ethHdr, tunnelId )
   if tunEnt is None:
      pwMsg( 'no tunnel entry' )
      qt0( pwHandler, 'no tunnel entry' )
      return noMatch
   # pylint: disable-next=isinstance-second-argument-not-valid-type
   if isinstance( tunEnt, RsvpLerTunnelTableEntry ):
      via = ( tunEnt.via if tunEnt.isPrimaryViaUsable and tunEnt.via
              else tunEnt.backupVia )
   else:
      via = ecmpHash( tunEnt.via, ethHdr.src, ethHdr.dst ) if tunEnt.via else None
   if not via:
      pwMsg( 'tunnel has no vias' )
      qt0( pwHandler, 'tunnel has no vias' )
      return noMatch
   # pylint: disable-next=isinstance-second-argument-not-valid-type
   if not isinstance( via, MplsVia ):
      pwMsg( 'wrong tunnel via type', type( via ) )
      qt0( pwHandler, 'wrong tunnel via type', qv( type( via ) ) )
      return noMatch
   viaHop = via.nexthop
   viaIntfId = via.intfId
   viaLabels = via.labels
   labelStack = []
   for idx in range( viaLabels.calculatedStackSize ):
      labelStack.insert( 0, viaLabels.calculatedLabelStack( idx ) )

   if ELI in labelStack:
      entropyLabel = computeFlowOrEntropyLabel( None, ethHdr, None, None )
      eliIndexes = [ i + 1 for i, x in enumerate( labelStack ) if x == ELI ]
      for idx in eliIndexes:
         labelStack[ idx ] = entropyLabel

   if not labelStack:
      pwMsg( 'no labels in label stack' )
      qt0( pwHandler, 'no labels in label stack' )
      return noMatch
   pwMsg( 'tunnel', tunEnt.tep, 'via', viaHop, viaIntfId, labelStack )
   return viaHop, viaIntfId, labelStack

def rewriteVlanTag( ethHdr, ethTpidHdr, tpid, vlanId, retainCos ):
   """Rewrite TPID, VID and -optionally- CoS of a dot1ad/dot1q tag.

   ethHdr - rewrite the TPID in this header. Note that this header is the one
            preceding ethTpidHdr and contains the TPID for ethTpidHdr. This header
            is either an EthHdr with source and destination MACs (when rewriting the
            outer tag) or another ethTpidHdr (when rewriting an inner tag).
   ethTpidHdr - rewrite VID and CoS in this header (the TPID in this header is the
                payload's TPID, which remains unchanged)
   """
   ethHdr.ethType = tpid
   ethTpidHdr.tagControlVlanId = vlanId
   if not retainCos:
      # use static mapping: 0 to 1 and 1 to 0
      ethTpidHdr.tagControlPriority = tcToMplsCosStatic(
         ethTpidHdr.tagControlPriority )

def maybeManipulateVlanTags( ethPkt, ethHdr, ethTpidHdrOuter, ethTpidHdrInner,
                             vlanIdOuter, vlanIdInner, tpidOuter, tpidInner,
                             retainOuterCos, retainInnerCos, cosForNewTags, logMsg ):
   """Perform tag manipulation of an Ethernet packet and return the new packet.

   See http://aid/6957 chapter "VLAN Tag Rewrite"

   ethPkt is the packet to process
   ethHdr is the packet's Ethernet header
   ethTpidHdrOuter is the packet's outer dot1ad/dot1q header, or None if the packet
                   was not matched on the outer header (e.g. port connector)
   ethTpidHdrInner is the packet's inner dot1ad/dot1q header, or None if the packet
                   was not matched on the inner header (e.g. single tagged match)
   vlanIdOuter is the VLAN value to push or to rewrite the outer tag to,
               or VlanTag.anyTagOrNone if tags should be removed (if present)
   vlanIdInner is the VLAN value to push or to rewrite the inner tag to,
               or VlanTag.anyTagOrNone if inner tag should be removed (if present)
   tpidOuter, tpidInner are the  TPID values to push or to rewrite
   retainOuterCos and retainInnerCos tell if the CoS should be retained or not
                  when rewriting VID in tags
   cosForNewTags is the CoS to use when pushing tags. The CoS is translated to a
                  priority via tcToMplsCosStatic which maps 0 to 1 and 1 to 0

   When modifying ethTpidHdrOuter or ethTpidHdrInner (e.g. tag rewrite),
   ethPkt gets updated accordingly.
   """

   def _manipulateDouble():
      """Perform VLAN tag manipulation on a packet that was matched on both its
      outer and inner tags."""
      if vlanIdOuter in ( VlanTag.anyTagOrNone, VlanTag.untagged ):
         # strip both outer and inner tags
         logMsg( 'double tag outer', ethTpidHdrOuter.tagControlVlanId,
                 ethHdr.ethType, 'inner', ethTpidHdrInner.tagControlVlanId,
                 ethTpidHdrOuter.ethType, '-> untagged' )
         return applyVlanChanges( ethPkt.stringValue, None, None,
                                  PKTEVENT_ACTION_REMOVE, numTagsToRemove=2 )
      elif vlanIdOuter <= VlanId.max:
         # valid outer tag value indicates that the target packet has at least
         # one tag (vlanIdInner decides if it has two tags)
         if vlanIdInner == VlanTag.anyTagOrNone:
            # outer, inner -> inner' (pop outer, rewriter inner)
            logMsg( 'double tag outer', ethTpidHdrOuter.tagControlVlanId,
                    ethHdr.ethType, 'inner', ethTpidHdrInner.tagControlVlanId,
                    ethTpidHdrOuter.ethType, '-> single tag', vlanIdOuter, 'tpid',
                    '0x%x' % tpidOuter, 'retain' if retainInnerCos else 'rewrite',
                    'cos' )
            # for compatibility with 'inner client' rewrite the inner tag first,
            # possibly retaining CoS, then pop the outer tag
            rewriteVlanTag( ethTpidHdrOuter, ethTpidHdrInner, tpidOuter, vlanIdOuter,
                            retainInnerCos )
            return applyVlanChanges( ethPkt.stringValue, None, None,
                                     PKTEVENT_ACTION_REMOVE, numTagsToRemove=1 )
         elif vlanIdInner <= VlanId.max:
            # outer, inner -> outer', inner' (rewrite both)
            logMsg( 'double tag outer', ethTpidHdrOuter.tagControlVlanId,
                    ethHdr.ethType, 'inner', ethTpidHdrInner.tagControlVlanId,
                    ethTpidHdrOuter.ethType, '-> double tag outer', vlanIdOuter,
                    'tpid', '0x%x' % tpidOuter,
                    'retain' if retainOuterCos else 'rewrite', 'cos',
                    'inner', vlanIdInner, 'tpid', '0x%x' % tpidInner,
                    'retain' if retainInnerCos else 'rewrite', 'cos' )
            rewriteVlanTag( ethHdr, ethTpidHdrOuter, tpidOuter, vlanIdOuter,
                            retainOuterCos )
            rewriteVlanTag( ethTpidHdrOuter, ethTpidHdrInner, tpidInner, vlanIdInner,
                            retainInnerCos )
            return ethPkt.stringValue
         else:
            # unsupported operation, drop packet
            logMsg( 'double tag outer', ethTpidHdrOuter.tagControlVlanId,
                    ethHdr.ethType, 'inner', ethTpidHdrInner.tagControlVlanId,
                    ethTpidHdrOuter.ethType, '-> drop: unsupported inner operation',
                    vlanIdInner )
            return None
      else:
         # unsupported operation, drop packet
         logMsg( 'double tag outer', ethTpidHdrOuter.tagControlVlanId,
                 ethHdr.ethType, 'inner', ethTpidHdrInner.tagControlVlanId,
                 ethTpidHdrOuter.ethType, '-> drop: unsupported outer operation',
                 vlanIdOuter )
         return None

   def _manipulateSingle():
      """Perform VLAN tag manipulation on a packet that was matched on its outer
      tag only."""
      if vlanIdOuter in ( VlanTag.anyTagOrNone, VlanTag.untagged ):
         # strip outer tag
         logMsg( 'single tag', ethTpidHdrOuter.tagControlVlanId, ethHdr.ethType,
                 '-> untagged' )
         return applyVlanChanges( ethPkt.stringValue, None, None,
                                  PKTEVENT_ACTION_REMOVE, numTagsToRemove=1 )
      elif vlanIdOuter <= VlanId.max:
         # valid outer tag value indicates that the outer tag needs to be rewritten
         # and possibly a new tag pushed on if vlanIdInner says so
         if vlanIdInner <= VlanId.max:
            # outer -> outer', inner (rewrite, push)
            prio = tcToMplsCosStatic( cosForNewTags )
            logMsg( 'single tag', ethTpidHdrOuter.tagControlVlanId, ethHdr.ethType,
                    '-> double tag outer', vlanIdOuter, 'tpid', '0x%x' % tpidOuter,
                    'retain' if retainOuterCos else 'rewrite', 'cos',
                    'inner', vlanIdInner, 'tpid', '0x%x' % tpidInner, 'prio', prio )
            rewriteVlanTag( ethHdr, ethTpidHdrOuter, tpidInner, vlanIdInner,
                            retainOuterCos )
            return applyVlanChanges( ethPkt.stringValue, prio, vlanIdOuter,
                                     PKTEVENT_ACTION_ADD, tpid=tpidOuter )
         elif vlanIdInner == VlanTag.anyTagOrNone:
            # outer -> outer' (rewrite)
            logMsg( 'single tag', ethTpidHdrOuter.tagControlVlanId,
                    ethHdr.ethType, '-> single tag', vlanIdOuter, 'tpid',
                    '0x%x' % tpidOuter,
                    'retain' if retainOuterCos else 'rewrite', 'cos' )
            rewriteVlanTag( ethHdr, ethTpidHdrOuter, tpidOuter, vlanIdOuter,
                            retainOuterCos )
            return ethPkt.stringValue
         else:
            # unsupported operation, drop packet
            logMsg( 'single tag', ethTpidHdrOuter.tagControlVlanId, ethHdr.ethType,
                    '-> drop: unsupported inner operation', vlanIdInner )
            return None
      else:
         # unsupported operation, drop packet
         logMsg( 'single tag', ethTpidHdrOuter.tagControlVlanId, ethHdr.ethType,
                 '-> drop: unsupported outer operation', vlanIdOuter )
         return None

   def _manipulateNoTag():
      """Perform VLAN tag manipulation on a packet that was not matched by any of
      its tags (if any were present at all)."""
      if vlanIdOuter in ( VlanTag.anyTagOrNone, VlanTag.untagged ):
         # no tag manipulation (port mode)
         return ethPkt.stringValue
      elif vlanIdOuter <= VlanId.max:
         if vlanIdInner <= VlanId.max:
            # any tag or untagged -> outer, inner (push two tags)
            prio = tcToMplsCosStatic( cosForNewTags )
            logMsg( 'any -> double tag outer', vlanIdOuter, 'tpid',
                    '0x%x' % tpidOuter, 'inner', vlanIdInner, 'tpid',
                    '0x%x' % tpidInner, 'prio', prio )
            return applyVlanChanges( ethPkt.stringValue, prio, vlanIdOuter,
                                     PKTEVENT_ACTION_ADD, tpid=tpidOuter,
                                     tpidInner=tpidInner, priorityInner=prio,
                                     vlanIdInner=vlanIdInner )
         elif vlanIdInner == VlanTag.anyTagOrNone:
            # any tag or untagged -> outer (push)
            prio = tcToMplsCosStatic( cosForNewTags )
            logMsg( 'any -> single tag', vlanIdOuter, 'tpid',
                    '0x%x' % tpidOuter, 'prio', prio )
            return applyVlanChanges( ethPkt.stringValue, prio, vlanIdOuter,
                                     PKTEVENT_ACTION_ADD, tpid=tpidOuter )
         else:
            # unsupported operation, drop packet
            logMsg( 'any -> drop: unsupported inner operation', vlanIdInner )
            return None
      else:
         # unsupported operation, drop packet
         logMsg( 'any -> drop: unsupported outer operation', vlanIdOuter )
         return None

   if ethTpidHdrOuter:
      if ethTpidHdrInner:
         newData = _manipulateDouble()
      else:
         newData = _manipulateSingle()
   else:
      newData = _manipulateNoTag()

   return newData

def encapEthOverMpls( bridge, hopMac, labelStack, viaIntf, pkt, ethHdr,
                      ethTpidHdrOuter, ethTpidHdrInner, vlanIdOuter, vlanIdInner,
                      tpidOuter, tpidInner, retainOuterCos, retainInnerCos,
                      addControlWord, flowLabelPresent ):
   """MPLS-encapsulate a packet and return it along with the destination VLAN.

   Arguments:
     hopMac -- MAC address of the next-hop
     labelStack -- MPLS label stack to use
     viaIntf -- interface to the next-hop
     pkt -- packet to encapsulate
     ethHdr -- ethernet header of the packet
     ethTpidHdrOuter -- outer 802.1ad/802.1q header of the packet, or None if packet
                        is untagged
     ethTpidHdrInner -- inner 802.1ad/802.1q header of the packet, or None if packet
                        does not have such header or if it was not used for matching
                        the pseudowire
     vlanIdOuter, vlanIdInner -- perform VLAN tag manipulation:
                         VLAN ID value to push or to rewrite the outer tag to,
                         or VlanTag.anyTagOrNone if tags should be removed
     tpidOuter, tpidInner -- perform VLAN tag manipulation:
                         TPID value to push or to rewrite the tags to
     retainOuterCos and retainInnerCos tell if the TC should be retained or not
                         when rewriting VID in tags
     addControlWord -- add a control word between label stack and encapsulated packet
   """
   noMatch = ( None, {} )

   destIntfVlan = getIntfVlan( bridge.brConfig, viaIntf )
   if destIntfVlan is None:
      pwMsg( 'no internal vlan for intf', viaIntf )
      qt0( pwHandler, 'no internal vlan for intf', qv( viaIntf ) )
      return noMatch

   tagControlPriority = ( ethTpidHdrOuter.tagControlPriority if ethTpidHdrOuter
                          else 0 )
   removeImpNullFromLabelStack( labelStack )
   mplsHdrBytes = constructMplsHeader( labelStack, tc=tagControlPriority,
                                       flowLabelPresent=flowLabelPresent )

   # Perform VLAN tag manipluation as specified
   innerPacketString = maybeManipulateVlanTags( pkt, ethHdr, ethTpidHdrOuter,
                                                ethTpidHdrInner, vlanIdOuter,
                                                vlanIdInner, tpidOuter, tpidInner,
                                                retainOuterCos, retainInnerCos, 0,
                                                pwMsg )
   if not innerPacketString:
      return noMatch

   # Rewrite outer ethernet header
   ethHdr.ethType = EthType().ethTypeMpls
   ethHdr.dst = hopMac
   ethHdr.src = bridge.bridgeMac_
   # Create MPLS-encapsulated packet
   controlWord = b'\x00' * 4 if addControlWord else b''
   newData = ( pkt.stringValue[ 0 : ethHdrLen ] + mplsHdrBytes + controlWord +
               innerPacketString )
   return newData, { destIntfVlan }

def mplsPwEncapIntfVlan( bridge, pkt, ethHdr, ethTpidHdrOuter, ethTpidHdrInner,
                         ipHdr, vlanIdOuter, vlanIdInner, tpidOuter, tpidInner,
                         retainOuterCos, retainInnerCos, tunnelId, encapPwLabel,
                         addControlWord, flowLabel ):
   """Resolve tunnel next-hop, create label stack for MPLS and forward packet."""

   noMatch = ( None, {} )

   if not bridge.mplsRoutingInfo_.mplsRouting:
      pwMsg( 'mplsRouting is disabled' )
      qt0( pwHandler, 'mplsRouting is disabled' )
      return noMatch

   tunnelInfo = resolvePwTunnel( bridge, tunnelId, ethHdr )
   if tunnelInfo is None:
      return noMatch
   viaHop, viaIntf, labelStack = tunnelInfo
   if viaHop.af in ( Af.ipv4, Af.ipv6 ):
      helper = bridge.fwdHelper
      l2Hop = helper.getPopSwapResolvedNexthopInfo( str( viaHop ), viaIntf )
   else:
      pwMsg( 'Invalid address family', viaHop.af )
      qt0( pwHandler, 'Invalid address family', qv( viaHop.af ) )
      return noMatch
   resolved, hopMac = l2Hop[ 0 : 2 ]
   if not resolved:
      pwMsg( 'unresolved nexthop' )
      qt0( pwHandler, 'unresolved nexthop' )
      return noMatch
   pwMsg( 'resolved tunnel nexthop MAC', hopMac, viaIntf )

   if labelStack and labelStack[ -1 ] == impNull():
      del labelStack[ -1 ]

   labelStack.append( encapPwLabel )
   if flowLabel:
      flowLabel = computeFlowOrEntropyLabel( pkt, ethHdr, None, ipHdr )
      labelStack.append( flowLabel )
   pwMsg( 'encap dst:', hopMac, 'label(top-to-bottom):', labelStack )
   return encapEthOverMpls( bridge, hopMac, labelStack, viaIntf, pkt, ethHdr,
                            ethTpidHdrOuter, ethTpidHdrInner, vlanIdOuter,
                            vlanIdInner, tpidOuter, tpidInner, retainOuterCos,
                            retainInnerCos, addControlWord, flowLabel )

def parentIntfFromServiceIntf( serviceIntfId ):
   if SubIntfId.isSubIntfId( serviceIntfId ):
      # use the parent port for pseudowires attached to sub-interfaces,
      # because Etba does not fully support sub-interfaces
      parentIntfId = SubIntfId.parentIntfId( serviceIntfId )
   elif PseudowireIntfId.isPwServiceIntfId( serviceIntfId ):
      # determine the parent port from the pw service interface
      parentIntfId = PseudowireIntfId.parentIntfId( serviceIntfId )
   else:
      # service interface is a port already
      parentIntfId = serviceIntfId
   return parentIntfId

def localLocalPwEncapIntfVlan( bridge, pkt, ethHdr, ethTpidHdrOuter, ethTpidHdrInner,
                               vlanIdOuter, vlanIdInner, tpidOuter, tpidInner,
                               retainOuterCos, retainInnerCos, destIntfId ):
   noMatch = ( None, {} )

   parentIntf = parentIntfFromServiceIntf( destIntfId )
   destIntfVlan = getIntfVlan( bridge.brConfig, parentIntf )
   if destIntfVlan is None:
      pwMsg( 'no internal vlan for intf', destIntfId )
      qt0( pwHandler, 'no internal vlan for intf', qv( destIntfId ) )
      return noMatch

   # Update packet with new TPIDs and vlan IDs, if applicable
   newPkt = maybeManipulateVlanTags( pkt, ethHdr, ethTpidHdrOuter, ethTpidHdrInner,
                                     vlanIdOuter, vlanIdInner, tpidOuter, tpidInner,
                                     retainOuterCos, retainInnerCos, 0, pwMsg )
   if not newPkt:
      return noMatch

   return newPkt, { destIntfVlan }

def _makeVlanTag( vid, tpid ):
   if vid is None:
      return VlanTag()
   elif tpid is None:
      return VlanTag( vid=vid )
   else:
      return VlanTag( vid=vid, tpid=tpid )

def _makeVlanTagEncap( outerVid, outerTpid=None, innerVid=None, innerTpid=None ):
   return VlanTagEncap( _makeVlanTag( outerVid, outerTpid ),
                        _makeVlanTag( innerVid, innerTpid ) )

def mplsPwGetAcToDestIntfEncap( bridge, acIntfId, ethHdr, ethTpidHdrOuter,
                                ethTpidHdrInner ):
   """Look for an acToDestIntfEncap for the packet with the given outer and inner
   dot1ad/dot1q headers. Return the encap and the headers that were used to match
   it."""

   vlanIdOuter = ethTpidHdrOuter.tagControlVlanId if ethTpidHdrOuter else None
   vlanIdInner = ethTpidHdrInner.tagControlVlanId if ethTpidHdrInner else None
   tpidOuter = ( Tac.enumValue( EthType(), ethHdr.ethType ) if ethTpidHdrOuter
                 else None )
   tpidInner = ( Tac.enumValue( EthType(), ethTpidHdrOuter.ethType )
                 if ethTpidHdrInner else None )
   acToDestIfEncap = None

   if ethTpidHdrInner:
      # look for double tag encap
      vlanTagEncap = _makeVlanTagEncap( vlanIdOuter, tpidOuter, vlanIdInner,
                                        tpidInner )
      intfEncap = IntfEncap( acIntfId, vlanTagEncap )
      acToDestIfEncap = bridge.pwStatus.acToDestIntfEncap.get( intfEncap )
      if acToDestIfEncap:
         pwMsg( 'matched double-tagged encap', acIntfId, 'outer vid', vlanIdOuter,
                'tpid', '0x%x' % tpidOuter, 'inner vid', vlanIdInner, 'tpid',
                '0x%x' % tpidInner )
         qt0( pwHandler, 'matched encap', qv( acIntfId ), 'outer vid',
              qv( vlanIdOuter ), 'tpid', qv( tpidOuter ), 'inner vid',
              qv( vlanIdInner ), 'tpid', qv( tpidInner ) )

   if not acToDestIfEncap and not ethTpidHdrOuter:
      # packet is untagged; look for an 'untagged' encap
      vlanTagEncap = _makeVlanTagEncap( VlanTag.untagged, innerVid=VlanTag.untagged )
      intfEncap = IntfEncap( acIntfId, vlanTagEncap )
      acToDestIfEncap = bridge.pwStatus.acToDestIntfEncap.get( intfEncap )
   elif not acToDestIfEncap and ethTpidHdrOuter and vlanIdOuter != VlanId.invalid:
      # look for single tag encap
      vlanTagEncap = _makeVlanTagEncap( vlanIdOuter, tpidOuter )
      intfEncap = IntfEncap( acIntfId, vlanTagEncap )
      acToDestIfEncap = bridge.pwStatus.acToDestIntfEncap.get( intfEncap )
      if acToDestIfEncap:
         pwMsg( 'matched single-tagged encap', acIntfId, 'outer vid', vlanIdOuter,
                'tpid', '0x%x' % tpidOuter )
         qt0( pwHandler, 'matched single-tagged encap', qv( acIntfId ), 'outer vid',
              qv( vlanIdOuter ), 'tpid', qv( tpidOuter ) )
      ethTpidHdrInner = None # not matched on inner header in any case

   if not acToDestIfEncap:
      # look for port encap
      intfEncap = IntfEncap( acIntfId, VlanTagEncap() )
      acToDestIfEncap = bridge.pwStatus.acToDestIntfEncap.get( intfEncap )
      if acToDestIfEncap:
         pwMsg( 'matched port encap', acIntfId )
         qt0( pwHandler, 'matched port encap', qv( acIntfId ) )
      ethTpidHdrOuter = None # not matched on outer header in any case

   if not acToDestIfEncap:
      if vlanIdInner is not None:
         pwMsg( 'no matching encap for double tagged packet with outer vid',
                vlanIdOuter, 'tpid', '0x%x' % tpidOuter, 'inner vid', vlanIdInner,
                'tpid', '0x%x' % tpidInner, 'on', acIntfId )
         qt0( pwHandler, 'no matching encap for double tagged packet with outer vid',
              qv( vlanIdOuter ), 'tpid', qv( tpidOuter ), 'inner vid',
              qv( vlanIdInner ), 'tpid', qv( tpidInner ), 'on', qv( acIntfId ) )
      elif vlanIdOuter is not None:
         pwMsg( 'no matching encap for single tagged packet with vid', vlanIdOuter,
                'tpid', '0x%x' % tpidOuter, 'on', acIntfId )
         qt0( pwHandler, 'no matching encap for single tagged packet with vid',
              qv( vlanIdOuter ), 'tpid', qv( tpidOuter ), 'on', qv( acIntfId ) )
      else:
         pwMsg( 'no matching encap for untagged packet on', acIntfId )
         qt0( pwHandler, 'no matching encap for untagged packet on', qv( acIntfId ) )

   return acToDestIfEncap, ethTpidHdrOuter, ethTpidHdrInner

def mplsPwRoutingHandlerNewApi( bridge, pkt, ethHdr, ethTpidHdrOuter,
                                ethTpidHdrInner, ipHdr, intf ):
   """Look for a matching pseudowire for the given packet using the new PwA->Ale
   API (acToDestIntfEncap) and forward the packet accordingly."""

   noMatch = ( None, {} )

   acIntfId = ( SubIntfId.parentIntfId( intf ) if SubIntfId.isSubIntfId( intf )
                else intf )
   acToDestIfEncap, ethTpidHdrOuter, ethTpidHdrInner = (
      mplsPwGetAcToDestIntfEncap( bridge, acIntfId, ethHdr, ethTpidHdrOuter,
                                  ethTpidHdrInner ) )
   if not acToDestIfEncap:
      return noMatch

   if PseudowireIntfId.isPwInstanceId( acToDestIfEncap.destIntfEncap.intfId ):
      pwAdjGroupId = bridge.pwStatus.acToPwEncapAdjGroup.get(
                        acToDestIfEncap.sourceIntfEncap, 0 )
      adjGroup = bridge.pwStatus.adjGroup.get( pwAdjGroupId )
      if not adjGroup or not adjGroup.pwAdjSet:
         pwMsg( 'no adjacency found for encap, group id', pwAdjGroupId )
         qt0( 'no adjacency found for encap, group id', qv( pwAdjGroupId ) )
         return noMatch
      pwEncapAdj = ecmpHash( sorted( adjGroup.pwAdjSet ), ethHdr.src, ethHdr.dst )
      return mplsPwEncapIntfVlan( bridge, pkt, ethHdr, ethTpidHdrOuter,
                                  ethTpidHdrInner, ipHdr,
                                  acToDestIfEncap.destIntfEncap.encap.outer.vid,
                                  acToDestIfEncap.destIntfEncap.encap.inner.vid,
                                  acToDestIfEncap.destIntfEncap.encap.outer.tpid,
                                  acToDestIfEncap.destIntfEncap.encap.inner.tpid,
                                  acToDestIfEncap.retainOuterCos,
                                  acToDestIfEncap.retainInnerCos,
                                  pwEncapAdj.getRawAttribute( 'tunnelId' ),
                                  pwEncapAdj.peerLabel,
                                  pwEncapAdj.controlWord,
                                  pwEncapAdj.flowLabel )
   else:
      return localLocalPwEncapIntfVlan(
                bridge, pkt, ethHdr, ethTpidHdrOuter, ethTpidHdrInner,
                acToDestIfEncap.destIntfEncap.encap.outer.vid,
                acToDestIfEncap.destIntfEncap.encap.inner.vid,
                acToDestIfEncap.destIntfEncap.encap.outer.tpid,
                acToDestIfEncap.destIntfEncap.encap.inner.tpid,
                acToDestIfEncap.retainOuterCos,
                acToDestIfEncap.retainInnerCos,
                acToDestIfEncap.destIntfEncap.intfId )

def getHeadersFromPkt( headers, dot1qOnly=True, withIpHdr=False ):
   ethTpidHdrOuter = None
   ethTpidHdrInner = None
   ipHdr = None
   tpidHdrs = ( ( 'EthDot1QHdr', ) if dot1qOnly
                else ( 'EthDot1QHdr', 'EthDot1adHdr' ) )
   for hdr in headers:
      if hdr[ 0 ] in tpidHdrs:
         if ethTpidHdrOuter:
            if withIpHdr:
               # return first inner header only, ignore subsequent ones
               # but keep scanning for ip header
               if not ethTpidHdrInner:
                  ethTpidHdrInner = hdr[ 1 ]
            else:
               ethTpidHdrInner = hdr[ 1 ]
               break # found all the headers
         else:
            ethTpidHdrOuter = hdr[ 1 ]
      elif withIpHdr and hdr[ 0 ] in ( 'IpHdr', 'Ip6Hdr' ):
         ipHdr = hdr[ 1 ]
         break # found all the headers
   if withIpHdr:
      return ethTpidHdrOuter, ethTpidHdrInner, ipHdr
   else:
      return ethTpidHdrOuter, ethTpidHdrInner

def mplsPwRoutingHandler( bridge, vlanId, destMac, data ):
   pwMsg( 'Processing pkt', tf=t9 )
   noMatch = [ ( None, {} ) ]

   if not bridge.pwStatus.acToDestIntfEncap:
      pwMsg( "No acToDestIntfEncap, pseudowire not enabled" )
      qt0( pwHandler, "No acToDestIntfEncap, pseudowire not enabled" )
      return noMatch

   if vlanId is None:
      pwMsg( 'vlanId is None' )
      qt0( pwHandler, 'vlanId is None' )
      return noMatch

   try:
      vlanConfig = bridge.brConfig.vlanConfig.get( vlanId )
   except IndexError:
      # Catch IndexError for Tac::RangeException with invalid vlanId value
      vlanConfig = None
   if vlanConfig is None:
      pwMsg( 'no vlanConfig for vlanId', vlanId )
      qt0( pwHandler, 'no vlanConfig for vlanId', qv( vlanId ) )
      return noMatch
   if not vlanConfig.internal:
      pwMsg( 'not supported on switch ports' )
      qt0( pwHandler, 'not supported on switch ports' )
      return noMatch

   ( pkt, headers, _ ) = PktParserTestLib.parsePktStr( data )
   ethHdr = PktParserTestLib.findHeader( headers, 'EthHdr' )
   if ethHdr is None:
      pwMsg( 'Ethernet header not found' )
      qt0( pwHandler, 'Ethernet header not found' )
      return noMatch
   ethTpidHdrOuter, ethTpidHdrInner, ipHdr = getHeadersFromPkt(
      headers, dot1qOnly=False, withIpHdr=True )

   ret = noMatch[ 0 ]
   for intf in vlanConfig.intf:
      # vlanId -> vlanConfig is the incoming interface for the frame, and
      # ethTpidHdrOuter/inner are the 802.1AD/802.1Q headers on the incoming frame,
      # so we need to get the actual interface for this internal vlanId to find out
      # if we have a pseudowire configured for (intf, ethTpidHdrs) to do the encap.
      if intf == 'Cpu':
         continue
      ret = mplsPwRoutingHandlerNewApi( bridge, pkt, ethHdr, ethTpidHdrOuter,
                                        ethTpidHdrInner, ipHdr, intf )
      if ret[ 0 ] is not None:
         break

   if ret[ 1 ]:
      pwMsg( 'Finished processing pkt', '-> VLAN', ret[ 1 ], tf=t9 )
   else:
      pwMsg( 'Finished processing pkt', tf=t9 )
   return [ ret ]

def decapMsg( *args, **kwargs ):
   tf = kwargs.get( 'tf', t8 )
   tf( 'mplsPwDecapHandler:', *args )

def lookupPwDecap( bridge, label ):
   routeKey = RouteKey.fromLabel( label )
   if not routeKey:
      decapMsg( 'no indexed route for label', label )
      return None

   route = bridge.pwLfib.lfibRoute.get( routeKey )
   if not route:
      decapMsg( 'no route for label', label )
      qt0( decapHandler, 'no route for label', qv( label ) )
      return None
   if route.viaSetKey == Tac.Value( 'Mpls::LfibViaSetKey' ):
      decapMsg( 'invalid route (viaSetKey)' )
      qt0( decapHandler, 'invalid route (viaSetKey)' )
      return None
   viaSet = bridge.pwLfib.viaSet.get( route.viaSetKey )
   if not viaSet:
      decapMsg( 'no viaSet available for', route.viaSetKey )
      qt0( decapHandler, 'no viaSet available for', route.viaSetKey )
      return None
   if not ( viaSet.hasViaType( LfibViaType.viaTypeMplsLdpPseudowire ) or
            viaSet.hasViaType( LfibViaType.viaTypeMplsBgpPseudowire ) or
            viaSet.hasViaType( LfibViaType.viaTypeMplsVpwsPseudowire ) or
            viaSet.hasViaType( LfibViaType.viaTypeMplsStaticPseudowire ) ):
      decapMsg( 'missing viaType' )
      qt0( decapHandler, 'missing viaType' )
      return None
   pwVia = bridge.pwLfib.pwVia.get( viaSet.viaKey[ 0 ] )
   if not pwVia:
      decapMsg( 'no via available for', viaSet.viaKey[ 0 ] )
      qt0( decapHandler, 'no via available for', viaSet.viaKey[ 0 ] )
      return None
   assert isinstance( pwVia, Tac.Type( 'Mpls::LfibPwVia' ) )
   return pwVia

def mplsPwDecapIntfVlan( bridge, mplsHdr, innerPkt, ethHdr, ethTpidHdrOuter,
                         ethTpidHdrInner, vlanIdOuter, vlanIdInner, tpidOuter,
                         tpidInner, retainOuterCos, retainInnerCos, parentIntf ):
   """Perform VLAN tag manipulation on decapped packet and select egress VLAN."""

   noMatch = ( None, {} )

   vlan = getIntfVlan( bridge.brConfig, parentIntf )
   if vlan is None:
      decapMsg( 'cannot find internal VLAN ID for parent intf', parentIntf )
      qt0( decapHandler, 'cannot find internal VLAN ID for parent intf',
           qv( parentIntf ) )
      return noMatch

   newData = maybeManipulateVlanTags( innerPkt, ethHdr, ethTpidHdrOuter,
                                      ethTpidHdrInner, vlanIdOuter, vlanIdInner,
                                      tpidOuter, tpidInner, retainOuterCos,
                                      retainInnerCos, mplsHdr.cos, decapMsg )
   if not newData:
      return noMatch

   return newData, { vlan }

def mplsPwGetPwToAcIntfEncap( bridge, ethHdr, ethTpidHdrOuter, ethTpidHdrInner,
                              pwVia ):
   """Look for an pwToAcIntfEncap for the packet with the given outer and inner
   dot1ad/dot1q headers. Return the encap and the headers that were used to match
   it."""

   pktOuterVid = ethTpidHdrOuter.tagControlVlanId if ethTpidHdrOuter else None
   pktInnerVid = ethTpidHdrInner.tagControlVlanId if ethTpidHdrInner else None
   pktOuterTpid = ( Tac.enumValue( EthType(), ethHdr.ethType ) if ethTpidHdrOuter
                    else None )
   pktInnerTpid = ( Tac.enumValue( EthType(), ethTpidHdrOuter.ethType )
                    if ethTpidHdrInner else None )
   pwToAcIntfEncap = None

   if ethTpidHdrInner:
      # look for double tag encap
      vlanTagEncap = _makeVlanTagEncap( pktOuterVid, pktOuterTpid,
                                        pktInnerVid, pktInnerTpid )
      intfEncap = IntfEncap( pwVia.pwIntfId, vlanTagEncap )
      pwToAcIntfEncap = bridge.pwStatus.pwToAcIntfEncap.get( intfEncap )
      if pwToAcIntfEncap:
         decapMsg( 'matched double-tagged decap', pwVia.pwIntfId, 'outer vid',
                   pktOuterVid, 'tpid', '0x%x' % pktOuterTpid, 'inner vid',
                   pktInnerVid, 'tpid', '0x%x' % pktInnerTpid )
         qt0( pwHandler, 'matched double-tagged decap', qv( pwVia.pwIntfId ),
              'outer vid', qv( pktOuterVid ), 'tpid', qv( pktOuterTpid ),
              'inner vid', qv( pktInnerVid ), 'tpid', qv( pktInnerTpid ) )

   if not pwToAcIntfEncap and not ethTpidHdrOuter:
      # packet is untagged; look for an 'untagged' encap
      vlanTagEncap = _makeVlanTagEncap( VlanTag.untagged, innerVid=VlanTag.untagged )
      intfEncap = IntfEncap( pwVia.pwIntfId, vlanTagEncap )
      pwToAcIntfEncap = bridge.pwStatus.pwToAcIntfEncap.get( intfEncap )
   elif not pwToAcIntfEncap and ethTpidHdrOuter and pktOuterVid != VlanId.invalid:
      # look for single tag encap
      vlanTagEncap = _makeVlanTagEncap( pktOuterVid, pktOuterTpid )
      intfEncap = IntfEncap( pwVia.pwIntfId, vlanTagEncap )
      pwToAcIntfEncap = bridge.pwStatus.pwToAcIntfEncap.get( intfEncap )
      if pwToAcIntfEncap:
         decapMsg( 'matched single-tagged decap', pwVia.pwIntfId, 'outer vid',
                   pktOuterVid, 'tpid', '0x%x' % pktOuterTpid )
         qt0( pwHandler, 'matched single-tagged decap', qv( pwVia.pwIntfId ),
              'outer vid', qv( pktOuterVid ), 'tpid', qv( pktOuterTpid ) )
      else:
         # look for "any tag" match (pwType4)
         vlanTagEncap = _makeVlanTagEncap( VlanTag.anyTag, pktOuterTpid )
         intfEncap = IntfEncap( pwVia.pwIntfId, vlanTagEncap )
         pwToAcIntfEncap = bridge.pwStatus.pwToAcIntfEncap.get( intfEncap )
         if pwToAcIntfEncap:
            decapMsg( 'matched any-tag decap', pwVia.pwIntfId )
            qt0( pwHandler, 'matched any-tag decap', qv( pwVia.pwIntfId ) )
      ethTpidHdrInner = None # not matched on inner header in any case

   if not pwToAcIntfEncap:
      # look for port encap
      intfEncap = IntfEncap( pwVia.pwIntfId, VlanTagEncap() )
      pwToAcIntfEncap = bridge.pwStatus.pwToAcIntfEncap.get( intfEncap )
      if pwToAcIntfEncap:
         decapMsg( 'matched port decap', pwVia.pwIntfId )
         qt0( pwHandler, 'matched port decap', qv( pwVia.pwIntfId ) )
      ethTpidHdrOuter = None # not matched on outer header in any case

   if not pwToAcIntfEncap:
      if pktInnerVid is not None:
         decapMsg( 'no matching encap for double tagged packet with outer vid',
                   pktOuterVid, 'tpid', '0x%x' % pktOuterTpid, 'inner vid',
                   pktInnerVid, 'tpid', '0x%x' % pktInnerTpid, 'on', pwVia.pwIntfId )
         qt0( decapHandler,
              'no matching encap for double tagged packet with outer vid',
              qv( pktOuterVid ), 'tpid', qv( pktOuterTpid ), 'inner vid',
              qv( pktInnerVid ), 'tpid', qv( pktInnerTpid ), 'on',
              qv( pwVia.pwIntfId ) )
      elif pktOuterVid is not None:
         decapMsg( 'no matching encap for single tagged packet with vid',
                   pktOuterVid, 'tpid', '0x%x' % pktOuterTpid, 'on', pwVia.pwIntfId )
         qt0( decapHandler, 'no matching encap for single tagged packet with vid',
              qv( pktOuterVid ), 'tpid', qv( pktOuterTpid ), 'on',
              qv( pwVia.pwIntfId ) )
      else:
         decapMsg( 'no matching encap for untagged packet on', pwVia.pwIntfId )
         qt0( decapHandler, 'no matching encap for untagged packet on',
              qv( pwVia.pwIntfId ) )

   return pwToAcIntfEncap, ethTpidHdrOuter, ethTpidHdrInner

def decapMplsPwFrameNewApi( bridge, mplsHdr, innerPkt, ethHdr, ethTpidHdrOuter,
                            ethTpidHdrInner, pwVia ):
   noMatch = ( None, {} )

   if not PseudowireIntfId.isPwInstanceId( pwVia.pwIntfId ):
      decapMsg( 'invalid pwVia.pwIntfId', pwVia.pwIntfId )
      qt0( decapHandler, 'invalid pwVia.pwIntfId', qv( pwVia.pwIntfId ) )
      return noMatch

   pwToAcIntfEncap, ethTpidHdrOuter, ethTpidHdrInner = (
      mplsPwGetPwToAcIntfEncap( bridge, ethHdr, ethTpidHdrOuter, ethTpidHdrInner,
                                pwVia ) )
   if not pwToAcIntfEncap:
      return noMatch

   parentIntf = parentIntfFromServiceIntf( pwToAcIntfEncap.destIntfEncap.intfId )
   return mplsPwDecapIntfVlan( bridge, mplsHdr, innerPkt, ethHdr, ethTpidHdrOuter,
                               ethTpidHdrInner,
                               pwToAcIntfEncap.destIntfEncap.encap.outer.vid,
                               pwToAcIntfEncap.destIntfEncap.encap.inner.vid,
                               pwToAcIntfEncap.destIntfEncap.encap.outer.tpid,
                               pwToAcIntfEncap.destIntfEncap.encap.inner.tpid,
                               pwToAcIntfEncap.retainOuterCos,
                               pwToAcIntfEncap.retainInnerCos,
                               parentIntf )

def decapMplsPwFrame( bridge, encapPkt, mplsHdr, pwVia ):
   noMatch = ( None, {} )

   # forward the packet depending on API version
   if not bridge.pwStatus.pwToAcIntfEncap:
      decapMsg( "No pwToAcIntfEncap, pseudowire not enabled" )
      qt0( decapHandler, "No pwToAcIntfEncap, pseudowire not enabled" )
      return noMatch

   decapMsg( 'pwVia', pwVia )

   if pwVia.pwType not in ( PseudowireType.pwType4, PseudowireType.pwType5 ):
      decapMsg( 'bad pwVia.pwType', pwVia.pwType )
      qt0( decapHandler, 'bad pwVia.pwType', qv( pwVia.pwType ) )
      return noMatch

   # PW label bos != True, when flow label is present

   # pwType4:
   #   ETHER-1 [ DOT1Q-1 ] MPLS_BOS ETHER-2 DOT1Q-2 [ PAYLOAD ]
   # pwType5:
   #   ETHER-1 [ DOT1Q-1 ] MPLS_BOS ETHER-2 [ DOT1Q-2 ] [ PAYLOAD ]
   innerPktOffset = mplsHdr.offset + 4 # pop MPLS label
   decapMsg( 'pop MPLS label', mplsHdr.label )
   if pwVia.flowLabelPresent:
      decapMsg( 'pop flow label' )
      innerPktOffset += 4

   if pwVia.controlWordPresent:
      decapMsg( 'pop control word' )
      innerPktOffset += 4 # pop control word

   innerData = encapPkt.stringValue[ innerPktOffset : ]
   ( innerPkt, headers, _ ) = PktParserTestLib.parsePktStr( innerData )
   innerEth = PktParserTestLib.findHeader( headers, 'EthHdr' )
   if not innerEth:
      decapMsg( 'no inner ethHdr' )
      qt0( decapHandler, 'no inner ethHdr' )
      return noMatch

   # pylint: disable-next=unbalanced-tuple-unpacking
   ethTpidHdrOuter, ethTpidHdrInner = getHeadersFromPkt( headers, dot1qOnly=False )

   return decapMplsPwFrameNewApi( bridge, mplsHdr, innerPkt, innerEth,
                                  ethTpidHdrOuter, ethTpidHdrInner, pwVia )

def maybeDecapMplsPwFrame( bridge, pkt, mplsHdr ):
   noMatch = ( None, {} )

   decapMsg( 'attempt PW decap entry for label', mplsHdr.label )
   if mplsHdr.ttl <= 1:
      decapMsg( 'TTL drop, ttl=', mplsHdr.ttl )
      qt0( decapHandler, 'TTL drop, ttl=', qv( mplsHdr.ttl ) )
      return noMatch

   # Find the PW decap entry for this label
   via = lookupPwDecap( bridge, mplsHdr.label )
   if not via:
      return noMatch

   return decapMplsPwFrame( bridge, pkt, mplsHdr, via )

def mplsPwDecapHandler( bridge, vlanId, destMac, data ):
   decapMsg( 'Processing pkt', tf=t9 )
   noMatch = [ ( None, {} ) ]

   if not bridge.pwLfib.lfibRoute:
      decapMsg( "No PW LFIB routes" )
      qt0( decapHandler, "No PW LFIB routes" )
      return noMatch

   # Check mpls routing is enabled
   if not bridge.mplsRoutingInfo_.mplsRouting:
      decapMsg( 'mplsRouting is disabled' )
      qt0( decapHandler, 'mplsRouting is disabled' )
      return noMatch

   if not isMpls( data ):
      decapMsg( "Non-MPLS frame (MplsPktHelper), skip" )
      qt0( decapHandler, "Non-MPLS frame (MplsPktHelper), skip" )
      return noMatch

   # Parse the packet
   ( pkt, headers, _ ) = PktParserTestLib.parsePktStr( data )
   ethHdr = PktParserTestLib.findHeader( headers, 'EthHdr' )
   mplsHdr = PktParserTestLib.findHeader( headers, "MplsHdr" )

   # Ensure it's an ether frame
   if not ethHdr:
      decapMsg( 'cannot find ethHdr' )
      qt0( decapHandler, 'cannot find ethHdr' )
      return noMatch

   # Check the MAC matches the bridgeMac
   if ethHdr.dst != bridge.bridgeMac_:
      decapMsg( 'dstMac', ethHdr.dst, 'is not bridgeMac', bridge.bridgeMac_ )
      qt0( decapHandler, 'dstMac', qv( ethHdr.dst ),
           'is not bridgeMac', qv( bridge.bridgeMac_ ) )
      return noMatch

   if not mplsHdr:
      decapMsg( 'not an MPLS frame' )
      qt0( decapHandler, 'not an MPLS frame' )
      return noMatch

   ret = maybeDecapMplsPwFrame( bridge, pkt, mplsHdr )
   if ret[ 1 ]:
      decapMsg( 'Finished processing pkt', '-> VLAN', ret[ 1 ], tf=t9 )
   else:
      decapMsg( 'Finished processing pkt', tf=t9 )
   return [ ret ]

def maybePwTrapOverride( bridge, pkt, ethHdr, ethTpidHdrOuter, ethTpidHdrInner,
                         intf ):
   func = maybePwTrapOverride.__name__
   noMatch = ( False, False, False )

   t8( func, "Checking Pw trap override for pkt" )

   # Look for a port encap match. The match must have been made without using the
   # outer header, i.e. the outer header returned by the function must be None
   acToDestIfEncap, ethTpidHdrOuter, _ = (
      mplsPwGetAcToDestIntfEncap( bridge, intf, ethHdr, ethTpidHdrOuter,
                                  ethTpidHdrInner ) )
   if not acToDestIfEncap or ethTpidHdrOuter:
      t2( func, "Port connector patch not configured" )
      return noMatch

   match = re.search( "01:80:c2:00:00:([0-9a-f]{2})", ethHdr.dst )
   if match:
      suffixHex = int( match.group( 1 ), 16 )
      if suffixHex == 0x03 or suffixHex == 0x07 or \
         ( 0x0A < suffixHex < 0x10 ):
         t1( func, "Override Trap for pkt" )
         return ( True, False, False )
   else:
      t2( func, "Mac address is not relevant for Pw trap override" )

   return noMatch

def mplsPwTrapOverrideHandler( bridge, routed, vlanId, destMac, data, srcPortName ):
   func = mplsPwTrapOverrideHandler.__name__
   noMatch = ( False, False, False )

   t8( func, 'Processing pkt' )

   if not bridge.pwStatus.acToDestIntfEncap:
      t8( func, "No acToDestIntfEncap, pseudowire not enabled" )
      qt0( func, "No acToDestIntfEncap, pseudowire not enabled" )
      return noMatch

   # Check mpls routing is enabled
   if not bridge.mplsRoutingInfo_.mplsRouting:
      t2( func, 'mplsRouting is disabled' )
      return noMatch

   try:
      vlanConfig = bridge.brConfig.vlanConfig.get( vlanId )
   except IndexError:
      # Catch IndexError for Tac::RangeException with invalid vlanId value
      vlanConfig = None
   if vlanConfig is None:
      t2( func, 'no vlanConfig for vlanId', vlanId )
      return noMatch
   if not vlanConfig.internal:
      t2( func, 'not supported on switch ports' )
      return noMatch

   # Parse the packet
   ( pkt, headers, _ ) = PktParserTestLib.parsePktStr( data )
   ethHdr = PktParserTestLib.findHeader( headers, 'EthHdr' )

   # Ensure it's an ether frame
   if not ethHdr:
      t2( func, 'cannot find ethHdr' )
      return noMatch

   # pylint: disable-next=unbalanced-tuple-unpacking
   ethTpidHdrOuter, ethTpidHdrInner = getHeadersFromPkt(
      headers, dot1qOnly=bool( not bridge.pwStatus.acToDestIntfEncap ) )

   for intf in vlanConfig.intf:
      # Iterate over interfaces, and check if the trap needs to be overidden
      if intf == 'Cpu':
         continue
      ret = maybePwTrapOverride( bridge, pkt, ethHdr, ethTpidHdrOuter,
                                 ethTpidHdrInner, intf )
      if ret[ 0 ]:
         break

   t8( func, 'Finished processing pkt for override' )
   return ret

def mplsBridgeInit( bridge ):
   t2( 'mplsBridgeInit' )

   em = bridge.em()
   bridge.mplsRoutingInfo_ = em.entity( 'routing/mpls/routingInfo/status' )
   bridge.mplsHwStatus_ = em.entity( 'routing/hardware/mpls/status' )
   bridge.mplsHwTunnelStatus_ = em.entity( 'routing/hardware/mpls/tunnel-status' )

   bridge.backupHwStatus_ = em.entity( 'routing/hardware/mpls/backup-status' )
   bridge.backupHwTunnelStatus_ = em.entity(
         'routing/hardware/mpls/backup-tunnel-status' )

   bridge.mplsHwCapability_ = em.entity( 'routing/hardware/mpls/capability' )

   shmemEm = SharedMem.entityManager( sysdbEm=em )
   bridge.proactiveArpNhgNexthopConfig = None
   if not inArfaMode:
      # ArfaProactiveArpHelper mounts it when in Arfa mode
      proactiveArpMountInfo = (
         Smash.mountInfo( 'writer',
                          collectionInfo=[ ( 'request', 1024 ) ] ) )
      bridge.proactiveArpNhgNexthopConfig = (
         shmemEm.doMount( 'arp/proactive/config/mplsNhg',
                          'Arp::ProactiveArp::ClientConfig',
                          proactiveArpMountInfo ) )
   bridge.lfib_ = shmemEm.doMount( 'mpls/transitLfib', 'Mpls::LfibStatus',
                                   Smash.mountInfo( 'keyshadow' ) )
   # Initialize SMs
   bridge.fwdHelper = mplsEtbaForwardingHelperFactory( bridge )

   bridge.pwRouteIndex = None
   bridge.mplsHardwareStatusSm = None
   bridge.mplsHwCapability_.mplsSupported = True
   bridge.mplsHwCapability_.mplsForwardIntoNexthopGroupSupported = True
   bridge.mplsHwCapability_.mplsEntropyLabelSupported = True
   bridge.mplsHwCapability_.mplsMultiLabelTerminationSupported = True
   bridge.mplsHwCapability_.mplsMulticastSupported = True
   bridge.mplsHwCapability_.mplsMvpnSupported = True

   nhgSm = mplsNhgHardwareStatusSmFactory( bridge )
   bridge.mplsNhgHardwareStatusSm = nhgSm

   if not inArfaMode:
      bridge.packetTracerSm = packetTracerSmFactory( bridge )

def mplsAgentInit( em ):
   t2( 'mplsAgentInit' )

   # Mouting Mpls config and status
   em.mount( 'routing/mpls/config', 'Mpls::Config', 'r' )
   em.mount( 'routing/mpls/status', 'Mpls::Status', 'rS' )
   em.mount( 'routing/mpls/routingInfo/status', 'Mpls::RoutingInfo', 'r' )
   em.mount( 'routing/hardware/mpls/status', 'Mpls::Hardware::Status', 'rS' )
   em.mount( 'routing/hardware/mpls/tunnel-status', 'Mpls::Hardware::Status', 'rS' )
   em.mount( 'routing/hardware/mpls/capability', 'Mpls::Hardware::Capability', 'w' )

   em.mount( 'routing/hardware/mpls/backup-status', 'Mpls::Hardware::Status', 'rS' )
   em.mount( 'routing/hardware/mpls/backup-tunnel-status', 'Mpls::Hardware::Status',
             'rS' )

   # PacketTracer SM mounts
   em.mount( 'packettracer/config', 'Tac::Dir', 'ri' )
   em.mount( 'packettracer/status', 'Tac::Dir', 'wi' )
   em.mount( 'packettracer/hwstatus', 'PacketTracer::HwStatus', 'w' )
   em.mount( 'packettracer/swstatus', 'PacketTracer::SwStatus', 'w' )

def mplsPwBridgeInit( bridge ):
   t2( 'mplsPwBridgeInit' )
   em = bridge.em()
   shmemEm = SharedMem.entityManager( sysdbEm=em )
   bridge.pwLfib = shmemEm.doMount( 'mpls/protoLfibInputDir/pseudowire',
                                    'Mpls::LfibStatus',
                                    Smash.mountInfo( 'keyshadow' ) )
   bridge.pwStatus = em.entity( 'pseudowire/status' )
   bridge.pwHwCapability = em.entity( 'routing/hardware/pseudowire/capability' )
   bridge.pfs = shmemEm.doMount( 'forwarding/srte/status',
                                 'Smash::Fib::ForwardingStatus',
                                 FibUtils.forwardingStatusInfo( 'reader' ) )
   bridge.pwHwCapability.mplsPseudowireSupported = True
   bridge.pwHwCapability.mplsPwachDecapSupported = True
   bridge.pwHwCapability.flowLabelSupported = True

def mplsPwAgentInit( em ):
   t2( 'mplsPwAgentInit' )
   em.mount( 'pseudowire/status', 'Pseudowire::Status', 'r' )
   em.mount( 'routing/hardware/pseudowire/capability',
             'Pseudowire::Hardware::Capability', 'w' )

def Plugin( ctx ):
   # In breadth tests, the ctx will be None, ignore it in this case
   if ctx is None:
      return
   global inArfaMode
   inArfaMode = ctx.inArfaMode()
   # MPLS pseudowire handlers
   if not ctx.inArfaMode():
      # Disable a bunch of these because we are not using them in cEOS land
      ctx.registerBridgeInitHandler( mplsPwBridgeInit )
      ctx.registerAgentInitHandler( mplsPwAgentInit )
      ctx.registerRoutingHandler( mplsPwRoutingHandler )
      ctx.registerRoutingHandler( mplsPwDecapHandler )
      ctx.registerTrapLookupHandler( mplsPwTrapOverrideHandler )
      ctx.registerRoutingHandler( mplsSwapPopRoutingHandler )

   # MPLS IP handlers
   ctx.registerBridgeInitHandler( mplsBridgeInit )
   ctx.registerAgentInitHandler( mplsAgentInit )
   FwdIntfDevice.registerFwdIntfRoutingHandler( mplsPushRoutingHandler )
   FwdIntfDevice.registerFwdIntfRoutingHandler( mplsFwdIntfSwapPopRoutingHandler )

   # MPLS FWD Interface (depends on mplsBridgeInit)
   FwdIntfEtba.pluginHelper( ctx )
   if not ctx.inArfaMode():
      MplsMldpEtbaDecap.pluginHelper( ctx )

   # L3 EVPN MPLS (depends on mplsBridgeInit)
   if not ctx.inArfaMode():
      MplsL3EvpnEtbaEncap.pluginHelper( ctx )

   MplsL3EvpnEtbaDecap.pluginHelper( ctx )
