#!/usr/bin/env python3
# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Tracing, Tac, Cell
from Arnet import  (
      PktParserTestLib,
      EthTestLib,
      MplsLib,
      IpTestLib,
      Subnet,
      IpGenAddr,
      IpGenPrefix,
)
from EbraTestBridge import IP_MCAST_BASE
from EbraTestBridge import IP_MCAST_END
import QuickTrace
from MrouteConsts import routingMulticastStatusSysdbPath
import SharedMem
import Smash
import MfibSmashTestLib
from EbraTestBridgeLib import (
      PKTEVENT_ACTION_NONE,
      PKTEVENT_ACTION_ADD,
      macAddrsAsStrings,
      applyVlanChanges )
from TypeFuture import TacLazyType
from MvpnLibTypes import tacMvpnRouteKey

PacketFormat = Tracing.HexDump
TunnelTableIdentifier = TacLazyType( "Tunnel::TunnelTable::TunnelTableIdentifier")
TunnelTableMounter = TacLazyType( "Tunnel::TunnelTable::TunnelTableMounter" )
VlanIntfId = TacLazyType( 'Arnet::VlanIntfId' )
MplsLabelConversion = Tac.Type( "Arnet::MplsLabelConversion" )
ArpKey = TacLazyType( 'Arp::Table::ArpKey' )

handle = Tracing.Handle( 'Mroute' )
t2 = handle.trace2
t8 = handle.trace8

qtInitialized = False

def qTrace():
   '''Initializes QuickTrace once, return the module itself'''
   global qtInitialized
   if not qtInitialized:
      QuickTrace.initialize( 'etba-routingHandlers.qt' )
      qtInitialized = True
   return QuickTrace

def qv( *args ):
   qTrace().Var( *args )

def qt0( *args ):
   qTrace().trace0( *args )

intfVlanIdTable = Tac.newInstance( "IntfVlanId::Table", "intfVlanIdTable" )

# How do I get these flags from the tac model?
flagForward = 1
flagAccept = 2

shmemEm = None

mrouteHandler = 'Mroute:'

def initVrfBidir( bridge, vrfName ):

   if vrfName in bridge.mrouteBidir_:
      return

   t2( "Mount bidir routestatus for vrf", vrfName )

   shmemEm.doMount( "routing/multicast/routestatus/%s/bidir" % vrfName,
                    "Smash::Multicast::Fib::Status", Smash.mountInfo( 'reader' ) )
   shmemEm.doMount( "routing/multicast/bidirstatus/%s" % vrfName,
                    "Smash::Multicast::Fib::BidirStatus",
                    Smash.mountInfo( 'reader' ) )
   shmemEm.doMount( "routing/multicast/bitmapperstatus/%s/bidir" % vrfName,
                    "BitMapper::SmashStatus", Smash.mountInfo( 'reader' ) )

   bridge.mrouteBidir_[ vrfName ] = bridge.sEm().getTacEntity(
         'routing/multicast/routestatus/%s/bidir' % vrfName )
   bridge.bidirStatus_[ vrfName ] = bridge.sEm().getTacEntity(
         'routing/multicast/bidirstatus/%s' % vrfName )
   bridge.bitMapperSmashBidir_[ vrfName ] = bridge.sEm().getTacEntity(
         'routing/multicast/bitmapperstatus/%s/bidir' % vrfName )

def initVrfPimsm( bridge, vrfName ):

   if vrfName in bridge.mroutePimsm_:
      return

   t2( "Mount pimsm routestatus for vrf", vrfName )

   shmemEm.doMount( "routing/multicast/routestatus/%s/sparsemode" % vrfName,
                    "Smash::Multicast::Fib::Status", Smash.mountInfo( 'reader' ) )
   shmemEm.doMount( "routing6/multicast/routestatus/%s/sparsemode" % vrfName,
                    "Smash::Multicast::Fib::Status", Smash.mountInfo( 'reader' ) )

   bridge.mroutePimsm_[ vrfName ] = bridge.sEm().getTacEntity(
         'routing/multicast/routestatus/%s/sparsemode' % vrfName )
   bridge.mroute6Pimsm_[ vrfName ] = bridge.sEm().getTacEntity(
         'routing6/multicast/routestatus/%s/sparsemode' % vrfName )

def initVrf( bridge, vrfName ):

   initVrfBidir( bridge, vrfName )
   initVrfPimsm( bridge, vrfName )

def RouteKey( s, g ):
   if s is None:
      s = IpGenPrefix( '0.0.0.0/0' )
   else:
      s = IpGenPrefix( s + '/32' )
   return Tac.Value( "Routing::Multicast::Fib::IpGenRouteKey",
                     s, IpGenPrefix( g +'/32' ) )

def IpGenRouteKey( s, g ):
   if s is None:
      s = IpGenPrefix( '0.0.0.0/0' )
   else:
      s = IpGenPrefix( s + '/32' )
   return Tac.Value( "Routing::Multicast::Fib::IpGenRouteKey",
                     s, IpGenPrefix( g +'/32' ) )


def lookup( bridge, s, g, bidir=True, vrfName='default' ):
   if bidir:
      routingTable = bridge.mrouteBidir_[vrfName].route
      s = None
   else:
      routingTable = bridge.mroutePimsm_[vrfName].route

   rt = routingTable.get( IpGenRouteKey(s, g) )
   # BUG106170 - Support *, G and prefix routes
   #  *, G routes not supported yet
   # if not rt:
   #    rt = routingTable.get( RouteKey( None, g ) )
   return rt

def prefixLen( p ):
   return int( p.split( '/' )[ 1 ] )

def hasActiveIpAddr( bridge, intfName, pmsiIntfId ):
   # Pmsi intf (SVI) is an internal SVI to deliver decapped mpls MVPN traffic into
   # the non default vrf. It is not expected to have an active ip address so, skip
   # hasActiveIpAddr check for pmsiIntfId
   if intfName == pmsiIntfId:
      return True
   ipIntfStatus = bridge.ipIntfStatus_.get( intfName )
   return ipIntfStatus and ( ipIntfStatus.activeAddrWithMask.address != "0.0.0.0" )

def getVrfNameFromIntf( bridge, intfName ):
   ipIntfStatus = bridge.ipIntfStatus_.get( intfName )
   if ipIntfStatus:
      return ipIntfStatus.vrf
   return 'default'

def getVrfId( bridge, vrfName ):
   if bridge.vrfNameStatus and bridge.vrfNameStatus.vrfIdExists( vrfName ) :
      return bridge.vrfNameStatus.nameToIdMap.vrfNameToId.get( vrfName )
   return None

def getPmsiIntfIdForVrfName( bridge, vrfName ):
   pmsiIntfId = None
   vrfId = getVrfId( bridge, vrfName )
   if vrfId is not None and bridge.mvpnIntfStatus and \
         bridge.mvpnIntfStatus.vrfIntfId.get( vrfId ):
      pmsiIntfId = bridge.mvpnIntfStatus.vrfIntfId[ vrfId ].pmsiIntfId
   return pmsiIntfId

def getVrfIdFromPmsiIntfId( bridge, vlanId ):
   for intfId, vrfId in bridge.mvpnIntfStatus.vrfIntfId.items():
      if intfId == vlanId:
         return vrfId
   return None

def getEncapTunnelId( bridge, vrfName, s, g ):
   routeKey = tacMvpnRouteKey( s, g, False )

   # From the MvpnTunnelEncapStatus , get MvpnTunnelVrfEncapStatus   
   vrfId = getVrfId( bridge, vrfName )
   mvpnTunnelVrfEncapStatus = bridge.mvpnTunnelEncapStatus.vrfStatus.get( vrfId )
   staticMvpnTunnelVrfEncapStatus = \
         bridge.staticMvpnTunnelEncapStatus.vrfStatus.get( vrfId )

   # From MvpnTunnelVrfEncapStatus , MvpnEncapTunnel
   if not mvpnTunnelVrfEncapStatus and not staticMvpnTunnelVrfEncapStatus:
      t2( 'Missing mvpnTunnelVrfEncapStatus and '\
            'staticMvpnTunnelVrfEncapStatus for vrfId: %s' % vrfId )
      return None

   mvpnTunnelVrfEncapStatus = mvpnTunnelVrfEncapStatus or\
         staticMvpnTunnelVrfEncapStatus

   encapTunnel = mvpnTunnelVrfEncapStatus.lookupEncapTunnel( routeKey )
   if not encapTunnel:
      t2( 'Missing encapTunnel for vrfId: %s, routeKey: %s ' % ( vrfId, routeKey ) )
      return None

   # From MvpnEncapTunnel, get MulticastTreeId
   multicastTreeId = encapTunnel.encapTunnel

   mldpP2mpTunnelRibEntry = bridge.mldpP2mpTunnelRib.entry.get( multicastTreeId )
   if not mldpP2mpTunnelRibEntry:
      t2( 'Missing mldpP2mpTunnelRibEntry for mcastTreeId: %s' %
          ( multicastTreeId ) )
      return None

   tId = mldpP2mpTunnelRibEntry.tunnelId
   return tId

def mvpnEncapFwd( bridge, vrfName, pmsiIntfId, ipHdr, data ):
   tId = getEncapTunnelId( bridge, vrfName, ipHdr.src, ipHdr.dst )
   if tId is None:
      return
   tunnelMfibEntry = bridge.tunnelMfib.entry.get( tId )
   if tunnelMfibEntry is None:
      t2( 'No entry found in bridge.tunnelMfib for', tId )
      return

   # Tunnel the packet
   for tId, tVia in tunnelMfibEntry.tunnelVia.items():
      nexthop, intfId, encapId = tVia.nexthop, tVia.intfId, tVia.encapId
      labelStackEncap = bridge.tunnelMfib.labelStackEncap.get( encapId )
      if labelStackEncap is None:
         t2( 'Could not find a label stack encap for', encapId )
         continue
      labelStack = MplsLabelConversion.unboundedOpToBoundedOp(
            labelStackEncap.labelStack )
      labelStack = MplsLib.getLabelStack( labelStack )
      mplsHdr = MplsLib.constructMplsHeader(
            labelStack, mplsTtl=ipHdr.ttl )
      srcMacAddr = bridge.bridgeMac()
      dstMacAddr = getDstMacAddr( bridge, intfId, nexthop )
      if not srcMacAddr or not dstMacAddr:
         t2( '%s %s: src MAC or dst MAC missing' % ( ipHdr.src, ipHdr.dst ) )
         continue
      _, ethHdr, _, _ = EthTestLib.newEthPkt( srcMacAddr, dstMacAddr,
            ethType = 'ethTypeMpls' )
      # Construct a tunnel packet and put it into this intf
      newData = ethHdr.pkt.bytesValue + mplsHdr + data[ ipHdr.offset : ]
      port = bridge.port.get( intfId )
      if port:
         t2( 'mvpnEncapFwd: port: ', port, ' pkt ', newData )
         port.sendFrame( newData, srcMacAddr, dstMacAddr, port.name(), None, None,
                         PKTEVENT_ACTION_NONE )
      elif VlanIntfId.isVlanIntfId( intfId ):
         vlanId = VlanIntfId.vlanId( intfId )
         t2( 'mvpnEncapFwd: vlanId: ', vlanId, ' pkt ', newData )
         vlanPri = 0
         vlanAct = PKTEVENT_ACTION_ADD
         ( srcMacAddr, dstMacAddr ) = macAddrsAsStrings( newData )
         vlanData = applyVlanChanges( newData, vlanPri, vlanId, vlanAct )
         srcPort = bridge.port[ 'Cpu' ]
         bridge.bridgeFrame( 
          vlanData, srcMacAddr, dstMacAddr, srcPort, False, None )

def getDstMacAddr( bridge, intf, hop ):
   '''
   Given an interface and hop, find the corresponding mac address.
   Note that the arp entry has a vrfId in the key
   '''
   if not hop or not intf:
      return None
   hop = IpGenAddr( str( hop ) )
   vrfName = getVrfNameFromIntf( bridge, intf )
   t8( 'getDstMacAddr vrfName', vrfName, 'intf: ', intf, 'hop: ', hop )
   vrfId = getVrfId( bridge, vrfName )
   arpKey = ArpKey( vrfId, hop, intf )
   afGetFunc = {
      'ipv4' : bridge.arpSmash.arpEntry.get,
      'ipv6': bridge.arpSmash.neighborEntry.get,
   }
   arpEntry = afGetFunc[ hop.af ]( arpKey )
   if arpEntry:
      t8( 'getDstMacAddr arfEntry: ', arpEntry )
      return arpEntry.ethAddr
   else:
      t2( 'getDstMacAddr no arpEntry found: ', arpEntry,
            ' for vrfName', vrfName, 'intf: ', intf, 'hop: ', hop )
      return None

def iif( r ):
   return r.iif

def iifFrr( r ):
   return r.iifFrr

def notify( r ):
   return r.notify

def isBidir( bridge, grpAddr, vrfName='default' ):
   # check group address in bidir range
   bidirGroup = bridge.bidirStatus_[vrfName].bidirGroup
   for prefix in bidirGroup:
      if prefix.contains( IpGenAddr( grpAddr ) ):
         return True
   return False

def iAmDf( bridge, intf, rpaId, vrfName='default' ):
   dfProfile = bridge.bidirStatus_[vrfName].dfProfile.get( intf )
   if not dfProfile:
      return False
   if rpaId > 64:
      return False
   dfBitmask = 1 << rpaId
   return ( dfProfile.dfBitmap & dfBitmask ) != 0

def oifs( bridge, r, bidir, vrfName='default' ):
   outgoingIntfIds = set()
   oifIndexNull = Tac.Value( "Smash::Multicast::Fib::OifIndex", 512 )
   curOifIndex = oifIndexNull
   curOifIndex = r.nextOif( curOifIndex )
   if bidir:
      mfibSmashTestEnv = MfibSmashTestLib.MfibSmashTestEnv(
         mfibSmash=bridge.mrouteBidir_[ vrfName ],
         bitMapperSmashStatus=bridge.bitMapperSmashBidir_[ vrfName ] )
      return set( mfibSmashTestEnv.oifs( r.key ) )
   else:
      oifIndexDict = bridge.mroutePimsm_[ vrfName ].oifIndexIntf

   while curOifIndex < 512:
      if oifIndexDict.get( curOifIndex ):
         intfId = oifIndexDict[ curOifIndex ].intf
         outgoingIntfIds.add( intfId )
      curOifIndex = r.nextOif( curOifIndex )
   return outgoingIntfIds

toCpuFlag = 1
# If we're not in forwardingStyleKernel or forwardingStyleBess, then
# remove the Cpu from the output interface list.
def floodsetIncludesCpu( bridge, vlanId, dstMacAddr, data ):
   if dstMacAddr < IP_MCAST_BASE or dstMacAddr > IP_MCAST_END:
      t8( 'floodsetIncludesCpu not claiming dest ', dstMacAddr )
      return None
   if ( bridge.mrouteEtbaStatus_.forwardingStyle in
           [ 'forwardingStyleKernel', 'forwardingStyleBess' ] ):
      return True
   # The rest of the function mimics the HW (FocalPoint) behavior
   intfName = intfVlanIdTable.vlanIdToIntfId( vlanId )
   vrfName = getVrfNameFromIntf( bridge, intfName )
   pmsiIntfId = getPmsiIntfIdForVrfName( bridge, vrfName )
   if not hasActiveIpAddr( bridge, intfName, pmsiIntfId ):
      return None

   # pylint: disable-msg=W0612
   ( _pkt, headers, _offset ) = PktParserTestLib.parsePktStr( data )

   ipHdr = PktParserTestLib.findHeader( headers, "IpHdr" )
   if not ipHdr:
      t8( 'Invalid ipHdr' )
      return None
   # BUG106169 - tracks adding this back
   # if intfName in bridge.mrouteStatus_.notifyLocalSources:
   #    ipIntfStatus = bridge.ipIntfStatus_[ intfName ]
   #    for addr in [ ipIntfStatus.activeAddrWithMask ] + \
   #        ipIntfStatus.activeSecondaryWithMask.keys():
   #       subnet = Subnet( '%s/%s' % ( addr.address, addr.len ) )
   #       if subnet.containsAddr( ipHdr.src ):
   #          t8( 'Including cpu due to notifyLocalSource' )
   #          return True
   #    return None
   initVrf( bridge, vrfName )

   bidir = isBidir( bridge, ipHdr.dst, vrfName )
   r = lookup( bridge, ipHdr.src, ipHdr.dst, bidir, vrfName )
   t8( 'Multicast pkt s, g: ', ipHdr.src, ipHdr.dst )
   if not r:
      t8( 'No matching route in mfib' )
      return True
   if intfName != iif( r ) and intfName != iifFrr( r ) and not bidir:
      # RPF failure. At some point we may want to send the
      # ETH_P_ARISTA_RPF_FAILURE pkt as well.
      t8( 'RPF failure %s, intfName %s, iif( r ) %s, iifFrr( r ) %s'
            % ( r.key, intfName, iif( r ), iifFrr( r ) ) )
      return True
   if r.routeFlags.toCpu:
      t8( '(toCpu) route wants the pkt to be copied to cpu.' )
      return True
   return None
      
# Currently we check the routeConfig object and check the incoming
# interface has an IP address.  Also currently we forward on all the
# egress interfaces regardless of their IP configuration.
def route( bridge, vlanId, destMac, data ):
   noMatch = [ ( None, {} ) ]
   t2( 'route vlanId ', vlanId, ' destMac ', destMac )
   if not vlanId or vlanId < 1 or vlanId > 4094:
      t8( 'vlanId is invalid ', vlanId )
      return noMatch

   if bridge.mrouteEtbaStatus_.forwardingStyle == 'forwardingStyleKernel':
      t8( 'deferring to kernel due to forwardingStyleKernel' )
      qt0( mrouteHandler, 'deferring to kernel due to forwardingStyleKernel' )
      return noMatch
   if bridge.mrouteEtbaStatus_.forwardingStyle == 'forwardingStyleBess':
      t8( 'deferring to BESS due to forwardingStyleBess' )
      qt0( mrouteHandler, 'deferring to BESS due to forwardingStyleBess' )
      return noMatch

   # Compare destMac to virtual Mac for Ip routing
   if destMac < IP_MCAST_BASE or destMac > IP_MCAST_END:
      t8( 'not claiming dest ', destMac )
      qt0( mrouteHandler, 'not claiming dest ', qv( destMac ) )
      return noMatch
   # Get Ip headers
   # pylint: disable-msg=W0612
   ( pkt, headers, _offset ) = PktParserTestLib.parsePktStr( data )
   ipHdr = PktParserTestLib.findHeader( headers, "IpHdr" )
   if not ipHdr:
      t8( 'no ip headers' )
      qt0( mrouteHandler, 'no ip headers' )
      return noMatch
   if not Subnet( '224.0.0.0/4' ).containsAddr( ipHdr.dst ):
      t8( 'not ip multicast dest ',  ipHdr.dst )
      qt0( mrouteHandler, 'not ip multicast dest ', qv( ipHdr.dst ) )
      return noMatch
   if ipHdr.ttl <= 1:
      t8( 'packet not routed because of ttl of 1' )
      qt0( mrouteHandler, 'packet not routed because of ttl of 1' )
      return noMatch
   intfName = intfVlanIdTable.vlanIdToIntfId( vlanId )
   t8( 'route intfName ', intfName )

   if intfName not in bridge.ipIntfStatus_:
      t8( "%s %s: intf %s not in IP intf status" %
          ( ipHdr.src, ipHdr.dst, intfName ) )
      return noMatch
   vrfName = getVrfNameFromIntf( bridge, intfName )
   t8( 'init vrf', vrfName )
   initVrf( bridge, vrfName )
   bidir = isBidir( bridge, ipHdr.dst, vrfName )
   t8( 'init vrf', vrfName, 'for bidir', bidir )

   pmsiIntfId = getPmsiIntfIdForVrfName( bridge, vrfName )
   if not hasActiveIpAddr( bridge, intfName, pmsiIntfId ):
      t8( 'no active ip on incoming interface' )
      qt0( mrouteHandler, 'no active ip on incoming interface' )
      return noMatch

   r = lookup( bridge, ipHdr.src, ipHdr.dst, bidir, vrfName )
   if not r:
      t8( 'no route match s, g: ', ipHdr.src, ipHdr.dst )
      qt0( mrouteHandler, 'no route match s, g: ', qv( ipHdr.src ), qv( ipHdr.dst ) )
      return noMatch
   if not bidir:
      routeStatus = bridge.mrouteHwStatus_.vrfStatus[ vrfName ].route.get( \
         RouteKey( ipHdr.src, ipHdr.dst ) )
      if routeStatus:
         routeStatus.lastPollTime = Tac.now()
   
   rpfIntfName = iif( r )
   rpfFrr = iifFrr( r )
   t8( 'match', r.key.s, r.key.g, 'iif', rpfIntfName, '/', intfName,
       'iifFrr', rpfFrr )

   if intfName == rpfFrr and not bidir:
      t8( 'Packet arrives on rpfFrr', rpfFrr, ', drop the pkt', r.key )
      qt0( mrouteHandler, 'Packet arrives on rpfFrr', qv( rpfFrr ),
           'drop the pkt', qv( r.key ) )
      return noMatch


   if intfName != rpfIntfName and not bidir and rpfIntfName != pmsiIntfId:
      t8( 'RPF failure, incoming', intfName,
          ', expecting', rpfIntfName, ', not routing the pkt', r.key, ' pmsiIntfId ',
          pmsiIntfId )
      qt0( mrouteHandler, 'RPF failure, incoming', qv( intfName ),
           ', expecting', qv( rpfIntfName ), ', not routing the pkt', qv( r.key ),
           ', pmsiIntfId ', qv( pmsiIntfId ) )
      return noMatch

   intfNameSet = oifs( bridge, r, bidir, vrfName )

   if bidir:
      # This is BiDIR route, remove ingress interface
      intfNameSet.discard( intfName )
      # check dfProfile
      if not iAmDf( bridge, intfName, r.rpaId, vrfName ):
         t8( 'Not DF for interface', intfName, ', rpaId', r.rpaId, ', drop packet' )
         qt0( mrouteHandler, 'Not DF for interface', qv( intfName ),
              ', rpaId', qv( r.rpaId ), ', drop packet' )
         return noMatch

   if not intfNameSet:
      t8( 'Empty oifs:', r.key )
      qt0( mrouteHandler, 'Empty oifs:', qv( r.key ) )
      return noMatch

   if r.routeFlags.toCpu:
      # Mimic the HW (FocalPoint) behavior.
      t8( 'route with toCpuFlag, so not routing the pkt' )
      qt0( mrouteHandler, 'route with toCpuFlag, so not routing the pkt' )
      return noMatch

   # Edit the packet src MAC and ttl
   pkt = Tac.newInstance( "Arnet::Pkt" )
   pkt.bytesValue = data
   
   ( headers, _offset ) = PktParserTestLib.parsePkt( pkt )
   hdrNames = [ i[ 0 ] for i in headers ]
   ethHdr = headers[ hdrNames.index( 'EthHdr' ) ][ 1 ]
   ipHdr = headers[ hdrNames.index( 'IpHdr' ) ][ 1 ]
   
   # update the src MAC address 
   ethHdr.src = bridge.bridgeMac_

   # update the dst MAC address for the multicast packet
   ethHdr.dst = IpTestLib.ipMcastMacAddr( ipHdr.dst )
   # if the ttl is less or equal to 1 we don't route
   if ipHdr.ttl <= 1:
      t8( 'not routing packet with ttl of ', ipHdr.ttl )
      qt0( mrouteHandler, 'not routing packet with ttl of ', qv( ipHdr.ttl ) )
      return noMatch
   ipHdr.ttl = ipHdr.ttl - 1
   # recompute the ip header checksum
   ipHdr.checksum = 0
   ipHdr.checksum =  ipHdr.computedChecksum

   # Check for PmsiIntfId in the Outgoing VlanIdSet
   if pmsiIntfId and pmsiIntfId in intfNameSet:
      intfNameSet.remove( pmsiIntfId )
      mvpnEncapFwd( bridge, vrfName, pmsiIntfId, ipHdr, pkt.bytesValue )
        
   vlanIdSet = { intfVlanIdTable.intfIdToVlanId( intfName )
                 for intfName in intfNameSet }

   # Its possible that we don't have an intfId to vlanId mapping as yet (or at all
   # as in the case of loopbacks)
   if 0 in vlanIdSet:
      vlanIdSet.remove( 0 )

   if not vlanIdSet:
      t8( 'Empty vlanIdSet:', r.key )
      qt0( mrouteHandler, 'Empty vlanIdSet:', qv( r.key ) )
      return noMatch

   data = pkt.bytesValue
   t8( 'vlans', vlanIdSet, 'data:', data )
   return [ ( data, vlanIdSet ) ]

def bridgeInit( bridge ):

   t2( 'bridgeInit' )
   t2( 'Mounting Smashed Mfib' )

   bridge.mroutePimsm_ = {}
   bridge.mroute6Pimsm_ = {}

   bridge.mrouteBidir_ = {}
   bridge.bitMapperSmashBidir_ = {}
   bridge.bidirStatus_ = {}

   bridge.mroutePimsm_[ 'default' ] = bridge.sEm().getTacEntity(
         'routing/multicast/routestatus/default/sparsemode' )
   bridge.mroute6Pimsm_[ 'default' ] = bridge.sEm().getTacEntity(
         'routing6/multicast/routestatus/default/sparsemode' )

   bridge.mrouteBidir_[ 'default' ] = bridge.sEm().getTacEntity(
         'routing/multicast/routestatus/default/bidir' )
   bridge.bitMapperSmashBidir_[ 'default' ] = bridge.sEm().getTacEntity(
         'routing/multicast/bitmapperstatus/default/bidir' )
   bridge.bidirStatus_[ 'default' ] = bridge.sEm().getTacEntity(
         'routing/multicast/bidirstatus/default' )
   bridge.mrouteEtbaStatus_ = bridge.em().entity( 'routing/multicast/etba/status' )

   bridge.mrouteStatus_ = bridge.mroutePimsm_[ 'default' ]
   bridge.mroute6Status_ = bridge.mroute6Pimsm_[ 'default' ]

   bridge.mrouteHwStatus_ = bridge.em().entity( 'routing/hardware/multicast/status' )
   bridge.mroute6HwStatus_ = \
      bridge.em().entity( 'routing6/hardware/multicast/status' )

   bridge.ipIntfStatus_ = bridge.em().entity( 'ip/status' ).ipIntfStatus
   bridge.allVrfStatusLocal_ = bridge.em().entity(
         Cell.path( 'ip/vrf/status/local' ) )

   # Routing hw status entities capabilities moved to ArfaPlugin to unblock cleanup
   # code used in some test infra when testing using Arfa

   bridgingConfig = bridge.em().entity( 'bridging/config' )
   bridge.intfVlanIdSm_ = Tac.newInstance(
      "IntfVlanId::ConfigSm", bridgingConfig, None, intfVlanIdTable )

   bridge.vrfNameStatus = bridge.em().entity( Cell.path( 'vrf/vrfNameStatus' ) )

   # Mvpn related status
   tacType = 'Routing::Multicast::MvpnVrfStatus'
   bridge.mvpnVrfStatus = shmemEm.doMount(
      Tac.Type( tacType ).mountPath( "ipv4", "bgp" ),
      tacType, Smash.mountInfo( 'shadow' ) )
   tacType = 'Routing::Multicast::MvpnIntfStatus'
   bridge.mvpnIntfStatus = shmemEm.doMount(
      Tac.Type( tacType ).mountPath(), tacType,
      Smash.mountInfo( 'keyshadow' ) )
   tacType = Tac.Type( 'Routing::Multicast::MvpnTunnelEncapStatus' )
   mountPath = tacType.mountPath( 'ipv4', 'bgp' )
   bridge.mvpnTunnelEncapStatus = bridge.em().entity( mountPath )
   mountPath = tacType.mountPath( 'ipv4', 'staticMvpn' )
   bridge.staticMvpnTunnelEncapStatus = bridge.em().entity( mountPath )
   tacType = 'Tunnel::TunnelTable::MldpP2mpTunnelRib'
   mountPath = Tac.Type( tacType ).mountPath()
   bridge.mldpP2mpTunnelRib = bridge.em().entity( mountPath )
   shmemEm.doMount( "tunnel/tunnelMfib",
                    "Tunnel::TunnelFib::TunnelFib", Smash.mountInfo( 'keyshadow' ) )
   bridge.tunnelMfib = bridge.sEm().getTacEntity( "tunnel/tunnelMfib" )
   bridge.arpSmash = shmemEm.doMount( 'arp/status',
                                      'Arp::Table::Status',
                                      Smash.mountInfo( 'shadow' ) )

def agentInit( em ):
   t2( 'agentInit' )
   global shmemEm
   shmemEm = SharedMem.entityManager( sysdbEm=em )

   shmemEm.doMount( "routing/multicast/routestatus/default/sparsemode",
                    "Smash::Multicast::Fib::Status", Smash.mountInfo( 'reader' ) )
   shmemEm.doMount( "routing6/multicast/routestatus/default/sparsemode",
                    "Smash::Multicast::Fib::Status", Smash.mountInfo( 'reader' ) )

   shmemEm.doMount( "routing/multicast/bitmapperstatus/default/bidir",
                    "BitMapper::SmashStatus", Smash.mountInfo( 'reader' ) )
   shmemEm.doMount( "routing/multicast/routestatus/default/bidir",
                    "Smash::Multicast::Fib::Status", Smash.mountInfo( 'reader' ) )
   shmemEm.doMount( "routing/multicast/bidirstatus/default",
                    "Smash::Multicast::Fib::BidirStatus",
                    Smash.mountInfo( 'reader' ) )

   shmemEm.doMount( "routing/multicast/status", "Smash::Multicast::Fib::Status",
                    Smash.mountInfo( 'shadow' ) )
   shmemEm.doMount( "routing6/multicast/status", "Smash::Multicast::Fib::Status",
                    Smash.mountInfo( 'shadow' ) )

   # mounting config and status.
   em.mount( routingMulticastStatusSysdbPath,
             'Routing::Multicast::Fib::Status', 'r' )
   em.mount( 'routing/multicast/etba/status',
             'Routing::Multicast::Etba::Status', 'r' )

   em.mount( 'routing/hardware/multicast/status',
             'Routing::Multicast::Fib::Hardware::Status', 'w' )
   em.mount( 'routing6/hardware/multicast/status',
             'Routing::Multicast::Fib::Hardware::Status', 'w' )

   em.mount( 'bridging/config', 'Bridging::Config', 'r' )
   mg = em.activeMountGroup()
   Tac.Type( "Ira::IraIpStatusMounter" ).doMountEntities( mg.cMg_, True, False )
   # this is to enable multicast routing support
   em.mount( 'routing/hardware/statuscommon',
             'Routing::Hardware::StatusCommon', 'w' )
   em.mount( 'routing/hardware/status', 'Routing::Hardware::Status', 'w' )
   em.mount( 'routing6/hardware/status', 'Routing6::Hardware::Status', 'w' )
   em.mount( Cell.path( 'ip/vrf/status/local' ), 'Ip::AllVrfStatusLocal', 'r' )
   em.mount( Cell.path( 'vrf/vrfNameStatus' ),
         'Vrf::VrfIdMap::NameToIdMapWrapper', 'r' )

   # MvpnTunnelEncapStatus & MvpnTunnelRib
   tacType = 'Routing::Multicast::MvpnTunnelEncapStatus'
   mountPath = Tac.Type( tacType ).mountPath( 'ipv4', 'bgp' )
   em.mount( mountPath, tacType, 'r' )
   mountPath = Tac.Type( tacType ).mountPath( 'ipv4', 'staticMvpn' )
   em.mount( mountPath, tacType, 'r' )
   tacType = 'Tunnel::TunnelTable::MulticastTunnelRib'
   mountPath = Tac.Type( tacType ).mountPath()
   em.mount( mountPath, tacType, 'r' )

def Plugin( ctx ):
   ctx.registerBridgeInitHandler( bridgeInit )
   ctx.registerAgentInitHandler( agentInit )
   if not ctx.inArfaMode():
      ctx.registerRoutingHandler( route )
   ctx.registerFloodsetIncludesCpuHandler( floodsetIncludesCpu )
