#!/usr/bin/env python3
# Copyright (c) 2008-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
# pylint: disable=consider-using-in

import Tac, Tracing, Ethernet, socket, struct, re
from binascii import hexlify
from cryptography.exceptions import InvalidTag
from cryptography.hazmat.backends import default_backend
from cryptography.hazmat.primitives.ciphers import Cipher, algorithms, modes
from EbraTestBridgePort import EbraTestPort
import Arnet.Device
from Arnet import IpGenAddr
from Arnet.PktParserTestLib import parsePktStr, findHeader
from Arnet.VxlanPktTestLib import VxlanHdrSize
from BridgingHostEntryType import isEntryTypeController
from EbraTestBridgePythonImpl import getPythonIntoCppShimContext
from EbraTestBridgeLib import (
   makeTap,
   macAddrsAsStrings,
   applyVlanChanges,
   PKTEVENT_ACTION_ADD,
)
from if_ether_arista import ETH_P_ARISTA_VXLAN_ENCAP, ETH_P_ARISTA_VXLAN_ARP, \
      VXLAN_DOT1QTUNNEL_FLAG, VXLAN_CPUPORT_FLAG, VXLAN_MLAG_SRCPORT_FLAG, \
      VXLAN_SRC_VTEP_EXT_FLAG, VXLAN_SRC_VTEP_V6_FLAG, VXLAN_MPLS_SRCPORT_FLAG
import VxlanLib
import SharedMem
import Smash
import Shark
import MlagMountHelper
import Toggles.VxlanToggleLib
import random

handle = Tracing.Handle( 'EbraVxlan' )
t0 = handle.trace0    #
t1 = handle.trace1    #
t2 = handle.trace2    #
t4 = handle.trace4    #
t5 = handle.trace5    #
t6 = handle.trace6    # less frequent tracing/exception cases
t7 = handle.trace7    # per-packet exception cases
t8 = handle.trace8    # per-packet tracing

helper = None

# Global handle on Vxlan agent devices
vxlanAgentDevs = None

# Global config directory
vxlanConfigDir = None

# Global status directory
vxlanStatusDir = None

# Global H/W status directory
vxlanHwStatusDir = None
vxlanFdbStatus = None
vxlanFdbMultiVtepStatus = None
swGroupIdColl = None

# Global Bridging and Vxlan H/W capabilities
bridgingHwCapabilities = None

# Global Routing H/W Status
routingHwStatus = None
routing6HwStatus = None

# Global varMac status directory
vrMacStatus = None

# Global Vxlan Learned Hosts (in HwStatusDir)
vxlanLearnedHost = None
multiVtepLearnedHost = None

# Global view of floodset published by L2Rib
l2RibOutput = None
l2RibLoadBalance = None
l2RibDest = None
l2RibFloodSet = None
vxlanStaticSource = 0x00000004
vxlanDynamicSource = 0x00000400

# Global mlagStatus
mlagStatusMount = None

# Global EVPN status for ethernet segment handling
evpnStatus = None

# global flag
mountCompleted = False

BridgingEntryType = Tac.Type( "Bridging::EntryType" )
VxlanEntryType = VxlanLib.EntryType()
BumReplicationMode = Tac.Type( "Vxlan::BumReplicationMode" )
VtepType = Tac.Type( "L2Rib::VtepType" )
VxlanTunnelType = Tac.Type( "Vxlan::VxlanTunnelType" )
destIdNotInSmash = Tac.Value( 'L2Rib::Constants' ).objIdNotInSmash
EthAddr = Tac.Type( 'Arnet::EthAddr' )
TunnelIpAddrList = Tac.Type( 'Vxlan::TunnelIpAddrList' )
SwTunnelGroupId = Tac.Type( 'Vxlan::SwTunnelGroupId' )

# global DMA driver seq No
encapSeqNo = 0

# Global L2Rib host and dest input of type "vxlan-dynamic"
vxlanDynamicL2RibHostInput = None
vxlanDynamicL2RibDestInput = None

# Global remote VTEP config
remoteVtepConfigDir = None

# Global vtep HW Status
vtepHwStatus = None

# Global IPSec status
ipsecStatus = None

#Global decap config
decapConfig = None
defaultVrfId  = Tac.newInstance( 'Vrf::VrfIdMap::VrfId' ).defaultVrf
defaultVrfName = Tac.Type( 'L3::VrfName' ).defaultVrf
UnderlayAddrType =  Tac.Type( 'Routing::Multicast::UnderlayAddr' )
UnderlayRouteType = Tac.Type( "Routing::Multicast::UnderlayRoute" )
uSrcV4 = UnderlayAddrType( "" )
uSrcV4.af = "ipv4"

# Global Arfa mode
inArfaMode = None

class VxlanAgentDevs:
   """ All interactions with Vxlan agent defined here """
   def __init__( self, bridge ):
      # XXX: the kernel won't be able to find these devices if not properly named
      self.vxlanTapFile_ = None
      self.txrawTapFile_ = None
      self.vxlanTapDevice_ = None
      self.vxlanTapPam_ = None

      dev = 'vxlan' if bridge.inNamespace() else '%s-vxlan' % bridge.name()

      self.vxlanTapDevice_ = makeTap( dev, hw=bridge.bridgeMac() )

      dev = 'txraw' if bridge.inNamespace() else '%s-txraw' % bridge.name()
      self.txrawTapDevice_ = makeTap( dev, hw=bridge.bridgeMac(), mtu=10000 )

      vxlanName = "None" if not self.vxlanTapDevice_ else self.vxlanTapDevice_.name
      t0( 'created Vxlan Tap Devices {} {} {}'.format(
         vxlanName, self.txrawTapDevice_.name, self.vxlanTapPam_ ) )
      self.bridge_ = bridge

   def onActive( self ):
      if not self.vxlanTapFile_ and not inArfaMode:
         self.vxlanTapFile_ = Tac.File( self.vxlanTapDevice_.fileno(),
                                        self._vxlanTapReadHandler,
                                        self.vxlanTapDevice_.name,
                                        readBufferSize=16000 )
      if not self.txrawTapFile_:
         self.txrawTapFile_ = Tac.File( self.txrawTapDevice_.fileno(),
                                        self._txrawTapReadHandler,
                                        self.txrawTapDevice_.name,
                                        readBufferSize=16000 )

   def __del__( self ):
      self.vxlanTapFile_ = None
      self.txrawTapFile_ = None
      del self.vxlanTapDevice_
      del self.txrawTapDevice_

   def _vxlanTapReadHandler( self, data ):
      """ Process frames from Vxlan Agent """
      # No packets expected on this tap, so if we receive any we drop
      # We still need to consume the packet to prevent memory leak
      t0( 'received unexpected frame from vxlan netdevice' ) 

   def _txrawTapReadHandler( self, data ):
      """ Process decap frames from Vxlan Agent """
      if self.bridge_ is None:
         t0( 'txraw dropping frame because there is no bridge configured' )
         return

      if not self.bridge_.vxlanPort_:
         t2( 'txraw dropping frame because vxlan port does not exist' )
         return

      #if inArfaMode:
      #   t0( "In Arfa mode, calling Arfa routeFrame" )
      #   shim = getPythonIntoCppShimContext()
      #   shim.routeFrame( data, "Vxlan1" )
      #   return
      
      ( srcMacAddr, dstMacAddr ) = macAddrsAsStrings( data )
      t0( 'txraw received frame for smac(%s)/dmac(%s)' % \
            ( srcMacAddr, dstMacAddr ) )
      t7( 'Raw frame is', data )
      self.bridge_.processFrame( data, srcMacAddr, dstMacAddr,
                                 self.bridge_.vxlanPort_[ 'Vxlan1' ], False, 0 )

class VxlanStatusDirReactor( Tac.Notifiee ):
   notifierTypeName = 'Vxlan::VxlanStatusDir'
   def __init__( self, bridge ):
      self.vxlanStatusDir_ = vxlanStatusDir
      self.bridge_ = bridge
      self.vxlanStatus_ = {}
      self.vxlanStatusReactor_ = {}
      Tac.Notifiee.__init__( self, vxlanStatusDir )

   @Tac.handler( 'vxlanStatus' )
   def handleVxlanStatus( self, intfName ):
      t2( 'VxlanStatusDirReactor::handleVxlanStatus::%s' % intfName )
      if intfName in self.vxlanStatusDir_.vxlanStatus:
         self.vxlanStatus_[ intfName ] = vxlanStatusDir.vxlanStatus[ intfName ]
         self.vxlanStatusReactor_[ intfName ] = VxlanStatusReactor(
               self.vxlanStatus_[ intfName ], self.bridge_ )

   def __del__( self ):
      for intf in self.vxlanStatus_:
         t2( 'VxlanStatusDirReactor::__del__() %s' % intf )
         self.vxlanStatus_[ intf ] = None

class VxlanStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Vxlan::VxlanStatus'
   def __init__( self, vxlanStatus, bridge ):
      t2( 'VxlanStatusReactor::__init__()' )
      self.vxlanStatus_ = vxlanStatus
      self.bridge_ = bridge
      self.vlanToLearnRestrictReactor_ = {}
      Tac.Notifiee.__init__( self, vxlanStatus )
      self.handleVlanToLearnRestrict( vlanId=None )

   @Tac.handler( 'vlanToLearnRestrict' )
   def handleVlanToLearnRestrict( self, vlanId=None ):
      t2( 'handleVlanToLearnRestrict %s' % ( vlanId ) )

      if vlanId is None:
         for vlan in self.vxlanStatus_.vlanToLearnRestrict:
            self.handleVlanToLearnRestrict( vlan )
         return

      t2( 'handleVlanToLearnRestrict create a new reactor for vlan=%d' % vlanId )
      # If the key doesn't exist, handleCollectionChange will delete the reactor
      Tac.handleCollectionChange( VlanToLearnRestrictReactor, vlanId,
                self.vlanToLearnRestrictReactor_,
                self.notifier_.vlanToLearnRestrict,
                reactorArgs=( vxlanHwStatusDir.learnStatus, self.vxlanStatus_,
                              self.bridge_ ),
                reactorTakesKeyArg=True,
                reactorFilter=None )

   @Tac.handler( 'vlanToClearCounterRequestTime' )
   def vlanToClearCounterRequestTime( self, vlanId ):
      t2( 'vlanToClearCounterRequestTime %s' % ( vlanId ) )

      learnList = vxlanHwStatusDir.learnStatus.learnStatusList
      if vlanId is None:
         for vlan in self.vxlanStatus_.vlanToClearCounterRequestTime:
            t4( 'calling vlanToClearCounterRequestTime %d' % vlan )
            self.vlanToClearCounterRequestTime( vlan )
         return

      if vlanId in self.vxlanStatus_.vlanToClearCounterRequestTime:
         time = self.vxlanStatus_.vlanToClearCounterRequestTime[ vlanId ]
         # Did we get a new clearCounterRequest?
         if vlanId in learnList and time != learnList[ vlanId ].lastClearTime:
            t4( 'handleVlanToClearCounterRequestTime: clear counters vlan %d' % 
                ( vlanId ) )
            for p in learnList[ vlanId ].numMatches:
               learnList[ vlanId ].numMatches[ p ] = 0
            learnList[ vlanId ].numMatchAny = 0
            learnList[ vlanId ].numMatchList = 0
            learnList[ vlanId ].numMatchFloodList = 0
            learnList[ vlanId ].numRejectList = 0
            learnList[ vlanId ].numRejectFloodList = 0
            learnList[ vlanId ].lastClearTime = time

# Minimal reactor, used only when in Arfa mode, to handle calling into the
# mlagHostTableReactor, which will remain in the Etba plugin until the work to move
# these MAC sources completely into L2Rib is completed.
class VtiStatusArfaReactor( Tac.Notifiee ):
   notifierTypeName = 'Vxlan::VtiStatus'

   def __init__( self, vtiStatus, bridge ):
      t2( 'VtiStatusArfaReactor::__init__()' )
      assert bridge
      self.vtiStatus_ = vtiStatus
      self.bridge_ = bridge
      Tac.Notifiee.__init__( self, vtiStatus )
      self.handleLinkStatus()

   @Tac.handler( 'linkStatus' )
   def handleLinkStatus( self ):
      t2( 'linkStatus %s' % self.notifier_.linkStatus )
      if self.notifier_.linkStatus == 'linkUp':
         if self.bridge_.mlagHostTableReactor_ is not None:
            self.bridge_.mlagHostTableReactor_.handleMlagHostEntry()

   @Tac.handler( 'extendedVlanToVniMap' )
   def handleExtVlanToVniMap( self, evid=None ):
      if evid is None:
         for extVlan in self.vtiStatus_.extendedVlanToVniMap:
            self.handleExtVlanToVniMap( extVlan )
         return

      t2( 'handleExtVlanToVniMap %s/%d' % ( self.vtiStatus_.intfId, evid ) )

      vniSource = self.vtiStatus_.extendedVlanToVniMap.get( evid )
      vni = vniSource.vni if vniSource and vniSource.vni != \
            Tac.Value( "Vxlan::VniOrNone" ) else None

      if vni is not None:
         tacEvid = Tac.Value( 'Bridging::ExtendedVlanId', evid )
         if tacEvid.inDot1qRange():
            if self.bridge_.mlagHostTableReactor_ is not None:
               self.bridge_.mlagHostTableReactor_.handleMlagHostEntry( vlanId=evid )

class VtiStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Vxlan::VtiStatus'

   def __init__( self, vtiStatus, bridge ):
      t2( 'VtiStatusReactor::__init__()' )
      assert bridge
      self.vtiStatus_ = vtiStatus
      self.bridge_ = bridge
      self.bridge_.controllerClientMode_ = False
      self.bridge_.datapathLearningMode_ = False

      if self.vtiStatus_:
         self.bridge_.controllerClientMode_ = self.vtiStatus_.controllerClientMode
         self.bridge_.datapathLearningMode_ = self.vtiStatus_.datapathLearning
         self.bridge_.controllerControlPlane_ = \
               self.vtiStatus_.controllerControlPlane
         self.bridge_.sharedRouterMac_ = self.vtiStatus_.mlagSharedRouterMacAddr
         for evid in self.vtiStatus_.extendedVlanToVniMap:
            self.handleExtVlanToVniMap( evid )
      Tac.Notifiee.__init__( self, vtiStatus )
      # process the current state
      self.handleLinkStatus()
      self.handleExtVlanToVniMap()

   @Tac.handler( 'controllerClientMode' )
   def handleControllerClientMode( self ):
      t2( 'controllerClientMode %s' % self.notifier_.controllerClientMode )
      self.bridge_.controllerClientMode_ = self.notifier_.controllerClientMode

   @Tac.handler( 'controllerControlPlane' )
   def handleControllerControlPlane( self ):
      t2( 'controllerControlPlane %s' % self.notifier_.controllerControlPlane )
      self.bridge_.controllerControlPlane_ = self.notifier_.controllerControlPlane

   @Tac.handler( 'datapathLearning' )
   def handleDatapathLearningMode( self ):
      t2( 'datapathLearningMode %s' % self.notifier_.datapathLearning )
      self.bridge_.datapathLearningMode_ = self.notifier_.datapathLearning
      if self.bridge_.datapathLearningMode_:
         cleanupReceivedRemoteMacs( self.bridge_ )
      else:
         deleteLearnedRemoteMacs( self.bridge_ )

   @Tac.handler( 'operStatus' )
   def handleOperStatus( self ):
      t2( 'operStatus %s' % self.notifier_.operStatus )
      # On intf up event, process L2Rib learned hosts
      if self.notifier_.operStatus == 'intfOperUp':
         if self.bridge_.l2RibOutputReactor_ is not None:
            self.bridge_.l2RibOutputReactor_.handleLearnedRemoteHosts()

   @Tac.handler( 'linkStatus' )
   def handleLinkStatus( self ):
      t2( 'linkStatus %s' % self.notifier_.linkStatus )
      if self.notifier_.linkStatus == 'linkUp':
         if self.bridge_.mlagHostTableReactor_ is not None:
            self.bridge_.mlagHostTableReactor_.handleMlagHostEntry()

   @Tac.handler( 'extendedVlanToVniMap' )
   def handleExtVlanToVniMap( self, evid=None ):
      if not evid:
         for extVlan in self.vtiStatus_.extendedVlanToVniMap:
            self.handleExtVlanToVniMap( extVlan )
         return

      t2( 'handleExtVlanToVniMap %s/%d' % ( self.vtiStatus_.intfId, evid ) )

      vniSource = self.vtiStatus_.extendedVlanToVniMap.get( evid )
      vni = vniSource.vni if vniSource and vniSource.vni != \
            Tac.Value( "Vxlan::VniOrNone" ) else None
      vtiIntf = self.vtiStatus_.intfId
      updateVniToVlanMap( self.bridge_, vni, vtiIntf, evid )

   @Tac.handler( 'vArpVtepAddr' )
   def handleVarpVtepAddr( self ):
      updateBpfFilterString( self.bridge_ )
      # When vARP MAC changes and control plane is VCS/EVPN, delete remote MACs.
      if not self.vtiStatus_.datapathLearning and \
            ( self.vtiStatus_.controllerClientMode or
              self.vtiStatus_.controllerControlPlane ):
         deleteLearnedRemoteMacs( self.bridge_ )

   @Tac.handler( 'vArpVtepAddr6' )
   def handleVarpVtepAddr6( self ):
      updateBpfFilterString( self.bridge_ )
      # When vARP MAC changes and control plane is VCS/EVPN, delete remote MACs.
      if not self.vtiStatus_.datapathLearning and \
            ( self.vtiStatus_.controllerClientMode or
              self.vtiStatus_.controllerControlPlane ):
         deleteLearnedRemoteMacs( self.bridge_ )

   @Tac.handler( 'mlagSharedRouterMacAddr' )
   def handleMlagSharedRouterMacAddr( self ):
      sharedMac = self.notifier_.mlagSharedRouterMacAddr
      t2( 'handleMlagSharedRouterMacAddr %s' % sharedMac )
      self.bridge_.sharedRouterMac_ = sharedMac

   @Tac.handler( "vtepToVtepBridging" )
   def handleVtepToVtepBridging( self ):
      if self.bridge_.vxlanPort_[ 'Vxlan1' ].shim:
         self.bridge_.vxlanPort_[ 'Vxlan1' ].shim.suppressInputPortValue = (
            not self.notifier_.vtepToVtepBridging )

   @Tac.handler( "localVtepAddr" )
   @Tac.handler( "mlagVtepAddr" )
   @Tac.handler( "localVtepAddr6" )
   @Tac.handler( "mlagVtepAddr6" )
   def handleAddrChange( self ):
      t2( "handleAddrChange {} {}".format( self.vtiStatus_.localVtepAddr,
                                       str( id( self ) ) ) )
      updateBpfFilterString( self.bridge_ )

   @Tac.handler( "udpPort" )
   def handleUdpPortChange( self ):
      updateBpfFilterString( self.bridge_ )

class VlanToLearnRestrictReactor( Tac.Notifiee ):
   notifierTypeName = 'Vxlan::LearnRestrict'
   def __init__( self, vlist, vlanId, learnStatus, vxlanStatus, bridge ):
      t2( 'VlanToLearnRestrictReactor init vlanId=', vlanId )
      assert vlanId == vlist.vlanId
      self.bridge_ = bridge
      self.vxlanStatus_ = vxlanStatus
      self.learnStatus_ = learnStatus
      self.vlanId_ = vlanId
      self.vlist_ = vlist
      Tac.Notifiee.__init__( self, vlist )
      self.handleLearnFrom()

   @Tac.handler( 'learnFrom' )
   def handleLearnFrom( self ):
      t2( 'handleLearnFrom: vlanId=', self.vlanId_ )
      if self.vlanId_ not in self.vxlanStatus_.vlanToLearnRestrict:
         if self.vlanId_ in self.learnStatus_.learnStatusList:
            del self.learnStatus_.learnStatusList[ self.vlanId_ ]
         return

      learnFrom = self.vxlanStatus_.vlanToLearnRestrict[ self.vlanId_ ].learnFrom
      if learnFrom == 'learnFromDefault':
         if self.vlanId_ in self.learnStatus_.learnStatusList:
            del self.learnStatus_.learnStatusList[ self.vlanId_ ]
         return

      prefixes = []
      vteps = []
      if learnFrom == 'learnFromFloodList':
         vteps = getFloodList( self.vlanId_ )
      elif learnFrom == 'learnFromList':
         if ( self.vxlanStatus_ and
              self.vlanId_ in self.vxlanStatus_.vlanToLearnRestrict ):
            prefixes = self.vxlanStatus_.vlanToLearnRestrict[
               self.vlanId_ ].prefixList

      modifyDefaultLearnRestrict( learnFrom, prefixes, vteps, self.vlanId_ )
      # TBD: Could avoid the cleanup when changing to learnFromAny
      cleanupRemoteMacsLearnRestrict( self.bridge_, self.vlanId_ )

   @Tac.handler( 'prefixList' )
   def handlePrefixList( self, prefix ):
      t2( 'handlePrefixList: vlanId=', self.vlanId_, ' prefix=', prefix )
      if prefix is None:
         if self.vlanId_ in self.learnStatus_.learnStatusList:
            ls = self.learnStatus_.learnStatusList[ self.vlanId_ ]
            if ls.learnFrom == 'learnFromDefault':
               del self.learnStatus_.learnStatusList[ self.vlanId_ ]
            else:
               ls.prefixList.clear()
               ls.numMatches.clear()
            cleanupRemoteMacsLearnRestrict( self.bridge_, self.vlanId_ )

         for p in self.vlist_.prefixList:
            t2( 'calling handlePrefixList %s' % ( p ) )
            self.handlePrefixList( p )
         return

      if prefix in self.vlist_.prefixList:
         if self.vlanId_ in self.learnStatus_.learnStatusList:
            ls = self.learnStatus_.learnStatusList[ self.vlanId_ ]
         else:
            ls = self.learnStatus_.learnStatusList.newMember( self.vlanId_ )
            if self.vlanId_ in self.vxlanStatus_.vlanToLearnRestrict:
               ls.learnFrom = \
                   self.vxlanStatus_.vlanToLearnRestrict[ self.vlanId_ ].learnFrom
            else:
               ls.learnFrom = 'learnFromDefault'

         ls.prefixList[ prefix ] = True
         ls.numMatches[ prefix ] = 0
         # Don't add with learnFromDefault
         if ls.learnFrom == 'learnFromDefault':
            if self.vlanId_ in self.learnStatus_.learnStatusList:
               del self.learnStatus_.learnStatusList[ self.vlanId_ ]
               cleanupRemoteMacsLearnRestrict( self.bridge_, self.vlanId_ )
         else:
            t2( 'handlePrefixList added %s' % ( prefix ) )
      else:
         if self.vlanId_ in self.learnStatus_.learnStatusList: 
            ls = self.learnStatus_.learnStatusList[ self.vlanId_ ] 
            if prefix in ls.prefixList:
               t2( 'handlePrefixList del %s' % ( prefix ) )
               del ls.prefixList[ prefix ]
            if prefix in ls.numMatches:
               del ls.numMatches[ prefix ]
            # Fewer VTEPs/prefixes allowed
            cleanupRemoteMacsLearnRestrict( self.bridge_, self.vlanId_ )

   def __del__( self ):
      t2( 'VlanToLearnRestrictReactor deleted vlanId=', self.vlanId_ )
      del self.learnStatus_.learnStatusList[ self.vlanId_ ]
      cleanupRemoteMacsLearnRestrict( self.bridge_, self.vlanId_ )

class VxlanIntfPort( EbraTestPort ):
   """ Manage the VTI port """

   def __init__( self, bridge, tapDevice, trapDevice, intfConfig, intfStatus ):
      """Initialize the port to its default state."""

      assert tapDevice is None
      assert trapDevice is None
      assert intfConfig
      assert intfStatus

      EbraTestPort.__init__( self, bridge, intfConfig, intfStatus )

      intfName = intfStatus.intfId
      num = re.search( r'Vxlan(\d+)', self.intfName_ ).group( 1 )
      if self.bridge_.inNamespace():
         devName = 'vx%s' % num
      else:
         devName = self.bridge_.name() + '-vx%s' % num

      global vxlanAgentDevs

      # When bringing up the first port, bring up the vxlan agent devices as well.
      if vxlanAgentDevs is None:
         vxlanAgentDevs = VxlanAgentDevs( self.bridge_ )

      self.trapDevice_ = Arnet.Device.Tap( devName, hw=self.bridge_.bridgeMac() )
      self.trapFile_ = None
      self.tapFile_ = None
      self.tapDevice_ = self.trapDevice_
      self.processFrame_ = None

      # Save the sysdb config reference this interface
      ent = bridge.em().entity
      self.vtiStatus_ = ent( 'interface/status/eth/vxlan' ).vtiStatus[ intfName ]
      self.vtiConfig_ = ent( 'interface/config/eth/vxlan' ).vtiConfig[ intfName ]
      self.vxlanStatusDir_ = ent( 'vxlan/status' )
      self.mlagStatus_ = ent( 'mlag/status' )
      self.vtiStatusReactor_ = None
      self.vxlanStatusDirReactor_ = None
      self.learnStatus_ = vxlanHwStatusDir.learnStatus
      self.shim = None

   def onActive( self ):
      vxlanAgentDevs.onActive()
      self.trapFile_ = Tac.File( self.trapDevice_.fileno(),
                                 self._trapDeviceReadHandler,
                                 self.trapDevice_.name,
                                 readBufferSize=16000 )
      self.tapFile_ = self.trapFile_
      t2( "Vxlan Port initialized: Device is: %s fileno %d" %
          ( self.trapDevice_.name, self.trapDevice_.fileno() ) )

      self.vtiStatusReactor_ = VtiStatusReactor( self.vtiStatus_, self.bridge_ )
      self.vxlanStatusDirReactor_ = VxlanStatusDirReactor( self.bridge_ )
      updateBpfFilterString( self.bridge_ )

      # Instantiate the vxlanHwStatus for this intfName
      vxlanHwStatusDir.vxlanHwStatus.newMember( self.intfName_ )

   def _tapDeviceReadHandler( self, data) :
      ( srcMacAddr, dstMacAddr ) = macAddrsAsStrings( data )
      t4( "%s: received TAP packet src %s/dst %s" % \
         ( self.name(), srcMacAddr, dstMacAddr ) )
      self.bridge_.processFrame( data, srcMacAddr, dstMacAddr, self, False )

   def _trapDeviceReadHandler( self, data) :
      ( srcMacAddr, dstMacAddr ) = macAddrsAsStrings( data )
      t4( "%s: received trap packet src %s/dst %s" % \
         ( self.name(), srcMacAddr, dstMacAddr ) )
      self.bridge_.processFrame( data, srcMacAddr, dstMacAddr, self, False )

   def close( self ):
      cleanupRemoteMacs( self.bridge_ )
      vxlanLearnedHost.clear()
      multiVtepLearnedHost.clear()
      swGroupIdColl.nuSwTunnelGroupIdToAddr.clear()
      self.bridge_.macTableFlushPort( self.name() )
      self.trapFile_.close()
      self.trapDevice_.close()
      self.vtiStatusReactor_ = None
      self.vxlanStatusDirReactor_ = None
      del self.bridge_.vxlanPort_[ self.intfName_ ]
      arfaVxlanPlugin = getPythonIntoCppShimContext().plugin( "Vxlan" )
      arfaVxlanPlugin.remoteVtepRef.ref.clear()
      if inArfaMode:
         arfaVxlanPlugin.vniVtiVlanMap.map.clear()
      self.bridge_.vxlanVniVlanMap_ = {}
      t2( "VxlanIntfPort", self.name(), "closed" )

   def updateStatusConfig( self, intfStatus, intfConfig ):
      EbraTestPort.updateStatusConfig( self, intfStatus, intfConfig )

   def checkVtepToVtepBridging( self ):
      return self.vtiStatus_.vtepToVtepBridging

   def sendFrame( self, data, srcMacAddr, dstMacAddr, srcPortName, priority, vlanId,
                  vlanAction ):
      """Send the frame up to Vxlan Agent for encap."""
      t8( "sendFrame - srcMacAddr: %s, dstMacAddr: %s, srcPort: %s, vlanId: %d" % \
            ( srcMacAddr, dstMacAddr, srcPortName, vlanId ) )

      if self.intfConfig_.enabled and \
         self.intfStatus_.operStatus == 'intfOperUp':
        
         # Discard frames received on mlag peer-link since it would
         # be processed by the other mlag peer on which the frame was
         # originally received.
         if self.mlagStatus_.peerLinkIntf is not None:
            if self.mlagStatus_.peerLinkIntf.intfId == srcPortName:
               t6( "Discarding frame on port %s received from peer link port %s" % \
                   ( self.name(), srcPortName ) )
               return

         t8( 'In data to sendFrame', data )
         # Need some additional information about the packet, such as whether or not
         # to strip the 802.1Q tag and encode the information in the header sent to
         # the agent.
         etherType = ( data[ 12 ] << 8 ) | data[ 13 ]
         vlanInfo = self.bridge_.vlanTagState( srcPortName, False, etherType,
                                               data, dstMacAddr, None )
         ( _, _, vlanTagNeedsUpdating, isTagged, _ ) = vlanInfo

         flags = VXLAN_CPUPORT_FLAG if srcPortName == 'Cpu' else 0
         srcIfIndex = 0

         # Add mlag flag if source is a MLAG port
         if self.mlagStatus_.intfStatus.get( srcPortName ):
            # add flag to indicate that the src port is from an MLAG interface
            flags |= VXLAN_MLAG_SRCPORT_FLAG
         elif srcPortName == 'MplsTrunk1':
            flags |= VXLAN_MPLS_SRCPORT_FLAG

         swintfconfig = self.bridge_.brConfig_.switchIntfConfig.get( srcPortName ) 
         # swintfconfig will not exist for Cpu, but Cpu will never be dot1qtunnel
         if swintfconfig:
            flags |= VXLAN_DOT1QTUNNEL_FLAG if \
               swintfconfig.switchportMode == 'dot1qTunnel' else 0

         # This uses vxlan_src_vtep_ext
         flags |= VXLAN_SRC_VTEP_EXT_FLAG

         # Extract and parse the source vtep ip
         context = self.bridge_.packetContext
         srcVtep = IpGenAddr() if 'vxlanSrcVtep' not in context else \
                   context[ 'vxlanSrcVtep' ]
         storage = srcVtep.privateV6Addr
         if srcVtep.af == 'ipv6':
            flags |= VXLAN_SRC_VTEP_V6_FLAG

         global encapSeqNo
         if isTagged or vlanTagNeedsUpdating or \
                srcPortName == 'Cpu':
            t6( "removing 802.1Q tag on port %s vlanId %d srcPort %s" %
                  ( self.name(), vlanId, srcPortName ) )
            data = data[ 0:12 ] + \
                     struct.pack( '!HHLLLLLLL', ETH_P_ARISTA_VXLAN_ENCAP, flags, \
                        vlanId, srcIfIndex, encapSeqNo, storage.word0, storage.word1,
                        storage.word2, storage.word3 ) + data[ 16: ]
         else:
            t6( "sending frame on port %s vlanId %d srcPort %s action %s" %
                  ( self.name(), vlanId, srcPortName, vlanAction ) )
            data = data[ 0:12 ] + \
                     struct.pack( '!HHLLLLLLL', ETH_P_ARISTA_VXLAN_ENCAP, flags, \
                        vlanId, srcIfIndex, encapSeqNo, storage.word0, storage.word1,
                        storage.word2, storage.word3 ) + data[ 12: ]
         t8( 'Send data to kernel ', data )
         vxlanAgentDevs.vxlanTapDevice_.send( data )
         encapSeqNo = encapSeqNo + 1
      else:
         t6( "Refusing to transmit on disabled port %s" % self.name() )

   def trapFrame( self, data ):
      """ We shouldn't be trapping the frame, but some other trap handlers can look
      at the ipHdr info (e.g. igmp) and think that a copyToCpu is necessary.  Ignore
      this trap request """
      t0( 'unexpected trap on port %s' % self.name() )
      t6( "trapping frame input on %s" % self.name(), Tracing.HexDump( data ) )

# Minimal version of the VxlanIntfDirReactor, used only in Arfa mode, to spawn
# the minimal Arfa version of the VtiStatusReactor.  See the comment above the
# VtiStatusArfaReactor for more info
class VxlanIntfDirArfaReactor:
   def __init__( self, bridge ):
      self.bridge_ = bridge
      self.ethIntfConfigDir_ = bridge.ethIntfConfigDir_
      self.ethIntfStatusDir_ = bridge.ethIntfStatusDir_

   def handleIntf( self, key ):
      if not key:
         for i in self.ethIntfConfigDir_:
            self.handleIntf( i )
         return

      if not key.startswith( 'Vxlan' ):
         return

      intfConfig = self.ethIntfConfigDir_.intfConfig.get( key )
      intfStatus = self.ethIntfStatusDir_.intfStatus.get( key )
      if intfConfig and intfStatus:
         self.bridge_.arfaVtiStatusReactors_[ key ] = VtiStatusArfaReactor(
               intfStatus, self.bridge_ )
      else:
         if key in self.bridge_.arfaVtiStatusReactors_:
            del self.bridge_.arfaVtiStatusReactors_[ key ]

# Minimal Arfa version of VxlanIntfStatusReactor.  See comment above
# VxlanIntfDirArfaReactor for more info
class VxlanIntfStatusArfaReactor( Tac.Notifiee, VxlanIntfDirArfaReactor ):
   notifierTypeName = 'Interface::EthIntfStatusDir'

   def __init__( self, bridge ):
      t0( "VxlanIntfStatusArfaReactor: __init__: ")
      VxlanIntfDirArfaReactor.__init__( self, bridge )
      Tac.Notifiee.__init__( self, bridge.ethIntfStatusDir_ )
      for intf in bridge.ethIntfStatusDir_.intfStatus:
         t0( "VxlanIntfStatusArfaReactor: __init__: handleStatus for %s" % intf )
         self.handleStatus( intf )

   @Tac.handler( 'intfStatus' )
   def handleStatus( self, key ):
      self.handleIntf( key )

# Minimal Arfa version of VxlanIntfConfigReactor.  See comment above
# VxlanIntfDirArfaReactor for more info
class VxlanIntfConfigArfaReactor( Tac.Notifiee, VxlanIntfDirArfaReactor ):
   notifierTypeName = 'Interface::EthIntfConfigDir'

   def __init__( self, bridge ):
      t0( "VxlanIntfConfigArfaReactor: __init__ " )
      VxlanIntfDirArfaReactor.__init__( self, bridge )
      Tac.Notifiee.__init__( self, bridge.ethIntfConfigDir_ )
      for intf in bridge.ethIntfConfigDir_.intfConfig:
         t0( "VxlanIntfConfigArfaReactor: __init__: handleConfig for %s" % intf )
         self.handleConfig( intf )

   @Tac.handler( 'intfConfig' )
   def handleConfig( self, key ):
      self.handleIntf( key )

class VxlanIntfDirReactor:
   """We wait until a vxlan intf appears in the interface/{config,status}/eth/intf
   collections before adding it to the test bridge.  We don't just watch the
   interface/{config,status}/eth/vxlan collections, because there is a time delay
   between showing up there and showing up in the 'intf' collections.  The test
   bridge looks in the 'intf' collections, so that's what we watch for.

   This class is a base class for the reactors to each of the config and status
   collections since the code for each is nearly identical.
   """

   def __init__( self, bridge ):
      self.bridge_ = bridge
      self.ethIntfConfigDir_ = bridge.ethIntfConfigDir_
      self.ethIntfStatusDir_ = bridge.ethIntfStatusDir_

   def handleIntf( self, key ):
      t2( "VxlanIntfDirReactor for", key )
      if not key:
         # In the unlikely event of deletion of multiple interfaces in a
         # multi-attribute notification, we will leave orphan interfaces
         # in the bridge port list, until a given interface gets re-added.
         for i in self.ethIntfConfigDir_:
            self.handleIntf( i )
         return

      if not key.startswith( 'Vxlan' ):
         return

      t2( "VxlanIntfDirReactor checking", key )
      intfConfig = self.ethIntfConfigDir_.intfConfig.get( key )
      intfStatus = self.ethIntfStatusDir_.intfStatus.get( key )
      if intfConfig and intfStatus:
         if not key in self.bridge_.port:
            t2( "VxlanIntfDirReactor adding", key, "to ports" )
            vxlanPort = VxlanIntfPort( self.bridge_, None, None,
                                       intfConfig, intfStatus )
            vxlanPort.onActive()
            self.bridge_.addPort( vxlanPort )

            assert key == vxlanPort.intfName_, "sanity check"
            self.bridge_.vxlanPort_[ key ] = vxlanPort
            updateBpfFilterString( self.bridge_ )
            t0( 'vxlan port iniitialized' )
            t0( 'port name %s: ' % vxlanPort.intfName_ )
         else:
            # intfStatus object changed.
            port = self.bridge_.port[ key ]
            t2( 'vxlan port', key )
            if not self.bridge_.vxlanPort_.get( key ):
               self.bridge_.vxlanPort_[ key ] = port
               updateBpfFilterString( self.bridge_ )
            
            oldIntfStatus = port.intfStatus_

            if intfStatus != oldIntfStatus:
               t2( 'VxlanIntfDirReactor, change in intfStatus, modify port' )
               port.updateStatusConfig( intfStatus, intfConfig )
      elif key in self.bridge_.port:
         # Clean up the underlying port
         t2( "VxlanIntfDirReactor removing", key, "from ports" )
         self.bridge_.delPort( key )

class VxlanIntfStatusReactor( Tac.Notifiee, VxlanIntfDirReactor ):
   notifierTypeName = 'Interface::EthIntfStatusDir'

   def __init__( self, bridge ):
      t0( "VxlanIntfStatusReactor: __init__: ")
      VxlanIntfDirReactor.__init__( self, bridge )
      Tac.Notifiee.__init__( self, bridge.ethIntfStatusDir_ )
      for intf in bridge.ethIntfStatusDir_.intfStatus:
         t0( "VxlanIntfStatusReactor: __init__: handleStatus for %s" % intf )
         self.handleStatus( intf )

   @Tac.handler( 'intfStatus' )
   def handleStatus( self, key ):
      self.handleIntf( key )

class VxlanIntfConfigReactor( Tac.Notifiee, VxlanIntfDirReactor ):
   notifierTypeName = 'Interface::EthIntfConfigDir'

   def __init__( self, bridge ):
      t0( "VxlanIntfConfigReactor: __init__ " )
      VxlanIntfDirReactor.__init__( self, bridge )
      Tac.Notifiee.__init__( self, bridge.ethIntfConfigDir_ )
      for intf in bridge.ethIntfConfigDir_.intfConfig:
         t0( "VxlanIntfConfigReactor: __init__: handleConfig for %s" % intf )
         self.handleConfig( intf )

   @Tac.handler( 'intfConfig' )
   def handleConfig( self, key ):
      self.handleIntf( key )

class VxlanVlanStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Bridging::VlanStatusDir'

   def __init__( self, bridgingVlanStatusDir, bridge ):
      self.bridge_ = bridge
      Tac.Notifiee.__init__( self, bridgingVlanStatusDir )

   @Tac.handler( 'vlanStatus' )
   def handleVlanStatus( self, vlanId ):
      # XXX: Just supporting one VTI for now
      if not self.bridge_.vxlanPort_.get( 'Vxlan1' ) or \
         self.bridge_.vxlanPort_[ 'Vxlan1' ].vtiStatus_.linkStatus != 'linkUp' or \
         self.bridge_.vxlanPort_[ 'Vxlan1' ].intfStatus_.operStatus != 'intfOperUp':
         t2( 'handleVlanStatus: VTI is down for vlan %d' % vlanId )
         return

      vtiStatus = self.bridge_.vxlanPort_[ 'Vxlan1' ].vtiStatus_

      if vlanId in vtiStatus.vlanToVniMap:
         vniSource = vtiStatus.vlanToVniMap.get( vlanId )
         vni = vniSource.vni if vniSource.vni != \
               Tac.Value( "Vxlan::VniOrNone" ) else None 
         vti = vniSource.vti
      else: 
         vni = None
         vti = None

      if vlanId in self.notifier_.vlanStatus and vni is not None:
         mapVlanId = self.bridge_.vxlanVniVlanMap_.get( ( vni, vti ) )
         if mapVlanId == vlanId:
            t2( "handleVlanStatus: vlan %d already mapped" % mapVlanId )
            return
         elif mapVlanId:
            # Not sure how this might happen, but print a trace statement
            # if it does
            t0( "handleVlanStatus: vlan %d/mapvlan %d mismatch" %
                  ( vlanId, mapVlanId ) )
            return
         t2( "handleVlanStatus: vlan %d added" % vlanId )
         updateVniToVlanMap( self.bridge_, vni, vti, vlanId )
      elif vni is not None:
         t2( "handleVlanStatus: vlan %d removed" % vlanId )
         updateVniToVlanMap( self.bridge_, None, vti, vlanId )

class EvpnMacTableReactor( Tac.Notifiee ):
   notifierTypeName = 'Bridging::RemoteMacTable'

   def __init__( self, evpnMacTable, bridge ):
      assert bridge
      self.bridge_ = bridge
      self.remoteMacEntry_ = evpnMacTable.remoteMacEntry
      Tac.Notifiee.__init__( self, evpnMacTable )
      self.handleRemoteMacEntry()

   @Tac.handler( 'remoteMacEntry' )
   def handleRemoteMacTable( self, macVlan ):
      t2( 'handleRemoteMacTable key mac %s vlan %d' %  \
            ( macVlan.macaddr, macVlan.vlanId ) )
      self.handleRemoteMacEntry( key=macVlan )

   def handleRemoteMacEntry( self, key=None ):
      # add the entry to both vxlan and etba
      keys = list( self.remoteMacEntry_ ) if key is None else [ key ]
      for k in keys:
         host = self.remoteMacEntry_.get( k )
         if not host:
            t4( f'deleting evpn found for key {k.macaddr}/{k.vlanId}' )
            deleteLearnedHost( self.bridge_, k.vlanId, k.macaddr,
                               entryType=BridgingEntryType.evpnDynamicRemoteMac,
                               deleteBridgeMac=True )
            continue

         assert not host.tunnelIpGenAddr.isAddrZero, 'remote vtep is zero'
         t4( 'evpn for key {}/{} tunnelIpAddr {}'.format( host.macaddr, host.vlanId,
                                                      host.tunnelIpGenAddr ) )
         addLearnedHost( self.bridge_,
                         host.vlanId,
                         host.macaddr,
                         host.tunnelIpGenAddr,
                         entryType=BridgingEntryType.evpnDynamicRemoteMac )

# Handle wraparound for U16 moveCount
def geMoveCount( m1, m2 ):
   allf = 0xFFFF
   halfway = 0x8000
   return ( ( m1 - m2 ) & allf ) < halfway

def gtMoveCount( m1, m2 ):
   return m1 != m2 and geMoveCount( m1, m2 )

def updateBpfFilterString( bridge ):
   if not inArfaMode:
      return
   t6( "Updating bpf filter" )

   try:
      filterObj = (
         bridge.arfaRoot_.pluginManager.pythonDecapHook.decapBpfFilters[ "Vxlan" ]
      )
   except ( AttributeError, KeyError ):
      t6( "Fast Etba not initialized" )
      return

   filterObj.bpfFilterString = generateBpfFilterString( bridge )

def generateBpfFilterString( bridge ):
   vxlanPort = bridge.vxlanPort_.get( 'Vxlan1' )
   if not vxlanPort:
      t6( "Bpf: no vxlan port" )
      return ""

   bpfFilter = "udp port %s" % ( vxlanPort.vtiStatus_.udpPort )
   ips = []
   for ip in (
         vxlanPort.vtiStatus_.localVtepAddr,
         vxlanPort.vtiStatus_.vArpVtepAddr,
         vxlanPort.vtiStatus_.mlagVtepAddr,
         vxlanPort.vtiStatus_.localVtepAddr6,
         vxlanPort.vtiStatus_.vArpVtepAddr6,
         vxlanPort.vtiStatus_.mlagVtepAddr6 ):
      ip = Arnet.IpGenAddr( str( ip ) )
      t6( "Bpf: ip ", ip )
      if not ip or ip.af == "ipunknown" or \
         ( ip.af == 'ipv4' and ip.v4Addr == "0.0.0.0" ) or \
         ( ip.af == 'ipv6' and ip.v6Addr == Arnet.Ip6Addr( "::" ) ):
         continue
      ips.append( "ip{} dst {}".format( "6" if ip.af == "ipv6" else "",
                                    ip.stringValue ) )

   if not ips:
      # If no ips were added, we can't function anyway
      t6( "Bpf: no ips" )
      return ""

   bpfFilter += " and ( %s )" % " or ".join( ips )

   t6( "New bpf filter", bpfFilter )

   return bpfFilter

def validateAndParseVxlanHeader( bridge, data, srcPort=None ):
   '''Validate Vxlan port and parse Vxlan Header'''

   decryptedData = None
   retval = ( False, 0, 0, 0, 0, 0, decryptedData )

   t8( 'Validate Vxlan port and parse header' )
   def vtiUp( vxlanPort ):
      if not vxlanPort or \
         vxlanPort.vtiStatus_.linkStatus != 'linkUp' or \
         vxlanPort.intfStatus_.operStatus != 'intfOperUp':
         return False
      return True

   vxlanPorts = bridge.vxlanPort_
   if not any( vtiUp( port ) for port in vxlanPorts.values() ):
      t7( "All Vxlan interfaces are DOWN" )
      return retval
   else:
      t7( "At least 1 VTI is UP" )

   ( pkt, headers, off ) = parsePktStr( data )
   ipHdr = findHeader( headers, "IpHdr" )
   ip6Hdr = findHeader( headers, "Ip6Hdr" )
   if ipHdr is None and ip6Hdr is None:
      t7( 'ignoring non ip packet' )
      return retval

   if ipHdr: # v4
      if ipHdr.protocolNum != 'ipProtoUdp':
         t7( f'ignoring v4 non udp frame (proto: {ipHdr.protocolNum})' )
         return retval
      srcVtep = ipHdr.src
      dstVtep = ipHdr.dst
   else:
      if ip6Hdr.protocolNum != 'ipProtoUdp':
         t7( f'ignoring v6 non udp frame (proto: {ip6Hdr.protocolNum})' )
         return retval
      srcVtep = ip6Hdr.src
      dstVtep = ip6Hdr.dst
   t8( 'validateAndParseVxlanHeader srcVtep', srcVtep, 'dstVtep', dstVtep )
   # validate if destination IP is mine or is multicast IP address in case of
   # multicast replication mode. If not, return
   uDst =  UnderlayAddrType( str( dstVtep ) ) 

   def dstVtepMatch( vxlanPort ):
      if dstVtep != vxlanPort.vtiStatus_.localVtepAddr and \
         dstVtep != vxlanPort.vtiStatus_.vArpVtepAddr and \
         dstVtep != vxlanPort.vtiStatus_.mlagVtepAddr and \
         dstVtep != vxlanPort.vtiStatus_.localVtepAddr6 and \
         dstVtep != vxlanPort.vtiStatus_.vArpVtepAddr6 and \
         dstVtep != vxlanPort.vtiStatus_.mlagVtepAddr6 and \
         not uDst.isMulticast :
         return False # Did not match
      return True

   dstVtiPort = None
   for port in vxlanPorts.values():
      if not dstVtepMatch( port ):
         continue
      dstVtiPort = port
      break

   if not dstVtiPort:
      t8( 'received packet, not intended for any Vtep IP %s ' % dstVtep )
      return retval
   elif uDst.isMulticast:
      t8( 'received mcast destIp packet %s ' % dstVtep  )
      # check dstVtep is in decapConfig underlay routes, supported for ipv4 only
      pimSsmDecap = UnderlayRouteType( defaultVrfId, uSrcV4, uDst,
            'tunnelTypePimSsm' )
      pimSmDecap = UnderlayRouteType( defaultVrfId, uSrcV4, uDst,
            'tunnelTypePimSm' )
      if pimSsmDecap not in decapConfig.decapRoute and \
         pimSmDecap not in decapConfig.decapRoute:
         t8( 'mcast destIp not in decap config, dropping it' )
         return retval 

   if ipHdr:
      # validate ip checksum and ttl.
      # Note: we are not asserting here on checksum mismatch
      checksum = ipHdr.computedChecksum
      if ( checksum != 0 or ipHdr.ttl == 0 ):
         t0( 'Invalid IP Checksum %d or zero TTL %d' % \
               ( checksum, ipHdr.ttl ) )
         return retval

   udpHdr = findHeader( headers, 'UdpHdr' )
   if not udpHdr:
      t7( 'Udp protocol but no udpHdr' )
      return retval

   # RFC 3947
   if udpHdr.dstPort == 4500:
      t8( 'Port', udpHdr.dstPort, 'matches NAT-T UDP port for Vxlansec,'
            ' try to decrypt' )
      decryptedData = handleVxlansecDecrypt( srcVtep, dstVtep, off, data )
      if decryptedData:
         t8( 'Vxlansec decryption successful, rewriting data' )
         data = decryptedData
         ( pkt, headers, off ) = parsePktStr( data )
         udpHdr = findHeader( headers, 'UdpHdr' )

   def validPort( vxlanPort ):
      if decryptedData:
         return udpHdr.dstPort == vxlanPort.vtiStatus_.secUdpPort
      
      if udpHdr.dstPort == vxlanPort.vtiStatus_.udpPort:
         return True
      
      if srcPort:
         # Check if the UDP dst port matches VxlanSec UDP port if the pkt is
         # received from VxlanSwFwd agent via txraw
         srcPortName = srcPort.intfId if inArfaMode else srcPort.intfName_         
         return ( srcPortName == vxlanPort.intfName_ and
                  udpHdr.dstPort == vxlanPort.vtiStatus_.secUdpPort )
      return False

   if not any( validPort( port ) for port in vxlanPorts.values() ):
      t7( 'Udp port mismatch got %d ' % ( udpHdr.dstPort ) )
      return retval
   
   vxlanHdr = Tac.newInstance( "Arnet::VxlanHdrWrapper", pkt, off )

   if not vxlanHdr.flagsVniValid:
      t0( 'got invalid vni %d' % vxlanHdr.vni )
      return retval
   vni = vxlanHdr.vni
   vti = dstVtiPort.intfName_

   try:
      vlan = bridge.vxlanVniVlanMap_[ ( vni, vti ) ]
   except KeyError:
      t0( 'Vni {} not found in VniVlan map ({}, {})'.format(
          vni, vti, list( bridge.vxlanVniVlanMap_ ) ) )
      return retval

   off += VxlanHdrSize
   srcVtep = Arnet.IpGenAddr( str( srcVtep ) )
   dstVtep = Arnet.IpGenAddr( str( dstVtep ) )
   t1( "Valid Vxlan header" )
   return ( True, srcVtep, dstVtep, vlan, off, dstVtiPort, decryptedData )

def handleVxlansecDecrypt( srcVtep, dstVtep, offset, data ):
   # Vxlansec UDP-ESP packet must have a ESP header (8 bytes), an IV (8 bytes), and
   # an ICV (16 bytes).
   espSize = 8
   spiSize = 4
   ivSize = 8
   icvSize = 16
   if len( data ) < offset + 33:
      t7( 'Packet is too short for Vxlansec UDP-ESP' )
      return None 
   espHdr = data[ offset : offset + espSize ]
   t8( 'ESP offset', offset, 'ESP', Tracing.HexDump( espHdr ) )
   spiHdr = espHdr[ : spiSize ]
   # Convert SPI byte order to be consistent with IPsec Sysdb status
   spi = socket.ntohl( int( hexlify( spiHdr ), base=16 ) )
   ivHdr = data[ offset + espSize : offset + espSize + ivSize ]
   cipherText = data[ offset + espSize + ivSize : -icvSize ]
   cryptoKey = getCryptoKey( spi, dstVtep )
   if not cryptoKey:
      t7( 'IPSec cryptoKey not present' )
      return None
   # Using AES GCM with 256-bit key (32 bytes)
   aesGcmKeyLen = 32
   saltLen = 4
   if len( cryptoKey ) != aesGcmKeyLen + saltLen:
      t7( 'Warn: IPSec cryptoKey len is %d, but should be %d' % ( 
         len( cryptoKey ), aesGcmKeyLen + saltLen ) )
      return None
   ipsecKey = cryptoKey[ : aesGcmKeyLen ]
   salt = cryptoKey[ aesGcmKeyLen : ]
   nonce = salt + ivHdr
   icvHdr = data[ -icvSize : ]
   plaintext = aesGcmDecrypt( ipsecKey, espHdr, nonce, cipherText, icvHdr )
   if not plaintext:
      return None
   # Delete UDP-ESP header
   return data[ : offset - 8 ] + plaintext

def makeVxlanDest( vtepAddr, vtepType=VtepType.controlPlaneVtep ):
   tacVtepAddr = vtepAddr
   if isinstance( vtepAddr, str ):
      tacVtepAddr = Tac.ValueConst( 'Arnet::IpGenAddr', vtepAddr )
   destId = Tac.Value( 'L2Rib::Constants' ).objIdNotInSmash
   return Tac.ValueConst(
      'L2Rib::Dest', destId, destType='destTypeVxlan',
      vxlan=Tac.ValueConst( 'L2Rib::DestDataVxlan', tacVtepAddr, vtepType ) )

def checkLearnFromVtep( vlanId, vtep, stats=True ):
   t8( "checkLearnFromVtep: vlan %d vtep %s" % ( vlanId, vtep ) )

   learnStatus = vxlanHwStatusDir.learnStatus
   if vlanId not in learnStatus.learnStatusList:
      t6( 'checkLearnFromVtep: NO VLAN vlan %d vtep %s' % ( vlanId, vtep ) )
      return True

   ls = learnStatus.learnStatusList[ vlanId ]
   if ls.learnFrom == 'learnFromAny' or ls.learnFrom == 'learnFromDefault':
      t6( 'checkLearnFromVtep: matchAny vlan %d vtep %s' % ( vlanId, vtep ) )
      if stats:
         ls.numMatchAny += 1
      return True
   elif ls.learnFrom == 'learnFromFloodList':
      # Check if VTEP is present in L2Rib::FloodSetOutput
      if vlanId in l2RibFloodSet.vlanFloodSet:
         vfs = l2RibFloodSet.vlanFloodSet[ vlanId ]
         fs = vfs.floodSet[ EthAddr.ethAddrZero ]
         dest = makeVxlanDest( vtep )
         if dest in fs.destSet:
            t6( 'checkLearnFromVtep: matchFlood vlan %d vtep %s' % ( vlanId, vtep ) )
            if stats:
               ls.numMatchFloodList += 1
               ls.numMatches[ vtepToPrefix( vtep.v4Addr ) ] += 1
            return True
      t6( 'checkLearnFromVtep: rejectFlood vlan %d vtep %s' % ( vlanId, vtep ) )
      if stats:
         ls.numRejectFloodList += 1
      return False
   elif ls.learnFrom == 'learnFromList':
      for p in ls.prefixList:
         if p.ipPrefix.contains( vtep ):
            t6( 'checkLearnFromVtep: matchList vlan %d vtep %s' % ( vlanId, vtep ) )
            if stats:
               ls.numMatchList += 1
               ls.numMatches[ p ] += 1
            return True
      t6( 'checkLearnFromVtep: rejectList vlan %d vtep %s' % ( vlanId, vtep ) )
      if stats:
         ls.numRejectList += 1
      return False

   t6( 'checkLearnFromVtep: unknown learnFrom %s' % ls.learnFrom )
   return False

def perVtepLearning( srcVtep ):
   """Check if perVtepLearning is enabled for a given vtep"""
   if srcVtep in remoteVtepConfigDir.vtepConfig:
      return remoteVtepConfigDir.vtepConfig[ srcVtep ].learningEnabled
   return True

def processInnerFrame( bridge, srcVtep, vlan, off, data ):
   """Extract inner frame data and learn the remote address if needed"""

   vxlanIntfCfg = bridge.brConfig_.switchIntfConfig.get( 'Vxlan1' ) 
   if vxlanIntfCfg and not vxlanIntfCfg.macLearningEnabled:
      t8( 'processInnerFrame: vxlan mac learning is disabled, not learning' )
      return

   if not bridge.datapathLearningMode_:
      t8( 'processInnerFrame: datapath learning is disabled, not learning' )
      return

   if not checkLearnFromVtep( vlan, srcVtep ):
      t8( 'processInnerFrame: rejected by checkLearnFromVtep' )
      return

   if not perVtepLearning( srcVtep ):
      t8( 'processInnerFrame: datapath learning disabled for %s' % srcVtep )
      return

   addr = struct.unpack( '>HHH', data[ off+6 : off+12 ] )
   macAddr = Tac.Value( "Arnet::EthAddr", addr[0], addr[1], addr[2] )

   t8( 'processInnerFrame: learning mac: %s, vlan: %d, vtep: %s' %
         ( macAddr, vlan, srcVtep ) )
   addLearnedHost( bridge, vlan, macAddr, srcVtep, 
                   entryType=BridgingEntryType.learnedRemoteMac )

def incRemoteVtepRef( vtep, vti, vlanId ):
   # increment/set the reference count for this vtep
   arfaVxlanPlugin = getPythonIntoCppShimContext().plugin( "Vxlan" )

   # If we're in Etba mode, just use the Arfa plugin as a data structure
   # to hold the ref count.  If we're in Arfa mode, call into the actual Arfa
   # refcount logic
   if not inArfaMode:
      refColl = arfaVxlanPlugin.remoteVtepRef.ref
      if vtep in refColl:
         refColl[ vtep ] += 1
      else:
         refColl[ vtep ] = 1
   else:
      arfaVxlanPlugin.vxlanLearnHelper.vtepRefInc( vtep, vti, vlanId )

def decRemoteVtepRef( vtep, vti ):
   arfaVxlanPlugin = getPythonIntoCppShimContext().plugin( "Vxlan" )
   refColl = arfaVxlanPlugin.remoteVtepRef.ref

   # If we're in Etba mode, just use the Arfa plugin as a data structure
   # to hold the ref count.  If we're in Arfa mode, call into the actual Arfa
   # refcount logic
   if not inArfaMode:
      if vtep in refColl:
         t2( "Vxlan dropping refcnt %s" % refColl[ vtep ] )
         if refColl[ vtep ] > 1:
            refColl[ vtep ] -= 1
            return refColl[ vtep ]

         refColl[ vtep ] = 0
         return 0
      return None
   else:
      # To match the return API for this function in Etba mode, we need to look
      # up the value of the refcount before we call the Arfa decrement function,
      # because else the Arfa decrement function might deleted the refcount entry
      # if it goes to 0
      if vtep in refColl:
         returnVal = refColl[ vtep ] - 1
      else:
         returnVal = None

      arfaVxlanPlugin.vxlanLearnHelper.vtepRefDec( vtep, vti )
      return returnVal

def incRemoteVtepFloodRef( bridge, vtep ):
   # increment/set the flood reference count for this vtep
   if vtep in bridge.remoteVtepFloodRef_:
      bridge.remoteVtepFloodRef_[ vtep ] += 1
   else:
      bridge.remoteVtepFloodRef_[ vtep ] = 1
   t2( 'Flood refcnt: %s = %d' % ( vtep, bridge.remoteVtepFloodRef_[ vtep ] ) )

def decRemoteVtepFloodRef( bridge, vtep ):
   if vtep in bridge.remoteVtepFloodRef_:
      newCount = bridge.remoteVtepFloodRef_[ vtep ] - 1
      t2( 'Dropping flood refcnt for vtep=%s, new count=%d' % ( vtep, newCount ) )
      bridge.remoteVtepFloodRef_[ vtep ] = newCount
      return bridge.remoteVtepFloodRef_[ vtep ] > 0
   return False

def mapVxlanEntryType( vxlanEntryType ):

   if vxlanEntryType == VxlanEntryType.configuredMac:
      return BridgingEntryType.configuredRemoteMac
   elif vxlanEntryType == VxlanEntryType.receivedMac:
      return BridgingEntryType.receivedRemoteMac
   elif vxlanEntryType == VxlanEntryType.evpnMac:
      return BridgingEntryType.evpnDynamicRemoteMac
   else:
      return BridgingEntryType.learnedRemoteMac

def getHostMoveDetails( bridge, vlan, macAddr ):
   bridgingFdbStatusDir = bridge.brStatus_.fdbStatus
   fdbStatus = bridgingFdbStatusDir.get( vlan )
   if fdbStatus:
      learnedHosts = fdbStatus.learnedHost
      if learnedHosts and macAddr in learnedHosts:
         host = learnedHosts[ macAddr ]
         return ( host.moves, host.lastMoveTime, host.intf )
   return ( None, None, None )

def addLearnedHost( bridge, vlan, macAddr, vtep, intf=None,
                    entryType=BridgingEntryType.learnedDynamicMac,
                    moveCount=None ):
   t0( 'addLearnedHost vlan:', vlan, 'macAddr:', macAddr, 'vtep:', vtep,
         'intf:', intf, 'type:', entryType, 'moveCount:', moveCount )

   if isinstance( vtep, str ) or not hasattr( vtep, '__iter__' ):
      vtep = [ vtep ]
   # make sure vteps are of the correct type
   vtep = [ Arnet.IpGenAddr( str( v ) ) for v in vtep ]
   
   vxlanPort = bridge.vxlanPort_.get( 'Vxlan1' )
   arfaVtiReactor = bridge.arfaVtiStatusReactors_.get( 'Vxlan1' )

   if not inArfaMode:
      linkUp = vxlanPort and vxlanPort.vtiStatus_.linkStatus == 'linkUp'
   else:
      linkUp = arfaVtiReactor and arfaVtiReactor.vtiStatus_.linkStatus == 'linkUp'

   # Check that link is up and the VLAN-VNI map exists
   if not linkUp:
      t2( 'addLearnedHost: VTI is down - returning' )
      return
   else:
      if not inArfaMode:
         vtiStatus = vxlanPort.vtiStatus_
      else:
         vtiStatus = arfaVtiReactor.vtiStatus_
      vni = None
      vti = None
      if vlan in vtiStatus.vlanToVniMap:
         vniSource = vtiStatus.vlanToVniMap[ vlan ]
         if vniSource and vniSource.vni != Tac.Value( "Vxlan::VniOrNone" ):
            vni = vniSource.vni
            vti = vniSource.vti
      # Check that reverse map exists
      if not inArfaMode:
         if vni is None or ( vni, vti ) not in bridge.vxlanVniVlanMap_:
            t0( 'addLearnedHost: VLAN-VNI map is missing or is not active'
                  '- returning' )
            return
      else:
         arfaVxlanPlugin = getPythonIntoCppShimContext().plugin( "Vxlan" )
         if vni is None or Tac.Value( 'Vxlan::VniVtiPair', vni=vni,
               vti=vtiStatus.intfId ) not in arfaVxlanPlugin.vniVtiVlanMap.map:
            t0( 'addLearnedHost: VLAN-VNI map is missing or is not active'
                  '- returning' )
            return

   vxlanEntryType = helper.mapBridgingEntryType( entryType )
   macVlanPair = Tac.Value( "Vxlan::MacVlanPair", macAddr, vlan )

   # Do not learn VARP MAC
   if str( macVlanPair.macAddr ) == str( vrMacStatus.varpVirtualMac ) and \
      vxlanEntryType != VxlanEntryType.configuredMac:
      t2( "macAddr is a varpMac. Skip learning ", macAddr )
      return
   
   # Get host move details for MAC move handling
   hostMovesCount, hostMoveTime, hostIntf = getHostMoveDetails( bridge, vlan, 
                                                                macAddr )
   updateLearnedHost = False
   # Check for remote to remote MAC moves or entry type changes
   if macVlanPair in vxlanLearnedHost:
      lh = vxlanLearnedHost[ macVlanPair ]
      mvlh = multiVtepLearnedHost.get( macVlanPair )
      if mvlh:
         l2SwId = mvlh.swTunnelGroupId.index
         vtepList = swGroupIdColl.nuSwTunnelGroupIdToAddr[ l2SwId ]
         mvlh = list( vtepList.tunnelIpGenAddr )
      else:
         # For ease of comparison, fake a single multi-vtep entry if none exists
         mvlh = [ lh.remoteVtepAddr ]

      t2( f"lh entryType: {lh.entryType} vxlanEntryType: {vxlanEntryType}" )

      # Update if learned hosts have changed
      remoteToRemoteMove = set( mvlh ) != set( vtep )
      # Update when learned VTEP(s) have not changed, but control plane indicates
      # move on learned VTEP(s)
      moveCountUpdate = not remoteToRemoteMove and not moveCount is None and \
            moveCount != hostMovesCount
      # Update for non-configured -> configured MAC move
      confEntry = vxlanEntryType == VxlanEntryType.configuredMac
      confMacMove = lh.entryType != VxlanEntryType.configuredMac and confEntry
      # Update for EVPN <-> received MAC move 
      evpnRcvdMacMove = ( lh.entryType == VxlanEntryType.receivedMac and \
                              vxlanEntryType == VxlanEntryType.evpnMac ) or \
                        ( lh.entryType == VxlanEntryType.evpnMac and \
                              vxlanEntryType == VxlanEntryType.receivedMac )
      evpnLearnedMacMove = ( lh.entryType == VxlanEntryType.learnedMac and \
                              vxlanEntryType == VxlanEntryType.evpnMac ) or \
                           ( lh.entryType == VxlanEntryType.evpnMac and \
                              vxlanEntryType == VxlanEntryType.learnedMac )

      # Determine whether MAC move has occurred
      if ( remoteToRemoteMove or confMacMove or evpnRcvdMacMove or moveCountUpdate or
           evpnLearnedMacMove ):
         t0( "updating learned host - mac: %s, vlan: %s, vtep: %s, entryType: %s " \
               "moveCount: %s, lastMoveTime: %s" % ( macAddr, vlan, vtep,
                  vxlanEntryType, moveCount, hostMoveTime ) )
         # MAC move or entry type change results in removing entry from FdbStatus
         # This breaks remote <-> remote MAC move logic, so update move count here
         if moveCount is None and remoteToRemoteMove and not confEntry:
            moveCount = hostMovesCount + 1 if hostMovesCount else 1
         deleteLearnedHost( bridge, vlan, macAddr,
                            mapVxlanEntryType( lh.entryType ),
                            deleteBridgeMac=True )
         updateLearnedHost = True
   else:
      # Update learned host if either
      # 1. Move count is not specified, Eg: MAC from CLI.
      # 2. We see a higher move count.
      hostMovesCount = hostMovesCount or 0
      updateLearnedHost = not moveCount or moveCount > hostMovesCount

   t2( "updateLearnedHost: %r " % ( updateLearnedHost ) )
   if updateLearnedHost:
      if len( vtep ) > 1:
         tunnelIpAddrList = TunnelIpAddrList()
         for v in vtep:
            tunnelIpAddrList.tunnelIpGenAddr[ vtep ] = True
         swId = random.randint( 0, 2**32 - 1 )
         while swId in swGroupIdColl.nuSwTunnelGroupIdToAddr:
            swId = random.randint( 0, 2**32 - 1 )
         swGroupIdColl.nuSwTunnelGroupIdToAddr[ swId ] = tunnelIpAddrList
         l2SwId = SwTunnelGroupId( swId )
         newMvlh = Tac.Value( 'Vxlan::MultiVtepLearnedHost', macVlanPair, l2SwId,
                              vxlanEntryType )
         multiVtepLearnedHost.addMember( newMvlh )

      if entryType == "evpnIntfDynamicMac" or \
         entryType == 'evpnIntfStaticMac':
         remoteVtepAddr = Arnet.IpGenAddr( "0.0.0.0" )
      else:
         remoteVtepAddr = vtep[ 0 ]

      lh = Tac.Value( 'Smash::Vxlan::LearnedHost', macVlanPair, remoteVtepAddr,
                      vxlanEntryType )

      if entryType == 'evpnIntfDynamicMac' or \
         entryType == 'evpnIntfStaticMac':
         hostIntf = intf
      else:
         hostIntf = vtiStatus.intfId
      # EbraTestBridge already handles incrementing move count when FdbStatus
      # contains MacVlan pair, so local to remote moves are covered
      bridge.learnAddr( vlan, str( macAddr ), hostIntf, entryType,
                        moves=moveCount )
      ( hostMoveCount, hostMoveTime, _ ) = getHostMoveDetails( bridge, vlan,
                                                               macAddr )

      t0( 'added lastMoveTime %s' % hostMoveTime )
      if hostMoveTime:
         lh.lastMoveTime = hostMoveTime
      if not hostMoveCount:
         hostMoveCount = 0

      vxlanLearnedHost.addMember( lh )
      helper.maybeAddToLearnedRemoteHostStatus( macVlanPair,
                                         vtiStatus.intfId,
                                         remoteVtepAddr, entryType, 
                                         lh.lastMoveTime, hostMoveCount )

      arfaVxlanPlugin = getPythonIntoCppShimContext().plugin( "Vxlan" )
      refColl = arfaVxlanPlugin.remoteVtepRef.ref
      for v in vtep:
         incRemoteVtepRef( v, vtiStatus.intfId, macVlanPair.vlanId )
         updateRemoteVtep( bridge, v, macVlanPair.vlanId, add=True )
         t0( 'added %s mac %s vlan %d vtep %s refcnt %d' % \
             ( entryType, macVlanPair.macAddr, macVlanPair.vlanId, v,
               refColl[ v ] ) )
   else:
      t7( "not updating host mac: %s, vlan: %d) " % ( macAddr, vlan ),
          f"vtep: {vtep}, entryType: {entryType}" )

# Handler is called when new entry is added to bridging::fdbStatus
# Handles remote to local MAC moves, as remote to remote and local to
# remote MAC moves are handled in preTunnelHandler
def macLearningHandler( vlanId, macAddr, portName, entryType ):
   t4( 'macLearningHandler mac %s vlan %d portName %s entryType %s' % \
      ( macAddr, vlanId, portName, entryType ) )
   
   peerRemote = isPeerRemoteEntry( entryType )
   if vxlanAgentDevs is None:
      return 'dontLearn' if peerRemote else None

   # If peer remote mac, then learn it via vxlan plugin 
   # and return 'dontLearn' so that Etba skips processing this entry
   # portName for peer remote macs is encoded as 
   # "<intf>:<vtepIp>" OR "<intf>:<vtepIp>:<preference>"
   # where "intf" is a vxlan interface such as "Vxlan1"
   #        "vtepIp" is the remote vtep IP and
   #        "preference" is the preference associated with the mac 
   #           entry VxlanController::PreferenceV2
   portNameSplit = portName.split(":")
   if peerRemote:
      # Note that this callback will also be called again while Vxlan plugin
      # is in the process of installing this mac. In that situation, the portName 
      # would have already been appropriately pruned to "Vxlan". If so, just return
      # and let etba do the real learning of peer remote macs
      if len( portNameSplit ) > 1:
         return 'dontLearn'
      else:
         return None

   # Reject datapath learning if learning from source VTEP is disabled
   srcVtep = vxlanAgentDevs.bridge_.packetContext.get( 'vxlanSrcVtep' )
   if ( portName == 'Vxlan1' and entryType == 'learnedRemoteMac' and
        srcVtep and not perVtepLearning( srcVtep ) ):
      return 'dontLearn'

   # Handle remote VXLAN to remote MPLS move
   mplsMove = entryType == BridgingEntryType.evpnDynamicRemoteMac and \
         portName == 'MplsTrunk1'
   if mplsMove or entryType in ( BridgingEntryType.learnedDynamicMac,
                                 BridgingEntryType.configuredStaticMac ):
      if ( mlagStatusMount.peerLinkIntf and
           mlagStatusMount.peerLinkIntf.intfId == portName ):
         # MAC was learned over the MLAG peer-link. Learning will be ignored
         # by MlagDut therefore MAC should not be deleted from the vxlan entry
         return None
      # Delete the vxlan Entry if this is a remote to local move
      deleteLearnedHost( vxlanAgentDevs.bridge_, vlanId, macAddr )
   return None

def deleteLearnedHost( bridge, vlanId, macAddr,
                       entryType=BridgingEntryType.learnedDynamicMac,
                       deleteBridgeMac=False ):
   t4( 'deleteLearnedHost mac %s vlan %d entryType %s' % \
      ( macAddr, vlanId, entryType ) )

   vxlanEntryType = helper.mapBridgingEntryType( entryType )
   macVlan = Tac.Value( "Vxlan::MacVlanPair", macAddr, vlanId )
   if macVlan in list( vxlanLearnedHost ):
      lh = vxlanLearnedHost[ macVlan ]
      # Do not remove configured MAC
      if ( lh.entryType == VxlanEntryType.configuredMac and 
            vxlanEntryType != VxlanEntryType.configuredMac ):
         t2( "Vxlan not removing host type %s, mac: %s, vlan: %d" % \
               ( lh.entryType, macAddr, vlanId ) )
         return

      t2( "Vxlan removing learned entry mac: %s, vlan: %d, entryType: %s" % \
            ( macAddr, vlanId, lh.entryType ) )

      helper.maybeDelFromLearnedRemoteHostStatus( macVlan )
      del vxlanLearnedHost[ macVlan ]

      # We'll get called back recursively from the bridge code (since this is also
      # the aging handler), but since we've already deleted the learned host, it will
      # be a no-op.
      if deleteBridgeMac:
         fid = bridge.brConfig_.vidToFidMap.get( vlanId )
         if not fid:
            fid = vlanId

         if fid in bridge.brStatus_.fdbStatus:
            bridge.deleteMacAddressEntry( vlanId, str( macAddr ), entryType )

      mvlh = multiVtepLearnedHost.get( macVlan )
      if mvlh:
         swId = mvlh.swTunnelGroupId.index
         tnlIpList = swGroupIdColl.nuSwTunnelGroupIdToAddr[ swId ]
         vtepList = list( tnlIpList.tunnelIpGenAddr )
      else:
         vtepList = [ lh.remoteVtepAddr ]
      for vtep in vtepList:
         # Hardcoding VTI here, since this code isn't multi-VTI aware
         if decRemoteVtepRef( vtep, Tac.Value( 'Arnet::IntfId', 'Vxlan1' ) ) == 0:
            # VTEP is only removed from VLAN flood-lists when there are no 
            # MACs learned behind VTEP for *ANY* VLAN
            updateRemoteVtep( bridge, vtep, vlanId, add=False )
      # If a learned host aged out, check if the host exists in the control plane
      host = Tac.const( Tac.Value( "Bridging::MacVlanPair", macAddr, vlanId ) )
      if( lh.entryType == VxlanEntryType.learnedMac and host in l2RibOutput.host and
          bridge.l2RibOutputReactor_ is not None ):
         bridge.l2RibOutputReactor_.handleLearnedRemoteHosts( key=host )
      # In campus tests, a deleted EVPN entry may be replaced by a
      # peerLearnedRemoteEntry
      peerHost = Tac.const( Tac.Value( "Mlag::HostEntryKey", vlanId, macAddr ) )
      if( lh.entryType == VxlanEntryType.evpnMac and
            bridge.mlagHostTableReactor_ is not None and
            peerHost in bridge.mlagHostTableReactor_.mlagHostTable_.hostEntry ):
         bridge.mlagHostTableReactor_.handleMlagHostEntry( hostKey=peerHost )

   if macVlan in multiVtepLearnedHost:
      mvlh = multiVtepLearnedHost.get( macVlan )
      swId = mvlh.swTunnelGroupId.index
      del multiVtepLearnedHost[ macVlan ]
      del swGroupIdColl.nuSwTunnelGroupIdToAddr[ swId ]

def updateRemoteVtep( bridge, vtep, vlanId, add=True ):
   """ If the Vxlan device exists, add the remote vtep to the config """
   vxlanPort = bridge.vxlanPort_.get( 'Vxlan1' )
   if vxlanPort and vxlanPort.intfName_ in vxlanHwStatusDir.vxlanHwStatus:
      hwstatus = vxlanHwStatusDir.vxlanHwStatus[ vxlanPort.intfName_ ]
      t2( "%s remote vtep %s vlan %s" %
          ( "adding" if add else "removing", vtep, vlanId ) )
      if add:
         # vxlanHwStatus.remoteVtep indicates the set of active remote VTEPs and 
         # the VLANs they are active in. This set is used to construct dynamic 
         # VXLAN flood-lists when configured.  For a vtep to be configured,
         # vlanList[ vlanId ] has to be set
         newVtep = hwstatus.newRemoteVtep( vtep )
         newVtep.vlanList[ vlanId ] = True
         vtepStatus = vtepHwStatus.vtepStatus.get( VxlanTunnelType.vxlanTunnel )
         if not vtepStatus:
            vtepStatus = vtepHwStatus.newVtepStatus( VxlanTunnelType.vxlanTunnel )
         vtepStatus.vtepList[ vtep ] = 1
      else:
         del hwstatus.remoteVtep[ vtep ]
         vtepStatus = vtepHwStatus.vtepStatus.get( VxlanTunnelType.vxlanTunnel )
         if vtepStatus:
            del vtepStatus.vtepList[ vtep ]

def updateVniToVlanMap( bridge, vni, vti, evid ):
   t2( "updateVniToVlanMap: vni=", vni, "vti=", vti, " extVlanId=", evid )
   arfaVxlanPlugin = getPythonIntoCppShimContext().plugin( "Vxlan" )
   tacEvid = Tac.Value( 'Bridging::ExtendedVlanId', evid )

   if vti is None:
      vti = 'Vxlan1'
   if vni is not None:
      t2( "updateVniToVlanMap: adding reverse lookup vni %d extVlanId %d" %
           ( vni, evid ) )
      for ( previousVni, previousVti ) in bridge.vxlanVniVlanMap_:
         if ( bridge.vxlanVniVlanMap_[ ( previousVni, previousVti ) ] == evid and
              previousVti == vti ):
            t2( "deleting previous entry for vni", previousVni, "extVlan", evid )
            del bridge.vxlanVniVlanMap_[ ( previousVni, previousVti ) ]
            pair = Tac.Value( "Vxlan::VniVtiPair", previousVni, previousVti )
            if inArfaMode:
               del arfaVxlanPlugin.vniVtiVlanMap.map[ pair ]
            break

      bridge.vxlanVniVlanMap_[ ( vni, vti ) ] = evid
      pair = Tac.Value( "Vxlan::VniVtiPair", vni, vti )
      if inArfaMode:
         arfaVxlanPlugin.vniVtiVlanMap.map[ pair ] = tacEvid
      vxlanStatus = vxlanStatusDir.vxlanStatus.get( vti )

      # VlanToLearnRestrict is still keyed by vlanId, so we cannot look up
      # ext range vlan ids in it.
      if ( tacEvid.inDot1qRange() and vxlanStatus and
           evid in vxlanStatus.vlanToLearnRestrict ):
         learnFrom = vxlanStatus.vlanToLearnRestrict[ evid ].learnFrom
         prefixes = vxlanStatus.vlanToLearnRestrict[ evid ].prefixList
      else:
         learnFrom = 'learnFromDefault'
         prefixes = []

      # L2Rib is also keyed by normal vlanId
      vteps = []
      if tacEvid.inDot1qRange() and evid in l2RibFloodSet.vlanFloodSet:
         vteps = getFloodList( evid )

      if tacEvid.inDot1qRange() and vxlanStatus:
         modifyDefaultLearnRestrict( learnFrom, prefixes, vteps, evid )
   else:
      for vniKey, vtiKey in list( bridge.vxlanVniVlanMap_ ):
         if ( bridge.vxlanVniVlanMap_[ ( vniKey, vtiKey ) ] == evid and
              vtiKey == vti ):
            if tacEvid.inDot1qRange():
               cleanupRemoteMacs( bridge, evid )
            # now its safe to delete the reverse lookup map 
            t2( "updateVniToVlanMap: removing reverse lookup vni %d extvlan %d" %
                ( vniKey, evid ) )
            del bridge.vxlanVniVlanMap_[ ( vniKey, vtiKey ) ]
            pair = Tac.Value( "Vxlan::VniVtiPair", vniKey, vtiKey )
            if inArfaMode:
               del arfaVxlanPlugin.vniVtiVlanMap.map[ pair ]
      if tacEvid.inDot1qRange():
         modifyDefaultLearnRestrict( 'learnFromDefault', [], [], evid )

   # Some vnis could be removed - might need to unlearn from vteps/mac addresses
   if tacEvid.inDot1qRange():
      cleanupRemoteMacsLearnRestrict( bridge, evid )

      if vni is not None:
         if bridge.mlagHostTableReactor_ is not None:
            bridge.mlagHostTableReactor_.handleMlagHostEntry( vlanId=evid )
         # Hosts learned during VLAN flap must be processed again
         if bridge.l2RibOutputReactor_ is not None:
            bridge.l2RibOutputReactor_.handleLearnedRemoteHosts()

def isPeerRemoteEntry( entryType ):
   if entryType == BridgingEntryType.peerConfiguredRemoteMac or \
         entryType == BridgingEntryType.peerReceivedRemoteMac or \
         entryType == BridgingEntryType.peerLearnedRemoteMac:
      return True
      
   return False

class MlagHostTableReactor( Tac.Notifiee ):
   notifierTypeName = 'Mlag::HostTable'
   def __init__( self, mlagHostTable, bridge ):
      t0( 'MlagHostTableReactor init ' )
      self.mlagHostTable_ = mlagHostTable
      self.bridge_ = bridge
      Tac.Notifiee.__init__( self, mlagHostTable )
      self.handleMlagHostEntry()

   @Tac.handler( 'hostEntry' )
   def handleMlagHostEntry( self, hostKey=None, vlanId=None ):
      if hostKey is None:
         for hKey in self.mlagHostTable_.hostEntry:
            self.handleMlagHostEntry( hKey, vlanId )
         return

      t0( "MlagHostTableReactor::handleMlagHostEntry: mac = %s vlanId = %d" % \
            ( hostKey.address, hostKey.vlanId ) )
      
      # pylint: disable-next=singleton-comparison
      if vlanId != None and vlanId != hostKey.vlanId:
         return

      macVlan = Tac.Value( "Vxlan::MacVlanPair",
                           hostKey.address, hostKey.vlanId )
      if hostKey in self.mlagHostTable_.hostEntry:
         hostEntry = self.mlagHostTable_.hostEntry[ hostKey ]
         
         t0( "   handleMlagHostEntry: evaluating add entryType = %s intf=%s" % \
               ( hostEntry.entryType, hostEntry.intf ) )

         if not isPeerRemoteEntry( hostEntry.entryType ):
            return

         # interface name for peer remote macs is encoded as
         # '<intf>:<vtepIp>:<preference>', where preference is optional.
         # IPv6 vtep address will be encoded in full format of
         # 'xxxx:xxxx:xxxx:xxxx:xxxx:xxxx:xxxx:xxxx'. Thus the sub list of
         # the intfNameSplit[ 1 : 9 ] will contain the IPv6 vtep address
         intfNameSplit = hostEntry.intf.split( ":" )
         if len( intfNameSplit ) > 8: # IPv6 address
            addr = ':'.join( intfNameSplit[ 1 : 9 ] )
         else:
            addr = intfNameSplit[ 1 ]
         vtep = Arnet.IpGenAddr( addr )
         addLearnedHost( self.bridge_, hostEntry.vlanId, hostEntry.address,
                         vtep, entryType=hostEntry.entryType,
                         moveCount=hostEntry.moves )
      else:
         fid = self.bridge_.brConfig_.vidToFidMap.get( hostKey.vlanId )
         if not fid:
            fid = hostKey.vlanId

         fdbStatus = self.bridge_.brStatus_.fdbStatus.get( fid )
         if not fdbStatus:
            return
         hostEntry = fdbStatus.learnedHost.get( hostKey.address )
         if ( hostEntry and isPeerRemoteEntry( hostEntry.entryType ) and
              macVlan in vxlanLearnedHost ):
            lh = vxlanLearnedHost[ macVlan ]
            deleteLearnedHost( self.bridge_, hostKey.vlanId, hostKey.address,
                               mapVxlanEntryType( lh.entryType ),
                               deleteBridgeMac=True )  

def cleanupReceivedRemoteMacs( bridge, vlanId=None ):
   t4( "cleanupReceivedRemoteMacs" )
   for macVlan in list( vxlanLearnedHost ):
      lh = vxlanLearnedHost[ macVlan ]
      if lh.entryType == VxlanEntryType.receivedMac:
         if ( vlanId is None ) or ( macVlan.vlanId == vlanId ):
            deleteLearnedHost( bridge, macVlan.vlanId, macVlan.macAddr,
                  BridgingEntryType.receivedRemoteMac,
                  deleteBridgeMac=True )  

def cleanupRemoteMacs( bridge, vlanId=None ):
   t4( "cleanupRemoteMacs vlan=", vlanId )
   for macVlan in list( vxlanLearnedHost ):
      if ( vlanId is None or vlanId == macVlan.vlanId ):
         lh = vxlanLearnedHost[ macVlan ]
         deleteLearnedHost( bridge, macVlan.vlanId, macVlan.macAddr,
                            mapVxlanEntryType( lh.entryType ),
                            deleteBridgeMac=True )

def deleteLearnedRemoteMacs( bridge ):
   t4( "deleteLearnedRemoteMacs" )
   for macVlan in list( vxlanLearnedHost ):
      lh = vxlanLearnedHost[ macVlan ]
      if lh.entryType == VxlanEntryType.learnedMac: 
         deleteLearnedHost( bridge, macVlan.vlanId, macVlan.macAddr,
                            mapVxlanEntryType( lh.entryType ),
                            deleteBridgeMac=True )

def getFloodList( vlanId ):
   t2( 'getFloodList: vlan =', vlanId )
   # Return all VTEPs present in flood-list for VLAN
   vteps = []
   if vlanId in l2RibFloodSet.vlanFloodSet:
      vfs = l2RibFloodSet.vlanFloodSet[ vlanId ]
      for fs in vfs.floodSet.values():
         for dest in fs.destSet:
            vteps.append( dest.vxlan.vtepAddr )
   return vteps

def vtepToPrefix( vtep ):
   vtep = Arnet.IpGenAddr( str( vtep ) )
   nmask = '/32' if vtep.af == 'ipv4' else '/128'
   return Tac.Value( 'Vxlan::VtepPrefix',
                     Tac.Value( 'Arnet::IpGenPrefix', str( vtep ) + nmask ) )

def updateLearnStatusFromFlood( vlanId, vteps, learnStatusList ):
   t2( 'updateLearnStatusFromFlood vlanId %d, vteps %s, learnStatusList %s' %
       ( vlanId, vteps, learnStatusList ) )
   # Which ones need to be deleted?
   if learnStatusList[ vlanId ]:
      for old in learnStatusList[ vlanId ].prefixList:
         t2( 'updateLearnStatusFromFlood: check remove prefix=', old )
         if not old.ipPrefix.v4Prefix.address in vteps:
            t4( 'updateLearnStatusFromFlood vlanId %d deleting %s' %
                ( vlanId, old ) )
            del learnStatusList[ vlanId ].prefixList[ old ]
            del learnStatusList[ vlanId ].numMatches[ old ]
   # Which ones to add?
   if vteps:
      for vl in vteps:
         tp = vtepToPrefix( vl )
         t2( 'updateLearnStatusFromFlood: check add addr=', vl )
         if not learnStatusList[ vlanId ].prefixList or \
                not tp in learnStatusList[ vlanId ].prefixList:
            t4( 'updateLearnStatusFromFlood vlanId %d adding %s aka %s' %
                ( vlanId, vl, tp ) )
            learnStatusList[ vlanId ].prefixList[ tp ] = True
            learnStatusList[ vlanId ].numMatches[ tp ] = 0

# Add/replace any existing info
def addDefaultLearnRestrict( learnFrom, prefixes, vteps, vlanId ):
   t2( 'addDefaultLearnRestrict: vlan %d learnFrom %s prefixes %s vteps %s' % \
          ( vlanId, learnFrom, prefixes, vteps ) )
   if vlanId in vxlanHwStatusDir.learnStatus.learnStatusList:
      del vxlanHwStatusDir.learnStatus.learnStatusList[ vlanId ]
   if learnFrom == 'learnFromDefault':
      return

   ls = vxlanHwStatusDir.learnStatus.learnStatusList.newMember( vlanId )
   ls.learnFrom = learnFrom
   if learnFrom == 'learnFromList':
      for p in prefixes:
         t2( 'addDefaultLearnRestrict: vlan %d add prefix %s' % ( vlanId, p ) )
         ls.prefixList[ p ] = True
         ls.numMatches[ p ] = 0
   elif learnFrom == 'learnFromFloodList':
      for v in vteps:
         p = vtepToPrefix( v )
         t2( 'addDefaultLearnRestrict: vlan %d add vtep %s' % ( vlanId, p ) )
         ls.prefixList[ p ] = True
         ls.numMatches[ p ] = 0

def modifyDefaultLearnRestrict( learnFrom, prefixes, vteps, vlanId ):
   t2( 'modifyDefaultLearnRestrict: vlan %d learnFrom %s prefixes %s vteps %s' % \
          ( vlanId, learnFrom, prefixes, vteps ) )
   learnList = vxlanHwStatusDir.learnStatus.learnStatusList
   if vlanId not in learnList:
      # New entry
      t4( 'modifyDefaultLearnRestrict: new vlan %d' % ( vlanId ) )
      addDefaultLearnRestrict( learnFrom, prefixes, vteps, vlanId )
      return

   if learnFrom == 'learnFromDefault':
      del learnList[ vlanId ]
   elif learnFrom != learnList[ vlanId ].learnFrom:
      # learnFrom changed - replace all
      t4( 'modifyDefaultLearnRestrict: learnFrom change from %s to %s' %
          ( learnList[ vlanId ].learnFrom, learnFrom ) )
      addDefaultLearnRestrict( learnFrom, prefixes, vteps, vlanId )
   elif learnList[ vlanId ].learnFrom == 'learnFromFloodList':
      t4( 'modifyDefaultLearnRestrict: learnFromFloodList' )
      learnList[ vlanId ].learnFrom = learnFrom
      updateLearnStatusFromFlood( vlanId, vteps, learnList )
   elif learnList[ vlanId ].learnFrom == 'learnFromList':
      t4( 'modifyDefaultLearnRestrict: learnFromList' )

      learnList[ vlanId ].learnFrom = learnFrom
      # What needs to be deleted?
      for old in learnList[ vlanId ].prefixList:
         if not old in prefixes:
            t4( 'modifyDefaultLearnRestrict: delete vlan %d prefix %s' %
                ( vlanId, old ) )
            del learnList[ vlanId ].prefixList[ old ]
            del learnList[ vlanId ].numMatches[ old ]
         # What needs to be added?
      for prefix in prefixes:
         if not prefix in learnList[ vlanId ].prefixList:
            t4( 'modifyDefaultLearnRestrict: add vlan %d prefix %s' %
                ( vlanId, prefix ) )
            learnList[ vlanId ].prefixList[ prefix ] = True
            learnList[ vlanId ].numMatches[ prefix ] = 0

# Remove any learned entries which matches VTEPs which we no longer accept
def cleanupRemoteMacsLearnRestrict( bridge, vlanId=None ):
   t2( 'cleanupRemoteMacsLearnRestrict vlan %s' %  vlanId )
   learnStatus = vxlanHwStatusDir.learnStatus
   # Determine which vlans are completely gone or have learnFromList with an empty
   # list. Those will be completely removed from vxlanLearnHost.
   # The vlans with learnFromAny will be skipped.
   deleteVlans = set()
   skipVlans = set()
   for vlan in vxlanHwStatusDir.learnStatus.learnStatusList:
      if learnStatus.learnStatusList[ vlan ].learnFrom == 'learnFromAny':
         skipVlans.add( vlan )
         t4( 'cleanupRemoteMacsLearnRestrict skipVlans %s' %  skipVlans )
      elif learnStatus.learnStatusList[ vlan ].learnFrom == 'learnFromList' and \
             not learnStatus.learnStatusList[ vlan ].prefixList:
         deleteVlans.add( vlan )
         t4( 'cleanupRemoteMacsLearnRestrict deleteVlans %s' %  deleteVlans )

   for macVlan in list( vxlanLearnedHost ):
      if ( vlanId is None or vlanId == macVlan.vlanId ) and \
             not macVlan.vlanId in skipVlans:
         lh = vxlanLearnedHost[ macVlan ]
         delete = False
         if macVlan.vlanId in deleteVlans:
            delete = True
         elif macVlan.vlanId not in learnStatus.learnStatusList:
            deleteVlans.add( macVlan.vlanId )
            delete = True
         else:
            delete = not checkLearnFromVtep(
                  macVlan.vlanId, lh.remoteVtepAddr, stats=False )

         t5( 'cleanupRemoteMacsLearnRestrict vlan %d vtep %s delete %s' %  \
                ( macVlan.vlanId, lh.remoteVtepAddr, delete ) )
         if delete:
            deleteLearnedHost( bridge, macVlan.vlanId, macVlan.macAddr,
                               mapVxlanEntryType( lh.entryType ),
                               deleteBridgeMac=True )

# vlanId has learnFromFloodList and the flood list changed. Remove any
# entries we should not have learned with new flood list.
def cleanupRemoteMacsLearnRestrictFlood( bridge, vlanId ):
   t2( 'cleanupRemoteMacsLearnRestrictFlood vlan %s' %  vlanId )

   for macVlan in list( vxlanLearnedHost ):
      if vlanId == macVlan.vlanId:
         lh = vxlanLearnedHost[ macVlan ]
         if not checkLearnFromVtep(
               macVlan.vlanId, lh.remoteVtepAddr, stats=False ):
            t5( 'cleanupRemoteMacsLearnRestrictFlood vlan %d vtep %s delete' % \
                   ( macVlan.vlanId, lh.remoteVtepAddr ) )
            deleteLearnedHost( bridge, macVlan.vlanId, macVlan.macAddr,
                               mapVxlanEntryType( lh.entryType ),
                               deleteBridgeMac=True )

def updateHwHerTunnel( bridge, vtep, delete=False ):
   t2( f'updateHwHerTunnel: vtep={vtep}, delete={delete!r}' )
   if not delete:
      vtepStatus = vtepHwStatus.vtepStatus.get( VxlanTunnelType.vxlanHwHerTunnel )
      if not vtepStatus:
         vtepStatus = vtepHwStatus.newVtepStatus( VxlanTunnelType.vxlanHwHerTunnel )
      incRemoteVtepFloodRef( bridge, vtep )
      vtepStatus.vtepList[ vtep ] = 1
   else:
      if not decRemoteVtepFloodRef( bridge, vtep ):
         vtepStatus = vtepHwStatus.vtepStatus.get( VxlanTunnelType.vxlanHwHerTunnel )
         if vtepStatus:
            t2( 'updateHwHerTunnel: deleting entry for vtep=%s' % vtep )
            del vtepStatus.vtepList[ vtep ]

# check of vxlan encapsulated arp and redirect to cpu
def redirectArpToCpu( vlanId, offset, pkt ):
   ( p, headers, off ) = parsePktStr( pkt[ offset : ] )
   ethHdr = findHeader( headers, "EthHdr" )
   if ethHdr and ethHdr.ethType == "ethTypeArpIp":
      t2( "vxlan encapsulated arp frame. Redirecting to cpu" )

   elif ethHdr and ethHdr.ethType == "ethTypeIp6" and \
         Toggles.VxlanToggleLib.toggleVxlanIpv6OverlayEncapsulatedNdAclEnabled():
      # Overlay IPv6 NS/NA pkts handled in VxlanSwFwd agent
      ip6Hdr = findHeader( headers, "Ip6Hdr" )
      if not ip6Hdr or ip6Hdr.protocolNum != 'ipProtoIcmpv6':
         return False
      icmp6Hdr = Tac.newInstance( "Arnet::Icmp6HdrWrapper", p, off )
      if not icmp6Hdr or ( icmp6Hdr.typeNum != 'icmp6TypeNeighborSolicitation' and
               icmp6Hdr.typeNum != 'icmp6TypeNeighborAdvertisement' ):
         return False
      t2( "vxlan encapsulated NS/NA frame. Redirecting to cpu" )
   else:
      return False

   redirectedPkt = pkt[ 0:12 ] + \
       struct.pack( '!HH', ETH_P_ARISTA_VXLAN_ARP, vlanId ) + \
       pkt[ 12: ]
   vxlanAgentDevs.vxlanTapDevice_.send( redirectedPkt )
   return True

# check of vxlan encapsulated bfd and redirect to cpu
def redirectBfdToCpu( offset, pkt ):
   ( _, headers, _ ) = parsePktStr( pkt[ offset : ] )
   udpHdr = findHeader( headers, 'UdpHdr' )
   if udpHdr and udpHdr.dstPort == 3784:
      ethHdr = findHeader( headers, "EthHdr" )
      if ethHdr and ethHdr.dst == vxlanAgentDevs.vxlanTapDevice_.hw_:
         t2( "vxlan encapsulated bfd frame. Redirecting inner bfd to cpu" )
         vxlanAgentDevs.vxlanTapDevice_.send( pkt[ offset : ] )
         return True
      else:
         t2( "vxlan encapsulated bfd frame not addressed to this cpu" )
   return False

def aesGcmDecrypt( key, associated_data, iv, ciphertext, tag ):
   t8( 'Try to decrypt Vxlansec using AES GCM, key', Tracing.HexDump( key ), 
       '\nassociated data', Tracing.HexDump( associated_data ), '\niv',
       Tracing.HexDump( iv ), '\nciphertext', Tracing.HexDump( ciphertext ), 
       '\ntag', Tracing.HexDump( tag ) )
   # Construct a Cipher object, with the key, iv, and additionally the
   # GCM tag used for authenticating the message.
   decryptor = Cipher(
                  algorithms.AES( key ),
                  modes.GCM( iv, tag ),
                  backend=default_backend()
               ).decryptor()

   # We put associated_data back in or the tag will fail to verify
   # when we finalize the decryptor.
   decryptor.authenticate_additional_data( associated_data )

   # Decryption gets us the authenticated plaintext.
   # If the tag does not match an InvalidTag exception will be raised.
   plaintext = None
   try:
      plaintext = decryptor.update( ciphertext ) + decryptor.finalize()
   except InvalidTag:
      t1( 'Packet authentication failed (Vxlansec UDP-ESP)' )
   return plaintext

def getCryptoKey( spi, dip, vrfName=defaultVrfName ):
   t8( 'Try to get IPSec cryptoKey, spi %d, dip %s, vrf %s' % ( spi, dip, vrfName ) )
   vrfStatus = ipsecStatus.vrfStatus.get( vrfName )
   if not vrfStatus:
      t7( 'IPSec VrfStatus not found' )
      return None
   dAddr = Arnet.IpGenAddr( dip )
   proto = Tac.enumValue( 'Ipsec::Ike::IpsecAuthProtocol', 'esp' )
   saId = Tac.newInstance( 'Ipsec::Ike::SAId', spi, dAddr, proto, 0 )
   addrFamily = socket.AF_INET6 if ':' in dip else socket.AF_INET
   saKey = Tac.newInstance( 'Ipsec::Ike::SecurityAssociationKey', saId,
                            int( addrFamily ) )
   sa = vrfStatus.sa.get( saKey )
   if not sa:
      t7( 'SA not found' )
      return None
   t8( 'cryptoKey', Tracing.HexDump( sa.cryptoKey ) )
   return sa.cryptoKey

def handlePreTunnelPkt( bridge, dstMacAddr, data, srcPort ):
   """
   If a vxlan unicast encapsulated packet, decap and return the inner packet with 
   a modified source port of the vxlan interface so learning is properly handled.
   """

   t8( "Vxlan handler for preTunnel packet, dstMacAddr={} and srcPort={}".format(
                                            dstMacAddr, srcPort ) )
   retval = ( False, False, False, None )
   drop = ( False, False, True, None )

   if not Ethernet.isUnicast( dstMacAddr ) and \
         not Ethernet.isMulticast( dstMacAddr ):
      t8( 'received potential BUM packet' )
      return retval

   ( valid, srcVtep, dstVtep, vlan, off, dstVtiPort, decryptedData ) = \
            validateAndParseVxlanHeader( bridge, data, srcPort )

   if not valid:
      t8( 'received invalid packet, dropping it' )
      return retval
   if decryptedData:
      data = decryptedData

   # When McastEvpn underlay multicast is configured, etba should bridge
   # only BULL packets.
   # Multicast data packets are bridged/routed by BessMgr, so etba should drop them.
   if Ethernet.isMulticast( dstMacAddr ):
      ( _, headers, _ ) = parsePktStr( data[ off : ] )
      ethHdr = findHeader( headers, "EthHdr" )
      if Ethernet.isMulticast( ethHdr.dst ):
         ipHdr = findHeader( headers, "IpHdr" )
         if ipHdr is None:
            t8( 'received multicast packet with non-ipv4 multicast encapsulation, '
                'dropping it' )
            return retval

         if isinstance( ipHdr.dst, str ):
            ipDst = Arnet.IpAddr( ipHdr.dst )
         else:
            ipDst = ipHdr.dst

         if not ipDst.isReservedMulticast:
            t8( 'received multicast packet with multicast encapsulation, drop it' )
            return retval

   vxlanPort = dstVtiPort # bridge.vxlanPort_.get( 'Vxlan1' )
   vtiStatus = bridge.em().entity(
                     'interface/status/eth/vxlan' ).vtiStatus[ vxlanPort.intfName_ ]

   processInnerFrame( bridge, srcVtep, vlan, off, data )
  
   # txraw now gets complete packet from VxlanSwFwd, it is checked against
   # Vxlan port to not send back to VxlanSwFwd
   if srcPort != vxlanPort and redirectArpToCpu( vlan, off, data ): 
      t8( 'packet is redirected to cpu' )
      return drop

   if srcPort != vxlanPort and redirectBfdToCpu( off, data ):
      t8( 'packet is redirected to cpu' )
      return drop

   # Verify that packet is encapsulated with VARP VTEP IP if and only if the inner
   # source mac is VARP MAC.
   def dropVirtVtep( vtep, mac, vtiStatus, virtMac ):
      if vtep.af == 'ipv4':
         virtVtep = vtiStatus.vArpVtepAddr
         if virtVtep == '0.0.0.0':
            return False
         vtep = vtep.v4Addr
      else:
         virtVtep = vtiStatus.vArpVtepAddr6
         if virtVtep.isUnspecified:
            return False
         vtep = vtep.v6Addr
      if ( vtep == virtVtep ) == ( str( mac ) == str( virtMac ) ):
         return False
      t8( 'dropping packet' )
      t8( 'dest vtep %s virt vtep %s inner dmac %s virt mac %s' %
          ( vtep, virtVtep, mac, virtMac ) )
      return True

   addr = struct.unpack( '>HHH', data[ off : off+6 ] )
   innerDstMacAddr = Tac.Value( "Arnet::EthAddr", addr[0], addr[1], addr[2] )
   if dropVirtVtep( dstVtep, innerDstMacAddr, vtiStatus,
         vrMacStatus.varpVirtualMac ):
      return drop

   decaped = data[ off: ]
   vlanPri = 0
   vlanAct = PKTEVENT_ACTION_ADD
   t8( f"applyVlanChanges pri={vlanPri} id={vlan} act={vlanAct}" )
   data = applyVlanChanges( decaped, vlanPri, vlan, vlanAct )

   highOrderVlanBits = None
   evid = Tac.Value( 'Bridging::ExtendedVlanId', vlan )
   if evid.inExtendedRange():
      highOrderVlanBits = evid.highOrderBits()

   bridge.packetContext[ 'vxlanSrcVtep' ] = srcVtep
   return ( data, vxlanPort, False, highOrderVlanBits )

def floodSuppress( bridge, finalIntfs, srcPort=None, dropReasonList=None,
                   vlanId=None, dstMacAddr=None, data=None ):
   srcVtep = bridge.packetContext.get( 'vxlanSrcVtep' )
   if not srcVtep:
      # Not a VXLAN packet.
      return

   if not dstMacAddr:
      return

   tacEvid = Tac.Value( 'Bridging::ExtendedVlanId', vlanId )
   if tacEvid.inExtendedRange():
      # no flooding in extended vlans
      return

   if Ethernet.isUnicast( dstMacAddr ):
      # TODO: With VXLAN GPE support, the B bit can make a more accurate assessment
      # of whether this was known or unknown unicast on the source VTEP.

      if bridge.destLookup( vlanId, dstMacAddr )[ 0 ]:
         # Destination is known. Assume known unicast on the source VTEP too, in
         # which case finalIntfs is already correct.
         return

   vlanFloodStatus = evpnStatus.vlanFloodStatus.get( vlanId )
   if vlanFloodStatus:
      peerFloodStatus = vlanFloodStatus.peerFilter.get( srcVtep )

      # Mutate existing list in a delete-over-iterator safe way.
      for intf in list( finalIntfs ):
         if intf in vlanFloodStatus.floodFilter or (
               peerFloodStatus and intf in peerFloodStatus.floodFilter ):
            del finalIntfs[ intf ]

def handleDecapPkt( bridge, routed, vlanId, dstMacAddr, data, srcPortName ):
   """ If the received packet is intended for Vxlan decap, learn the remote
   address """

   retval = ( False, False )

   t8( 'Vxlan Decap packet processing' )
   if not routed:
      t7( 'received non-decap packet' )
      return retval

   ( valid, srcVtep, _, vlan, off, _, decryptedData ) = \
         validateAndParseVxlanHeader( bridge, data )
   if not valid:
      t7( "Invalid vxlan header" )
      return retval
   if decryptedData:
      data = decryptedData

   processInnerFrame( bridge, srcVtep, vlan, off, data )
   return retval

def rewriteSharedRouterDstMac( bridge, routed, vlanId,
                               dstMacAddr, data, srcPortName ):
   """If the received packet dst MAC is MLAG shared router MAC,
      rewrite it to bridge MAC"""

   retval = ( False, False )

   t8( 'MLAG Shared Router MAC processing' )
   if bridge.sharedRouterMac_ and dstMacAddr == bridge.sharedRouterMac_:
      t2( 'Frame with shared router MAC destination. Sending to bridgeMac instead' )
      newDstMac = struct.pack( '>BBBBBB', *( int( n, 16 ) for n in
                                          bridge.bridgeMac_.split( ':' ) ) )
      data = newDstMac + data[ 6: ]
      retval = ( data, bridge.bridgeMac_ )
   return retval

def rewriteEvpnDefaultGwMac( bridge, routed, vlanId,
                             dstMacAddr, data, srcPortName ):
   """If the received packet dst MAC is one of the Evpn Default Gateway MAC,
      and vlan ID matches one of the bridgeMAC's bridge IDs,
      rewrite it to bridge MAC"""

   retval = ( False, False )

   t8( 'EVPN Default Gateway Router MAC processing' )
   if dstMacAddr in bridge.evpnGatewayMacDb_.gatewayMac:
      gwMacData = bridge.evpnGatewayMacDb_.gatewayMac[ dstMacAddr ]
      allowedVlans = [ brId.vlanId for brId in gwMacData.bridgeId ]
      if vlanId in allowedVlans:
         t2( 'Frame with Evpn Default Gateway MAC dest.'
            ' Sending to bridgeMac instead' )
         newDstMac = struct.pack( '>BBBBBB', *( int( n, 16 ) for n in
                                             bridge.bridgeMac_.split( ':' ) ) )
         data = newDstMac + data[ 6: ]
         retval = ( data, bridge.bridgeMac_ )
   return retval

def destLookup( bridge, vlanId, destMac, data ):
   fabricMacs = { vrMacStatus.varpVirtualMac: 'varpMac',
                  bridge.sharedRouterMac_: 'sharedRouterMac' }
   if destMac in fabricMacs:
      t2( "dest mac matches the", fabricMacs[ destMac ], destMac )
      return( True, [ 'Cpu', ] )

   # For EVPN default GW MAC, only forward to CPU if vlanId matched.
   if destMac in bridge.evpnGatewayMacDb_.gatewayMac:
      gwMacData = bridge.evpnGatewayMacDb_.gatewayMac[ destMac ]
      allowedVlans = [ brId.vlanId for brId in gwMacData.bridgeId ]
      if vlanId in allowedVlans:
         t2( "dest mac matches the defaultGatewayMac", destMac )
         return( True, [ 'Cpu', ] )
   return( False, None )

def learning( vlanId, macAddr, portName, entryType ):
   # disable learning varpMac
   if macAddr == vrMacStatus.varpVirtualMac:
      t2( "macAddr is a varpMac. Skip learning ", macAddr )
      return 'dontLearn'
   return 'learn'

class L2RibDestReactor( Tac.Notifiee ):
   notifierTypeName = 'L2Rib::DestOutput'

   def __init__( self, l2RibDest_, hostReactor ):
      t8( 'L2RibDestReactor initialized' )
      self.l2RibDest_ = l2RibDest_
      self.hostReactor = hostReactor
      Tac.Notifiee.__init__( self, self.l2RibDest_ )

   def __del__( self ):
      t8( 'L2RibDestReactor destroyed' )
   
   @Tac.handler( 'dest' )
   def handleDest( self, key ):
      t8( 'handle destId ', key )
      if key not in self.hostReactor.destIdBacklog:
         t8( 'no hosts' )
         return
      backlog = self.hostReactor.destIdBacklog[ key ]
      self.hostReactor.destIdBacklog[ key ] = []
      for host in backlog:
         t8( 'update host', host )
         self.hostReactor.handleLearnedRemoteHosts( key=host )

class L2RibLBReactor( Tac.Notifiee ):
   notifierTypeName = 'L2Rib::LoadBalanceOutput'

   def __init__( self, l2RibLoadBalance_, hostReactor ):
      t8( 'L2RibLBReactor initialized' )
      self.l2RibLoadBalance_ = l2RibLoadBalance_
      self.hostReactor = hostReactor
      Tac.Notifiee.__init__( self, self.l2RibLoadBalance_ )

   def __del__( self ):
      t8( 'L2RibLBReactor destroyed' )
   
   @Tac.handler( 'lb' )
   def handleLoadBalance( self, key ):
      t8( 'handle lbId', key )
      if key not in self.hostReactor.hostsByLbId:
         t8( 'no hosts' )
         return
      for host in list( self.hostReactor.hostsByLbId[ key ] ):
         t8( 'update host', host )
         self.hostReactor.handleLearnedRemoteHosts( key=host )

class L2RibOutputReactor( Tac.Notifiee ):
   notifierTypeName = 'L2Rib::HostOutput'

   def __init__( self, l2RibOutput_, l2RibLoadBalance_, l2RibDest_, bridge ):
      assert bridge
      t8( 'L2RibOutputReactor initialized' )
      self.bridge_ = bridge
      self.l2RibOutput_ = l2RibOutput_
      self.hosts_ = self.l2RibOutput_.host
      self.l2RibLoadBalance_ = l2RibLoadBalance_
      self.loadBalance_ = self.l2RibLoadBalance_.lb
      self.l2RibDest_ = l2RibDest_
      self.dest_ = self.l2RibDest_.dest
      Tac.Notifiee.__init__( self, self.l2RibOutput_ )
      self.hostsByLbId = {}
      self.lbIdByHost = {}
      self.lbReactor = L2RibLBReactor( l2RibLoadBalance_, self )
      self.destIdBacklog = {}
      self.destReactor = L2RibDestReactor( l2RibDest_, self )
      self.hostSource = []
      self.handleLearnedRemoteHosts()

   def __del__(self):   
      t8( 'L2RibOutputReactor destroyed' )

   def getHostEntryType( self, key=None ):
      fid = self.bridge_.brConfig_.vidToFidMap.get( key.vlanId )
      if not fid:
         fid = key.vlanId

      fdbStatus = self.bridge_.brStatus_.fdbStatus[ fid ]
 
      learnedHosts = fdbStatus.learnedHost
      hostEntry = learnedHosts.get( key.macaddr )
      if hostEntry:
         return hostEntry.entryType
      else:
         return None

   @Tac.handler( 'host' )
   def handleHost( self, hostKey ):
      t8( 'handleHost key mac %s vlan %d' % ( hostKey.macaddr, hostKey.vlanId ) )
      self.handleLearnedRemoteHosts( key=hostKey )

   def checkDest( self, nt, key ):
      vtepAddrs = []
      intf = None
      destType = 'notVxlan'
      d = self.dest_.get( nt.destId() )
      if not d:
         t0( 'dest entry not ready yet' )
         if not nt.destId() in self.destIdBacklog:
            self.destIdBacklog[ nt.destId() ] = []
         self.destIdBacklog[ nt.destId() ].append( key )
      elif d.destType == 'destTypeVxlan':
         destType = d.destType
         vtepAddrs.append( d.vxlan.vtepAddr )
      elif d.destType == 'destTypeIntf':
         destType = d.destType
         intf = d.intf.intfId
      return ( vtepAddrs, destType, intf )

   def checkLoadBalance( self, nt, key ):
      vtepAddrs = []
      destType = 'notVxlan'
      lb = self.loadBalance_.get( nt.lbId() )
      if not lb:
         if not nt.lbId() in self.hostsByLbId:
            self.hostsByLbId[ nt.lbId() ] = {}
         t8( "Adding host %s to lbId" % key, nt.lbId() )
         self.hostsByLbId[ nt.lbId() ][ key ] = True
         if key not in self.lbIdByHost:
            self.lbIdByHost[ key ] = {}
         self.lbIdByHost[ key ][ nt.lbId() ] = True
         t0( 'lb entry not ready yet' )
         return ( vtepAddrs, destType )
      for n in lb.lb.next.values():
         if n.tableType == 'tableTypeLoadBalance':
            vts, dT = self.checkLoadBalance( n, key )
         elif n.tableType == 'tableTypeDest':
            vts, dT, _ = self.checkDest( n, key )
         else:
            continue
         vtepAddrs.extend( vts )
         destType = dT
      if destType == 'destTypeVxlan':
         if not nt.lbId() in self.hostsByLbId:
            self.hostsByLbId[ nt.lbId() ] = {}
         t8( "Adding host %s to lbId" % key, nt.lbId() )
         self.hostsByLbId[ nt.lbId() ][ key ] = True
         if key not in self.lbIdByHost:
            self.lbIdByHost[ key ] = {}
         self.lbIdByHost[ key ][ nt.lbId() ] = True
      return ( vtepAddrs, destType )

   
   def handleLearnedRemoteHosts( self, key=None ):
      keys = list( self.hosts_ ) if key is None else [ key ]

      for k in keys:
         # verify that the host has been removed from l2RibOutput
         # if so remove it from fdbStatus
         host = self.hosts_.get( k )
         t8( 'handleLearnedRemote host', k )
         if not host:
            # clear old lb info
            lbs = self.lbIdByHost.get( key )
            if lbs:
               for lb in lbs:
                  t8( f'removing key {key} from lb {lb}' )
                  hosts = self.hostsByLbId.get( lb )
                  del hosts[ key ]
            self.lbIdByHost[ key ] = {}
            if k in self.hostSource:
               t5( 'Deleting entry found for key {}/{}'.format( k.macaddr,
                                                            k.vlanId ) )
               fid = self.bridge_.brConfig_.vidToFidMap.get( k.vlanId )
               if not fid:
                  fid = k.vlanId

               if fid in self.bridge_.brStatus_.fdbStatus:
                  entryType = self.getHostEntryType( k )
                  t5( 'Host entry type for key {}/{} is {}'.format( k.macaddr,
                                                                k.vlanId,
                                                                entryType ) )
                  # Only entry types managed by L2Rib can be removed from
                  # vxlanLearnedHost.
                  if entryType and ( isEntryTypeController( entryType ) or \
                                     entryType == 'configuredRemoteMac' ):
                     deleteLearnedHost(
                        self.bridge_, k.vlanId, k.macaddr,
                        entryType, deleteBridgeMac=True )
               self.hostSource.remove( k )
            t8( 'skipping host, not found' )
            continue
         
         vtepAddrs = []
         intf = None
         destType = 'notVxlan'
         # Traverse the multi-table model if next is valid
         # L2Rib guarantees that the install order is correct, i.e. a host is not
         # installed if the ObjTuple that  host.next points to isn't also installed,
         # so we don't have to worry about incomplete L2Rib routes.
         if host.nextIsStored:
            t0( 'Checking multitable host for Vxlan' )
            nextTuple = host.next
            if nextTuple.tableType == 'tableTypeLoadBalance':
               vtepAddrs, destType = self.checkLoadBalance( nextTuple, key )
            elif nextTuple.tableType == 'tableTypeDest':
               vtepAddrs, destType, intf = self.checkDest( nextTuple, key )
         elif host.intfIsStored:
            destType = 'destTypeIntf'
            intf = host.intf
         t5( 'host source {} destType {} entryType {}'.format( host.source,
             destType, host.entryType ) )
         if ( host.source in ( 'sourceBgp', 'sourceVcs', 'sourceVxlanStatic' ) and
              destType in ( 'destTypeVxlan', 'destTypeIntf' ) ):
            assert( host.entryType == 'evpnDynamicRemoteMac' or
                    host.entryType == 'evpnConfiguredRemoteMac' or
                    host.entryType == 'evpnIntfDynamicMac' or
                    host.entryType == 'evpnIntfStaticMac' or
                    host.entryType == 'receivedRemoteMac' or
                    host.entryType == 'configuredRemoteMac' )
            self.hostSource.append( key )
            vtepAddrs = [ addr for addr in vtepAddrs if not addr.isAddrZero ] 
            if destType == 'destTypeVxlan':
               assert vtepAddrs, 'remote vtep is zero'
            t4( 'source {} for key {}/{} tunnelIpAddr {}'.format(
                  host.source, host.macAddr, host.vlanId, vtepAddrs ) )

            addLearnedHost( self.bridge_,
                            host.vlanId,
                            host.macAddr,
                            vtepAddrs,
                            intf=intf,
                            entryType=host.entryType,
                            moveCount=host.seqNo if host.seqNo else None )

class L2RibVlanFloodSetReactor( Tac.Notifiee ):
   notifierTypeName = 'L2Rib::FloodSetOutput'
   def __init__( self, l2RibFloodSetOutput, bridge ):
      t2( 'L2RibVlanFloodSetReactor::init' )
      self.l2RibFloodSet = l2RibFloodSetOutput
      self.bridge_ = bridge
      self.l2RibFloodSetReactor_ = {}
      Tac.Notifiee.__init__( self, self.l2RibFloodSet )
      self.handleVlanFloodSets()

   def handleVlanFloodSets( self ):
      t2( 'L2RibVlanFloodSetReactor::handleVlanFloodSets' )
      for vlanId in self.l2RibFloodSet.vlanFloodSet:
         self.handleVlanFloodSet( vlanId )

   @Tac.handler( 'vlanFloodSet' )
   def handleVlanFloodSet( self, vlanId ):
      t2( 'L2RibVlanFloodSetReactor::handleVlanFloodSet for vlan=%s' % vlanId )
      if vlanId not in self.l2RibFloodSet.vlanFloodSet:
         return

      t2( 'handleVlanFloodSet create a new reactor for vlan=%d' % vlanId )
      # When vlanFoodSet is deleted, L2RibFloodSetReactor will be destroyed
      Tac.handleCollectionChange( L2RibFloodSetReactor, vlanId,
                                  self.l2RibFloodSetReactor_,
                                  self.notifier_.vlanFloodSet,
                                  reactorArgs=( self.bridge_, ),
                                  reactorTakesKeyArg=True,
                                  reactorFilter=None )

   def delAllFloodSetReactors( self ):
      t2( 'L2RibVlanFloodSetReactor::delAllFloodSetReactors' )
      self.l2RibFloodSetReactor_.clear()

   def __del__( self ):
      t2( 'L2RibVlanFloodSetReactor::del' )
      self.delAllFloodSetReactors()

class L2RibFloodSetReactor( Tac.Notifiee ):
   notifierTypeName = 'L2Rib::VlanFloodSetOut'
   def __init__( self, vlanFloodSet, vlanId, bridge ):
      t2( 'L2RibFloodSetReactor::init' )
      self.vlanFloodSet = vlanFloodSet
      self.vlanId = vlanId
      self.bridge_ = bridge
      self.l2RibDestSetReactor_ = {}
      Tac.Notifiee.__init__( self, self.vlanFloodSet )
      self.handleFloodSets()

   def handleFloodSets( self ):
      t2( 'L2RibFloodSetReactor::handleFloodSets' )
      for macAddr in self.vlanFloodSet.floodSet:
         self.handleFloodSet( macAddr )

   @Tac.handler( 'floodSet' )
   def handleFloodSet( self, macAddr ):
      t2( 'L2RibFloodSetReactor::handleFloodSet for mac=%s' % macAddr )
      if macAddr not in self.vlanFloodSet.floodSet:
         return

      t2( 'handleFloodSet create a new reactor for vlan=%d macAddr=%s' % \
          ( self.vlanId, macAddr ) )
      # When floodSet is deleted, L2RibDestSetReactor will be destroyed
      Tac.handleCollectionChange( L2RibDestSetReactor, macAddr,
                                  self.l2RibDestSetReactor_,
                                  self.notifier_.floodSet,
                                  reactorArgs=( self.vlanId, 
                                                self.bridge_ ),
                                  reactorTakesKeyArg=True,
                                  reactorFilter=None )

      # Handle updates to learnStatus
      learnStatus = vxlanHwStatusDir.learnStatus
      if self.vlanId in learnStatus.learnStatusList and \
         learnStatus.learnStatusList[ self.vlanId ].learnFrom == \
            'learnFromFloodList':
         # Get vteps for flood-list
         vteps = []
         for dest in self.vlanFloodSet.floodSet[ macAddr ].destSet:
            vteps.append( dest.vxlan.vtepAddr )
         updateLearnStatusFromFlood( self.vlanId, vteps, 
               learnStatus.learnStatusList )
         cleanupRemoteMacsLearnRestrictFlood( self.bridge_, self.vlanId )

   def delAllDestSetReactors( self ):
      t2( 'L2RibFloodSetReactor::delAllDestSetReactors' )
      self.l2RibDestSetReactor_.clear()

   def close( self ):
      t2( 'L2RibFloodSetReactor::close' )
      # Remove destSet reactors
      self.delAllDestSetReactors()
      t2( 'L2RibFloodSetReactor::close complete' )
      Tac.Notifiee.close( self )

class L2RibDestSetReactor( Tac.Notifiee ):
   notifierTypeName = 'L2Rib::FloodSetOut'
   def __init__( self, floodSet, macAddr, vlanId, bridge ):
      t2( 'L2RibDestSetReactor::init' )
      self.floodSet = floodSet
      self.macAddr = macAddr
      self.vlanId = vlanId
      self.learnStatus_ = vxlanHwStatusDir.learnStatus
      self.bridge_ = bridge
      Tac.Notifiee.__init__( self, self.floodSet )
      self.handleDestSets()

   def handleDestSets( self ):
      t2( 'L2RibDestSetReactor::handleFloodSets' )
      for dest in self.floodSet.destSet:
         self.handleDestSet( dest )

   @Tac.handler( 'destSet' )
   def handleDestSet( self, dest ):
      if dest.destType != 'destTypeVxlan':
         return

      vtepAddr = dest.vxlan.vtepAddr
      # Increment or decrement HER tunnel ref count for dest
      destPresent = dest in self.floodSet.destSet
      t2( 'L2RibDestSetReactor::handleFloodSet for vtep=%s, vlan=%s' % 
          ( vtepAddr, self.vlanId ) )
      updateHwHerTunnel( self.bridge_, vtepAddr, delete=not destPresent )

      # Update learnStatus if necessary
      if self.vlanId not in self.learnStatus_.learnStatusList:
         return

      learnStatus = self.learnStatus_.learnStatusList[ self.vlanId ]
      if learnStatus.learnFrom != 'learnFromFloodList':
         return

      vtepPrefix = vtepToPrefix( vtepAddr.v4Addr )
      if destPresent:
         if vtepPrefix not in learnStatus.prefixList:
            learnStatus.prefixList[ vtepPrefix ] = True
            learnStatus.numMatches[ vtepPrefix ] = 0
      else:
         if vtepPrefix in learnStatus.prefixList:
            del learnStatus.prefixList[ vtepPrefix ]
            del learnStatus.numMatches[ vtepPrefix ]
            cleanupRemoteMacsLearnRestrict( self.bridge_, self.vlanId )
      
   def __del__( self ):
      t2( 'L2RibDestSetReactor::del' )

class EtbaStatusReactor( Tac.Notifiee ):
   notifierTypeName = 'Bridging::Etba::Status'
   def __init__( self, etbaStatus, bridge ):
      t0( 'EtbaStatusReactor: __init__ ' )
      assert bridge
      self.etbaStatus_ = etbaStatus
      self.bridge_ = bridge
      Tac.Notifiee.__init__( self, etbaStatus )
      self.handleEtbaStatusInitialized()

   @Tac.handler( 'initialized' )
   def handleEtbaStatusInitialized( self ):
      etbaInit = self.etbaStatus_.initialized
      t0( 'handleEtbaStatusInitialized initialized %d' % etbaInit )
      if not etbaInit:
         return

      bridge = self.bridge_
      em = bridge.em()
      assert em

      if not inArfaMode:
         # Do not set up a port until the VTI is up
         bridge.vxlanIntfStatusReactor_ = VxlanIntfStatusReactor( bridge )
         bridge.vxlanIntfConfigReactor_ = VxlanIntfConfigReactor( bridge )

         bridge.l2RibOutputReactor_ = \
               L2RibOutputReactor( l2RibOutput, l2RibLoadBalance, l2RibDest, bridge )

         bridge.l2RibVlanFloodSetReactor_ = \
               L2RibVlanFloodSetReactor( l2RibFloodSet, bridge )

         bridge.vxlanVlanStatusReactor_ = \
               VxlanVlanStatusReactor( em.entity( 'bridging/vlan/status' ), bridge )

         bridge.evpnMacTableReactor_ = \
               EvpnMacTableReactor( em.entity( 'vxlan/input/macaddr/IpRib' ),
                     bridge )

         bridge.mlagHostTableReactor_ = \
               MlagHostTableReactor( em.entity( 'mlag/hostTable' ), bridge )
      else:
         # In Arfa mode, we need to instantiate a minimal vxlanIntfConfig/Status
         # reactor to handle triggering the static FDB and MLAG reactors, until
         # those have been fully ported to L2Rib
         bridge.vxlanIntfStatusArfaReactor_ = VxlanIntfStatusArfaReactor( bridge )
         bridge.vxlanIntfConfigArfaReactor_ = VxlanIntfConfigArfaReactor( bridge )

def postBridging( bridge, srcMac, destMac, vlanId,
                   data, srcPortName, destIntf, finalIntfs ):
   if not bridge.vxlanPort_:
      return ( None, None, None, None )

   if vlanId:
      vlanConfig = bridge.brConfig_.vlanConfig.get( vlanId )
   else:
      return ( None, None, None, None )
 
   if vlanConfig and vlanConfig.etreeRole == 'etreeRoleLeaf':
      leafBridging = "Vxlan" not in srcPortName and \
                     "Vxlan" not in destIntf and \
                     "Cpu" not in srcPortName and \
                     "Cpu" not in destIntf
      if leafBridging:
         t0( "etree leaf: local bridging is disabled" )
         return ( 'deny', None, None, None )
   return ( None, None, None, None )

def bridgeInit( bridge ):
   t2( "Vxlan plugin bridgeInit" )
   assert bridge
   assert mountCompleted, 'Vxlan mounts not completed '

   # initialize Vxlan specific data structures attached to the bridge
   bridge.vxlanVniVlanMap_ = {}
   bridge.vxlanPort_ = {} # keyed by intfId
   bridge.vxlanVlanStatusReactor_ = None
   bridge.controllerControlPlane_ = False
   bridge.mlagHostTableReactor_ = None
   bridge.evpnMacTableReactor_ = None
   bridge.l2RibOutputReactor_ = None
   bridge.l2RibVlanFloodSetReactor_ = None
   bridge.l2RibOutput_ = l2RibOutput
   bridge.vrMacStatus_ = vrMacStatus
   bridge.sharedRouterMac_ = None
   bridge.remoteVtepFloodRef_ = {}

   # These are only used in Arfa mode
   bridge.arfaVtiStatusReactors_ = {} # keyed by intfId
   bridge.vxlanIntfStatusArfaReactor_ = None
   bridge.vxlanIntfConfigArfaReactor_ = None

   em = bridge.em()
   assert em
   bridge.etbaStatusReactor_ = \
         EtbaStatusReactor( em.entity( 'bridging/etba/status' ), bridge )
   bridge.mlagStatus_ = em.entity( 'mlag/status' )
   bridge.evpnGatewayMacDb_ = em.entity( "evpn/gatewayMacDb" )

def vxlanMountCompleted():
   t0( 'All mounts completed ' )
   global mountCompleted
   mountCompleted = True

   # Finish setting up the globals
   global vxlanLearnedHost, multiVtepLearnedHost
   vxlanLearnedHost = vxlanFdbStatus.learnedHost
   multiVtepLearnedHost = vxlanFdbMultiVtepStatus.multiVtepLearnedHost
   assert vxlanLearnedHost is not None
   assert multiVtepLearnedHost is not None
   bridgingHwCapabilities.vxlanBfdSupported = True
   bridgingHwCapabilities.vxlanLogicalRouterSupported = True
   bridgingHwCapabilities.vxlanNdSnoopingSupported = True
   bridgingHwCapabilities.vxlanIpv6UnderlaySupported = True
   if not inArfaMode:
      bridgingHwCapabilities.vxlanTxRawSupported = True
      bridgingHwCapabilities.vxlanTxFwdSupported = False
   bridgingHwCapabilities.vxlanEncapIpv6ConfigSupported = True
   bridgingHwCapabilities.vxlanVtepToVtepBridgingSupported = True
   bridgingHwCapabilities.vtepClassificationBridgingSupported = True
   bridgingHwCapabilities.vxlanMaxTunnelInterfacesSupported = 7
   routingHwStatus.vxlanMaxTunnelInterfacesSupported = 7
   routing6HwStatus.vxlanMaxTunnelInterfacesSupported = 7
   bridgingHwCapabilities.vxlanMultiVtepMlagSupported = True
   bridgingHwCapabilities.vxlanDecapFilterSupported = False
   bridgingHwCapabilities.vxlanSecuritySupported = True
   bridgingHwCapabilities.remoteDomainVtepBridgingSupported = True
   bridgingHwCapabilities.evpnDciGatewayMultihomingSupported = True
   bridgingHwCapabilities.evpnMacAliasDefaultGatewaySupported = True
   bridgingHwCapabilities.vxlanEncapsulatedNdSnoopingSupported = True

def agentInit( em ):
   t2( "Vxlan plugin agentInit" )

   mg = em.mountGroup()
   shmemEm = SharedMem.entityManager( sysdbEm=em )
   shmemMg = shmemEm.getMountGroup()

   # interface/{config,status}/eth/intf is mounted by the etba, no need to do it here

   # React to the mlag Host table
   mg.mount( 'mlag/hostTable', 'Mlag::HostTable', "rO" )

   # React to this dir to get the vxlan1 up event
   mg.mount( "interface/status/eth/vxlan", "Vxlan::VtiStatusDir", "r" )

   # React to vlan changes
   mg.mount( "bridging/vlan/status", "Bridging::VlanStatusDir", "r" )

   global vrMacStatus
   vrMacStatus = mg.mount( "routing/fhrp/vrMacStatus",
                           "Routing::Fhrp::VirtualRouterMacStatus", "r" )

   # this is set by us to learn the user configured vxlan dmac+vlans
   # needs to be set during init time and during when decap
   global vxlanHwStatusDir
   vxlanHwStatusDir = \
         mg.mount( "vxlan/hardware/status", "Vxlan::VxlanHwStatusDir", "w" )

   global vxlanFdbStatus
   smashTableLen = 16 * 1024
   ci = Tac.Value( 'TacSmash::CollectionInfo', "learnedHost", smashTableLen )
   ci.prevValue = True
   mountInfo = Smash.mountInfo( 'writer' )
   mountInfo.collectionInfo.addMember( ci )

   vxlanFdbStatus = shmemEm.doMount( "vxlan/hardware/fdbStatus",
                                     "Smash::Vxlan::FdbStatus",
                                     mountInfo )

   global vxlanFdbMultiVtepStatus
   vxlanFdbMultiVtepStatus = \
         mg.mount( "vxlan/hardware/fdbMultiVtepStatus", "Vxlan::FdbMultiVtepStatus",
                   "w" )
   global swGroupIdColl
   swGroupIdColl = \
      mg.mount( "vxlan/hardware/swTunnelGroupIdColl", "Vxlan::SwTunnelGroupIdColl",
                "w" )

   global bridgingHwCapabilities
   bridgingHwCapabilities = \
       mg.mount( "bridging/hwcapabilities", "Bridging::HwCapabilities", "w" )

   global routingHwStatus
   routingHwStatus = \
       mg.mount( "routing/hardware/status", "Routing::Hardware::Status", "w" )

   global routing6HwStatus
   routing6HwStatus = \
       mg.mount( "routing6/hardware/status", "Routing6::Hardware::Status", "w" )

   global vxlanConfigDir
   vxlanConfigDir = mg.mount( "vxlan/config", "Vxlan::VxlanConfigDir", "r" )

   global vxlanStatusDir
   vxlanStatusDir = mg.mount( "vxlan/status", "Vxlan::VxlanStatusDir", "r" )

   mg.mount( "vxlan/input/macaddr/IpRib", "Bridging::RemoteMacTable", "rO" )

   # React to etba status
   mg.mount( "bridging/etba/status", "Bridging::Etba::Status", "r" )

   # React to Mlag status
   global mlagStatusMount
   # Mount mlag/status, Mlag::Status and its dependent paths
   mlagStatusMount = MlagMountHelper.mountMlagStatus( mg )

   global vxlanDynamicL2RibHostInput
   smashHostTableLen = 16 * 1024
   vxlanDynamicL2RibHostInput = \
      shmemEm.doMount( "bridging/l2Rib/vxlanDynamic/hostInput", "L2Rib::HostInput",
                       Smash.mountInfo( 'writer',
                          [ ( "host", smashHostTableLen ) ] ) )

   global vxlanDynamicL2RibDestInput
   smashDestTableLen = 1024
   vxlanDynamicL2RibDestInput = \
      shmemEm.doMount( "bridging/l2Rib/vxlanDynamic/destTable", "L2Rib::DestTable",
                       Smash.mountInfo( 'writer',
                          [ ( "dest", smashDestTableLen ) ] ) )

   global l2RibOutput
   l2RibOutput = shmemEm.doMount( "bridging/l2Rib/hostOutput", "L2Rib::HostOutput",
                                  Smash.mountInfo( 'keyshadow' ) )
   global l2RibLoadBalance
   l2RibLoadBalance = shmemEm.doMount( "bridging/l2Rib/lbOutput",
                                   "L2Rib::LoadBalanceOutput",
                                   Smash.mountInfo( 'keyshadow' ) )
   global l2RibDest
   l2RibDest = shmemEm.doMount( "bridging/l2Rib/destOutput",
                            "L2Rib::DestOutput",
                            Smash.mountInfo( 'keyshadow' ) )
   global decapConfig
   decapConfig = shmemEm.doMount( "routing/multicast/tunnel/ip/decapConfig",
                              "Routing::Multicast::IpTunnelDecapConfig",
                              Smash.mountInfo( 'keyshadow' ) )
   global l2RibFloodSet
   l2RibFloodSet = shmemMg.doMount( 'bridging/l2Rib/floodOutput',
                                    'L2Rib::FloodSetOutput',
                                    Shark.mountInfo( 'shadow' ) )
   shmemMg.doClose()

   global evpnStatus
   evpnStatus = mg.mount( 'evpn/status', 'Evpn::EvpnStatus', 'r' )

   # Mount gatewayMacDb for default gw macs
   mg.mount( 'evpn/gatewayMacDb', 'Evpn::GatewayMacDb', 'r' )

   global remoteVtepConfigDir
   remoteVtepConfigDir = mg.mount( 'vxlan/remoteVtepHwConfig',
                                   'Vxlan::RemoteVtepHwConfigDir', 'r' )

   global vtepHwStatus
   vtepHwStatus = mg.mount( 'vxlan/vtepHwStatus',
                            'Vxlan::VtepHwStatusDir', 'w' )

   global ipsecStatus
   ipsecStatus = mg.mount( 'ipsec/ike/status', 'Ipsec::Ike::Status', 'r' )

   global helper
   helper = Tac.newInstance( "VxlanImpl::Helper", vxlanDynamicL2RibHostInput,
         vxlanDynamicL2RibDestInput )

   def onMountComplete():
      vxlanMountCompleted()
   mg.close( onMountComplete )

def Plugin( ctx ):
   t2( "Vxlan plugin registering" )

   global inArfaMode
   inArfaMode = ctx.inArfaMode()

   ctx.registerBridgeInitHandler( bridgeInit )
   ctx.registerAgentInitHandler( agentInit )

   bpfFilterInfo = ( "Vxlan", "" )
   if not inArfaMode:
      ctx.registerRewritePacketOnTheWayToTheCpuHandler( handleDecapPkt )
      ctx.registerRewritePacketOnTheWayToTheCpuHandler( rewriteSharedRouterDstMac )
      ctx.registerRewritePacketOnTheWayToTheCpuHandler( rewriteEvpnDefaultGwMac )
      ctx.registerDestLookupHandler( destLookup )
      ctx.registerPacketReplicationHandler( floodSuppress )
      bpfFilterInfo = None

   if not inArfaMode:
      # bpf filter will be generated when Vxlan port becomes active
      ctx.registerPreTunnelHandler( handlePreTunnelPkt, bpfFilterInfo=bpfFilterInfo )
      ctx.registerPostAgingHandler( deleteLearnedHost )
      ctx.registerLearningHandler( macLearningHandler )
      ctx.registerLearningHandler( learning )
      ctx.registerInterfaceHandler( 'Vxlan::VtiStatus', VxlanIntfPort )
      # Post bridging modifications and changing action based on egress intf
      ctx.registerPostBridgingHandler( postBridging )

