# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re

def isFixedSystem( entityMibRoot ):
   return "FixedSystem" in entityMibRoot.tag

def formatFanDirection( entityMibRoot ):
   forwardSuffixes = ( 'F', 'RED', 'FAN' )
   reverseSuffixes = ( 'R', 'BLUE' )
   allSuffixes = forwardSuffixes + reverseSuffixes
   fanSuffix = None
   fanTraySlots = entityMibRoot.fanTraySlot
   for fanTraySlot in fanTraySlots.values():
      currentFanSuffix = None
      if not fanTraySlot.fanTray: # no fan in slot
         continue

      # Systems can have 2 kinds of Fans either -F,-R fans or -RED,-BLUE fans
      fanSuffixMatch = re.search( f"-({'|'.join( allSuffixes )})$",
            fanTraySlot.fanTray.modelName )

      if not fanSuffixMatch: # unrecognized type (or modular chassis)
         # Set fanSuffix in the case that there are mismatched fantrays
         # (where at one has an invalid name and the other has a valid name)
         fanSuffix = "Unknown"
         continue

      currentFanSuffix = fanSuffixMatch.group( 1 )
      if not fanSuffix:
         fanSuffix = currentFanSuffix

      if fanSuffix != currentFanSuffix:
         return 'Unknown'

   if fanSuffix in forwardSuffixes:
      return 'Forward'
   elif fanSuffix in reverseSuffixes:
      return 'Reverse'
   else: # Shouldn't happen
      return 'Unknown'

def formatModelNameExtended( entityMibRoot ):
   fanDirection = formatFanDirection( entityMibRoot )
   modelName = entityMibRoot.modelName
   if modelName:
      if fanDirection == 'Unknown':
         extendedName = modelName
      else:
         # Some SKUs end in -2F or -2R (eg CCS-720DP-24S-2F)
         if re.search( r"-\d$", modelName ):
            extendedName = f"{modelName}{fanDirection[ 0 ]}"
         else:
            extendedName = f"{modelName}-{fanDirection[ 0 ]}"
   else:
      extendedName = "(unknown model name)"
   return extendedName

def updateRootExtendedFields( entityMibRoot ):
   entityMibRoot.fanDirection = formatFanDirection( entityMibRoot )
   entityMibRoot.modelNameExtended = formatModelNameExtended( entityMibRoot )
