# Copyright (c) 2006-2010 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""This is the Python module for the Errdisable module.
It contains the init function for the errdisable aggregation
state machine, the errdisable cause state machine, and the
cause status creation"""

from ErrdisableCliLib import causeNameWithoutSliceId
import Tac
import Tracing
from TypeFuture import TacLazyType

t0 = Tracing.trace0

cgTypeEnum = Tac.Type( 'Errdisable::CauseGroupType' )
cgPriorityEnum = Tac.Type( 'Errdisable::CauseGroupPriority' )

CliStatus = TacLazyType( 'Interface::CliStatus::CliStatus' )
CliStatusStringHelper = TacLazyType( 'Interface::CliStatusStringHelper' )

def ErrdisableCauseStatusInit( entityManager, causeGroup, causeName ):
   def causeStatusFunc():
      return entityManager.lookup(
            # pylint: disable-next=consider-using-f-string
            'interface/errdisable/cause/%s' % causeName )
   return ErrdisableCauseStatusInitEntities( entityManager,
                                             causeGroup,
                                             causeStatusFunc,
                                             causeName )

def ErrdisableCauseStatusInitEntities( entityManager,
                                       causeGroup,
                                       causeStatusFunc,
                                       causeName ):

   """See ErrdisableCauseGroupInit for description of how to use this"""

   redundancyStatus = entityManager.redundancyStatus()
   if redundancyStatus.protocol == 'sso' and redundancyStatus.mode == 'standby':
      t0( "Errdisable cause entity:", causeName, "can't be created in SSO/STANDBY "
          "mode." )
      return None 

   t0( "ErrdisableCauseStatusInit, causeGroup:", causeGroup.name, "causeName",
       causeName )

   # If called from a plugin running on the standby supervisor in SSO mode, this
   # should just be a no-op.
   assert( not causeGroup.singleCause ) or ( causeGroup.name == causeName )
   causeStatus = causeStatusFunc()
   if causeGroup.type == cgTypeEnum.maintdisable:
      causeStatus.status = CliStatus.maintDown
   else:
      causeStatus.status = CliStatus.errdisabled
   causeStatus.reason = CliStatusStringHelper.cliStatusReasonFromString(
      causeNameWithoutSliceId( causeName ) )
   causeGroup.addCauseStatus( causeStatus )

   t0( "Errdisable entity:", causeName, "created for causeGroup:", causeGroup.name )
   return causeStatus

def ErrdisableCauseGroupInit( entityManager, causeGroupName, 
                              installCauseEnableCliCmd, description, 
                              installCauseRecoveryCliCmd=None, hiddenCause=False,
                              syslogEnabled=True, singleCause=True,
                              causeGroupType=cgTypeEnum.errdisable,
                              causeGroupPriority=cgPriorityEnum.high,
                              enabledStateReason="errdisabled" ):
   causeGroup = entityManager.lookup(
         # pylint: disable-next=consider-using-f-string
         'interface/errdisable/causegroup/%s' % causeGroupName )

   def causeStatusFunc():
      return entityManager.lookup(
            # pylint: disable-next=consider-using-f-string
            'interface/errdisable/cause/%s' % causeGroupName )
   return ErrdisableCauseGroupInitEntities( entityManager,
                                            causeGroupName,
                                            installCauseEnableCliCmd,
                                            description,
                                            causeGroup,
                                            causeStatusFunc,
                                            installCauseRecoveryCliCmd,
                                            hiddenCause,
                                            syslogEnabled,
                                            singleCause,
                                            causeGroupType,
                                            causeGroupPriority,
                                            enabledStateReason )

def ErrdisableCauseGroupInitEntities( entityManager,
                                      causeGroupName,
                                      installCauseEnableCliCmd,
                                      description,
                                      causeGroup,
                                      causeStatusFunc,
                                      installCauseRecoveryCliCmd=None,
                                      hiddenCause=False,
                                      syslogEnabled=True,
                                      singleCause=True,
                                      causeGroupType=cgTypeEnum.errdisable,
                                      causeGroupPriority=cgPriorityEnum.high,
                                      enabledStateReason="errdisabled" ):

   """Creates the CauseGroup entity in Sysdb. This function has to be
   called from the agent's Sysdb plugin.  If singleCause is set to true
   (default), then only a single CauseStatus is allowed in the group,
   with the same name as the group, and it is automatically created.
   If singleCause is explicitly set to false, then the agent must create
   each desired causeStatus using  ErrdisableCauseStatusInit, above."""

   t0( "ErrdisableCauseGroupInit, causeGroup:", causeGroupName )

   # If called from a plugin running on the standby supervisor in SSO mode, this
   # should just be a no-op.
   redundancyStatus = entityManager.redundancyStatus()
   if redundancyStatus.protocol == 'sso' and redundancyStatus.mode == 'standby':
      t0( "Errdisable causeGroup entity:", causeGroupName, "can't be created in "
          "SSO/STANDBY mode." )
      return None 

   causeGroup.installCauseEnableCliCmd = installCauseEnableCliCmd
   if installCauseRecoveryCliCmd is not None:
      causeGroup.installCauseRecoveryCliCmd = installCauseRecoveryCliCmd
   causeGroup.hiddenCause = hiddenCause
   causeGroup.description = description
   causeGroup.singleCause = singleCause
   causeGroup.syslogEnabled = syslogEnabled
   causeGroup.type = causeGroupType
   causeGroup.priority = causeGroupPriority
   causeGroup.enabledStateReason = enabledStateReason

   t0( "Errdisable causeGroup:", causeGroupName, "created." )

   # for singles, go ahead and make the solitary cause with the same
   # name as the group
   if singleCause:
      ErrdisableCauseStatusInitEntities( entityManager,
                                         causeGroup,
                                         causeStatusFunc,
                                         causeGroupName )

   return causeGroup

def ErrdisableCauseSMInit( agent, entityManager, cause, causeGroupName,
                           agentCauseStatus,
                           intfConfigDir=None, intfStatusDir=None ):
   """ Mounts the errdisable cause status entity, and instantiates the
   errdisable cause state machine. This function is called by
   the agent that owns the corresponding errdisable cause """

   # If called from a plugin running on the standby supervisor in SSO mode, this
   # should just be a no-op.
   redundancyStatus = entityManager.redundancyStatus()
   if redundancyStatus.protocol == 'sso' and redundancyStatus.mode == 'standby':
      t0( "Errdisable no SM for", causeGroupName, " in SSO/STANDBY mode" )
      return
   #
   # Instantiate the cause state machine
   # An agent can have multiple causes, and so multiple state
   # machines. Each of them is stored in a dict with the cause
   # name as the key. The dict is initialized below, during
   # the first cause init call.
   #
   if not hasattr( agent, 'errdisableCauseSM' ):
      agent.errdisableCauseSM = { }

   if causeGroupName is None:
      causeGroupName = cause
         
   agent.errdisableCauseSM[ cause ] = Tac.newInstance(
      'ErrdisableLib::CauseAgentSM', cause, causeGroupName,
      Tac.activityManager.clock, entityManager.cEntityManager(), 
      agentCauseStatus, intfConfigDir, intfStatusDir )

