#!/usr/bin/env python3
# Copyright (c) 2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
# 
# Module contains utilities for accessing EthIntf information from sysdb.

from Intf.IntfRange import allIntfNames

ethernetIntfPrefix = "Ethernet"
switchIntfPrefix = "Switch"
mgmtIntfPrefix = "Management"
cpuNicIntfPrefix = "Cpu"

switchPortIntfTypes = {}

def registerSwitchPortIntfType( intfType ):
   switchPortIntfTypes[ intfType.tagLong ] = intfType

def isEthernetIntf( intfName ):
   return ( intfName.startswith( ethernetIntfPrefix ) or
            intfName.startswith( switchIntfPrefix ) )

def isMgmtIntf( intfName ):
   return intfName.startswith( mgmtIntfPrefix )

def isCpuNicIntf( intfName ):
   return intfName.startswith( cpuNicIntfPrefix )

def isSubIntf( intfName ):
   return '.' in intfName

# -----------------------------------------------------------------------------
# Returns an iterator of all physical interface names
# -----------------------------------------------------------------------------
def allPhyInterfaceNames( phyIntfConfigDir ):
   return ( intf for intfConfig in phyIntfConfigDir.values()
            for intf in intfConfig.intfConfig
            if isEthernetIntf( intf ) or isMgmtIntf( intf ) or isCpuNicIntf( intf ) )

#-----------------------------------------------------------------------------
# Returns an iterator of all physical Ethernet interface names
#-----------------------------------------------------------------------------
def allPhyEthernetInterfaceNames( phyIntfConfigDir ):
   return ( intf for intfConfig in phyIntfConfigDir.values()
            for intf in intfConfig.intfConfig if isEthernetIntf( intf ) )

#-----------------------------------------------------------------------------
# Returns an iterator of all physical Management interface names
#-----------------------------------------------------------------------------
def allPhyMgmtInterfaceNames( phyIntfConfigDir ):
   return ( intf for intfConfig in phyIntfConfigDir.values()
            for intf in intfConfig.intfConfig if isMgmtIntf( intf ) )

#----------------------------------------------------------------------------
# Returns an iterator of all switchport interface names.
#
# If 'includeEligible' is set to True then the names for all interfaces capable
# of becoming switchport are returned. Else only currently configured switchport
# names are returned.
#
# By default subintfs are not included (since they are not really switchports) unless
# the caller explicitly wants them.
#---------------------------------------------------------------------------
def allSwitchportNames( requireMounts,
                        includeEligible=False,
                        includeSubIntf=False ):
   """
   Return all switchport interfaces in config. This is used by CliSave plugins
   to get configured switchports interfaces for saveAll/saveAllDetail.

   If includeEligible is False, exclude routed ports. Typically, includeEligible=True
   is used by saveAllDetail case.
   """
   allPorts = allIntfNames( requireMounts,
                            explicitIntfTypes=switchPortIntfTypes,
                            includeSubIntf=includeSubIntf )
   if includeEligible:
      return allPorts

   bridgeCliConfig = requireMounts.getValue( 'bridging/input/config/cli' )

   def isSwitchPort( intf ):
      # this is just sanity check
      if intf in bridgeCliConfig.switchIntfConfig:
         return bridgeCliConfig.switchIntfConfig[ intf ].enabled
      else:
         return bridgeCliConfig.defaultSwitchportMode != 'routed'

   return ( port for port in allPorts if isSwitchPort( port ) )
