# Copyright (c) 2008, 2009, 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string
import re, struct, socket, random
import ctypes
from SysConstants.if_ether_h import (
      ETH_P_8021Q,
      ETH_HLEN,
      ETH_FCS_LEN,
      ETH_P_IP,
      ETH_ZLEN,
      ETH_ALEN )
from SysConstants.in_h import IPPROTO_IPIP, IPPROTO_UDP, IPPROTO_TCP, IPPROTO_ICMP
IP_HDR_LEN = 20
IP6_HDR_LEN = 40
IP6_FRAG_EXTENSION_LEN = 8
UDP_HDR_LEN = 8
TCP_HDR_LEN = 20
TCP_MSS_OPTION_LEN = 4
ICMP_HDR_LEN = 8
VXLAN_HDR_LEN = 8
UDP_SPORT_DEFAULT = 1999
UDP_DPORT_DEFAULT = 0xAAAA
TCP_SPORT_DEFAULT = 1999
TCP_DPORT_DEFAULT = 0xAAAA
UDP_DPORT_VXLAN = 0x12B5
UDP_DPORT_ROCE = 4791
E_TAG_ETHER_TYPE = 0x893f
VN_TAG_ETHER_TYPE = 0x8926
ARISTA_ETHER_TYPE = 0xD28B
ARISTA_ETHER_SUBTYPE_TIMESTAMP = 0x0001

# NDP related defines
ND_OPTION_SRC_LL_ADDR = 1
ND_OPTION_DST_LL_ADDR = 2
ND_TYPE_RTR_SOLICIT = 133
ND_TYPE_NBR_SOLICIT = 135
ND_TYPE_NBR_ADVT = 136
IP6_EXT_HDR_ICMP6 = 0x3A

def macAddrToBytes( addr ):
   pair = '([0-9a-fA-F]{1,2})'
   quad = '([0-9a-fA-F]{1,4})'
   macAddrColnPattern = f'{pair}:{pair}:{pair}:{pair}:{pair}:{pair}$'
   macAddrDotPattern = fr'{quad}\.{quad}\.{quad}$'
   colonMatch = re.match( macAddrColnPattern, addr )
   dotMatch = re.match( macAddrDotPattern, addr )
   if colonMatch:
      bytes_ = b''.join( bytes( ( int( s, 16 ), ) ) for s in colonMatch.groups() )
   elif dotMatch:
      # zero pad on left of each group of numbers until length of group is 4 
      # this way an addr of 1.2.3 becomes 0001.0002.0003 and will be valid
      zeroPaddedMatch = ''.join( [ s.zfill( 4 ) for s in dotMatch.groups() ] )
      bytes_ = b''
      for i in range( 0, 12, 2 ):
         byte = zeroPaddedMatch[ i:i+2 ]
         bytes_ += bytes( ( int( byte, 16 ), ) )
   else:
      raise ValueError( "invalid MAC address: %s" % addr )
   return bytes_

def bytesToMacaddr( s ):
   return "%x:%x:%x:%x:%x:%x" % tuple( s[:6] )

def strContains( string, c ):
   # Return True if string contains c.
   if c in string: # pylint: disable=simplifiable-if-statement
      return True
   else:
      return False

def buildVlan( vlan, vpri, tpid=ETH_P_8021Q, dei=0 ):
   if ( vlan is not None ) or ( vpri is not None ):
      tci = ( ( vpri or 0 ) << 13 ) | ( dei << 12 ) | ( vlan or 0 )
      vlanTagStr = struct.pack( '>HH', tpid, tci )
   else:
      vlanTagStr = b""
   return vlanTagStr

def buildSvTag( svTagRaw ):
   if not svTagRaw:
      return b""
   svTagStr = struct.pack( '>I', svTagRaw )
   return svTagStr

# Versions that support 2B seconds and 4B nanoseconds(aid/3309)
supported6ByteL2TimestampVer = [ 0x0020, 0x0120, 0x0021, 0x0121 ]
# Versions that support 4B value of both seconds and nanoseconds(aid/3309)
supported8ByteL2TimestampVer = [ 0x0010, 0x0110, 0x0011, 0x0111 ]
supportedL2TimestampVer = supported6ByteL2TimestampVer + supported8ByteL2TimestampVer
def buildL2TimestampTagStr( l2_ts_ver, l2_ts_sec, l2_ts_nsec ):
   # Arista L2 timestamp tag(aid/2681 and aid/7947) is 12/14B long and consists of:
   # 2B ethType = 0xD28B
   # 2B subType = 0x0001
   # 2B version in [ 0x0010, 0x0020, 0x0110, 0x0120, 0x0011, 0x0021, 0x0111, 0x0121 ]
   # 6B/8B timestamp: 2B/4B seconds value depending on version + 4B nanoseconds value

   if l2_ts_ver is None and l2_ts_sec is None and l2_ts_nsec is None:
      return b""
   elif not ( l2_ts_ver is not None and l2_ts_sec is not None and
              l2_ts_nsec is not None ):
      raise ValueError( 'must specify all of --l2-ts-ver, --l2-ts-sec and '
                        '--l2-ts-nsec, or none of them' )

   if l2_ts_ver not in supportedL2TimestampVer:
      raise ValueError( 'version number not supported, must be one of %s' %
                        ', '.join( [ hex( ver ) for ver in
                                     supportedL2TimestampVer ] ) )

   # pylint: disable-next=chained-comparison
   if not ( l2_ts_nsec >= 0 and l2_ts_nsec <= 2**32 - 1 ):
      raise ValueError( 'timestamp nanoseconds should be between %s and %s' %
                        ( 0, 2**32 - 1 ) )

   if l2_ts_ver in supported8ByteL2TimestampVer:
      # Versions that support 4B value of both seconds and nanoseconds
      # pylint: disable-next=chained-comparison
      if not ( l2_ts_sec >= 0 and l2_ts_sec <= 2**32 - 1 ):
         raise ValueError( 'timestamp seconds should be between %s and %s' %
                           ( 0, 2**32 - 1 ) + ' for version %s' % l2_ts_ver )
      timeStamp = struct.pack( '>II', l2_ts_sec, l2_ts_nsec )
   else:
      # Versions that support 2B seconds and 4B nanoseconds
      # pylint: disable-next=chained-comparison
      if not ( l2_ts_sec >= 0 and l2_ts_sec <= 2**16 - 1 ):
         raise ValueError( 'timestamp seconds should be between %s and %s' %
                           ( 0, 2**16 - 1 ) + ' for version %s' % l2_ts_ver )
      timeStamp = struct.pack( '>HI', l2_ts_sec, l2_ts_nsec )

   header = struct.pack( '>HHH', ARISTA_ETHER_TYPE, ARISTA_ETHER_SUBTYPE_TIMESTAMP,
                         l2_ts_ver )
   return header + timeStamp

def buildBrTag( br=None ):
   # 802.1br tag header is 8B long and consists of:
   # ethType = 0x893f( 16bits )
   # e-pcp : priority bits ( 3bits )
   # DEI : drop eligible indicator ( 1bit ) 
   # ingress ECID : ingress extended port ID ( 12bits )
   #                [ source Vif in vn-tag ]
   # reserve ( 2bits )
   # ECID : extended port ID ( 14bits )
   #        [ destination Vif in vn-tag ]
   # reserve ( 16bits )
   # the content is simplified in this funciton for testing: 
   #     only set ethType and set 0 to all rest bits
   eTagStr = b""
   if br is not None:
      ethType = E_TAG_ETHER_TYPE 
      eTagHeader = 0
      reserve = 0
      eTagStr = struct.pack('>HIH', ethType, eTagHeader, reserve)
   return eTagStr

def buildVNTag( sVif=None, dVif=None ):
   # vn tag header is 6B( 48bits ) long and consits of :
   # ethType = 0x8926 ( 16bits )
   # destination bit ( 1bit ) : 0 from IV to bridge, 1: from the bridge to IV
   # pointer bit ( 1bit ): 1: a vif_list_id is in the tag, 0: dVif_id in the tage
   # destination Vif ( 12bits ), or vif_list_id( 14bits )
   #      dVif_id/vif_list_id is reserved if d is 0
   # looped ( 1bit ): 1 indicates that this is a multicast fram out the bridge
   # Reserved ( 3bit )
   # source Vif ( 12bits ): vif_id of the port that add the vntag,
   #                        reserved if d=1 and l=0 
   # version ( 2bits ): is 0
   # unicast d = 1, multicast p = 1
   vnTag = b""
   if( sVif is not None ) or ( dVif is not None ):
      ethType = VN_TAG_ETHER_TYPE
      if sVif is not None:
         sVif = sVif << 2
         dVif = 0 << 18
         vnTag = dVif |  sVif
         vnTagStr = struct.pack('>HI', ethType, vnTag )
         return vnTagStr
      if dVif is not None:
         # non multicast case:
         dstBit = 1 << 31
         pointer = 0 << 30
         dVif = dVif << 18
         sVif = 0 << 2
         vnTag = dstBit | pointer | dVif | sVif
         vnTagStr = struct.pack('>HI', ethType, vnTag )
         return vnTagStr
   return vnTag
         
def buildMplsHeader( mpls_label=None, mpls_ttl=None, mpls_exp=None, mpls_bos=True ):
   if mpls_label >= 0:
      # MPLS header is 32 bits long and consits of (in order):
      # - 20 bit label
      # - 3 bit QoS and ECN
      # - 1 bit bottom of stack flag (if set means the current label is the last)
      # - 8 bit TTL field
      # We set the bottom of stack flag because we are sending one label
      if mpls_exp is None:
         mpls_exp = 0
      bos = 1 if mpls_bos else 0
      value = 0
      value |= mpls_label << 12
      value |= mpls_exp << 9
      value |= bos << 8
      value |= mpls_ttl

      mplsStr = struct.pack('>I', value )
   else:
      mplsStr = b""
   return mplsStr

def buildEsmcHeader( eventFlag ):
   """
   Helper method for constructing a portion of the ESMC packet
   ITU G.8264 11.3.1.1
   +---------+----------------+-----------------------------------------------+
   |  Octet  |   Size/bits    |                     Field                     |
   +---------+----------------+-----------------------------------------------+
   | 1-6     | 6 octets       | Destination address = 01-80-C2-00-00-02 (hex) |
   | 7-12    | 6 octets       | Source address                                |
   | 13-14   | 2 octets       | Slow protocol Ethertype = 88-09 (hex)         |
   | 15      | 1 octet        | Slow protocol subtype = 0A (hex)              |
   | 16-18   | 3 octets       | ITU-OUI = 00-19-A7 (hex)                      |
   | 19-20   | 2 octets       | ITU subtype                                   |
   | 21      | bit 7:4        | Version                                       |
   | 21      | bit 3          | Event flag                                    |
   | 21      | bit 2:0        | Reserved                                      |
   | 22-24   | 3 octets       | Reserved                                      |
   +---------+----------------+-----------------------------------------------+
   """
   slowProtoSubtype = 0x0A
   oui = 0x0019A7
   slowProtoSubtypeAndOui = struct.pack( ">I", slowProtoSubtype << 24 | oui )
   # ITU G.8264 11.3.1.1 section f
   ouiSubtype = struct.pack( ">H", 0x01 )
   # ITU G.8264 11.3.1.1 section g
   version = 0x1
   # ITU G.8264 11.3.1.1 section h - Information: 0, Event: 1
   eventFlag = strContains( eventFlag, "E" ) or strContains( eventFlag, "event" )
   versionEventFlag = struct.pack( "B", ( version << 4 ) | ( eventFlag << 3 ) )
   reserved = struct.pack( ">HB", 0, 0 )
   return slowProtoSubtypeAndOui + ouiSubtype + versionEventFlag + reserved

def buildEsmcSsmTlv( ssm ):
   """
   Helper method for constructing the ESMC SSM TLV
   ITU G.8264 11.3.1.2
   +-------+-----------+---------------+
   | Octet | Size/bits |     Field     |
   +-------+-----------+---------------+
   | 1     | 8 bits    | Type: 0x1     |
   | 2-3   | 16 bits   | Length: 00-04 |
   | 4     | bits 7:4  | 0x0 (unused)  |
   |       | bits 3:0  | SSM code      |
   +-------+-----------+---------------+
   """
   assert 0 <= ssm <= 15
   ssmType = 0x01
   length = 0x0004
   unused = 0x0
   return struct.pack( ">BHB", ssmType, length, ( unused << 4 ) | ssm )

def buildEsmcPacket( headerSize, eventFlag, ssm ):
   """
   ITU G.8264 11.3.1.1
   +---------+----------------+-----------------------------------------------+
   |  Octet  |   Size/bits    |                     Field                     |
   +---------+----------------+-----------------------------------------------+
   | 1-6     | 6 octets       | Destination address = 01-80-C2-00-00-02 (hex) |
   | 7-12    | 6 octets       | Source address                                |
   | 13-14   | 2 octets       | Slow protocol Ethertype = 88-09 (hex)         |
   | 15      | 1 octet        | Slow protocol subtype = 0A (hex)              |
   | 16-18   | 3 octets       | ITU-OUI = 00-19-A7 (hex)                      |
   | 19-20   | 2 octets       | ITU subtype                                   |
   | 21      | bit 7:4        | Version                                       |
   |         | bit 3          | Event flag                                    |
   |         | bit 2:0        | Reserved                                      |
   | 22-24   | 3 octets       | Reserved                                      |
   | 25-1532 | 36-1490 octets | Data and padding                              |
   | Last 4  | 4 octets       | Frame check sequence (FCS)                    |
   +---------+----------------+-----------------------------------------------+
   """
   esmcPkt = buildEsmcHeader( eventFlag ) + buildEsmcSsmTlv( ssm )
   # ITU G.8264 11.3.1.1 section j
   MIN_ESMC_SIZE = 64
   padLen = max( MIN_ESMC_SIZE - headerSize - ETH_FCS_LEN - len( esmcPkt ), 0 )
   esmcPkt += b'\x00' * padLen
   return esmcPkt

def buildEthernetHeader( dmac, smac, ethertype, vlan=None,
                         vpri=None, dei=0, innerVlan=None,
                         innerVpri=None, innerDei=0,
                         vlanTpid=ETH_P_8021Q, innerVlanTpid=ETH_P_8021Q,
                         sVif=None, dVif=None,
                         br=None, svTagRaw=None,
                         l2TimestampInfo=None ):
   dmacStr = macAddrToBytes( dmac )
   smacStr = macAddrToBytes( smac )
   svTagStr  = buildSvTag( svTagRaw )
   if l2TimestampInfo:
      assert len( l2TimestampInfo ) == 4
      l2TsTagStr = buildL2TimestampTagStr( l2TimestampInfo[ 0 ],
            l2TimestampInfo[ 1 ], l2TimestampInfo[ 2 ] )
      l2TsAfterVlan = l2TimestampInfo[ 3 ]
   else:
      l2TsTagStr = b""
      l2TsAfterVlan = False
   vlanTagStr = buildVlan( vlan, vpri, vlanTpid, dei )
   innerVlanTagStr = buildVlan( innerVlan, innerVpri, innerVlanTpid, innerDei )
   ethertypeStr = struct.pack( '>H', ethertype )

   if br is not None:
      tagStr = buildBrTag( br )
   elif sVif is not None or dVif is not None: 
      tagStr = buildVNTag( sVif, dVif )
   else:
      tagStr = b""

   if l2TsAfterVlan:
      return dmacStr + smacStr + svTagStr + tagStr + vlanTagStr + innerVlanTagStr + \
            l2TsTagStr + ethertypeStr
   else:
      return dmacStr + smacStr + l2TsTagStr + svTagStr + tagStr + vlanTagStr + \
          innerVlanTagStr + ethertypeStr

def buildPayload( payloadLen, dataType='incrementing', dataValue=b'0' ):
   payload = b''

   if dataType == 'constant':
      data = int( dataValue )
      if data < 0 or data > 255 :
         raise ValueError( "Data value out of range: must be between 0 and 255" )
      bytes_ = [ data for i in range( payloadLen ) ]
   elif dataType == 'random':
      bytes_ = [ random.randrange( 256 ) for i in range( payloadLen ) ]
   elif dataType == 'raw':
      bytes_ = [ int( dataValue[ i:i + 2 ], base=16 )
                  for i in range( 0, len( dataValue ), 2 ) ]
      padding = [ 0 ] * max( ( payloadLen - len( bytes_ ) ), 0 )
      bytes_ = bytes_ + padding
   elif dataType == 'incrementing':
      bytes_ = [ i % 256  for i in range( payloadLen ) ]
   else:
      assert False

   payload = b''.join( bytes( ( b, ) ) for b in bytes_ )
   return payload

def buildEthernetPacket( size, initialData=b'',
                         dataType='incrementing', dataValue=0, extraHeaderSize=0 ):
   minSize = ETH_HLEN + len( initialData ) + ETH_FCS_LEN + extraHeaderSize
   if size < minSize:
      raise ValueError( "size too small: must be at least %d" % minSize )

   dataLen = size - minSize
   bytes_ = buildPayload( dataLen, dataType=dataType, dataValue=dataValue )
   return initialData + bytes_

def buildArpPacket( srcMacAddr, targetIpAddr, dstMacAddr='ff:ff:ff:ff:ff:ff', 
                    srcIpAddr='1.1.1.1', requestOrReply='request' ):
   hardwareAddrSpace = 1
   protocolAddrSpace = ETH_P_IP
   hardwareAddrLen = 6
   protocolAddrLen = 4
   if requestOrReply == 'request':
      opcode = 1
   elif requestOrReply == 'reply':
      opcode = 2
   else:
      opcode = 0
   dmacStr = macAddrToBytes( dstMacAddr )
   smacStr = macAddrToBytes( srcMacAddr )
   arpHeader = struct.pack( '>HHBBH',
                            hardwareAddrSpace,
                            protocolAddrSpace,
                            hardwareAddrLen,
                            protocolAddrLen,
                            opcode )
   ipSrcAddr = struct.pack( '>I', struct.unpack( '>L', 
                            socket.inet_aton( srcIpAddr ) )[ 0 ] )
   ipDstAddr = struct.pack( '>I', struct.unpack( '>L', 
                            socket.inet_aton( targetIpAddr ) )[ 0 ] )

   arpPacket = arpHeader + smacStr + ipSrcAddr + dmacStr + ipDstAddr
   return arpPacket

# OR packed Ipv6 addressses 
def ip6AddrOR( addr1, addr2 ):
   addr1 = struct.unpack( '4I', addr1 )
   addr2 = struct.unpack( '4I', addr2 )
   ret = list( map( lambda x, y: x | y, addr1, addr2 ) )
   t = b''.join( struct.pack( 'I', x ) for x in ret )
   return t

# AND packed Ipv6 addressses 
def ip6AddrAND( addr1, addr2 ):
   addr1 = struct.unpack( '4I', addr1 )
   addr2 = struct.unpack( '4I', addr2 )
   return b''.join( struct.pack( 'I', x & y ) for x, y in zip( addr1, addr2 ) )

# returns NS multicast IP address (packed bin)
def getNsMcastAddr( addr ):
   # Multicast addr format ff02::1:ff00:0/104
   r = ip6AddrAND( addr, socket.inet_pton( socket.AF_INET6, '::ff:ffff' ) )
   r = ip6AddrOR( socket.inet_pton( socket.AF_INET6, 'ff02::1:ff00:0' ), r )
   return r

# returns NS multicast MAC address (string)
def getNsMcastMac( addr ):
   addr = struct.unpack( '16B', addr )[ -4: ]
   mac = '33:33:'       # Multicast MAC
   mac += ':'.join( '%.2x' %x for x in addr )
   return mac

# ICMPv6 pseudo Hdr for checksum calculation
def buildIcmp6PseudoHdr( targetIpAddr, srcIpAddr, upperLayerLen, extHdr ):
   extHdr = struct.pack( ">I", extHdr )
   upperLayerLen = struct.pack( ">I", upperLayerLen )
   # pylint: disable-next=superfluous-parens
   return ( srcIpAddr + targetIpAddr + upperLayerLen + extHdr )


def buildNdpPacket( ndpType, srcMacAddr, dstMacAddr, targetIpAddr, srcIpAddr ):

   ipSrcAddr = socket.inet_pton( socket.AF_INET6, srcIpAddr )
   ipDstAddr = socket.inet_pton( socket.AF_INET6, targetIpAddr )

   # pylint: disable-next=consider-using-in
   if ndpType == "nbr-solicit" or ndpType == "rtr-solicit":
      if ndpType == "nbr-solicit":
         typeData = ND_TYPE_NBR_SOLICIT # Type code for Nbr Solicit
         # The address of the target neighbor
         reqIpDstAddr = ipDstAddr
      else:
         typeData = ND_TYPE_RTR_SOLICIT # Type code for Rtr Solicit
         reqIpDstAddr = b""

      code = 0
      chkSum = 0        # Initially setting to 0
      reserved = 0
      hdr = struct.pack( ">BBHI", typeData, code, chkSum, reserved )

      # As per RFC4861 section 4.3: Packets with an unspecified Ip6 addr
      # cannot provide a src link layer addr option
      # options Field in the solicitation packet
      optType = ND_OPTION_SRC_LL_ADDR
      optLen = 1        # 8 bytes : size of TLD
      optData = macAddrToBytes( srcMacAddr )
      optHdr = struct.pack( ">BB", optType, optLen )
      if srcIpAddr != '::':
         data = hdr + reqIpDstAddr + optHdr + optData
      else:
         data = hdr + reqIpDstAddr
      upperLayerLen = len( data )

      # If dmac is broadcast, get Multicast Addr from the dest-ip, otherwise use
      # the unmodified target IP. Update the target IP of the pkt accordingly.
      dstIp = getNsMcastAddr( ipDstAddr ) if dstMacAddr == 'ff:ff:ff:ff:ff:ff' \
              else ipDstAddr
      targetIpAddr = socket.inet_ntop( socket.AF_INET6, dstIp )

      # Needed for ICMP6 checksum
      icmp6PseudoHdr = buildIcmp6PseudoHdr( dstIp, ipSrcAddr, upperLayerLen,
                                             IP6_EXT_HDR_ICMP6 )
      chkSum = ipChecksum( icmp6PseudoHdr + data )

      # Repack the data stream with calculated checksum
      hdr = struct.pack( ">BBHI", typeData, code, chkSum, reserved )
      if srcIpAddr != '::':
         data = hdr + reqIpDstAddr + optHdr + optData
      else:
         data = hdr + reqIpDstAddr

   elif ndpType == "nbr-advt":
      typeData = ND_TYPE_NBR_ADVT # Type code for Nbr Advt
      code = 0
      chkSum = 0
      flag = 0xE000     # 1-Router, 1-Solicited reply, 1-Override
      reserved = 0
      hdr = struct.pack( ">BBHHH", typeData, code, chkSum, flag, reserved )

      # options Field in the Advt packet
      optType = ND_OPTION_DST_LL_ADDR 
      optLen = 1
      optHdr = struct.pack( ">BB", optType, optLen )
      optData = macAddrToBytes( srcMacAddr )
      data = hdr + ipSrcAddr + optHdr + optData
      upperLayerLen = len( data )

      # Needed for ICMP6 checksum
      icmp6PseudoHdr = buildIcmp6PseudoHdr( ipDstAddr, ipSrcAddr, upperLayerLen,
                                             IP6_EXT_HDR_ICMP6 )
      chkSum = ipChecksum( icmp6PseudoHdr + data )

      # Repack the data stream with calculated checksum
      hdr = struct.pack( ">BBHHH", typeData, code, chkSum, flag, reserved )
      data = hdr + ipSrcAddr + optHdr + optData

   else:
      # Should not reach here
      raise ValueError( "invalid NDP rquest type: %s" % ndpType )
  
   ip6Hdr = buildIp6Header( srcIpAddr, targetIpAddr, IP6_EXT_HDR_ICMP6,
                           upperLayerLen + IP6_HDR_LEN,
                           hopLimit=255, tos=0x00 )

   return ( ip6Hdr + data ) # pylint: disable=superfluous-parens

def ipChecksum( packet ):
   """Compute checksum for the packet. This algorithm is used by
   both TCP (header+pkt) and IP (header only)."""
   total = 0
   while len( packet ) > 1:
      total += struct.unpack( ">H", packet[ :2 ] )[ 0 ]
      packet = packet[ 2: ]
   if packet:
      total += packet[ 0 ]
   total = ( total & 0xFFFF ) + ( total >> 16 )
   total = ( total & 0xFFFF ) + ( total >> 16 )
   total &= 0xFFFF
   return ( 0xFFFF & ~total ) # pylint: disable=superfluous-parens

def buildIpHeader( ipSrc, ipDst, ipProtocol, ttl, totalLength, tos,
                   ipOptionData=b'', dontFragFlag=False, moreFragsFlag=False,
                   fragOffset=0, ipId=0, ipCkSum=0, ipVer=0, ipHl=0 ):
   ipSrcAddr = struct.unpack( '>L', socket.inet_aton( ipSrc ) )[ 0 ]
   ipDstAddr = struct.unpack( '>L', socket.inet_aton( ipDst ) )[ 0 ]
   assert len( ipOptionData ) % 4 == 0
   if ipVer == 0 and ipHl == 0:
      versionAndIhl = 0x40 | ( ( 20 + len( ipOptionData ) ) // 4 )
   else:
      versionAndIhl = ipVer | ( ipHl + ( len( ipOptionData ) ) // 4 )
   flags = 0
   if dontFragFlag:
      flags |= 0x2
   if moreFragsFlag:
      flags |= 0x1
   flagsAndFragOffset = ( flags << 13 ) | ( fragOffset & 0x1fff )
   ipHeader = struct.pack( '>BBHHHBBHII',
                           versionAndIhl,
                           tos,
                           totalLength,
                           ipId, # Id (should be unique, but 0 should be OK)
                           flagsAndFragOffset, # Flags and fragment offset
                           ttl,
                           ipProtocol,
                           0, # Header checksum
                           ipSrcAddr,
                           ipDstAddr ) + ipOptionData
   if ipCkSum == 0:
      checksum = ipChecksum( ipHeader )
   else:
      checksum = ipCkSum

   ipHeader = ipHeader[ :10 ] + struct.pack( '>H', checksum ) + ipHeader[ 12: ]

   return ipHeader


def buildIp6FragExtension( ipProtocol, moreFragsFlag, fragOffset, ipId ):
   fragBits = ( fragOffset << 3 ) | moreFragsFlag
   fragHdr = struct.pack( ">BBHI", ipProtocol, 0, fragBits, ipId )
   return fragHdr

def buildIp6Header( ipSrc, ipDst, ipProtocol, payloadLen, hopLimit, tos=0,
                    moreFragsFlag=False, fragOffset=0, ipId=0, flowLabel=0 ):
   fragHdr = None
   if ( fragOffset != 0 ) or moreFragsFlag:
      fragHdr = buildIp6FragExtension( ipProtocol, moreFragsFlag,
                                       fragOffset, ipId )
      ipProtocol = 44
   srcAddr = socket.inet_pton( socket.AF_INET6, ipSrc )
   dstAddr = socket.inet_pton( socket.AF_INET6, ipDst )
   hi = ( tos & 0xF0 ) >> 4
   lo = ( tos & 0x0F ) << 4
   flowLabelHi = ( flowLabel & 0xF0000 ) >> 16
   flowLabelLo = flowLabel & 0x0FFFF
   ipHdr = struct.pack( ">BBHHBB", ( 0x60 | ( hi & 0x0F ) ) ,
                        ( ( lo & 0xF0 ) | ( flowLabelHi & 0x0F ) ), flowLabelLo,
                        payloadLen - IP6_HDR_LEN,
                        ipProtocol, hopLimit ) + srcAddr + dstAddr

   if fragHdr:
      ipHdr += fragHdr
   return ipHdr

# The extraHeaderSize parameter is to account for any extra headers such as MPLS or
# GRE that need to be taken into account in the minimum packet size calculation
def buildIpPacket( size, ipSrc, ipDst,
                   ipProtocol=63, # 63 = "any local network"
                   ttl=64, tos=0, ipOptionData=b'',
                   dataType='incrementing', dataValue=0, initialData=b'', version=4,
                   dontFragFlag=False,
                   moreFragsFlag=False,
                   fragOffset=0,
                   ipId=0,
                   extraHeaderSize=0,
                   ipCkSum=0,
                   ipVer=0,
                   ipHl=0,
                   flowLabel=0 ):

   dataLen = size  - ( ETH_HLEN + ETH_FCS_LEN ) - extraHeaderSize
   ipHeader = None
   if version == 4:
      ipHeader = buildIpHeader( ipSrc,
                                ipDst,
                                ipProtocol,
                                ttl,
                                dataLen,
                                tos,
                                ipOptionData,
                                dontFragFlag=dontFragFlag,
                                moreFragsFlag=moreFragsFlag,
                                fragOffset=fragOffset,
                                ipId=ipId,
                                ipCkSum=ipCkSum,
                                ipVer=ipVer,
                                ipHl=ipHl )
   else:
      ipHeader = buildIp6Header( ipSrc,
                                 ipDst,
                                 ipProtocol,
                                 dataLen,
                                 ttl,
                                 tos=tos,
                                 moreFragsFlag=moreFragsFlag,
                                 fragOffset=fragOffset,
                                 ipId=ipId,
                                 flowLabel=flowLabel )

   initialData = ipHeader + initialData
   return buildEthernetPacket( size, initialData=initialData, dataType=dataType,
                               dataValue=dataValue,
                               extraHeaderSize=extraHeaderSize )

def buildIpInIpPacket( size, outerIpSrc, outerIpDst, ipSrc, ipDst,
                       ipProtocol=63, # 63 = "any local network"
                       ttl=64, tos=0, ipOptionData=b'',
                       dataType='incrementing', dataValue=0,
                       outerDontFragFlag=False,
                       outerMoreFragsFlag=False,
                       outerFragOffset=0,
                       outerIpId=0 ):
   outerDataLen = size - ( ETH_HLEN + ETH_FCS_LEN )
   outerIpHeader = buildIpHeader( outerIpSrc,
                                  outerIpDst,
                                  IPPROTO_IPIP,
                                  ttl,
                                  outerDataLen,
                                  tos,
                                  ipOptionData,
                                  dontFragFlag=outerDontFragFlag,
                                  moreFragsFlag=outerMoreFragsFlag,
                                  fragOffset=outerFragOffset,
                                  ipId=outerIpId )

   dataLen = outerDataLen - len( outerIpHeader )
   ipHeader = buildIpHeader( ipSrc,
                             ipDst,
                             ipProtocol,
                             ttl,
                             dataLen,
                             tos )

   return buildEthernetPacket( 
      size, initialData=outerIpHeader + ipHeader,
      dataType=dataType, dataValue=dataValue )

def buildUdpHeader( size, udpSport, udpDport, version=4,
                    ipOptionData=b'', ipFrag=False ):
   if version == 4:
      ipHdrLen = IP_HDR_LEN
   else:
      ipHdrLen = IP6_HDR_LEN
      if ipFrag:
         ipHdrLen += IP6_FRAG_EXTENSION_LEN

   udpLength = size - ( ETH_HLEN + ipHdrLen + len( ipOptionData ) + ETH_FCS_LEN )
   return struct.pack( '>HHHH', udpSport, udpDport, udpLength, 0 )

def buildUdpPacket( size, udpSport, udpDport, ipSrc, ipDst, ipProtocol=IPPROTO_UDP,
                    ttl=64, tos=0, ipOptionData=b'', initialData=b'',
                    dataType='incrementing', dataValue=0, version=4, 
                    dontFragFlag=False,
                    moreFragsFlag=False,
                    fragOffset=0,
                    ipId=0,
                    flowLabel=0 ):
   ipFrag = ( ( fragOffset != 0 ) or moreFragsFlag )
   udpHeader = buildUdpHeader( size, udpSport, udpDport, version,
                               ipOptionData, ipFrag )
   initialData = udpHeader + initialData
   return buildIpPacket( size, ipSrc, ipDst, ipProtocol, ttl, tos, ipOptionData,
                         dataType=dataType, dataValue=dataValue,
                         initialData=initialData, version=version, 
                         dontFragFlag=dontFragFlag,
                         moreFragsFlag=moreFragsFlag,
                         fragOffset=fragOffset,
                         ipId=ipId,
                         flowLabel=flowLabel )

def buildTcpHeader( sPort=TCP_SPORT_DEFAULT, dPort=TCP_DPORT_DEFAULT, seqNum=1,
                    ackNum=0, tcpFlags='', tcpMss=0, windowSize=1024, urgPointer=0 ):
   checksum = 0
   tcpHdrLen = TCP_HDR_LEN

   flagUrg = strContains( tcpFlags, "U" )
   flagAck = strContains( tcpFlags, "A" )
   flagPsh = strContains( tcpFlags, "P" )
   flagRst = strContains( tcpFlags, "R" )
   flagSyn = strContains( tcpFlags, "S" )
   flagFin = strContains( tcpFlags, "F" )

   if flagAck:
      ackNo = ackNum
   else:
      ackNo = 0
   if flagUrg:
      urgPtr = urgPointer
   else:
      urgPtr = 0
   flags = ( int( flagUrg ) << 5 ) | \
           ( int( flagAck ) << 4 ) | \
           ( int( flagPsh ) << 3 ) | \
           ( int( flagRst ) << 2 ) | \
           ( int( flagSyn ) << 1 ) | \
           ( int( flagFin ) )

   if tcpMss:
      if tcpMss < 0 or tcpMss > 65535:
         raise ValueError(
            "tcp-mss value out of range: must be between 0 and 65535" )
      mssOption = 0x02040000 | tcpMss
      tcpHdrLen += TCP_MSS_OPTION_LEN
      dataOffset = ( tcpHdrLen // 4 ) << 4
      header = struct.pack( '>HHIIBBHHHI', sPort, dPort, seqNum, ackNo,
                             dataOffset, flags, windowSize, checksum, urgPtr,
                             mssOption )
   else:
      dataOffset = ( tcpHdrLen // 4 ) << 4
      header = struct.pack( '>HHIIBBHHH', sPort, dPort, seqNum, ackNo,
                            dataOffset, flags, windowSize, checksum, urgPtr )
   return header

def buildTcpPacket( size, ipSrc, ipDst, ipVersion=4, tos=0, ipId=0,
                    dontFragFlag=False, moreFragsFlag=False, fragOffset=0, ttl=64,
                    ipOptionData=b'', tcpSPort=TCP_SPORT_DEFAULT,
                    tcpDPort=TCP_DPORT_DEFAULT, ackNum=0, tcpFlags='', tcpMss=0,
                    initialData=b'', seqNum=1 ):

   tcpHeader = buildTcpHeader( sPort=tcpSPort, dPort=tcpDPort, ackNum=ackNum,
                               tcpFlags=tcpFlags, tcpMss=tcpMss, seqNum=seqNum )
   if ( ipVersion == 4 ): # pylint: disable=superfluous-parens
      ipSrcAddr = struct.unpack( '>L',
                                 socket.inet_pton( socket.AF_INET, ipSrc ) )[ 0 ]
      ipDstAddr = struct.unpack( '>L',
                                 socket.inet_pton( socket.AF_INET, ipDst ) )[ 0 ]
      pseudoHeader = struct.pack( '>IIBBH', ipSrcAddr, ipDstAddr, 0,
                                  IPPROTO_TCP,
                                  len( tcpHeader ) + len( initialData ) )
   elif ( ipVersion == 6 ): # pylint: disable=superfluous-parens
      ipSrcAddr = struct.unpack( '>QQ',
                                 socket.inet_pton( socket.AF_INET6,  ipSrc ) )[ 0 ]
      ipDstAddr = struct.unpack( '>QQ',
                                 socket.inet_pton( socket.AF_INET6,  ipDst ) )[ 0 ]
      pseudoHeader = struct.pack( '>QQBBH', ipSrcAddr, ipDstAddr, 0,
                                  IPPROTO_TCP,
                                  len( tcpHeader ) + len( initialData ) )
   else:
      assert ( False ), "ipVersion = %r. IP version must be 4 or 6\n"\
            % ipVersion

   headerForChecksum = pseudoHeader + tcpHeader + initialData
   tcpChecksum = ipChecksum ( headerForChecksum )

   tcpHeader = tcpHeader[ :16 ] + struct.pack( '>H', tcpChecksum ) + tcpHeader[ 18: ]
   initialData = tcpHeader + initialData
   return buildIpPacket( size, ipSrc, ipDst, ipProtocol=IPPROTO_TCP, ttl=ttl,
                         tos=tos, ipOptionData=ipOptionData,
                         dataType='incrementing', dataValue=0,
                         initialData=initialData,
                         version=ipVersion, dontFragFlag=dontFragFlag,
                         moreFragsFlag=moreFragsFlag, fragOffset=fragOffset,
                         ipId=ipId )

def buildIcmpHeader( icmpType, icmpCode ):
   icmpHeader = struct.pack( '>BBHL', icmpType, icmpCode, 0, 0 )
   checksum = ipChecksum( icmpHeader )
   icmpHeader = struct.pack( '>BBHL', icmpType, icmpCode, checksum, 0 )
   return icmpHeader

def buildIcmpPacket( ipSrc, ipDst, ipProtocol=IPPROTO_ICMP,
                     icmpType=8, icmpCode=0, ttl=64, tos=0, ipOptionData=b'',
                     version=4, dontFragFlag=False, moreFragsFlag=False,
                     fragOffset=0, ipId=0, flowLabel=0 ):
   # We don't support data payload here. Because ICMP checksum
   # includes the entire packet, not just the header. The payload is
   # not even generated yet when we build the ICMP header. The payload
   # generation code will need to be restructured to support this.
   size = ETH_HLEN + len( ipOptionData ) + ICMP_HDR_LEN + \
          ETH_FCS_LEN
   if version == 6:
      size = size + IP6_HDR_LEN
   elif version == 4:
      size = size + IP_HDR_LEN
   else:
      assert ( False ), f'Unhandled IP version={version}'
   icmpHeader = buildIcmpHeader( icmpType, icmpCode )
   return buildIpPacket( size, ipSrc, ipDst, ipProtocol, ttl, tos,
                         ipOptionData, dataType='incrementing', dataValue=0,
                         initialData=icmpHeader, version=version,
                         dontFragFlag=False, moreFragsFlag=False,
                         fragOffset=fragOffset, ipId=ipId, flowLabel=flowLabel )

def buildVxlanHeader( flags=0x08, vni=None ):
   hdrPart1 = struct.pack( '>BBH' , flags, 0, 0 )
   # 5 bytes - 4 of VNI and 1 of Reserved
   temp1 = struct.pack( '>IB' , vni, 0 ) 
   # Truncate first byte and discard it
   ( _discard, temp2 ) = struct.unpack( '>BI', temp1 ) 
   hdrPart2 = struct.pack( '>I', temp2 ) 
   return hdrPart1 + hdrPart2

def buildGreHeader( greVersion, greProtocol,
                    greChecksum=None, greKey=None, greSequence=None ):
   flags = ( 1 if greChecksum else 0 ) << 7
   flags += ( 1 if greKey else 0 ) << 5
   flags += ( 1 if greSequence else 0 ) << 4
   version = greVersion & 0x0FFF
   protocol = greProtocol & 0xFFFF
   header = struct.pack( '>BBH', flags, version, protocol )
   if greChecksum:
      header += struct.pack( '>HH', greChecksum, 0 )
   if greKey:
      header += struct.pack( '>I', greKey )
   if greSequence:
      header += struct.pack( '>I', greSequence )
   
   return header

def buildDzGreHeader( greVersion, greKey, dzGreVersion, switchId, portId, policyId,
                      tsSeconds, tsNanoseconds ):
   greChecksum = 0
   greKeyPresent = 1
   greSequence = 0

   flags = greChecksum << 7
   flags += greKeyPresent << 5
   flags += greSequence << 4

   version = greVersion & 0x0FFF

   header = struct.pack( '>BBH', flags, version, ARISTA_ETHER_TYPE )
   header += struct.pack( '>HH', greKey, 0 )

   subType = 9 if tsSeconds or tsNanoseconds else 7
   header += struct.pack( '>HH', subType, dzGreVersion )

   switchId = 0 if not switchId else switchId
   portId = 0 if not portId else portId
   policyId = 0 if not policyId else policyId

   header += struct.pack( '>HH', switchId, portId )
   header += struct.pack( '>HH', policyId, 0 )

   if tsSeconds:
      header += struct.pack( '>I', tsSeconds )

   if tsNanoseconds:
      header += struct.pack( '>I', tsNanoseconds )

   return header

def buildRoceV2Packet(
      size, udpSport, udpDport, ipSrc, ipDst,
      bthOpcode, bthQueuePair,
      bthFlags=0, bthPKey=0, bthAckReq=0, bthPsn=0,
      bthReserved8b=0, bthReserved7b=0,
      aethSyndrome=None, aethMsn=None,
      rethWriteLength=None, rethRKey=None, rethVirtualAddr=None,
      ttl=64, tos=0, ipOptionData=b'',
      dataType='incrementing', dataValue=0, version=4,
      dontFragFlag=False,
      moreFragsFlag=False,
      fragOffset=0,
      ipId=0,
      flowLabel=0 ):
   additionalIbHdr = None
   if aethSyndrome is not None or aethMsn is not None:
      additionalIbHdr = buildRoceAeth( aethSyndrome, aethMsn )
   if( rethWriteLength or rethRKey or
        rethVirtualAddr ):
      assert additionalIbHdr is None
      additionalIbHdr = buildRoceReth( rethWriteLength,
                                       rethRKey,
                                       rethVirtualAddr )
   bth = buildRoceBth(
      bthOpcode, bthFlags, bthPKey,
      bthQueuePair, bthAckReq, bthPsn, bthReserved8b,
      bthReserved7b, additionalIbHdr )
   ipHdrSize = IP6_HDR_LEN if version == 6 else IP_HDR_LEN
   additionalHdrLen = len( additionalIbHdr ) if additionalIbHdr else 0
   minUdpPktSize = ETH_HLEN + ipHdrSize + \
                   len( ipOptionData ) + UDP_HDR_LEN + \
                   len( bth ) + additionalHdrLen + ETH_FCS_LEN
   pktSize = max( size, minUdpPktSize )
   if udpDport is None:
      dport = UDP_DPORT_ROCE
   else:
      dport = udpDport

   if udpSport is None:
      sport = UDP_SPORT_DEFAULT
   else:
      sport = udpSport

   ipProtocol = IPPROTO_UDP
   data = buildUdpPacket(
      pktSize, sport, dport,
      ipSrc, ipDst,
      ipProtocol=ipProtocol, ttl=ttl, tos=tos,
      ipOptionData=ipOptionData, dataType=dataType,
      dataValue=dataValue, initialData=bth,
      version=version,
      dontFragFlag=dontFragFlag,
      moreFragsFlag=moreFragsFlag,
      fragOffset=fragOffset,
      ipId=ipId,
      flowLabel=flowLabel )
   return data

def buildRoceBth( opcode, flags, pkey, queuePair,
                  ackRequired, psn, bthReserved8b, bthReserved7b, additionalIbHdrs ):
   if opcode is None:
      opcode = 17 # Acknowledge
   if flags is None:
      flags = 0
   if pkey is None:
      pkey = 0xffff
   if queuePair is None:
      queuePair = 1000
   if ackRequired is None:
      ackRequired = 0
   if psn is None:
      psn = 0
   if bthReserved8b is None:
      bthReserved8b = 0
   if bthReserved7b is None:
      bthReserved7b = 0


   hdr = struct.pack( ">BBHII",
                      opcode,
                      flags,
                      pkey,
                      ( ( bthReserved8b & 0xff ) << 24 ) | ( queuePair & 0xffffff ),
                      ( ( ackRequired << 31 ) | ( ( bthReserved7b & 0x7f ) << 24 ) |
                        ( psn & 0xffffff ) ) )
   if additionalIbHdrs:
      hdr += additionalIbHdrs
   return hdr

def buildRoceAeth( syndrome, msn ):
   if syndrome is None:
      syndrome = 0
   if msn is None:
      msn = 0

   syndrome = syndrome & 0xff
   msn = msn & 0xffffff
   return struct.pack( ">I",
                       syndrome << 24 | msn )

def buildRoceReth( writeLength, remoteKey, virtualAddress ):
   if writeLength is None:
      writeLength = 0
   if remoteKey is None:
      remoteKey = 0
   if virtualAddress is None:
      virtualAddress = 0

   writeLength = writeLength & 0xffffffff
   remoteKey = remoteKey & 0xffffffff
   virtualAddress = virtualAddress & 0xffffffffffffffff

   return struct.pack( '!QLL', virtualAddress, remoteKey, writeLength )

def buildControlWord( data, raw=False ):
   # if in raw format, data is 4-byte hex string,
   # else, it is ( pwType, flags, frg, seqno ) tuple
   if raw:
      assert len( data ) == 8
      assert isinstance( data, str )
      bytes_ = [ int( data[ i:i+2 ], base=16 ) for i in range( 0, len( data ), 2 ) ]
      return bytes( bytes_ )
   else:
      assert len( data ) == 4
      assert isinstance( data, tuple )
      ( pwType, flags, frg, seqno ) = data
      pwTypeFlags = ( ( pwType & 0xF ) << 4 ) + ( flags & 0xF )
      frgLength = ( frg & 0x3 ) << 6
      return struct.pack( '>BBH', pwTypeFlags, frgLength, seqno )

def insertIsl( orig, f1=None, f2=None, sglort=None, dglort=None ):
   ''' Pack the f1, f2, sglort, and dglort into four 16-bit big-endin 
   integers and insert them after the ethernet MAC addresses '''
   args = [ f1, f2, sglort, dglort ]
   for idx in range( len( args ) ): # pylint: disable=consider-using-enumerate
      if args[ idx ] is None:
         args[ idx ] = '0'
      args[ idx ] = int( args[ idx ] )

   isl = struct.pack( '>HHHH', args[ 0 ], args[ 1 ], args[ 2 ], args[ 3 ] )
   new = orig[:ETH_ALEN*2] + isl + orig[ETH_ALEN*2:]
   return new

def insertPtchItmh( origHeader, origData, txrawFap=None, sysDestPort=None,
                    sysSrcPort=None, cpuChannel=None ):
   ''' Prepend a txrawHeader followed by a 2 byte PTCH and optionally a 4 byte ITMH.
   The txrawHeader contains a 64 byte string identifying the source FAP, then
   a 2 byte field for the CPU channel (which is hardcoded).
   Hardcoded PTCH specifies a pp_ssp of 128 and programs the Parse_Program_Control
   (bit 15) to zero if an ITMH header follows the PTCH.
   For the ITMH, hardcoded dropPrecedence=1, traffic class=7,
   and only permitting a dest_sys_port for the FWD_DEST_INFO field'''
   assert isinstance( txrawFap, str ), "txraw must be a string"
   if sysDestPort is not None:
      assert isinstance( sysDestPort, int ), "sysDestPort must be an integer"
   if sysSrcPort is not None:
      assert isinstance( sysSrcPort, int ), "sysDestPort must be an integer"
   dropPrecedence = 1
   trafficClass = 7
   fillLen = 64 - len( txrawFap )
   txrawHeaderDeviceName = bytes( txrawFap, 'utf-8' ) + b'\x00' * fillLen
   rawCpuChannel = 1 << 9
   if ( txrawFap.startswith( 'Fap' )  and cpuChannel is not None ):
      rawCpuChannel = cpuChannel

   txrawHeaderCpuChannel = struct.pack( '<H', rawCpuChannel )

   includeItmh = False
   itmhBit = ( 1 << 15 )
   if sysSrcPort is None:
      includeItmh = True
      itmhBit = 0
      sysSrcPort = 128 # pp_ssp
   ptch = struct.pack( '>H', itmhBit + sysSrcPort )
   if includeItmh:
      # Include an itmh only if we're using 128 as source port.
      # otherwise, we inject without an itmh to let the packet
      # forward in hardware
      if txrawFap.startswith( 'Jericho' ):
         # Gen3 devices use a prefix of 'Jericho'
         # ( see sand-dma/sand-dma-drv/cmic-dma.c )
         itmh = struct.pack( '>L',
                             ( sysDestPort | 0xc0000 ) |
                             ( dropPrecedence << 20 ) |
                             ( trafficClass << 22 ) )
      elif txrawFap.startswith( 'Fap' ):
         # Gen4 devices use a prefix of 'Fap'
         # ( see sand-dma/sand-dma-drv/cmic-dma.c )
         ptch = ptch + struct.pack( '>B', 0 )
         itmh = struct.pack( '>L',
                             ( trafficClass << 1 ) |
                             ( ( sysDestPort | 0xc0000 ) << 9 ) |
                             ( dropPrecedence << 30 ) )
      else:
         assert False, "unknown device type: %s" % txrawFap
      ptch = ptch + itmh
   txrawHeader = txrawHeaderDeviceName + txrawHeaderCpuChannel + ptch
   data = origData
   if txrawFap.startswith( 'Fap' ):
      # sand-dma expects a minimum frame size of (J2_ITMHSZ + PTCHSZ +
      # ETH_ZLEN) for a txraw packet. PTCH is always there. ITMH may or may not
      # be there. We make sure that we add enough padding to make the case
      # without ITMH to work.
      J2_ITMHSZ = 5
      payloadLen = ( ETH_ZLEN + J2_ITMHSZ ) - len( origHeader ) - len( origData )
      if payloadLen > 0:
         data += buildPayload( payloadLen, dataType='constant' )
   return txrawHeader + origHeader, data

cfmPduMap = {
   'ccm' : {
      'opcode' : 1,
      'version' : 0,
      'flags' : None,
      'tlvOffset' : 70,
   },
   'lmm' : {
      'opcode' : 43,
      'version' : 1,
      'flags' : 1,
      'tlvOffset' : 12,
   },
   'lmr' : {
      'opcode' : 42,
      'version' : 1,
      'flags' : 1,
      'tlvOffset' : 12,
   },
   'dmm' : {
      'opcode' : 47,
      'version' : 1,
      'tlvOffset' : 32,
      'flags' : 1,
   },
   'dmr' : {
      'opcode' : 46,
      'version' : 1,
      'tlvOffset' : 32,
      'flags' : 1,
   },
   'slm' : {
      'opcode' : 55,
      'version' : 0,
      'flags' : 0,
      'tlvOffset' : 16,
   },
   'slr' : {
      'opcode' : 54,
      'version' : 0,
      'flags' : 0,
      'tlvOffset' : 16,
   },
}

def buildCfmHeader( options ):
   cfmHeader, cfmPdu = None, None
   cfmPduType = options.cfm_opcode
   mdl = options.cfm_mdl
   version = cfmPduMap[ cfmPduType ][ 'version' ]
   opcode = cfmPduMap[ cfmPduType ][ 'opcode' ]
   flags = cfmPduMap[ cfmPduType ][ 'flags' ]
   tlvOffset = cfmPduMap[ cfmPduType ][ 'tlvOffset' ]

   if cfmPduType == 'ccm':
      assert False, "Not supported yet." 
      # flags = ( rdiBitSet << 7 | txInterval )
   # Common CFM header.
   cfmHeader = struct.pack( '>BBBB', ( mdl << 5 ) | version, opcode, flags,
                            tlvOffset )
   if cfmPduType in [ 'dmm', 'dmr' ]:
      dmm_rx = 0 if cfmPduType == 'dmm' else options.dm_rxF
      dmr_tx = 0 if cfmPduType == 'dmm' else options.dm_txB
      reserved_timestamp_dmr = 0
      endTlv = 0
      cfmPdu = struct.pack( '>QQQQB', options.dm_txF, dmm_rx, dmr_tx,
                            reserved_timestamp_dmr, endTlv )
   elif cfmPduType in [ 'lmm', 'lmr' ]:
      lmm_rxFcf = 0 if cfmPduType == 'lmm' else options.lm_rxFcf
      lmm_txFcb = 0 if cfmPduType == 'lmm' else options.lm_txFcb
      endTlv = 0
      cfmPdu = struct.pack( '>IIIB',
                            options.lm_txFcf,
                            lmm_rxFcf,
                            lmm_txFcb,
                            endTlv )
   elif cfmPduType in [ 'slm', 'slr' ]:
      slm_txFcb = 0 if cfmPduType == 'slm' else options.lm_txFcb
      endTlv = 0
      cfmPdu = struct.pack( '>HHIIIB',
                            options.src_mep,
                            options.rsp_mep,
                            options.slm_test_id,
                            options.lm_txFcf,
                            slm_txFcb,
                            endTlv )
   return cfmHeader + cfmPdu

_et = None # cached libethxmit cdll

# A wrapper around the C++ implementation of socksend, that makes sure it is called
# with the correct number of arguments
def socksend( sock, pkt, stopat=1, burst=1, sleep=0.0, seq=False, progress=False,
              incSmac=0, incDmac=0, incSrcIp=0, incDstIp=0, incSrcPort=0,
              incDstPort=0, adjHdrCSums=False, incBthQp=0 ):
   global _et
   if not _et:
      _et = ctypes.cdll.LoadLibrary( "libethxmit.so" )

   return _et.socksend( sock.fileno(), pkt, len( pkt ), stopat, burst,
                        ctypes.c_float( sleep ), ctypes.c_bool( seq ),
                        ctypes.c_bool( progress ), incSmac, incDmac,
                        incSrcIp, incDstIp, incSrcPort, incDstPort, adjHdrCSums,
                        incBthQp )
