# Copyright (c) 2020 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.
#
# Common tools for reading and parsing information about eusb devices

from __future__ import absolute_import, division, print_function
import re

def parseUdevOutput( udevOutput ):
   '''Create key-value pairs from the output of udevadm info'''

   result = {}
   for line in udevOutput.splitlines():
      array = line.split( '=' )
      if len( array ) != 2:
         continue
      result[ array[ 0 ] ] = array[ 1 ]
   return result

def getRevision( udevInfo ):
   return udevInfo.get( 'ID_REVISION', 'Unknown' )

def getSerial( udevInfo ):
   '''Extract the serial number from udevInfo'''

   if 'ID_SERIAL_SHORT' in udevInfo:
      return udevInfo[ 'ID_SERIAL_SHORT' ]
   elif 'ID_SERIAL' in udevInfo:
      return udevInfo[ 'ID_SERIAL' ]
   else:
      return 'Unknown'

def getModel( udevInfo ):
   '''Extract the model from udevInfo by first extracting diskId and serial number'''

   diskId = getDiskId( udevInfo )
   serial = getSerial( udevInfo )

   # Most likely the model is extracted from diskId
   if serial and diskId:
      regex = r'\w+-(?P<model>.*).%s.*' % serial
      matcher = re.match( regex, diskId )
      if matcher:
         return matcher.group( 'model' )

   # Fall back to other info sources of flash model
   if 'ID_NAME' in udevInfo:
      return udevInfo[ 'ID_NAME' ]
   elif 'ID_MODEL' in udevInfo:
      return udevInfo[ 'ID_MODEL' ]
   elif diskId:
      return diskId
   return 'Unknown'

def getDiskId( udevInfo ):
   '''Extract the disk ID from udevInfo. This contains model and serial number'''

   if 'DEVLINKS' in udevInfo:
      array = udevInfo[ 'DEVLINKS' ].split()
      prefix = '/dev/disk/by-id/'
      for path in array:
         if path.startswith( prefix ):
            return path[ len( prefix ) : ]
   return 'Unknown'
