#!/usr/bin/env arista-python
# Copyright (c) 2010,2015 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import errno
import os
import socket
import struct
import sys
from TacUtils import readNbytes

import FastServ

INTEGER_STRUCT = struct.Struct( 'I' )

# pickle protocol that FastServ servers/clients should use to be compatible
# in a mixed python version environment.
PICKLE_PROTO = 2

def readInteger( fd ):
   """ Given a file descriptor it will read an integer. Will return None if unable
   to read an integer """
   bytesToRead = readNbytes( fd, INTEGER_STRUCT.size )
   if not bytesToRead:
      return None
   return getIntFromIntStruct( bytesToRead )

def readString( fd ):
   """ Given a file descriptor, we will read an integer which is the length of the
   bytes, followed by the bytes itself. Will return None if unable to read the
   bytes. The bytes are assumed to be encoded utf-8 and will be decoded into a
   string """
   totalBytesToRead = readInteger( fd )
   if totalBytesToRead == None: # pylint: disable=singleton-comparison
      return None
   value = readNbytes( fd, totalBytesToRead )
   return None if value is None else value.decode()

def readBytes( fd ):
   """ Given a file descriptor, we will read an integer which is the length of the
   bytes, followed by the bytes itself. Will return None if unable to read the
   bytes """
   totalBytesToRead = readInteger( fd )
   if totalBytesToRead == None: # pylint: disable=singleton-comparison
      return None
   return readNbytes( fd, totalBytesToRead )

def getIntStruct( num ):
   """ Given an integer we return a c struct containing an int"""
   return INTEGER_STRUCT.pack( *( num, ) )

def getIntFromIntStruct( intStruct ):
   """ Given a c struct that contains an integer we unpack it to get a python int """
   return INTEGER_STRUCT.unpack_from( intStruct, 0 )[ 0 ]

def sendAll( fd, string ):
   bytesWritten = 0
   bytesToWrite = len( string )
   while bytesWritten < bytesToWrite:
      try:
         bytesSent = fd.send( string[ bytesWritten : ] )
      except OSError as e:
         if e.errno == errno.EINTR:
            continue
         raise
      if bytesSent == 0:
         raise RuntimeError("socket connection broken")
      bytesWritten += bytesSent
   return bytesWritten

def writeInteger( fd, num ):
   """ Given a file descriptor it will write an integer. Will throw an exception
   if message is not entirely sent"""
   sendAll( fd, getIntStruct( num ) ) 

def writeString( fd, string ):
   """ Given a file descriptor, we will write an integer with the length of the
   string, followed by the string itself. Will throw an exception if message is not
   entirely sent """
   writeBytes( fd, string.encode() )

def writeBytes( fd, data ):
   """ Given a file descriptor, we will write an integer with the length of the
   bytes, followed by the bytes itself. Will throw an exception if message is not
   entirely sent """
   writeInteger( fd, len( data ) )
   sendAll( fd, data )

def recvSock( fd ):
   # pylint: disable-next=c-extension-no-member
   remoteSockFd = FastServ.recvFds( fd.fileno(), 1 )
   if not remoteSockFd:
      return None

   remoteSock = socket.fromfd( remoteSockFd[ 0 ], socket.AF_UNIX,
                               socket.SOCK_STREAM, 0 )
   os.close( remoteSockFd[ 0 ] )
   return remoteSock

def processEnvArgs( fd, setIds=True ):
   """Process the environment and arguments passed from fastclient to
   the server.  Takes a file descriptor, as returned by FastServ.serve
   and reads the environment and arguments that are sent through the
   socket by fastclient.  It sets up the environment, changes uid and
   gid and returns a list of the argv values sent by the client, or []
   if the client specified no arguments."""

   environ = os.environb  # pylint: disable=no-member


   # At this point, we are running in the child process and
   # our stdin, stdout, and stderr are a copy of the child's
   dataLen0 = os.read( fd, 4 )
   dataLen, = struct.unpack( "I", dataLen0 )
   data = os.read( fd, dataLen )

   # Read the passive mount related message. There are 2 read calls below
   # The first 4 bytes sent will be the # length of the message that is going to 
   # follow, with the next read containing the message about whether passive mount
   # is going to be used or not. It should never be used in this test.
   passiveMountDataLen0 = os.read( fd, 4 )
   passiveMountDataLen, = struct.unpack( "I", passiveMountDataLen0 )
   passiveMountData = os.read( fd, passiveMountDataLen )
   assert( passiveMountData[ : 1 ] == b'n' ) # pylint: disable=superfluous-parens

   if data:
      envArgs = data.split( b'\0\0\0' )

      if len( envArgs ) == 2:
         envs, args = envArgs
      else:
         envs = data
         args = None

   if envs:
      envs = envs.split( b'\0' )
      envs = envs[1:]
      for i in range( 0, len( envs ), 2 ):
         var = envs[ i ]
         val = envs[ i + 1 ]
         environ[ var ] = val
      if b'GID' in envs:
         old = os.getegid()
         new = int( environ[ b'GID' ] )
         if old != new and setIds:
            try:
               os.setgid( new )
               os.setegid( new )
            except OSError as e:
               if e.errno == errno.EPERM:
                  print( "Server does not have permissions to change to groupid",
                         new )
                  sys.exit( 255 )
               else:
                  raise
      if b'UID' in envs:
         old = os.geteuid()
         new = int( environ[ b'UID' ] )
         if old != new and setIds:
            try:
               os.setuid( new )
               os.seteuid( new )
            except OSError as e:
               if e.errno == errno.EPERM:
                  print( "Server does not have permissions to change to userid",
                         new )
                  sys.exit( 255 )
               else:
                  raise
      if b'PWD' in envs:
         try:
            os.chdir( environ[ b'PWD' ] )
         except OSError:
            pass

   if args.endswith( b'\0' ):
      args = args[:-1]
   argv = args and args.split( b'\0' ) or [] # pylint: disable=consider-using-ternary
   return [ a.decode() for a in argv ]

def popEnvFromArgv( argv ):
   """Pop any leading values of the form 'A=B' from argv, and push
   them into os.environ[A]=B.  Return the argv with the leading
   environment variables popped off.  This is used to allow the client
   program to specify environment variables on the command line:
      fastclient foo PATH=/sbin:$PATH ifconfig eth0 up"""
   while argv and ('=' in argv[0]):
      arg = argv.pop(0)
      (var, val) = arg.split( '=', 1 )
      os.environ[var] = val
   return argv
