#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, FdlInventoryInitializer
import re

class LedInitializer( FdlInventoryInitializer.Initializer ):
   managedTypeName = "Inventory::Led"

   def initInventoryEntry( self, **kwargs ):
      pass

class LedSystemDirInitializer( FdlInventoryInitializer.Initializer ):
   managedTypeName = "Inventory::LedSystemDir"

   def initInventoryEntry( self, **kwargs ):
      # Initialize default Arista palette
      if not bool( self.invEntity_.palette ):
         self.invEntity_.palette = ( "default", )
     
      # pylint: disable-msg=W0201
      self.defaultEthIntfThemeName_ = "defaultEthIntf"
      
      # Then add 'default' Arista theme if not yet available
      self.addLightSettingTheme( self.defaultEthIntfThemeName_ )

   def addEthIntfLed( self, intfId ):
      # First add 'default' Arista palette
      LedColor = Tac.Type( "Led::Color" )
      LedStatus = Tac.Type( "Led::PlutoStatus" )
      if intfId not in self.invEntity_.palette.ledCapability:
         self.addLedCapability( intfId )
         self.addPaletteLightSetting( intfId, color=LedColor.GREEN, green=True,
                                      plutoStatus=LedStatus.plutoStatusGood )
         self.addPaletteLightSetting( intfId, color=LedColor.YELLOW, yellow=True,
                                      plutoStatus=LedStatus.plutoStatusWarning )
         self.addPaletteLightSetting( intfId, color=LedColor.OFF,
                                      plutoStatus=LedStatus.plutoStatusOff )

      #
      # All LedStates are generated in the FDL.
      #
      # The reason is the SKU Owners should also be able to alter the 'default'
      # theme (beside the Customers), even though it's not recommended to do so
      # (unless Customers demand it out of the box in which case a startup-config
      # file can be provided along with the PSP)
      #
      # Thus, the 'default' theme must be available (pre-created) in the FDL
      # somehow.
      #
      invLed = self.invEntity_.leds.newMember( intfId )
      self.doGenerateEthIntfLedState( invLed )
      invLed.theme = self.defaultEthIntfThemeName_

   def doGenerateEthIntfLedState( self, invLed ):
      LinkStatus = Tac.Type( 'Interface::LinkStatus' )
      EthLinkMode = Tac.Type( 'Interface::EthLinkMode' )
      LedColor = Tac.Type( "Led::Color" )

      self.addThemeSetting( self.addState( "linkDisabled", invLed ).name,
                            self.getPaletteLightSetting( invLed.name,
                                                         LedColor.YELLOW ),
                            self.defaultEthIntfThemeName_ )
      self.addThemeSetting( self.addState( "linkInactive", invLed ).name,
                            self.getPaletteLightSetting( invLed.name,
                                                         LedColor.YELLOW ),
                            self.defaultEthIntfThemeName_ )
      self.addThemeSetting( self.addState( "linkErrdisabled", invLed ).name,
                            self.getPaletteLightSetting( invLed.name,
                                                         LedColor.YELLOW ),
                            self.defaultEthIntfThemeName_ )

      for linkStatusStr in LinkStatus.attributes:
         if linkStatusStr == LinkStatus.linkUp:
            color = LedColor.GREEN
         else:
            color = LedColor.OFF

         for linkModeStr in EthLinkMode.attributes:
            linkModeStrMatch = re.search( r'linkMode(?:Forced)?(.*)',
                                          linkModeStr )
            if linkModeStrMatch:
               linkModeStr = linkModeStrMatch.group( 1 )

            activityList = [ "Activity", "NoActivity" ]

            for activity in activityList:
               stateName = linkStatusStr + linkModeStr + activity
               if stateName not in invLed.state:
                  self.addThemeSetting(
                     self.addState( stateName, invLed ).name,
                     self.getPaletteLightSetting( invLed.name, color ),
                     self.defaultEthIntfThemeName_ )

   def doGenerateFanStatusLedState( self, invLed ):
      # Then add 'default' Arista theme if not yet available
      self.addLightSettingTheme( 'defaultFanStatus' )

      FanStatus = Tac.Type( "Environment::Cooling::FanStatus::HwStatus" )
      for stateName in FanStatus.attributes:
         if stateName not in invLed.state:
            invLed.newState( stateName, True )

   def doGenerateSupplyStatusLedState( self, invLed ):
      # Then add 'default' Arista theme if not yet available
      self.addLightSettingTheme( 'defaultSupplyStatus' )

      PowerSupplyStatus = Tac.Type( "Environment::Power::SupplyStatus::State" )
      for stateName in PowerSupplyStatus.attributes:
         if stateName not in invLed.state:
            invLed.newState( stateName, True )
   
   def doGenerateThermostatLedState( self, invLed ):
      # Then add 'default' Arista theme if not yet available
      self.addLightSettingTheme( 'defaultThermostat' )

      CoolingAlarmLevel = Tac.Type( "Environment::Thermostat::CoolingAlarmLevel" )
      for stateName in CoolingAlarmLevel.attributes:
         if stateName not in invLed.state:
            invLed.newState( stateName, True )

   def doGenerateLocalSupeLedState( self, invLed ):
      # Then add 'default' Arista theme if not yet available
      self.addLightSettingTheme( 'defaultLocalSupe' )

      RedundancyMode = Tac.Type( "Election::RedundancyMode" )
      for stateName in RedundancyMode.attributes:
         if stateName not in invLed.state:
            invLed.newState( stateName, True )

   def doGenerateStatusLedState( self, invLed ):
      # Then add 'default' Arista theme if not yet available
      self.addLightSettingTheme( 'defaultStatus' )

      stateList = [ "unknown", "danger", "warning", "good" ]
      for stateName in stateList:
         if stateName not in invLed.state:
            invLed.newState( stateName, True )

   def doGenerateCardStatusLedState( self, invLed ):
      # Then add 'default' Arista theme if not yet available
      self.addLightSettingTheme( 'defaultCardStatus' )

      CardState = Tac.Type( "Hardware::Card::CardStatus::CardState" )
      for stateName in CardState.attributes:
         if stateName not in invLed.state:
            invLed.newState( stateName, True )

   def doGenerateRemoteSupeLedState( self, invLed ):
      # Then add 'default' Arista theme if not yet available
      self.addLightSettingTheme( 'defaultRemoteSupe' )

      stateList = [ "inative" ]
      for stateName in stateList:
         if stateName not in invLed.state:
            invLed.newState( stateName, True )

   def addLightSettingTheme( self, themeName='default' ):
      if not self.invEntity_.theme or themeName not in self.invEntity_.theme:
         self.invEntity_.newTheme( themeName )

   def addState( self, stateName, invLed, available=False ):
      if invLed:
         if stateName not in invLed.state:
            invLed.newState( stateName, available )
      return invLed.state[ stateName ]

   def filterState( self, name, pattern, available ):
      if name in self.invEntity_.leds:
         for state in self.invEntity_.leds[ name ].state.values():
            if re.match( pattern, state.name ):
               del self.invEntity_.leds[ name ].state[ state.name ]
               self.invEntity_.leds[ name ].newState( state.name, available )

   def addThemeSetting( self, stateName, lightSetting, themeName='default' ):
      themeSettings = self.invEntity_.theme[ themeName ].setting
      if stateName not in themeSettings:
         themeSetting = self.invEntity_.theme[ themeName ].newSetting( stateName )
         themeSetting.value = lightSetting

   def addLedCapability( self, deviceName ):
      return self.invEntity_.palette.newLedCapability( deviceName )

   def removeLedCapability( self, deviceName ):
      del self.invEntity_.palette.ledCapability[ deviceName ]

   def getLedCapability( self, deviceName ):
      return self.invEntity_.palette.ledCapability[ deviceName ]

   def addPaletteLightSetting( self, deviceName, **lightSettingArgs ):
      color = lightSettingArgs.get( "color" )
      assert color is not None, "'color' must be provided with all light settings " \
                                "going on the palette"
      lightSettingArgs.setdefault( "flashRate", 0 )
      if not self.getLedCapability( deviceName ):
         self.addLedCapability( deviceName )
      ledCapability = self.getLedCapability( deviceName )
      ledCapability.lightSetting[ color ] = Tac.Value( "Led::LightSetting",
                                                       **lightSettingArgs )
      return ledCapability.lightSetting[ color ]
   
   def removePaletteLightSetting( self, deviceName, color ):
      if color in self.getLedCapability( deviceName ).lightSetting:
         del self.getLedCapability( deviceName ).lightSetting[ color ]

   def getPaletteLightSetting( self, deviceName, color ):
      return self.getLedCapability( deviceName ).lightSetting[ color ]

def Plugin( context ):
   context.registerInitializer( LedInitializer )
   context.registerInitializer( LedSystemDirInitializer )
