#!/usr/bin/env python3
# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import FdlInventoryInitializer 

class UcdInitializer( FdlInventoryInitializer.Initializer ):
   '''
   This is an inventory initializer for the Ucd9012 agent. For more information
   on inventory initializers, please refer to aid/214. 

   Example usage in fdl:

      import Plugins, FdlInventoryInitializer
      invContext = FdlInventoryInitializer.InitializerFactory()
      Plugins.loadPlugins( "FdlPlugin", plugins=[ "Ucd9012" ], context=invContext )

      pcDir = system.component.newEntity( "Inventory::Ucd9012ControllerDir",
                                          "powerControllerDir" )
      invPc = pcDir.newPowerController( name, addr )
      pcInit = invContext.newInitializer( invPc )
      invPc = pcInit.initInventory( bus=bus, rails=rails, causes=causes,
                                    getNextSensorId=getNextSensorId )

   The following keyword arguments are accepted by this initializer, most of which
   directly represent inventory attributes:
      
      Attribute                Default 
      ---------                --------
      location                 ""
      busTimeout               "busTimeout1000ms"
      modelName                "Ucd90120", "Ucd90320", "Psoc"
      isMainController         False
      inMainPowerDomain        False
      hasWatchdog              False
      reloadCauseSortKey       0
      shutdownEnabled          True
      pollingEnabled           True
      resetGpioPin             None
      autoStartPowerProgram    True
      agentControlsPower       True
      *getNextSensorId         None
      *rails                   N/A (Required)
      externalVoltageRails     None
      causes                   None
      pageFaultCauses          None
      bus                      None
      overrideRegs             None

   *getNextSensorId is a callable that generates unique global sensorIds
   *rails expects a list of tuples where each tuple provides one of the following 
    sets of voltage rail attributes:
       - railId
       - description
       - expectedVoltage
       - maxMarginHighPercent
       - maxMarginLowPercent
       - defaultMarginPercent
       - marginSource (optional)
       - overrideVoutUvLimit (optional. If present, must also include the latter two)
       - overrideVoutOvLimit (optional. See above)
       - overridePowerGoodOff (optional. See above)
   '''
   defaultModelName = None
   defaultBusTimeout = "busTimeout1000ms"

   def _instReloadCause( self, causes ):
      # causeDesc of type: ( reloadCauseKey, id, type [, isRemote] )
      for rId, causeDesc in enumerate( causes ):
         cause = Tac.Value( "Inventory::Cause", reloadCauseKey=causeDesc[ 0 ],
                            id=causeDesc[ 1 ], type=causeDesc[ 2 ] )
         if len( causeDesc ) == 4:
            cause.isRemote = causeDesc[ 3 ]
         self.invEntity_.cause[ rId ] = cause

   def _instPageFaultCause( self, causes ):
      for rId, ( key, idType, faultMask ) in enumerate( causes ):
         cause = Tac.Value( "Inventory::PageFaultCause", reloadCauseKey=key,
                            id=idType, faultMask=faultMask )
         self.invEntity_.pageFaultCause[ rId ] = cause

   def initInventory( self, **kwargs ):
      self.invEntity_.location = kwargs.get( 'location', "" )
      self.invEntity_.busTimeout = kwargs.get( 'busTimeout', self.defaultBusTimeout )
      self.invEntity_.modelName = kwargs.get( 'modelName', self.defaultModelName )
      self.invEntity_.isMainController = kwargs.get( 'isMainController', False )
      self.invEntity_.inMainPowerDomain = kwargs.get( 'inMainPowerDomain', False )
      self.invEntity_.hasWatchdog = kwargs.get( 'hasWatchdog', False )
      self.invEntity_.reloadCauseSortKey = kwargs.get( 'reloadCauseSortKey', 0 )
      self.invEntity_.shutdownEnabled = kwargs.get( 'shutdownEnabled', True )
      self.invEntity_.pollingEnabled = kwargs.get( 'pollingEnabled', True )
      self.invEntity_.resetGpioPin = kwargs.get( 'resetGpioPin', None )
      self.invEntity_.autoStartPowerProgram = kwargs.get( 'autoStartPowerProgram',
                                                          True )
      self.invEntity_.agentControlsPower = kwargs.get( 'agentControlsPower', True )

      getNextSensorIdFunc = kwargs.get( 'getNextSensorId', None )

      rails = kwargs[ 'rails' ]
      for rail in rails:
         # args holds overrideVoutUvLimit, overrideVoutOvLimit, overridePowerGoodOff
         # that are optional to provide
         ( railId, description, expectedVoltage, maxMarginHighPercent,
           maxMarginLowPercent, defaultMargin, args ) = ( rail[ 0 ], rail[ 1 ],
                                                          rail[ 2 ], rail[ 3 ],
                                                          rail[ 4 ], rail[ 5 ],
                                                          rail[ 6: ] )
         invRail = self.invEntity_.newRail( railId )
         invRail.description = description
         invRail.expectedVoltage = expectedVoltage
         invRail.maxMarginHighPercent = maxMarginHighPercent
         invRail.maxMarginLowPercent = maxMarginLowPercent
         invRail.defaultMarginPercent = defaultMargin
         if getNextSensorIdFunc is not None:
            invRail.sensorId = getNextSensorIdFunc()

         if args:
            if len( args ) == 1:
               invRail.marginSource = args[ 0 ]
            elif len( args ) == 2:
               invRail.marginSource = args[ 0 ]
               invRail.tolerance = args[ 1 ]
            elif len( args ) == 3:
               invRail.marginSource = "None"
               invRail.overrideVoutUvLimit = args[ 0 ]
               invRail.overrideVoutOvLimit = args[ 1 ]
               invRail.overridePowerGoodOff = args[ 2 ]
            elif len( args ) == 4:
               invRail.marginSource = args[ 0 ]
               invRail.overrideVoutUvLimit = args[ 1 ]
               invRail.overrideVoutOvLimit = args[ 2 ]
               invRail.overridePowerGoodOff = args[ 3 ]
            else:
               assert False, ( "Invalid number of values to unpack. To see how the "
                               "rail definitions should be provided, please refer "
                               "to //src/Ucd9012/FdlPlugin/Ucd9012.py" )
         else:
            invRail.marginSource = "None"
      
      externalVoltageRails = kwargs.get( 'externalVoltageRails', None )
      if externalVoltageRails:
         for rId, ( railNum, railType ) in enumerate( externalVoltageRails ):
            self.invEntity_.externalVoltageRail[ rId ] = Tac.Value( 
                  "Inventory::ExternalVoltageRail", railNum, railType )

      causes = kwargs.get( 'causes', None )
      if causes:
         self._instReloadCause( causes )

      pageFaultCauses = kwargs.get( 'pageFaultCauses', None )
      if pageFaultCauses:
         self._instPageFaultCause( pageFaultCauses )

      bus = kwargs[ 'bus' ]
      if bus is not None:
         self.invEntity_.upstreamBus = bus
         bus.device.addMember( self.invEntity_ )

      overrideRegs = kwargs.get( 'overrideRegs', [] )
      for addr, value in overrideRegs:
         overrideReg = Tac.Value(
            "Inventory::OverrideReg",
            addr,
            bytearray( value ) )
         self.invEntity_.overrideReg.enq( overrideReg )

      return self.invEntity_

class Ucd9012Initializer( UcdInitializer ):
   managedTypeName = "Inventory::Ucd9012Controller"
   defaultModelName = "Ucd90120"

class Ucd9032Initializer( UcdInitializer ):
   managedTypeName = "Inventory::Ucd9032Controller"
   defaultModelName = "Ucd90320"

class PsocPowerInitializer( UcdInitializer ):
   managedTypeName = "Inventory::PsocPowerController"
   defaultModelName = "Psoc"

   def _instReloadCause( self, causes ):
      for rId, ( key, idType, origin ) in enumerate( causes ):
         cause = Tac.Value( "Inventory::PsocCause", reloadCauseKey=key,
                            bitNum=idType, origin=origin )
         self.invEntity_.psocCause[ rId ] = cause

class PsocPmbusInitializer( PsocPowerInitializer ):
   managedTypeName = "Inventory::PsocPmbusController"

def Plugin( context ):
   context.registerInitializer( Ucd9012Initializer )
   context.registerInitializer( Ucd9032Initializer )
   context.registerInitializer( PsocPowerInitializer )
   context.registerInitializer( PsocPmbusInitializer )

