#!/usr/bin/env python3

# pylint: disable=consider-using-f-string

# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from enum import IntEnum
import errno
import fcntl
import re
import six
import subprocess
import sys
import tempfile

#--------------------------------------------------------------------------------
# Error codes
#--------------------------------------------------------------------------------

class FlashUtilErrorCode( IntEnum ):
   SUCCESS = 0
   FAIL_GENERIC = 1
   FAIL_UNKNOWN_SECTION = 2
   FAIL_SPI_WP_ACTIVE = 3

#--------------------------------------------------------------------------------
# Utility functions
#--------------------------------------------------------------------------------

def flashUtil():
   return "flashUtil"

def flashrom():
   return "flashrom"

# reading flash can return binary data, so default to text=False
def run( cmd, verboseOut=False, stdoutSetting=subprocess.PIPE, text=False,
         stderrSetting=subprocess.DEVNULL ):
   if verboseOut:
      cmd += [ '-VV' ]
      return subprocess.run( cmd, text=text, stdout=subprocess.PIPE,
                             stderr=subprocess.STDOUT, check=True ).stdout

   return subprocess.run( cmd,
            stdout=stdoutSetting,
            stderr=stderrSetting,
            text=text,
            check=True ).stdout

#--------------------------------------------------------------------------------
# FlashUtil lock
#--------------------------------------------------------------------------------

class FlashUtilLock( object ): # pylint: disable=useless-object-inheritance
   flashUtilLock = None
   printStatus = False

   def __init__( self, printStatus=False ):
      self.printStatus = printStatus

   def __enter__( self ):
      self.flashUtilLock = open( '/var/lock/flashUtil', 'w' )

      try:
         fcntl.flock( self.flashUtilLock.fileno(), fcntl.LOCK_EX | fcntl.LOCK_NB )
      except IOError as e:
         if e.errno != errno.EAGAIN:
            raise

         if self.printStatus:
            print( "Failed to acquire flashUtil exclusive lock. "
                   "We will block until we get it." )
         fcntl.flock( self.flashUtilLock.fileno(), fcntl.LOCK_EX )

      if self.printStatus:
         print( "flashUtil exclusive lock acquired." )

      return self.flashUtilLock

   def __exit__( self, exceptType, exceptValue, exceptTraceback ):
      self.flashUtilLock.close()

      if self.printStatus:
         print( "flashUtil exclusive lock released." )

#--------------------------------------------------------------------------------
# Flashrom helpers
#--------------------------------------------------------------------------------

def flashromDetect():
   try:
      out = run( [ flashrom() ], verboseOut=True )
   except subprocess.CalledProcessError as e:
      # Write debug log on failure
      sys.stderr.flush()
      sys.stderr.buffer.write( e.output + b'\n' )
      raise
   m = re.search( br"Found (.*?) flash chip \"(.*?)\" \((\d+) kB, .*\)", out )
   if not m:
      return ( None, None, 0 )

   # * 1024 -> KB to B
   return ( six.ensure_str( m.group( 1 ) ), six.ensure_str( m.group( 2 ) ),
            int( m.group( 3 ) ) * 1024 )

def getRomToFlash( fileToWrite, start, spiTotalLen ):
   # pylint: disable-next=consider-using-with
   tmpFile = tempfile.NamedTemporaryFile( delete=False )

   # first we make the image patch
   # All regions should be aligned at the 1KB boundary, if that is not the
   # case, we need adjust the block size we used.
   blockSize = 1024
   while blockSize > 1:
      if spiTotalLen % blockSize == 0 and start % blockSize == 0:
         break
      blockSize = blockSize / 2

   run( [ 'dd', 'if=/dev/zero', 'of=%s' % tmpFile.name,
          'bs=%d' % blockSize, 'count=%d' % ( spiTotalLen / blockSize ) ] )
   run( [ 'dd', 'if=' + fileToWrite, 'of=' + tmpFile.name,
          'bs=%d' % blockSize, 'seek=%d' % ( start / blockSize ),
          'conv=notrunc' ] )

   return tmpFile.name

def parseLayoutFile( layoutPath ):
   if not layoutPath:
      return {}

   try:
      with open( layoutPath ) as f:
         layoutRaw = f.read()
   except IOError:
      print( 'Error: failed to read %s' % layoutPath )
      return None

   layoutDict = {}

   for line in layoutRaw.splitlines():
      if line.startswith( '#' ) or not line:
         continue
      m = re.match( r'([0-9A-Fa-f]+):([0-9A-Fa-f]+) (\w+)', line )
      if not m:
         print( 'Error: malformed layout line "%s"' % line )
         return None

      start = int( m.group( 1 ), 16 )
      end = int( m.group( 2 ), 16 )
      size = end - start + 1
      name = m.group( 3 )
      if size < 0:
         print( 'Error: entry "%s" has negative size' % name )
         return None
      layoutDict[ name ] = {
         'start': start,
         'length': size,
      }

   return layoutDict
