# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from datetime import datetime
import decimal

from FlowTrackerConst import (
   activeInterval,
   congestionInterval,
   constants,
   dampingTimeout,
   flowTableSize,
   hoHeuristicsConstants,
   inactiveTimeout,
   ipfixMtu,
   ipfixPort,
   sampleRate,
   spaceConst,
   templateInterval,
   reservedGroupNames,
   accessListDefault,
   mirrorConfigDefault,
   initialCopy,
   mirrorIntervalFixed,
   mirrorIntervalRandom,
)
from TypeFuture import TacLazyType
import Tac
from Toggles.FlowTrackerToggleLib import (
   toggleSftIpoVxlanEnabled,
   toggleSftGreExportEnabled,
   toggleSftBgpExportCliEnabled,
)

FtrTypeEnum = TacLazyType( 'FlowTracking::FlowTrackingType' )
ftrTypeSampled = FtrTypeEnum.sampled
ftrTypeHardware = FtrTypeEnum.hardware
ftrTypeInbandTelemetry = FtrTypeEnum.inbandTelemetry
ftrTypeMirrorOnDrop = FtrTypeEnum.mirrorOnDrop
ftrTypeCpuQueue = FtrTypeEnum.cpuQueue
ftrTypeDfw = FtrTypeEnum.dfw
AddressFamily = TacLazyType( "Arnet::AddressFamily" )
exporterInactiveReasonEnum = Tac.Type( "FlowTracking::ExporterInactiveReasonEnum" )
ExpFormat = TacLazyType( 'FlowTracking::ExportFormat' )
ftrConfig = TacLazyType( 'FlowTracking::Config' )
PacketBufferRate = TacLazyType( 'FlowTracking::PacketBufferRate' )
PacketBufferBurst = TacLazyType( 'FlowTracking::PacketBufferBurst' )
SampleProbability = Tac.Type( 'FlowTracking::SampleProbability' )

ftrTypes = [
   ftrTypeSampled,
   ftrTypeHardware,
   ftrTypeInbandTelemetry,
   ftrTypeMirrorOnDrop,
   ftrTypeCpuQueue,
   ftrTypeDfw,
]

ftrTypeKwStr = {
   ftrTypeSampled : 'sampled',
   ftrTypeHardware : 'hardware',
   ftrTypeInbandTelemetry : 'telemetry inband',
   ftrTypeMirrorOnDrop : 'mirror-on-drop',
   ftrTypeCpuQueue : 'cpu-queue',
   ftrTypeDfw : 'firewall distributed',
}

egressStr = 'egress'

ftrTypeByName = { value : key for key, value in ftrTypeKwStr.items() }

ftrTypeShowStr = {
   ftrTypeSampled : 'Sampled',
   ftrTypeHardware : 'Hardware',
   ftrTypeInbandTelemetry : 'Inband telemetry',
   ftrTypeMirrorOnDrop : 'Mirror on drop',
   ftrTypeCpuQueue : 'CPU queue',
   ftrTypeDfw : 'Distributed Firewall',
}

encapTypeEnum = TacLazyType( 'FlowTracking::EncapType' )
encapNone = encapTypeEnum.encapNone
encapIpv4 = encapTypeEnum.encapIpv4
encapIpv6 = encapTypeEnum.encapIpv6
encapVxlan = encapTypeEnum.encapVxlan
encapMpls = encapTypeEnum.encapMpls
encapGre = encapTypeEnum.encapGre

encapBase = {                                                                        
   'ipv4' : 'IPv4 flows',                                                            
   'ipv6' : 'IPv6 flows',                                                            
}
encapTypeKwStr = {
   encapIpv4 : 'ipv4',
   encapIpv6 : 'ipv6',
   encapVxlan : 'vxlan',
   encapMpls : 'mpls',
   encapGre : 'gre',
}
encapTypeShowStr = {
   encapIpv4 : 'IPv4',
   encapIpv6 : 'IPv6',
   encapVxlan : 'VXLAN',
   encapMpls : 'MPLS',
   encapGre : 'GRE',
}

filterTypeEnum = TacLazyType( 'FlowTracking::FilterType' )
filterRpf = filterTypeEnum.filterRpf
filterTypeKwStr = {
   filterRpf : 'rpf',
}
filterTypeShowStr = {
   filterRpf : 'IPv4 uRPF, IPv6 uRPF',
}

flowTrackerSampleMode = TacLazyType( 'FlowTracking::FlowTrackerSampleMode' )
sampleModeAll = flowTrackerSampleMode.sampleModeAll
sampleModeFiltered = flowTrackerSampleMode.sampleModeFiltered

exportFormatEnum = TacLazyType( 'FlowTracking::ExportFormat' )
formatDefault = exportFormatEnum.formatDefault
formatNone = exportFormatEnum.formatNone
formatDropReport = exportFormatEnum.formatDropReport
formatPcap = exportFormatEnum.formatPcap
formatSflow = exportFormatEnum.formatSflow
formatIpfix = exportFormatEnum.formatIpfix

exportFormatShowStr = {
   formatDefault : '',
   formatNone : 'None',
   formatDropReport : 'Drop Report',
   formatPcap : 'PCAP',
   formatSflow : 'Sflow',
   formatIpfix : 'IPFIX',
}

fgEncaps = [ encapIpv4, encapIpv6, encapVxlan ]

ftrDefaultEncaps = [ encapIpv4, encapIpv6 ]

seqnoOffsetType = 'FlowTracking::StaticGroupSeqnoOffset'
GroupSeqnoOffsetEnum = TacLazyType( seqnoOffsetType )
reservedFgSeqnoBase = constants.reservedFgSeqnoBase
reservedGroupToSeqnoMap = {
   reservedGroupNames.groupHwIpv4 : \
         constants.reservedGroupSeqno( GroupSeqnoOffsetEnum.offsetIpv4 ),
   reservedGroupNames.groupHwIpv6 : \
         constants.reservedGroupSeqno( GroupSeqnoOffsetEnum.offsetIpv6 ),
   reservedGroupNames.groupHwVxlanIpv4 : \
         constants.reservedGroupSeqno( GroupSeqnoOffsetEnum.offsetVxlanIpv4 ),
   reservedGroupNames.groupHwVxlanIpv6 : \
         constants.reservedGroupSeqno( GroupSeqnoOffsetEnum.offsetVxlanIpv6 ),
}

packetBufferStatusEnum = TacLazyType( 'FlowTracking::PacketBuffer::Status' )
packetBufferStatusShowStr = {
   packetBufferStatusEnum.disabled : 'disabled',
   packetBufferStatusEnum.enabled : 'enabled',
   packetBufferStatusEnum.insufficientBufferSpace :
      'insufficient hardware resources',
}

# hwFgSeqno is computed based on the equation
#       U32 hwFgSeqno = configSeqno * hwFgSeqnoMultiple + offset;
# So for hwFgSeqno to be less than reservedFgSeqnoBase, we need
#       configSeqno < ( reservedFgSeqnoBase - offsetMax ) / hwFgSeqnoMultiple
groupSeqnoOffset = Tac.Value( "FlowTracking::GroupSeqnoOffset" )
configSeqnoMax = ( ( ( reservedFgSeqnoBase - groupSeqnoOffset.maxOffset ) //
                     constants.hwFgSeqnoMultiple ) - 1 )

collectorInactiveReason = TacLazyType( 'FlowTracking::CollectorInactiveReason' )
exporterInactiveReason = TacLazyType( 'FlowTracking::ExporterInactiveReason' )
trackerInactiveReason =  TacLazyType( 'FlowTracking::TrackerInactiveReason' )
hoState = TacLazyType( 'FlowTracking::HoState' )

ftrCapabilitiesPathPrefix = 'hardware/flowtracking/capabilities/'
ftrCapabilitiesType = 'HwFlowTracking::Capabilities'

mplsHwCapabilitiesPath = 'routing/hardware/mpls/capability'
bridgingHwCapabilitiesPath = 'bridging/hwcapabilities'

ftrConfigType = 'FlowTracking::Config'
ftrMirrorOnDropConfigType = "FlowTracking::MirrorOnDropConfig"

ftrStatusPathPrefix = 'flowtracking/status/'
ftrStatusType = 'FlowTracking::Status'

ftrConfigReqPathPrefix = 'flowtracking/configReq/'
ftrConfigReqType = 'FlowTracking::ConfigReq'

hwFtrConfigPathPrefix = 'hardware/flowtracking/config/'
hwFtrConfigType = 'HwFlowTracking::Config'

hwFtrStatusPathPrefix = 'hardware/flowtracking/status/'
hwFtrStatusType = 'HwFlowTracking::Status'

swExportCmd = 'record format ipfix standard timestamps counters'
hwOffloadCmd = 'hardware offload'
hwOffloadIpv4Cmd = hwOffloadCmd + ' ipv4'
hwOffloadIpv6Cmd = hwOffloadCmd + ' ipv6'
hwOffloadIpBothCmd = hwOffloadCmd + ' ipv4 ipv6'
hwOffloadMinSamplesCmd = hwOffloadCmd + ' threshold minimum %d samples'

rewriteDscpCmd = 'record rewrite dscp'

encapFilterModeCmd = 'encapsulation filter'
encapFilterRpfCmd = 'ipv4 ipv6 verify unicast source'

def inactiveTimeoutSupportedForType( ftrType ):
   return ftrType != ftrTypeCpuQueue

def activeIntervalSupportedForType( ftrType ):
   return ftrType != ftrTypeCpuQueue

def encapMplsSupportedForType( ftrType, mplsHwCaps, ftrCaps ):
   if not mplsHwCaps or not mplsHwCaps.mplsSupported:
      return False
   return ( ( ftrType == ftrTypeSampled ) or
            ( ftrType == ftrTypeMirrorOnDrop and ftrCaps.modMplsEncapSupported ) )

def encapVxlanSupportedForType( ftrType, bridgingHwCapabilities ):
   if not bridgingHwCapabilities:
      return False
   isftrTypeSampled = ( ftrType == ftrTypeSampled )
   return ( bridgingHwCapabilities.vxlanSupported and
            bridgingHwCapabilities.vxlanRoutingSupported and
            toggleSftIpoVxlanEnabled() and isftrTypeSampled )

def encapGreSupportedForType( ftrType ):
   isftrTypeSampled = ( ftrType == ftrTypeSampled )
   return ( isftrTypeSampled and toggleSftGreExportEnabled() )

def getEncapOptions( ftrType, ftrCaps, mplsHwCaps, bridgingHwCaps ):
   options = list( encapBase.keys() )
   if encapMplsSupportedForType( ftrType, mplsHwCaps, ftrCaps ):
      options.append( 'mpls' )
   if encapVxlanSupportedForType( ftrType, bridgingHwCaps ):
      options.append( 'vxlan' )
   if encapGreSupportedForType( ftrType ):
      options.append( 'gre' )
   return options

def exportMplsSupportedForType( ftrType, mplsHwCaps ):
   return mplsHwCaps.mplsSupported and ( ftrType == ftrTypeSampled )

def flowTableSizeSupportedForType( ftrType ):
   return ftrType == ftrTypeSampled or ftrType == ftrTypeInbandTelemetry

def congestionSupportedForType( ftrSwCaps ):
   return ftrSwCaps.congestionSupported if ftrSwCaps else False

def supportedExportFormatsForType( ftrCapabilities, ftrSwCapabilities ):
   # Each flow tracking type in its software capabilities defines a list of:
   # 1) Supported export formats, and
   # 2) Supported export formats which in addition require hardware-specific support
   # In the hardware capabilities there is another list defining which export formats
   # are supported by the hardware.
   # This method computes the final set of all supported export formats based on the
   # above criteria.
   swSupportedFormats = set( ftrSwCapabilities.exportFormatSupported )
   hwSupportRequired = set( ftrSwCapabilities.exportFormatRequiresHwSupport )
   # All export formats which do not require hardware support
   hwSupportNotRequired = swSupportedFormats.difference( hwSupportRequired )
   # Export formats which required hardware support and are supported by the hardware
   hwSupported = hwSupportRequired.intersection(
      set( ftrCapabilities.exportFormatHwSupported ) )
   supported = hwSupportNotRequired.union( hwSupported )

   return supported

def dscpSupportedForType( ftrCapabilities, ftrSwCapabilities ):
   supportedExportFormats = supportedExportFormatsForType( ftrCapabilities,
                                                           ftrSwCapabilities )
   dscpSupportedFormats = set( ftrSwCapabilities.exportFormatDscpSupported.keys() )
   anyValidFormat = len(
      supportedExportFormats.intersection( dscpSupportedFormats ) ) > 0

   if not anyValidFormat:
      return False
   return True

def templateSupportedForType( ftrCapabilities, ftrSwCapabilities ):
   # BUG849840 - Use FlowTracking::Capabilities instead
   return ftrSwCapabilities.ftrType != ftrTypeCpuQueue

def localIntfSupportedForType( ftrCapabilities, ftrSwCapabilities ):
   # BUG849840 - Use FlowTracking::Capabilities instead
   return ftrSwCapabilities.ftrType != ftrTypeCpuQueue

def collectorSupportedForType( ftrCapabilities, ftrSwCapabilities ):
   # BUG849840 - Use FlowTracking::Capabilities instead
   return ftrSwCapabilities.ftrType != ftrTypeCpuQueue

def defaultExportFormatForType( ftrCapabilities, ftrSwCapabilities ):
   if ( ( defaultFormat := ftrCapabilities.modDefaultExportFormat ) !=
        ExpFormat.formatDefault ):
      return defaultFormat
   return ftrSwCapabilities.defaultExportFormat

def showFlowTracking( config, ftrType, ftrTypeCaps=None, cliSave=True,
                     saveAll=False, saveAllDetail=False, space='',
                     mplsHwCaps=None, bridgingHwCaps=None ):
   detail = saveAll or saveAllDetail

   sampleLimitSupported = False
   defaultSampleLimit = None
   sampleLimitCfgStr = None
   if ftrTypeCaps:
      sampleLimitSupported = ftrTypeCaps.sampleLimitSupported
      defaultSampleLimit = ftrTypeCaps.defaultSampleLimit

   lines = []
   if not cliSave:
      space += spaceConst

   # 'sample limit' was originally only supported for MOD and it was rendered
   # in-between other MOD-specific commands. In order to preserve the ordering,
   # we first compute the sample limit config str (if applicable) and append it
   # in the ftrType handlers.
   if sampleLimitSupported:
      if config.sampleLimit is not None:
         # This means that sampleLimit is not set to nullopt.
         # So, CLI would have set some non default sample rate
         if config.sampleLimit != 0:
            sampleLimitCfgStr = f'{space}sample limit {config.sampleLimit} pps'
         else:
            sampleLimitCfgStr = f'{space}sample limit disabled'
      elif detail:
         if defaultSampleLimit:
            sampleLimitCfgStr = f'{space}sample limit {defaultSampleLimit} pps'
         else:
            # There's either no default (nullopt) or the default is 0 (disabled)
            sampleLimitCfgStr = f'{space}sample limit disabled'

   if ftrType == ftrTypeSampled:
      if toggleSftBgpExportCliEnabled():
         bgpExportStr = 'information element bgp disabled'
         if not config.bgpExport:
            lines.append( f'{space}{bgpExportStr}' )

      encapStr = 'encapsulation'
      if config.encap:
         for encap in sorted( config.encap ):
            encapStr += ' %s' % encapTypeKwStr[ encap ]
         lines.append( f'{space}{encapStr}' )
      elif detail:
         options = getEncapOptions( ftrType, ftrTypeCaps,
                                    mplsHwCaps, bridgingHwCaps )
         for encap in sorted( options ):
            encapStr += ' %s' % encap
         lines.append( f'{space}no {encapStr}' )

      if ( config.sampleRate != sampleRate.rateDefault or detail ):
         lines.append( '%ssample %d' % ( space, config.sampleRate ) )

      hwOffloadStr = hwOffloadCmd
      eitherOffload = False

      if config.hwOffloadIpv4:
         hwOffloadStr += ' ipv4'
         eitherOffload = True
      if config.hwOffloadIpv6:
         hwOffloadStr += ' ipv6'
         eitherOffload = True

      if not eitherOffload and detail:
         lines.append( f'{space}no {hwOffloadIpBothCmd}' )
      elif eitherOffload:
         lines.append( f'{space}{hwOffloadStr}' )

      if config.rewriteDscp:
         lines.append( f'{space}{rewriteDscpCmd}' )
      elif detail:
         lines.append( f'{space}no {rewriteDscpCmd}' )

      if detail or \
         config.minSamplesBeforeHwOffload != hoHeuristicsConstants.minSamplesDefault:
         lines.append( '%s%s' %
            ( space, hwOffloadMinSamplesCmd % config.minSamplesBeforeHwOffload ) )

   if ftrType == ftrTypeMirrorOnDrop:
      defaultEncapCmd = 'encapsulation ipv4 ipv6'
      mplsEncapCmd = 'encapsulation mpls'
      allEncapCmd = 'encapsulation ipv4 ipv6 mpls'
      defaultEncaps = [ encapIpv4, encapIpv6 ]
      allEncaps = [ encapIpv4, encapIpv6, encapMpls ]
      if ( set( config.encap ) == set( defaultEncaps ) ) and detail:
         lines.append( f'{space}{defaultEncapCmd}' )
      elif set( config.encap ) == { encapMpls }:
         lines.append( f'{space}{mplsEncapCmd}' )
      elif set( config.encap ) == set( allEncaps ):
         lines.append( f'{space}{allEncapCmd}' )
      filterConfig = len( config.filter )
      if filterConfig >= 1:
         lines.append( f'{space}{encapFilterModeCmd}' )
         for elem in sorted( config.filter ):
            if elem == filterRpf:
               lines.append( f'{space + spaceConst}{encapFilterRpfCmd}' )

      if sampleLimitCfgStr is not None:
         lines.append( sampleLimitCfgStr )

      if ftrTypeCaps and ftrTypeCaps.trapSuppressionSupported:
         if ftrTypeCaps.modDefaultTrapSuppression is None:
            if config.trapSuppression is False:
               lines.append( '%strap suppression disabled' % ( space ) )
            elif config.trapSuppression is True:
               lines.append( '%strap suppression' % ( space ) )
            elif config.trapSuppression is None and detail:
               lines.append( '%sdefault trap suppression' % ( space ) )
         else:
            if config.trapSuppression is True:
               if not ftrTypeCaps.modDefaultTrapSuppression:
                  lines.append( '%strap suppression' % ( space ) )
               elif detail:
                  lines.append( '%sdefault trap suppression' % ( space ) )
            elif config.trapSuppression is False:
               if ftrTypeCaps.modDefaultTrapSuppression:
                  lines.append( '%strap suppression disabled' % ( space ) )
               elif detail:
                  lines.append( '%sdefault trap suppression' % ( space ) )
                  
            elif config.trapSuppression is None and detail:
               lines.append( '%sdefault trap suppression' % ( space ) )

      if ftrTypeCaps and ftrTypeCaps.packetBufferDropsSupport:
         if ftrTypeCaps.packetBufferShaperSupport:
            if config.packetBufferShaperRate != PacketBufferRate.rateDefault:
               lines.append( f'{space}packet-buffer drop rate '
                             f'{config.packetBufferShaperRate} pps' )
            elif detail:
               lines.append( f'{space}no packet-buffer drop rate' )

            if config.packetBufferShaperBurst != PacketBufferBurst.burstDefault:
               lines.append( f'{space}packet-buffer drop burst '
                             f'{config.packetBufferShaperBurst} packets' )
            elif detail:
               lines.append( f'{space}no packet-buffer drop burst' )

      if ftrTypeCaps and ftrTypeCaps.packetBufferDropsSupport and \
            ftrTypeCaps.packetBufferSamplingProbSupport:
         if config.packetBufferSampleProb != SampleProbability.probabilityDefault:
            lines.append( f'{space}packet-buffer drop sample probability '
                          f'{config.packetBufferSampleProb}' )
         elif detail:
            lines.append( f'{space}no packet-buffer drop sample probability' )
         
         if config.packetBufferSampleProb25 is not None:
            lines.append( f'{space}packet-buffer drop queue usage 0%-25% '
                          'sample probability '
                          f'{config.packetBufferSampleProb25}' )
         elif detail:
            lines.append( f'{space}no packet-buffer drop queue usage 0%-25% '
                          'sample probability' )

         if config.packetBufferSampleProb50 is not None:
            lines.append( f'{space}packet-buffer drop queue usage 25%-50% '
                          'sample probability '
                          f'{config.packetBufferSampleProb50}' )
         elif detail:
            lines.append( f'{space}no packet-buffer drop queue usage 25%-50% '
                          'sample probability' )
         
         if config.packetBufferSampleProb75 is not None:
            lines.append( f'{space}packet-buffer drop queue usage 50%-75% '
                          'sample probability '
                          f'{config.packetBufferSampleProb75}' )
         elif detail:
            lines.append( f'{space}no packet-buffer drop queue usage 50%-75% '
                          'sample probability' )
         
         if config.packetBufferSampleProb100 is not None:
            lines.append( f'{space}packet-buffer drop queue usage 75%-100% '
                          'sample probability '
                          f'{config.packetBufferSampleProb100}' )
         elif detail:
            lines.append( f'{space}no packet-buffer drop queue usage 75%-100% '
                          'sample probability' )
   else:
      if sampleLimitCfgStr is not None:
         lines.append( sampleLimitCfgStr )

   if not cliSave:
      if lines:
         lines = [ 'flow tracking ' + ftrTypeKwStr[ ftrType ] ] + lines

   return lines

def showFlowTrackingTrailer( config, ftrType, cliSave=True,
                           saveAll=False, saveAllDetail=False, space='' ):
   detail = saveAll or saveAllDetail

   lines = []
   if not cliSave:
      space += spaceConst

   if ftrType == ftrTypeHardware:
      if config.swExport:
         lines.append( f"{space}{swExportCmd}" )
      elif detail:
         lines.append( f"{space}no {swExportCmd}" )
   if config.enabled:
      lines.append( "%sno shutdown" % space )
   elif detail:
      lines.append( "%sshutdown" % space )
   return lines

def showFtr( output, trName, ftr, ftrType, mplsHwCaps, ftrTypeCaps=None,
             ftrTypeSwCaps=None, cliSave=True, detail=False, space='' ):
   if not cliSave:
      output.write( f'{space}tracker {trName}\n' )
      space += spaceConst

   for cpuQueueName in ftr.monitoredCpuQueue:
      output.write( f'{space}monitor cpu-queue {cpuQueueName}\n' )

   if flowTableSizeSupportedForType( ftrType ) and (
         ftr.flowTableSize != flowTableSize.sizeDefault or detail ):
      output.write( '%sflow table size %d entries\n' %
            ( space, ftr.flowTableSize ) )

   if inactiveTimeoutSupportedForType( ftrType ):
      if ftr.inactiveTimeout != inactiveTimeout.timeoutDefault or detail:
         output.write( '%srecord export on inactive timeout %d\n' %
               ( space, ftr.inactiveTimeout ) )

   if activeIntervalSupportedForType( ftrType ):
      if ftr.activeInterval != activeInterval.intervalDefault or detail:
         output.write( '%srecord export on interval %d\n'
                  % ( space, ftr.activeInterval ) )

   cmd = 'record export on tcp state change'
   if ftr.tcpStateChangeExport != constants.tcpStateChangeExportDefault:
      output.write( f'{space}{cmd}\n' )
   elif detail and ( ftrTypeCaps is not None ) and \
        ftrTypeCaps.exportOnTcpStateSupported:
      output.write( f'{space}no {cmd}\n' )

   if exportMplsSupportedForType( ftrType, mplsHwCaps ):
      mplsExportCmd = "record export mpls"
      if ftr.mplsExport != constants.mplsExportDefault:
         output.write( f"{space}{mplsExportCmd}\n" )
      elif detail:
         output.write( f"{space}no {mplsExportCmd}\n" )

def showCongestion( output, congestionConfig, ftrSwCaps,
                    cliSave=True, detail=False, space='' ):
   # output: out stream to write to
   # congestionConfig: FlowTracking::congestionConfig | congestionContext
   # ftrSwCaps: flow tracking software capabilities
   # cliSave: whether or not this is called by cliSavePlugin
   # detail: whether or not this was called with detail, all or all detail
   # space: how much space to indent by
   if not congestionSupportedForType( ftrSwCaps ):
      return
   congestionCmd = 'record export on congestion'
   if congestionConfig:
      if not cliSave:
         space += spaceConst
         output.write( f'{space}{congestionCmd}\n' )
         space += spaceConst
      conConf = congestionConfig
      if conConf.dampingTimeout is not None:
         cmd = f'{space}damping timeout {conConf.dampingTimeout} seconds\n'
         output.write( cmd )
      elif detail:
         cmd = f'{space}damping timeout {dampingTimeout.defaultTimeout} seconds\n'
         output.write( cmd )
      if conConf.interval is not None:
         cmd = ( f'{space}interval before {conConf.interval.before} seconds'
                 f' after {conConf.interval.after} seconds\n' )
         output.write( cmd )
      elif detail:
         cmd = ( f'{space}interval before {congestionInterval.defaultBefore} seconds'
                 f' after {congestionInterval.defaultAfter} seconds\n' )
         output.write( cmd )

def showExporter( output, expName, exp, ftrTypeStr, ftrTypeCaps, ftrSwCaps, /, *,
                  cliSave=True, saveAll=False, saveAllDetail=False, space='' ):
   detail = saveAll or saveAllDetail
   if not cliSave:
      space += spaceConst
      output.write( f'{space}exporter {expName}\n' )
      space += spaceConst

   supportedExportFormats = supportedExportFormatsForType( ftrTypeCaps, ftrSwCaps )
   sflowExportSupported = ExpFormat.formatSflow in supportedExportFormats
   defaultExportFormat = defaultExportFormatForType( ftrTypeCaps, ftrSwCaps )
   exportFormat = exp.exportFormat
   defaultIpfixMtu = exp.ipfixMtu == ipfixMtu.mtuDefault
   if ( exportFormat in [ defaultExportFormat, ExpFormat.formatDefault ] and
        defaultExportFormat != ExpFormat.formatIpfix ):
      if detail:
         output.write( f'{space}default format\n' )
   elif exportFormat == ExpFormat.formatSflow:
      output.write( f'{space}format sflow\n' )
   elif exportFormat == ExpFormat.formatDropReport:
      output.write( f'{space}format drop-report\n' )
   elif exportFormat == ExpFormat.formatIpfix:
      output.write( f'{space}format drop-report\n' )
   elif ( exportFormat == ExpFormat.formatIpfix or
          ( defaultExportFormat == ExpFormat.formatIpfix and
            ( detail or not defaultIpfixMtu ) ) ):
      fmtStr = f'format ipfix version {exp.ipfixVersion}'
      if detail or not defaultIpfixMtu:
         fmtStr += f' max-packet-size {exp.ipfixMtu}'
      if exportFormat == ExpFormat.formatDefault and defaultIpfixMtu:
         fmtStr = f'default {fmtStr}'
      output.write( f'{space}{fmtStr}\n' )
   elif exportFormat == ExpFormat.formatPcap:
      output.write( f'{space}format pcap\n' )

   if sflowExportSupported:
      if exp.useSflowCollectorConfig:
         output.write( f'{space}collector sflow\n' )
      elif detail:
         output.write( f'{space}no collector sflow\n' )

   for host in sorted( exp.collectorHostAndPort.keys() ):
      hnp = exp.collectorHostAndPort[ host ]
      fmtStr = 'collector %s' % hnp.hostname
      if detail or hnp.port != ipfixPort.ipfixPortDefault:
         fmtStr += ' port %d' % hnp.port
      output.write( f'{space}{fmtStr}\n' )

   if dscpSupportedForType( ftrTypeCaps, ftrSwCaps ):
      if detail or exp.dscpValue != constants.dscpValueDefault:
         output.write( '%sdscp %d\n' % ( space, exp.dscpValue ) )

   if localIntfSupportedForType( ftrTypeCaps, ftrSwCaps ):
      if exp.localIntfName and exp.localIntfName != constants.intfNameDefault:
         output.write( f'{space}local interface {exp.localIntfName}\n' )
      elif detail:
         output.write( '%sno local interface\n' % ( space ) )

   if templateSupportedForType( ftrTypeCaps, ftrSwCaps ):
      if detail or exp.templateInterval != templateInterval.intervalDefault:
         output.write( '%stemplate interval %d\n' % ( space, exp.templateInterval ) )

def showGroup( output, groupName, group, cliSave=True,
               saveAll=False, saveAllDetail=False, space='' ):
   detail = saveAll or saveAllDetail
   if not cliSave:
      output.write( f'{space}group {groupName}\n' )
      space += spaceConst

   if group.encapType:
      encapStr='encapsulation'
      for encapType in sorted( group.encapType ):
         encapStr += ' %s' % encapTypeKwStr[ encapType ]
      output.write( f'{space}{encapStr}\n' )
   elif detail and groupName not in reservedGroupToSeqnoMap:
      output.write( '%sno encapsulation\n' % space )

   if group.ipAccessList != accessListDefault:
      output.write( '%sip access-list %s\n'
                    % ( space, group.ipAccessList.aclName ) )
   elif detail and groupName not in reservedGroupToSeqnoMap:
      output.write( '%sno ip access-list\n' % space )

   if group.ip6AccessList != accessListDefault:
      output.write( '%sipv6 access-list %s\n'
                    % ( space, group.ip6AccessList.aclName ) )
   elif detail and groupName not in reservedGroupToSeqnoMap:
      output.write( '%sno ipv6 access-list\n' % space )

   if group.expName:
      for exp in sorted( group.expName ):
         output.write( f'{space}exporter {exp}\n' )
   elif detail:
      output.write( '%sno exporter\n' % space )

   if group.mirrorConfig != mirrorConfigDefault:
      mirrorStr = ''
      if group.mirrorConfig.initialCopy != initialCopy.pktDefault:
         mirrorStr += ' initial packets %d' % group.mirrorConfig.initialCopy
      if group.mirrorConfig.mirrorIntervalFixed != \
         mirrorIntervalFixed.intervalDefault:
         mirrorStr += ' sample interval fixed %d' \
                      % group.mirrorConfig.mirrorIntervalFixed
      elif group.mirrorConfig.mirrorIntervalRandom != \
           mirrorIntervalRandom.intervalDefault:
         mirrorStr += ' sample interval random %d' \
                      % group.mirrorConfig.mirrorIntervalRandom
      if mirrorStr:
         mirrorStr = 'mirror %s' % group.mirrorConfig.sessionName + mirrorStr
         output.write( f'{space}{mirrorStr}\n' )
   elif detail:
      output.write( '%sno mirror\n' % space )

def showActiveGroups( output, trConfig, space='' ):
   if trConfig.fgConfig:
      space += spaceConst
      output.write( '%sgroups\n' % space )
      space += spaceConst
      seqnoToGroupMap = {}
      for groupConfig in trConfig.fgConfig.values():
         seqnoToGroupMap[ groupConfig.seqno ] = groupConfig
      for groupSeqno in sorted( seqnoToGroupMap ):
         groupConfig = seqnoToGroupMap[ groupSeqno ]
         showGroup( output, groupConfig.fgName, groupConfig,
                    cliSave=False, space=space )

def getFtrTypeFromArgs( args ):
   ftrType = None
   if egressStr in args:
      ftrType = ftrTypeSampled
   elif ftrTypeKwStr[ ftrTypeSampled ] in args:
      ftrType = ftrTypeSampled
   elif ftrTypeKwStr[ ftrTypeHardware ] in args:
      ftrType = ftrTypeHardware
   elif ftrTypeKwStr[ ftrTypeInbandTelemetry ].split()[ 1 ] in args:
      ftrType = ftrTypeInbandTelemetry
   elif ftrTypeKwStr[ ftrTypeMirrorOnDrop ] in args:
      ftrType = ftrTypeMirrorOnDrop
   elif ftrTypeKwStr[ ftrTypeCpuQueue ] in args:
      ftrType = ftrTypeCpuQueue
   elif ftrTypeKwStr[ ftrTypeDfw ].split()[ 1 ] in args:
      ftrType = ftrTypeDfw
   else:
      assert 0

   return ftrType

def getFtrPromptFromFtrTypeStr( ftrTypeStr ):
   if ftrTypeStr == ftrTypeKwStr[ ftrTypeDfw ]:
      return 'dfw'
   return ftrTypeStr.replace( ' ', '-' )

def protocolStr( protocol, protocolNumber ):
   if protocol:
      if protocol.startswith( 'ipProto' ):
         return protocol[ len( 'ipProto' ) : ].upper()
      else:
         return protocol
   else:
      return str( protocolNumber )

def tcpFlagStr( tcpFlags ):
   tcpFlagMap = { 'syn' : 'S',
                  'fin' : 'F',
                  'rst' : 'R',
                  'ack' : 'A',
                  'psh' : 'P',
                  'urg' : 'U',
                  'ns' : 'N',
                  'cwr' : 'C',
                  'ece' :'E',
   }
   flags = []
   for flag, symbol in tcpFlagMap.items():
      if getattr( tcpFlags, flag ):
         flags.append( symbol )
   return '|'.join( flags ) or 'none'

def addressStr( ip, port ):
   ipString = ipStr( ip )
   template = "[%s]:%s" if ":" in ipString else "%s:%s"
   return template % ( ipString, port )

def ipStr( ip ):
   return getattr( ip, 'stringValue', ip )

def timeStr( time ):
   return datetime.fromtimestamp( time ).strftime( "%Y-%m-%d %H:%M:%S.%f" )

def collectorInactiveReasonStr( reason ):
   if reason == collectorInactiveReason.maxCollectorsLimit:
      return " (inactive, reason - maximum collectors limit reached)"
   if reason == collectorInactiveReason.collectorNotProcessed:
      return " (inactive, reason - waiting configuration processing)"
   return " (inactive)"

def getExpReasonFlagsFromExpReasonModel( expReasonModel ):
   expInactiveReason = Tac.newInstance(
         "FlowTracking::ExporterInactiveReason", 0 )
   for expReason in expReasonModel.expReasons:
      if expReason.expReason == exporterInactiveReasonEnum.exporterNotProcessed:
         expInactiveReason.exporterNotProcessed = True
      if expReason.expReason == exporterInactiveReasonEnum.maxExportersLimit:
         expInactiveReason.maxExportersLimit = True
      if expReason.expReason == exporterInactiveReasonEnum.invalidCollector:
         expInactiveReason.invalidCollector = True
      if expReason.expReason == exporterInactiveReasonEnum.invalidVrf:
         expInactiveReason.invalidVrf = True
      if expReason.expReason == exporterInactiveReasonEnum.invalidLocalIntf:
         expInactiveReason.invalidLocalIntf = True
      if expReason.expReason == exporterInactiveReasonEnum.waitingForUDPPort:
         expInactiveReason.waitingForUDPPort = True
      if expReason.expReason == exporterInactiveReasonEnum.invalidExportFormat:
         expInactiveReason.invalidExportFormat = True
   return expInactiveReason

def getExpReasonEnumList( reasonInt ):
   expReasonEnumList = []
   if not reasonInt:
      expReasonEnumList.append(
            exporterInactiveReasonEnum.exporterInactiveReasonUnknown )
      return expReasonEnumList

   expReason = Tac.newInstance(
         "FlowTracking::ExporterInactiveReason", reasonInt )
   if expReason.exporterNotProcessed:
      expReasonEnumList.append( exporterInactiveReasonEnum.exporterNotProcessed )
   if expReason.maxExportersLimit:
      expReasonEnumList.append( exporterInactiveReasonEnum.maxExportersLimit )
   if expReason.invalidCollector:
      expReasonEnumList.append( exporterInactiveReasonEnum.invalidCollector )
   if expReason.invalidVrf:
      expReasonEnumList.append( exporterInactiveReasonEnum.invalidVrf )
   if expReason.invalidLocalIntf:
      expReasonEnumList.append( exporterInactiveReasonEnum.invalidLocalIntf )
   if expReason.waitingForUDPPort:
      expReasonEnumList.append( exporterInactiveReasonEnum.waitingForUDPPort )
   if expReason.invalidExportFormat:
      expReasonEnumList.append( exporterInactiveReasonEnum.invalidExportFormat )
   return expReasonEnumList

def exporterInactiveReasonStr( reason, ipVersion='' ):
   if ipVersion == AddressFamily.ipunknown:
      ipVersion = ''
   if reason.maxExportersLimit:
      return " (inactive, reason - maximum exporters limit reached)"
   if reason.exporterNotProcessed:
      return " (inactive, reason - waiting configuration processing)"
   if reason.invalidExportFormat:
      return " (inactive, reason - missing export format configuration)"
   if reason.waitingForUDPPort:
      return " (inactive, reason - waiting for UDP port for local interface)"
   if reason.value == 0:
      return " (inactive)"

   reasonStr = " (inactive, reason - invalid/missing %s" % ipVersion
   addComma = False
   if reason.invalidCollector:
      reasonStr += " collector config"
      addComma = True

   addLocalIntf = True
   if reason.invalidLocalIntf:
      if addComma:
         reasonStr += ','
      reasonStr += ' local interface IP address'
      addLocalIntf = False
      addComma = False

   if reason.invalidVrf:
      if addComma:
         reasonStr += ','
      if addLocalIntf:
         reasonStr += ' local interface VRF config'
      else:
         reasonStr += ' and VRF config'
   reasonStr += ')'
   return reasonStr

def trackerInactiveReasonStr( reason ):
   if reason == trackerInactiveReason.maxTrackersLimit:
      return " (inactive, reason - maximum trackers limit reached)"
   if reason == trackerInactiveReason.trackerNotProcessed:
      return " (inactive, reason - waiting configuration processing)"
   return " (inactive)"

def sampledHoInactiveReasonStr( reason ):
   if reason == hoState.counterFeatureDisabled:
      return " Hardware Counter feature is disabled"
   if reason == hoState.invalidTcamProfile:
      return " Invalid TCAM profile"
   return " Unknown"

def renderCounter( counter, humanReadableOnly=False ):
   suffix = [ '', 'K', 'M', 'B', 'T', 'Q' ]
   ctr = counter
   i = 0
   while ( decimal.Decimal( ctr ).quantize( decimal.Decimal( '0.001' ),
         rounding=decimal.ROUND_HALF_UP ) >= 1000 and i < len( suffix ) - 1 ):
      ctr = ctr / 1000.0
      i += 1
   if counter >= 1000:
      ctrStr = ( '%.3f' % ctr ).rstrip( '0' ).rstrip( '.' )
      ctrStr += suffix[ i ]
      if not humanReadableOnly:
         ctrStr += ' (' + str( counter ) + ')'
      return ctrStr
   else:
      return str( counter )

def getFlowTrackingCliConfigPath( ftrType ):
   return ftrConfig.flowTrackingCliConfigPath( ftrType )

def getSampledFlowTrackingCliConfigPath():
   return ftrConfig.sampledFlowTrackingCliConfigPath()

def getHardwareFlowTrackingCliConfigPath():
   return ftrConfig.hardwareFlowTrackingCliConfigPath()

def getHardwareFlowTrackingFlowWatcherConfigPath():
   return ftrConfig.hardwareFlowTrackingFlowWatcherConfigPath()

def getInbandTelemetryFlowTrackingCliConfigPath():
   return ftrConfig.inbandTelemetryFlowTrackingCliConfigPath()

def getMirrorOnDropFlowTrackingCliConfigPath():
   return ftrConfig.mirrorOnDropFlowTrackingCliConfigPath()

def getCpuQueueFlowTrackingCliConfigPath():
   return ftrConfig.cpuQueueFlowTrackingCliConfigPath()

def getDfwFlowTrackingCliConfigPath():
   return ftrConfig.dfwFlowTrackingCliConfigPath()
