#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''Contains code that are common to Flowspec CliPlugin and
CliSavePlugin'''

from MultiRangeRule import multiRangeToCanonicalString
import Tac

ethIntfId = Tac.Type( 'Arnet::EthIntfId' )
subIntfId = Tac.Type( 'Arnet::SubIntfId' )
vlanIntfId = Tac.Type( 'Arnet::VlanIntfId' )
portChannelIntfId = Tac.Type( 'Arnet::PortChannelIntfId' )

#----------------------------------------------------------------------------
# For an interface to be capable of flow-spec, it must be an ethernet,
# port-channel or vlan interface. NOTE: sub-interfaces also evaluate
# as ethernet or port-channel.
#----------------------------------------------------------------------------
def flowspecCapable( intfName ):
   return ethIntfId.isEthIntfId( intfName ) or \
        portChannelIntfId.isPortChannelIntfId( intfName ) or \
        vlanIntfId.isVlanIntfId( intfName )

#----------------------------------------------------------------------------
# Helper to fetch the flow-spec command based on Flowspec::IpVersion
#----------------------------------------------------------------------------
def getFlowspecCmdFrom( ipVersion ):
   if not ipVersion.ipv4 and not ipVersion.ipv6:
      return "no flow-spec ipv4 ipv6"
   elif ipVersion.ipv4 and not ipVersion.ipv6:
      return "flow-spec ipv4"
   elif not ipVersion.ipv4 and ipVersion.ipv6:
      return "flow-spec ipv6"
   else:  # ipVersion.ipv4 and ipVersion.ipv6
      return "flow-spec ipv4 ipv6"

# ----------------------------------------------------------------------------
# Helper to generate flow-spec interface-set command
# ----------------------------------------------------------------------------
def getFlowspecIntfInterfaceSetCmdFrom( interfaceSetConfig ):
   # Using multiRangeToCanonicalString to generate a string for the interface-set
   # range. For example, if range=[1,2,3,4,8], the output string is "1-4,8"
   if not interfaceSetConfig or not interfaceSetConfig.interfaceSet:
      return "no flow-spec interface-set"
   else:
      return "flow-spec interface-set " + \
         multiRangeToCanonicalString( interfaceSetConfig.interfaceSet.keys() )
