# Copyright (c) 2024 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''Implements `frozendict`, see class docstring for more information'''

from typing import Any, Hashable, NoReturn, Optional, TypeVar, Union, overload
from collections.abc import Iterable, Mapping, Sequence

# Type parameter for the `cls` argument of decorators:
_C = TypeVar( '_C', bound=type )

def __raiseTypeError( self: Any, *args: Any, **kwargs: Any ) -> NoReturn:
   '''Raises TypeError telling that current object is frozen'''
   raise TypeError( f'{type(self).__name__} is frozen and cannot be changed' )

def __freeze( cls: _C ) -> _C:
   '''Replaces all mutator methods of `cls` with `__raiseTypeError`'''

   # Names of attributes that are known not to be mutators (also `__setattr__` and
   # `__delattr__` since we don't have public attributes, while subclasses may need
   # these methods):
   allowedNames_ = ( '__class__ __class_getitem__ __contains__ __delattr__ __dir__ '
         '__doc__ __eq__ __format__ __ge__ __getattribute__ __getitem__ __gt__ '
         '__init__ __init_subclass__ __iter__ __le__ __len__ __lt__ __module__ '
         '__ne__ __new__ __or__ __reduce__ __reduce_ex__ __repr__ __reversed__ '
         '__ror__ __setattr__ __sizeof__ __slots__ __str__ __subclasshook__ copy '
         'fromkeys get items keys values' )
   allowedNames = set( allowedNames_.split() )
   attributes = { name : getattr( cls, name ) for name in dir( cls ) }
   # Dictionary of callable attributes (methods) not defined in current module.
   inheritedMethodNames = { name for name, method in attributes.items()
                            if callable( method ) and
                               getattr( method, '__module__', None ) != __name__ }
   # Names of inherited methods that are not in `allowedNames`.
   # As of Python 3.9 these are the following forbidden methods:
   # `__delitem__`, `__setitem__`, `clear`, `pop`, `popitem`, `setdefault`, `update`
   # If any new `dict` methods appear in the future, they are going to be forbidden
   # until explicitly added to `allowedNames`. This is the safer option.
   forbiddenMethodNames = inheritedMethodNames - allowedNames
   for methodName in forbiddenMethodNames:
      setattr( cls, methodName, __raiseTypeError )
   return cls

# Making it protected, not private, since subclasses may want to use it:
def _initializeEmpty( cls: _C ) -> _C:
   '''Sets `cls.__EMPTY` to a `cls` instance created from empty iterator'''
   # Set `__EMPTY` to `None` before creating new instance since `__init__` checks it.
   # Since `__EMPTY` is a private attribute, mangle the name to access it.
   setattr( cls, f'_{cls.__name__}__EMPTY', None )
   # Use iterator to trick class into actually constructing new instance
   # instead of returning one stored in `cls.__EMPTY`.
   setattr( cls, f'_{cls.__name__}__EMPTY', cls( iter( () ) ) )
   return cls

# Key and value type parameters for class definition
#
# Type parameters of `dict` are invariant: this class is covariant for reads
# and contravariant for writes. Since `frozendict` does not support writes,
# we mark its type parameters as covariant.
_K_co = TypeVar( '_K_co', bound=Hashable, covariant=True )
_V_co = TypeVar( '_V_co', covariant=True )

# Key and value type parameters for method arguments:
_K = TypeVar( '_K', bound=Hashable )
_V = TypeVar( '_V' )

@__freeze
@_initializeEmpty
# Ignoring type parameters variance mismatch explained above:
class frozendict( dict[ _K_co, _V_co ] ):  # type: ignore[type-var]
   '''Immutable dictionary

   Instances still return true if checked with `isinstance` against `dict` or
   `MutableMapping`, but raise `TypeError` on any attempt to modify them.
   This is unlike `frozenset` (that simply lacks mutator methods) but like `tuple`
   (that still has `__setitem__` and `__delitem__` methods but raises on them).

   The instance is hashable as long as all the values are hashable.
   It can also be safely used in function argument defaults.

   Constructor and some of the methods may return existing instance instead of
   creating a new one, but full deduplication is not implemented and must not be
   relied upon. Always use `==`, not `is` for comparing instances.'''

   __slots__ = ( '__hash', )

   __EMPTY: 'frozendict[ NoReturn, NoReturn ]'  # initialized by `@_initializeEmpty`

   @overload
   def __new__( cls: type[ 'frozendict[ _K, _V ]' ],
         ) -> 'frozendict[ NoReturn, NoReturn ]':
      ...

   @overload
   def __new__( cls: type[ 'frozendict[ _K, _V ]' ],
                *args: Union[ Iterable[ tuple[ _K, _V ] ], Mapping[ _K, _V ] ],
         ) -> 'frozendict[ _K, _V ]':
      ...

   @overload
   def __new__( cls: type[ 'frozendict[ str, _V ]' ], **kwargs: _V,
         ) -> 'frozendict[ str, _V ]':
      ...

   @overload
   def __new__( cls: type[ 'frozendict[ Union[ str, _K ], _V ]' ],
                *args: Union[ Iterable[ tuple[ _K, _V ] ], Mapping[ _K, _V ] ],
                **kwargs: _V,
         ) -> 'frozendict[ Union[ str, _K ], _V ]':
      ...

   def __new__( cls: type[ 'frozendict[ Union[ str, _K ], _V ]' ],
                *args: Union[ Iterable[ tuple[ _K, _V ] ], Mapping[ _K, _V ] ],
                **kwargs: _V,
         ) -> 'frozendict[ Union[ str, _K ], _V ]':
      if not kwargs:
         if not args and cls is frozendict:
            # If called with empty arguments, return pre-created empty instance:
            return cls.__EMPTY
         if len( args ) == 1:
            [ arg ] = args
            # Cannot use `isinstance` below since we want to allow downgrading.
            if type( arg ) is cls:  # pylint: disable=unidiomatic-typecheck
               # If called for collection of the right class just return it:
               return arg
            if not arg and cls is frozendict:
               # If called for empty collection return pre-created empty instance:
               return cls.__EMPTY
            # We are not trying to catch empty iterators, it's too expensive.
      return super().__new__( cls, *args, **kwargs )

   @overload
   def __init__( self ) -> None:
      ...

   @overload
   def __init__( self,
                 *args: Union[ Iterable[ tuple[ _K, _V ] ], Mapping[ _K, _V ] ],
         ) -> None:
      ...

   @overload
   def __init__( self, **kwargs: _V ) -> None:
      ...

   @overload
   def __init__( self,
                *args: Union[ Iterable[ tuple[ _K, _V ] ], Mapping[ _K, _V ] ],
                **kwargs: _V ) -> None:
      ...

   def __init__( self,
                 *args: Union[ Iterable[ tuple[ _K, _V ] ], Mapping[ _K, _V ] ],
                 **kwargs: _V ) -> None:
      if self is self.__EMPTY or ( args and args[ 0 ] is self ):
         # If we are here `__new__` returned an already-initialized object
         # (no need to check `kwargs` and `len( args )` since there is no way
         # `self` could be in the first argument if not for `__new__`).
         return
      super().__init__( *args, **kwargs )
      self.__hash: Optional[ int ] = None  # initialized on demand

   @overload
   @classmethod
   def fromkeys( cls, iterable: Iterable[ _K ], value: None = None, /,
         ) -> 'frozendict[ _K, None ]':
      ...

   @overload
   @classmethod
   def fromkeys( cls, iterable: Iterable[ _K ], value: _V, /,
         ) -> 'frozendict[ _K, _V ]':
      ...

   @classmethod
   def fromkeys( cls, iterable: Iterable[ _K ], value: Optional[ _V ] = None, /,
         ) -> Union[ 'frozendict[ _K, None ]', 'frozendict[ _K, _V ]' ]:
      return cls( dict.fromkeys( iterable, value ) )

   def __reduce__( self,
         ) -> tuple[ type[ 'frozendict[ _K_co, _V_co ]' ],
                     tuple[ Sequence[ tuple[ _K_co, _V_co ] ] ] ]:
      # Unpickle by passing items to the constructor:
      return ( type( self ), ( tuple( self.items() ), ) )

   # This attribute was set to `None` in the parent:
   def __hash__( self ) -> int:  # type: ignore[override]
      # Cached result is stored in `self.__hash`.
      result = self.__hash
      if result is None:
         self.__hash = result = hash( frozenset( self.items() ) )
      return result

   # mypy's `dict.__or__` only accepts `dict`, so `__ior__` must do the same
   def __ior__( self,  # type: ignore[override]
                other: dict[ _K, _V ],
         ) -> 'frozendict[ Union[ _K_co, _K ], Union[ _V_co, _V ] ]':
      # Will cause `lhs` to be reassigned to `lhs | rhs`:
      return NotImplemented

   def __or__( self, other: dict[ _K, _V ],
         ) -> 'frozendict[ Union[ _K_co, _K ], Union[ _V_co, _V ] ]':
      # If one of the arguments is empty and the other is the right type, return it.
      # (Right type includes subclasses, override if this is not desired behaviour.)
      # If both arguments are empty return the second one like logical `or` does.
      if not self and isinstance( other, type( self ) ):
         return other
      if not other:
         return self
      return type( self )( super().__or__( other ) )

   # Intentionally not overriding `__ror__` to preserve left hand side type:
   # `dict() | frozendict()` will produce `dict` the same way sets work.

   def __repr__( self ) -> str:
      # Return 'frozendict({...})', like `frozenset` does:
      return f'{type(self).__name__}({super().__repr__()})'

   def copy( self ) -> 'frozendict[ _K_co, _V_co ]':
      return self
