# Copyright (c) 2017 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Cell
import EntityMib
import Tac
import Tracing
from EntityMib import ChipType, IndexAllocator
 
__defaultTraceHandle__ = Tracing.Handle( 'Fru.AleMib' )

t0 = Tracing.trace0
t1 = Tracing.trace1

def asicMibChipId( chipType, localChipId ):
   assert chipType in ( ChipType.switchAsic, ChipType.fabricAsic ), \
      f'Invalid Ale Chip type {chipType}'
   assert localChipId < 1000, f'localChipId: {localChipId} must be less than 1000'
   return IndexAllocator.getChipId( chipType, localChipId )

def populateAsicChipMib( parentMib, invFapOrFe, chipType, localChipId,
                         tag, description ):
   assert chipType in [ ChipType.switchAsic, ChipType.fabricAsic ], \
          "Invalid Ale Chip type %s" % chipType
   chipId = asicMibChipId( chipType, localChipId )
   chipMib = parentMib.chip.get( chipId )
   if chipMib is None:
      physicalIndex = \
         IndexAllocator.collectionItemPhysicalIndex( parentMib.chip, chipId )
      chipMib = parentMib.newChip( physicalIndex, chipId, tag )
   chipMib.label = str( chipId )
   chipMib.description = f"{description} {chipId}"
   EntityMib.populateMib( chipMib, invFapOrFe )

entmib = None
redStatus = None
sliceMibDir = None
redStatusReactor = None
sliceMibDirReactor = None

class MibInfoReactor( Tac.Notifiee ):
   notifierTypeName = "Ale::MibInfo"

   def __init__( self, sliceMibInfo ):
      Tac.Notifiee.__init__( self, sliceMibInfo )
      self.cardMib_ = None
      self.handleSlotId()

   def cardMib( self ):
      slotId = self.notifier_.slotId
      assert slotId
      entmibRoot = entmib.root
      if entmib.chassis:
         cardSlot = entmibRoot.cardSlot.get( slotId )
         assert cardSlot is not None
         cardMib = cardSlot.card
      else:
         cardMib = entmibRoot
      return cardMib

   @Tac.handler( 'slotId' )
   def handleSlotId( self ):
      if not self.notifier_.slotId:
         return
      self.cardMib_ = self.cardMib()
      for key in self.notifier_.chip:
         self.handleChip( key )

   @Tac.handler( 'chip' )
   def handleChip( self, key ):
      if key not in self.notifier_.chip or not self.cardMib_:
         return

      chipInfo = self.notifier_.chip[ key ]
      chipMibId = asicMibChipId( chipInfo.chipType, chipInfo.chipId )
      chipMib = self.cardMib_.chip.get( chipMibId )
      if not chipMib:
         return
      chipMib.modelName = chipInfo.chipModel
      chipMib.firmwareRev = chipInfo.chipFirmwareRev
      chipMib.initStatus = 'ok'
      t0( "Published firmware info", chipMib.firmwareRev, "for", key )

class MibInfoSliceDirReactor( Tac.Notifiee ):
   notifierTypeName = "Tac::Dir"

   def __init__( self, mibInfoSliceDir ):
      Tac.Notifiee.__init__( self, mibInfoSliceDir )
      self.sliceMibReactor_ = {}
      for key in self.notifier_:
         self.handleSlice( key )

   @Tac.handler( 'entityPtr' )
   def handleSlice( self, key ):
      sliceInfo = self.notifier_.get( key )

      if not sliceInfo:
         t0( "Deleting MibInfo reactor for", key )
         del self.sliceMibReactor_[ key ]
      elif sliceInfo:
         t0( "Creating MibInfo reactor for", key )
         self.sliceMibReactor_[ key ] = MibInfoReactor( self.notifier_[ key ] )

class RedundancyStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Redundancy::RedundancyStatus"

   def __init__( self, redundancyStatus ):
      Tac.Notifiee.__init__( self, redundancyStatus )
      self.protocol = redundancyStatus.protocol
      self.updateReactors()

   @Tac.handler( 'mode' )
   def updateReactors( self ):
      global sliceMibDirReactor
      sysMode = self.notifier_.mode
      t1( 'mode:', sysMode, 'protocol:', self.protocol )
      if sysMode == 'active' or ( self.protocol != 'sso' and sysMode == 'standby' ):
         sliceMibDirReactor = MibInfoSliceDirReactor( sliceMibDir )

def setupMibReactors():
   global redStatusReactor
   redStatusReactor = RedundancyStatusReactor( redStatus )

def Plugin( context ):
   global sliceMibDir, entmib, redStatus

   mg = context.entityManager.mountGroup()
   entmib = mg.mount( 'hardware/entmib', 'EntityMib::Status', 'wi' )
   sliceMibDir = mg.mount( 'hardware/ale/mibInfo', 'Tac::Dir', 'ri' )
   # redundancy/status is already mounted through CAgent layer
   redStatus = context.entityManager.lookup( Cell.path( 'redundancy/status' ) )
   mg.close( setupMibReactors )
