#!/usr/bin/env python3
# Copyright (c) 2013 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import Fru

import Tracing
__defaultTraceHandle__ = Tracing.Handle( "Fru.Cooling" )
t0 = Tracing.trace0 
t3 = Tracing.trace3

class CoolingDomainDirDriver( Fru.FruDriver ):
   managedTypeName = "Inventory::CoolingDomainDir"
   provides = [ Fru.FruDriver.environmentInit ]

   def __init__( self, invCoolingDomainDir, parentMibEntity, parentDriver,
                 driverCtx ):
      super().__init__(
         invCoolingDomainDir, parentMibEntity, parentDriver, driverCtx )

      envConfig = driverCtx.sysdbRoot.entity[
         "environment/thermostat/config" ]
      envCoolingDomainDir = driverCtx.sysdbRoot.entity[
         "environment/thermostat/coolingDomain" ]

      profileName = ""
      if invCoolingDomainDir.profiles:
         envCoolingDomainDir.defaultCoolingDomainProfileName = \
               invCoolingDomainDir.profiles.defaultProfileName
         profiles = list( invCoolingDomainDir.profiles.profile.values() )
         assert len( set( profiles ) ) == len( profiles ), "More than " \
               "one profile with the same number of cooling domains in profile"

         for profile in profiles:
            envCoolingDomainDir.supportedProfile[ profile.name ] = \
                  profile.numDomainsInProfile
            if envConfig.cliNumCoolingZones == profile.numDomainsInProfile:
               profileName = profile.name

      if envConfig.cliNumCoolingZones and not profileName:
         t0( "Cooling domain profile (", envConfig.cliNumCoolingZones,
             ") configured in Cli is not supported",
             [ ( p.name, p.numDomainsInProfile )
                 for p in envCoolingDomainDir.supportedProfile.values() ] )

      if profileName:
         t0( "Configured in cli cooling domain profile is", profileName )
      elif invCoolingDomainDir.profiles:
         assert invCoolingDomainDir.profiles.defaultProfileName, \
            "Default cooling domain profile is not configured"
         profileName = invCoolingDomainDir.profiles.defaultProfileName
         assert profileName in invCoolingDomainDir.profiles.profile, \
               "Unknown cooling domains profile %s" % profileName
         t0( "Use default cooling domain profile", profileName )

      envCoolingDomainDir.coolingDomainProfileName = profileName
      
      profile = {}
      if profileName:
         profile = \
            invCoolingDomainDir.profiles.profile[ profileName ].coolingDomainMapping

      coolingDomains = [ coolingDomain.name for coolingDomain in profile.values() ]
      for coolingDomain in invCoolingDomainDir.coolingDomain.values():
         # Create cooling domains only from current cooling domains profile
         if coolingDomain.name in profile and \
               coolingDomain.name not in coolingDomains:
            t3( "Cooling domain '%s' is not in current profile" % \
                  coolingDomain.name )
            continue

         t3( "Created cooling domain", coolingDomain.name )
         envCoolingDomain = (
            envCoolingDomainDir.coolingDomain.get( coolingDomain.name ) )
         if not envCoolingDomain:
            # Only create the cooling domain if it doesn't already
            # exist. We may dynamically change minFanSpeed after
            # initialization, and after a switchover if we clobber the
            # existing cooling domain it may reset minFanSpeed to an
            # undesired value.
            envCoolingDomain = envCoolingDomainDir.newCoolingDomain(
               coolingDomain.name,
               coolingDomain.overheatThresholdRatioBase,
               coolingDomain.minFanSpeed )

         envCoolingDomain.targetToOverheatMargin = (
            coolingDomain.targetToOverheatMargin )
         envCoolingDomain.useAggressiveThresholds = (
            coolingDomain.useAggressiveThresholds )

         for ( fanKey, offset ) in coolingDomain.fanSpeedOffset.items():
            slope = coolingDomain.fanSpeedSlope.get( fanKey, 1.0 )
            envCoolingDomain.fanSpeedOffset[ fanKey ] = offset
            envCoolingDomain.fanSpeedSlope[ fanKey ] = slope
            envCoolingDomain.maxFanSpeed[ fanKey ] = 100

         envCoolingDomain.baseFansDefined = coolingDomain.baseFansDefined
         for ( fanKey, priority ) in ( coolingDomain.baseFanKeyPriority
                                     ).items():
            envCoolingDomain.baseFanKeyPriority[ fanKey ] = priority

         envCoolingDomain.fanSlopeOffsetDefined = coolingDomain.fanSlopeOffsetDefined
         envCoolingDomain.initialFanSpeed = coolingDomain.initialFanSpeed

      for domainNameFrom, domainTo in profile.items():
         envCoolingDomainDir.coolingDomainMapping[ domainNameFrom ] = \
            envCoolingDomainDir.coolingDomain[ domainTo.name ]

      for slot, coolingDomain in invCoolingDomainDir.slotToCoolingDomain.items():
         coolingDomainName = coolingDomain.name
         if coolingDomainName in profile:                         
            coolingDomainName = profile[ coolingDomainName ].name

         envCoolingDomainDir.slotToCoolingDomain[ slot ] = (
            envCoolingDomainDir.coolingDomain[ coolingDomainName ] )

      if parentMibEntity.powerSupplySlot or parentMibEntity.fanTraySlot:
         # If no power supply slots or fan tray slots are defined, assume
         # we're defining PLSlots, which will be initialized in the PLSlot FruPlugin
         envCoolingDomainDir.ready = True

def Plugin( ctx ):
   ctx.registerDriver( CoolingDomainDirDriver )
   mg = ctx.entityManager.mountGroup()
   mg.mount( "environment/thermostat/coolingDomain",
             "Environment::Thermostat::CoolingDomainDir", "w" )
   mg.mount( "environment/thermostat/config",
             "Environment::Thermostat::Config", "r" )
   mg.close( None )
