# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac
import Tracing
import Fru
import EntityMib
# pylint: disable-next=consider-using-from-import
import FruPlugin.TempSensorFru as TempSensorFru
from EntityMib import IndexAllocator

th = Tracing.Handle( "Fru.Denali" )
t0 = th.trace0

class DenaliPhyDriver( Fru.FruDriver ):
   managedTypeName = 'Inventory::Denali::Ac400Dir'
   provides = []
   requires = [ Fru.FruDriver.systemInit, Fru.FruDriver.interfaceInit, "xcvr" ]

   def __init__( self, invPhyDir, fruEntMib, parentDriver, driverCtx ):
      super().__init__( invPhyDir, fruEntMib,
                        parentDriver, driverCtx )
      t0( "Handling Inventory::Denali::Ac400Dir for",
          Fru.fruBaseName( invPhyDir ) )

      sliceId = Fru.fruBase( invPhyDir ).sliceId
      assert sliceId

      hwDir = driverCtx.sysdbRoot[ "hardware" ]
      chipConfigDir = hwDir.entity[ "phyChip/config/ac400" ]
      phyTopologyDir = hwDir.entity[ "phy/topology/allPhys" ]
      # pylint: disable-next=consider-using-f-string
      mdioEngine = hwDir.entity[ "managedMdio/slice/%s" % sliceId ]
      environmentDir = driverCtx.sysdbRoot[ 'environment' ]

      assert chipConfigDir
      assert phyTopologyDir
      assert mdioEngine
      assert environmentDir

      # Setup phy coherent config generation
      coherentSliceDir = hwDir.entity[ "archer/phy/config/data/coherent/slice" ]
      coherentConfig = Fru.Dep( coherentSliceDir, invPhyDir ).newEntity(
            "Phy::Coherent::PhyCoherentConfig", sliceId )
      coherentConfig.generation = Tac.Value( "Ark::Generation",
                                             Fru.powerGenerationId( invPhyDir ),
                                             True )

      # Setup FruPlugin helper
      fruHelper = Tac.newInstance( "Denali::FruPluginHelper", "" )
      fruHelper.fruUtil = Fru.fruUtil()
      fruHelper.invDir = invPhyDir
      fruHelper.phyChipConfigDir = chipConfigDir
      fruHelper.fruEntMib = fruEntMib
      fruHelper.phyTopology = phyTopologyDir
      fruHelper.mdioEngine = mdioEngine
      fruHelper.generationId = Fru.powerGenerationId( invPhyDir )

      # Create the temp sensors for the Denali DSPs
      for phy in invPhyDir.phy.values():
         tempSensorInv = phy.tempSensor
         tempSensorId = tempSensorInv.sensorId # BUG1731
         tempSensorName = \
                EntityMib.componentNameFromParent( fruEntMib,
                                                   "TempSensor",
                                                   tempSensorId )
         TempSensorFru.createSensorEnvConfig( environmentDir[ 'temperature' ],
                                              tempSensorName, tempSensorInv )

         fruHelper.tempSensorName[ phy.name ] = tempSensorName

         if not fruEntMib:
            # Running breadth test
            continue

         # Create the sensor EntityMib object
         sensorMib = fruEntMib.sensor.get( tempSensorId )
         if sensorMib is None:
            physicalIndex = IndexAllocator.collectionItemPhysicalIndex \
                                 ( fruEntMib.sensor, tempSensorId )
            sensorMib = Fru.Dep(
               fruEntMib.sensor, tempSensorInv ).newMember(
               physicalIndex, tempSensorId, "TempSensor" )

            sensorMib.description = tempSensorInv.description
            sensorMib.groupName = tempSensorInv.groupName
            sensorMib.label = str( tempSensorId )
            # Declare success for the sensor. This MUST be done after the
            # TempSensorEnvConfig has been created
            sensorMib.initStatus = "ok"

      fruHelper.doPopulateHardware()
      self.entityMibXcvrSlotReactor = fruHelper.entityMibXcvrSlotReactor

def Plugin( context ):
   context.registerDriver( DenaliPhyDriver )
   mg = context.entityManager.mountGroup()
   mg.mount( 'hardware/phyChip/config', 'Tac::Dir', 'wi' )
   mg.mount( 'hardware/phy/topology/allPhys',
             'Hardware::PhyTopology::AllPhyIntfStatuses', 'w' )
   mg.mount( 'hardware/managedMdio', 'Tac::Dir', 'wi' )
   mg.mount( 'hardware/archer/phy/config/data/coherent/slice', 'Tac::Dir', 'wi' )
   mg.close( None )
