# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

from EntityMib import ChipType, IndexAllocator
import Fru
import Tac
import Tracing
from TypeFuture import TacLazyType

traceHandle = Tracing.Handle( 'Fru.Dpll' )
t0 = traceHandle.trace0

InvSmbusDpll = TacLazyType( 'Inventory::SmbDpll' )
PllGpioConfig = TacLazyType( 'Hardware::Dpll::PllGpioConfig' )

def maybeMarkDefaultPll( inv, config, name ):
   if inv.pllRole in [ 'pllPhy', 'pllSystem' ]:
      Fru.Dep( config.defaultPll, inv )[ name ] = True

def populatePllGpioConfig( inv: InvSmbusDpll, gpioConfig: PllGpioConfig ):
   '''
   Populate the PllGpioConfig with the systemName of the relevant GPIO pins as set
   in the inventory SmbDpll object
   '''
   invGpio = inv.smbusPllGpio
   if invGpio:
      # The systemName is used here since it is unique across all GPIO pins in the
      # system. These names are populated by the GPIO FruPlugin
      if invGpio.clockGenSmbusEnable is not None:
         gpioConfig.clockGenSmbusEnableName = invGpio.clockGenSmbusEnable.systemName
      if invGpio.clockGenStatus is not None:
         gpioConfig.clockGenStatusName = invGpio.clockGenStatus.systemName
      if invGpio.clockGenReset is not None:
         gpioConfig.clockGenResetName = invGpio.clockGenReset.systemName

modelsToOffsets = {
   'Renesas8a34001': 0,
   'lmk05318': 10,
   'lmk5b33216': 20,
   'rc32312a': 30,
}

def dpllChipId( dpllModel, localId ):
   t0( f'Generating chipId for dpll model: {dpllModel}, localId: {localId}.' )
   assert dpllModel in modelsToOffsets, f'invalid dpllModel {dpllModel}'
   assert localId < 10, 'localId must be less than 10 to avoid index collisions'
   chipIndex = modelsToOffsets[ dpllModel ] + localId
   chipId = IndexAllocator.getChipId( ChipType.dpll, chipIndex )
   t0( f'Generated chipId: {chipId}' )
   return chipId

class DpllStatusDirReactor( Tac.Notifiee ):
   """Generic Status Dir Reactor for Dpll Drivers to use"""
   notifierTypeName = "Hardware::Dpll::Status"
   pllStatusType = TacLazyType( "Hardware::Dpll::PllStatus" )

   def __init__( self, statusDir, pllNameToChipMib, versionReactor ):
      Tac.Notifiee.__init__( self, statusDir )
      self.statusDir_ = statusDir
      self.pllNameToChipMib = pllNameToChipMib
      self.versionReactor = versionReactor
      self.pllNameToVersionReactor = {}
      for status in self.notifier_.pll:
         self.handlePll( status )

   @Tac.handler( 'pll' )
   def handlePll( self, key ):
      status = self.statusDir_.pll.get( key )
      if ( not status and
           key in self.pllNameToVersionReactor ):
         t0( f"Deleting reactor for: { key }" )
         del self.pllNameToVersionReactor[ key ]
      elif key not in self.pllNameToChipMib:
         t0( f"Waiting for mapping for key { key }" )
      elif ( status and
             # pylint: disable-next=isinstance-second-argument-not-valid-type
             isinstance( status, self.pllStatusType ) and
             key not in self.pllNameToVersionReactor ):
         t0( f"Creating reactor for: { key }" )
         t0( f"Pll is : { self.pllNameToChipMib[ key ] }" )
         self.pllNameToVersionReactor[ key ] = \
                                  self.versionReactor( status,
                                                       self.pllNameToChipMib[ key ] )
