# Copyright (c) 2021 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import os
import re
import json

import Tac
import Cell
import Tracing
import Arnet
import FirmwareRev
from EntityMib import ChipType, IndexAllocator
import EosVersion
import Fru
from Fru.FruBaseVeosDriver import (
      FruBaseVeosDriver,
)
from FruPlugin.GenericPc import generateSystemMacAddr
from FruPlugin.Health import registerHealthSource
from SysConstants.if_arp_h import ARPHRD_ETHER

__defaultTraceHandle__ = Tracing.Handle( "Fru.vEOScEOSLab" )
t0 = Tracing.trace0
t8 = Tracing.trace8

uuidForTest = None
jsonIntfConfigPathForTest = None

class IntfType:
   management = 'management'
   ethernet = 'ethernet'
   #VEosLab only
   ethernetLinuxFwd = 'ethernet (linux)'
   unknown = 'unknown'

class Config:

   def getSerialNumber( self ):
      raise NotImplementedError

   def getSystemMacAddr( self ):
      raise NotImplementedError

   def getMode( self ):
      raise NotImplementedError

   def handleEtbaOverride( self ):
      raise NotImplementedError

class VEosLabCEosLabBaseDriver( FruBaseVeosDriver ):
   '''
   This base driver implements the common functionality between the cEOSLab and
   vEOSLab drivers. It acquires the appropriate entities and handles the basic
   tasks of setting the system mac address, eos version, and device name.

   cEOSLab and vEOSLab initialize their interfaces differently, so this area
   is very generic and the implementation for each intfType are contained in the
   derived types.
   '''

   def getValidDevices( self ):
      '''
      Get the set of kernel devices to be adopted by EOS.
      '''
      raise NotImplementedError

   def handleDevice( self, devName, intfType, label, subLabel, subSubLabel,
                     mac, portId, etbaConfig, ethPortDir, pciPortDir,
                     phyEthtoolPhyDir ):
      '''
      Called for each valid kernel device with the results from
      parseKernelDevices. Entities are created in sysdb for each interface here.
      '''
      raise NotImplementedError

   def setDeviceName( self, phyEthtoolPhyDir ):
      raise NotImplementedError

   def setBridging( self, etbaConfig, linuxConfig, config ):
      raise NotImplementedError

   def setHostname( self, netConfig ):
      raise NotImplementedError

   def parseKernelDevices( self, devNames, config ):
      '''
      Given the list of valid kernel devices (from getValidDevices), return a
      list of tuples of the interface type (type IntfType) and label, subLabel,
      and subSubLabel. This optionally renames the kernel devices (VEos) and is
      also where explicit device mapping takes place.
      '''
      raise NotImplementedError

   def __init__( self, fruBase, parentMibEntity, parentDriver, driverCtx,
                 modelName, description, config ):
      t0( "VEosCEosBaseDriver: Populating the Entity Mib" )
      assert parentMibEntity is None
      sysdbRoot = driverCtx.sysdbRoot
      entityMibStatus = sysdbRoot[ "hardware" ][ "entmib" ]
      entityMibStatus.fixedSystem = ( 1, 0, "FixedSystem" )
      systemMib_ = entityMibStatus.fixedSystem
      systemMib_.mfgName = "Arista"
      systemMib_.modelName = modelName
      systemMib_.modelNameExtended = modelName
      systemMib_.description = description
      systemMib_.serialNum = config.getSerialNumber()
      systemMib_.swappability = "notSwappable"
      entityMibStatus.nextPhysicalIndex = max( entityMibStatus.nextPhysicalIndex, 2 )
      config.handleEtbaOverride()

      # Emulate the FixedConfig FruPlugin, i.e.
      #  - create per-cell state ("sys/config/cell/<cellId>",
      #    "hardware/cell/<cellId>")
      #  - map the appropriate roles to our cell
      #  - set the default port roles we support
      myCellId = Cell.cellId()
      roles = [ "AllCells", "AllSupervisors", "ActiveSupervisor" ]
      Fru.createAndConfigureCell( sysdbRoot, fruBase, myCellId, roles )

      for portRole in [ "Management", "Switched" ]:
         fruBase.portRole[ portRole ] = portRole

      # Emulate a FixedConfig system with a mock single FAP instance
      chip = systemMib_.chip
      # create chip with (physicalIndex, relPos, tag)
      chipId = IndexAllocator.getChipId( ChipType.switchAsic, 0 )
      physicalIndex = IndexAllocator.collectionItemPhysicalIndex( chip, chipId )
      chipTag = f"SwitchAsic{chipId}"
      firmwareRev = FirmwareRev.abootFirmwareRev()
      m = chip.newMember( physicalIndex, chipId, chipTag )
      chipattrs =  {
         "description": f"Switching ASIC Chip {chipId}",
         "firmwareRev": firmwareRev,
         "label": str( chipId ),
         "modelName": modelName,
         "initStatus": "ok",
      }
      for attr, val in chipattrs.items():
         setattr( m, attr, val )

      systemMib_.firmwareRev = firmwareRev

      # Grab some entities we use to configure interfaces
      etbaConfig = sysdbRoot[ 'bridging' ][ 'etba' ][ 'config' ]
      linuxConfig = sysdbRoot[ 'bridging' ][ 'linux' ][ 'config' ]
      ethPortDir = fruBase.component.newEntity(
            "Inventory::VirtualEthPortDir", "ethPortDir" )
      pciPortDir = fruBase.component.newEntity(
            "Inventory::PciPortDir", "pciPortDir" )
      phyEthtoolPhyDir = fruBase.component.newEntity(
            "Inventory::Phy::EthtoolDir", "phy" )
      intfConfigDir = sysdbRoot[ 'interface' ][ 'config' ]\
            [ 'eth' ][ 'phy' ][ 'default' ]
      # Hack - change the default switched interface config
      # to be linkModeForced1GbpsFull instead of
      # linkModeForced10GbpsFull, as otherwise the PhyEthtool
      # agent will fail trying to manage front-panel ports.
      defaultSwitchedEthIntfConfig = intfConfigDir.defaultIntfConfig[
            "DefaultEthSwitchedPort" ]
      defaultSwitchedEthIntfConfig.linkModeLocal = "linkModeForced1GbpsFull"

      t0( "VEosCEosBaseDriver: Creating interfaces" )
      lowestMacAddr = None
      devNames = self.getValidDevices()
      devConfig = self.parseKernelDevices( devNames, config )
      for portId, ( devName, intfType, label, subLabel, subSubLabel ) \
            in enumerate( devConfig ):
         with open( os.path.join( '/sys/class/net', devName, "address" ) ) as f:
            mac = f.read().strip()
         t0( ( "VEosCEosBaseDriver: devName %s, intfType %s, label %d, "
               + "subLabel %d, subSubLabel %d, mac %s" )
             % ( devName, intfType, label, subLabel, subSubLabel, mac ) )
         if not lowestMacAddr or mac < lowestMacAddr:
            lowestMacAddr = mac
         self.handleDevice( devName, intfType, label, subLabel, subSubLabel,
                            mac, portId, etbaConfig, ethPortDir, pciPortDir,
                            phyEthtoolPhyDir )

      FruBaseVeosDriver.__init__( self,
                                  fruBase,
                                  parentMibEntity,
                                  parentDriver,
                                  driverCtx )
      registerHealthSource( fruBase, "vRoot" )
      # Now that we've built up the full inventory tree, run
      # drivers across it
      self.instantiateChildDrivers( fruBase, systemMib_ )

      t0( 'VEosCEosBaseDriver: Generating system MAC address' )
      # Set the system mac address. We have no choice but to choose
      # this randomly - see the comment in generateSystemMac for details.
      #
      # We allow the mac address to be overwritten by creating the
      # file /mnt/flash/system_mac_address. This gives us a hook in
      # case the automatically generated mac address is not
      # sufficient. Care must be taken when cloning systems with
      # a /mnt/flash/system_mac_address file.
      systemMacAddr = config.getSystemMacAddr()
      if lowestMacAddr and not systemMacAddr:
         systemMacAddr = generateSystemMacAddr( lowestMacAddr )
      if systemMacAddr:
         entityMibStatus.systemMacAddr = systemMacAddr
         if systemMib_.serialNum == "":
            t0( "VEosCEosBaseDriver: Setting serialNum from system mac ",
                systemMacAddr )
            serialGenerator = Tac.newInstance( 'License::SerialNumberGenerator' )
            if uuidForTest is not None:
               serialGenerator.uuidForTest = uuidForTest
            serialNum = serialGenerator.getSerialNumberFromMac( systemMacAddr )
            systemMib_.serialNum = serialNum

      t0( 'VEosCEosBaseDriver: Generating EOS version' )
      vi = EosVersion.VersionInfo( sysdbRoot=None )
      if vi.version() is None:
         systemMib_.softwareRev = ""
      else:
         systemMib_.softwareRev = vi.version()

      t0( 'VEosCEosBaseDriver: Setting deviceName in PhyEthtool' )
      phyEthtoolPhyDir = fruBase.component.get( "phy" )
      self.setDeviceName( phyEthtoolPhyDir )

      t0( 'VEosCEosBaseDriver: Setting bridging' )
      self.setBridging( etbaConfig, linuxConfig, config )

      t0( 'VEosCEosBaseDriver: Setting hostname' )
      netConfig = sysdbRoot[ 'sys' ][ 'net' ][ 'config' ]
      self.setHostname( netConfig )

      t0( 'VEosCEosBaseDriver: Finalizing' )
      hwCellDir = sysdbRoot[ 'hardware' ][ 'cell' ][ '%d' % myCellId ]
      assert hwCellDir
      hwCellDir.newEntity( 'Tac::Dir', 'FruReady' )

      # Declare success at the end
      systemMib_.initStatus = "ok"

   @staticmethod
   def getDevices( validDevicePfxs=None ):
      '''
      Gathers a list of kernel devices to be adopted by EOS.
      This is used by both cEOS-lab and vEOS-lab.
      Intended as a helper method for getValidDevices, which is virtual.

      Arguments:
       - validDevicePfxs: a list of strings used to filter the intf names to
                          the subset of those to be considered. This is done
                          by checking the prefix.
      Returns:
         List of strings (device names)
      '''
      t0( 'Getting interfaces' )
      devNames = []
      # Get all ethernet interface device names
      sysClassNet = os.path.join( os.path.abspath( os.sep ), 'sys', 'class', 'net' )
      for devName in os.listdir( sysClassNet ):
         if not os.path.isdir( os.path.join( sysClassNet, devName ) ):
            # Possibly a file, like "bonding_masters".
            continue
         with open( os.path.join( sysClassNet, devName, "type" ) ) as f:
            devType = int( f.read() )
         # This type apparently corresponds to include/linux/if_arp.h
         # ARPHRD_* values.
         t8( devName, 'is type', devType )
         validDevice = ( any( devName.startswith( devNamePfx )
                            for devNamePfx in validDevicePfxs )
                         if validDevicePfxs else True )
         if devType != ARPHRD_ETHER or not validDevice:
            t8( 'skipping', devName, 'devType good =', devType == ARPHRD_ETHER,
                                     'validDevice =', validDevice,
                                     'validDevicePfxs =', validDevicePfxs )
            continue
         devNames.append( devName )
      return devNames

   @staticmethod
   def getLabels( intfOrDevName ):
      '''
      Expand name into its type and labels.
      Normalize slashes to underscores so this works with both device names
         and interface names (eth1_2_3, Ethernet1/2/3).
      '''
      intfOrDevName = intfOrDevName.replace( '/', '_' )
      rexpr = r"(\D+)(\d+)(?:_(\d+)(?:_(\d+))?)?$"
      rexprMatch = re.match( rexpr, intfOrDevName )
      if not rexprMatch:
         t0( 'Unsupported device %s' % intfOrDevName )
         return None
      intfOrDevType, label, subLabel, subSubLabel = rexprMatch.groups()
      subLabel = subLabel if subLabel is not None else 0
      subSubLabel = subSubLabel if subSubLabel is not None else 0
      return intfOrDevType, int( label ), int( subLabel ), int( subSubLabel )

   @staticmethod
   def intfTypeFromDevType( devType, config, ethernetPrefixes ):
      '''
      Determine the desired interface type from the name of the kernel device.
      '''
      intfTypeMap = {
            'ma': IntfType.management,
      }
      for etPfx in ethernetPrefixes:
         intfTypeMap [ etPfx ] = ( IntfType.ethernet if config.getMode() == 'test'
                                   else IntfType.ethernetLinuxFwd )
      return intfTypeMap.get( devType, IntfType.unknown )

   @staticmethod
   def intfTypeFromDesiredIntfPrefix( intfPrefix, config ):
      '''
      If there is a json intf mapping, determine the desired interface type from
      the desired interface name.
      '''
      intfTypeMap = {
            'Management': IntfType.management,
            'Ethernet': IntfType.ethernet if config.getMode() == 'test'
                  else IntfType.ethernetLinuxFwd,
      }
      return intfTypeMap[ intfPrefix ]


# Inside a namespace dut, we need to look at FILESYSTEM_ROOT
defaultJsonIntfConfigPath=os.path.join( os.environ.get( 'FILESYSTEM_ROOT', '/mnt' ),
                                        "flash/EosIntfMapping.json" )

class JsonIntfConfig:
   def __init__( self, jsonIntfConfigPath=defaultJsonIntfConfigPath ):
      self.jsonIntfConfigPath = jsonIntfConfigPath
      if jsonIntfConfigPathForTest:
         self.jsonIntfConfigPath = jsonIntfConfigPathForTest
      self.intfConfig = None
      self.getIntfOverrides()

   def getIntfOverrides( self ):
      if os.path.exists( self.jsonIntfConfigPath ):
         with open( self.jsonIntfConfigPath ) as f:
            try:
               intfConfig = json.loads( f.read() )
            except ValueError as e:
               t0( "Interface mapping is not valid json: %s" % e )
               return False
         if self.validate( intfConfig ):
            t0( "Found valid Interface mapping:", self.jsonIntfConfigPath )
            self.intfConfig = intfConfig
            return True
         else:
            t0( "Invalid Interface mapping file", self.jsonIntfConfigPath )
            return False
      else:
         t0( "Intface mapping file", self.jsonIntfConfigPath, "does not exist" )
         return False

   def validate( self, intfMapConfig ):
      try:
         for interfaceType in intfMapConfig:
            if interfaceType not in [ 'EthernetIntf', 'ManagementIntf' ]:
               t0( 'Invalid interface type %s' % interfaceType )
               return False
            for eosName in intfMapConfig[ interfaceType ].values():
               requiredPfx = ( 'Ethernet' if interfaceType == 'EthernetIntf'
                               else 'Management' )
               if not eosName.startswith( requiredPfx ):
                  t0( 'Invalid interface name %s in section %s'
                      % ( eosName, interfaceType ) )
                  return False
               try:
                  Arnet.IntfId( eosName )
               except IndexError:
                  t0( '%s is not a valid EOS IntfName' % eosName )
                  return False
      # There a couple of ways dicts in intfMapConfig could be messed up,
      # hence the broader exception catch. just log the config as invalid
      # and move on
      except Exception as e: # pylint: disable=W0703
         t0( 'Invalid interface mapping configuration: %s' % e )
         return False

      return True

   # Flattens the the Config into just name -> override
   def getNameMapping( self ):
      if self.intfConfig is None:
         return None
      mapping = {}
      for intfType in self.intfConfig:
         mapping.update( self.intfConfig[ intfType ] )
      return mapping
