# Copyright (c) 2006-2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

import json
import os
import re

from AsuPStore import getAsuPStorePath
from EthIntf import MAX_SUPPORTED_MTU
from TypeFuture import TacLazyType
import Fru
from FruPlugin import (
   L1IntfFactory,
   L1PortDriver,
)
import Tac
import Tracing

__defaultTraceHandle__ = Tracing.Handle( "Fru.EthPort" )
t0 = Tracing.trace0
t1 = Tracing.trace1

ethIntfConfigSliceDir = None
ethIntfStatusSliceDir = None
ethIntfDefaultConfigDir = None
ethIntfSliceDefaultConfigDir = None
ethIntfCapabilitiesConfigSliceDir = None
ethPhyAutonegConfigSliceDir = None
ethPhyIntfDescSliceDir = None
sysName = None
ethLinkModes = Tac.Type( "Interface::EthLinkMode" )
DescType = TacLazyType( 'Interface::DescType' )

# Early reading of the PSTORE - this happens well before the PStore restorer
# code runs. Seconds before. Need to do it now so we get right info for MTU
# before early agents try using it. Not a problem in cold boot but when we do
# ASU boot connected ports are already up (thanks to # AsuFastPktRestore)
# before the correct information is written into Sysdb.
def getEthIntfPStoreMtus():
   mtuConfigs = {}
   cmdLinePath = os.environ.get( "ASU_CMDLINE_PATH", "/proc/cmdline" )
   with open( cmdLinePath ) as cmdFile:
      m = re.search( r"arista\.asu_(reboot|hitless)", cmdFile.read() )
   if m is not None:
      t0( "ASU Reboot" )
      cfgPath = getAsuPStorePath()
      try:
         with open( cfgPath ) as f:
            jsonConfig = json.load( f )
            ethIntfCfg = jsonConfig.get( "EthIntf", None )
            if ethIntfCfg is not None:
               mtuConfigs = ethIntfCfg.get( "intfMtus", {} )
               if not mtuConfigs:
                  t0( "intfMtus not in PSTORE or is empty. Using defaults." )
            else:
               t0( "EthIntf not in PSTORE. Using defaults." )
      except ValueError:
         t0( "File %s not valid JSON format. Using defaults." % ( cfgPath ) )
      except OSError as e:
         t0( "Could not read %s: %s. Using defaults." % ( cfgPath, e ) )
   else:
      t0( "Not an ASU reboot. Not looking at PSTORE. Using defaults." )
   return mtuConfigs

class EthIntfFactory( L1IntfFactory.L1IntfFactory ):
   """Ultimately provides createIntfStatus() and handlePortRole() methods
   for creating the appropriate EthIntfStatus object for a Fru::Port.
   This class also is responsible for creating the default EthIntfConfig objects,
   and thus is a reactor to the Fru::Status in order to be notified when the
   supported port roles are set """

   managedTypeName = "Inventory::EthPort"
   managedPortRoles = [ "Switched", "Internal" ]

   EthTypesApi = Tac.Type( "Interface::EthTypesApi" )

   mtuConfigs = getEthIntfPStoreMtus()

   @staticmethod
   def defaultSwitchedIntfConfigName( linkMode ):
      return EthIntfFactory.EthTypesApi.linkModeToDefaultSwitchedIntfConfig(
            linkMode )

   @staticmethod
   def getEthIntfDefaultConfigDir():
      return ethIntfDefaultConfigDir.defaultIntfConfig

   def defaultIntfConfigName( self, port ):
      if port.role == "Switched":
         return EthIntfFactory.defaultSwitchedIntfConfigName(
            port.defaultLinkMode )
      else:
         return 'DefaultEth%sPort' % port.role

   @staticmethod
   def getIntfCfgMtu( intfName ):
      mtuCfg = MAX_SUPPORTED_MTU
      intfCfg = EthIntfFactory.mtuConfigs.get( intfName, None )
      if intfCfg is not None:
         mtuCfg = intfCfg.get( "mtu", MAX_SUPPORTED_MTU )
      return mtuCfg
   
   def createIntfStatus( self, fruBase, port, intfName ):
      super().createIntfStatus( fruBase, port, intfName )

      t0( "Create EthIntfStatus for", intfName )
      if fruBase.managingCellId != 0:
         sliceName = str( fruBase.managingCellId )
         capsSliceName = 'FixedSystem'
         phyAnConfigSliceName = 'FixedSystem'
      else:
         sliceName = fruBase.sliceId
         capsSliceName = sliceName
         phyAnConfigSliceName = sliceName

      # Create the intfStatus object under the corresponding slice dir
      # Create the slice dir first if it does not already exist
      # XXX_BUG96584
      # For breadth tests, if the celltype is not explicitly set to 
      # 'supervisor', we assume its a fixed system and set the slice
      # name to the cell id
      # This is a temporary change till we can figure out how to set 
      # celltype to 'supervisor' in breadth tests
      # Currently if we attempt to set the CELLTYPE before calling
      # Artest.beginTest(), we hit an exception
      import Cell # pylint: disable=import-outside-toplevel
      if os.environ.get( "SIMULATION_VMID" ) and\
         os.environ.get( "CELLTYPE" ) != 'supervisor':
         sliceName = str( Cell.cellId() )
         t0( "Forcing sliceName to", sliceName, "for breadth tests" )

      # Create the slice for the EthPhyIntfDesc entity
      ethPhyIntfDescDir = ethPhyIntfDescSliceDir.get( sliceName )
      if not ethPhyIntfDescDir:
         Fru.Dep( ethPhyIntfDescSliceDir, port ).newEntity(
               'Interface::EthPhyIntfDescDir', sliceName )
         t0( "Created status slice dir for", sliceName )
         ethPhyIntfDescDir = ethPhyIntfDescSliceDir[ sliceName ]
         ethPhyIntfDescDir.generation = Tac.Value(
               "Ark::Generation", Fru.powerGenerationId( fruBase ), True )

      intfStatusDir = ethIntfStatusSliceDir.get( sliceName )
      if not intfStatusDir:
         Fru.Dep( ethIntfStatusSliceDir, port ).newEntity(
            'Interface::EthPhyIntfStatusDir', sliceName )
         t0( "Created status slice dir for ", sliceName )
         intfStatusDir = ethIntfStatusSliceDir[ sliceName ]

      ethIntfCapConfig = ethIntfCapabilitiesConfigSliceDir.get( capsSliceName )
      if not ethIntfCapConfig:
         # Create CapabilitiesConfig
         Fru.Dep( ethIntfCapabilitiesConfigSliceDir, fruBase ).newEntity(
            'Interface::Capabilities::CapabilitiesConfig',
            capsSliceName )
         ethIntfCapConfig = ethIntfCapabilitiesConfigSliceDir[ capsSliceName ]
         ethIntfCapConfig.generation = Tac.Value( "Ark::Generation",
                                                  Fru.powerGenerationId( fruBase ),
                                                  True )
      ethPhyAnConfig = ethPhyAutonegConfigSliceDir.get( phyAnConfigSliceName )
      if not ethPhyAnConfig:
         # Create PhyAutonegConfig
         Fru.Dep( ethPhyAutonegConfigSliceDir, fruBase ).newEntity(
                  'Interface::Autoneg::PhyAutonegConfig', phyAnConfigSliceName )
         ethPhyAnConfig = ethPhyAutonegConfigSliceDir[ phyAnConfigSliceName ]
         ethPhyAnConfig.generation = Tac.Value( 'Ark::Generation',
                                                Fru.powerGenerationId( fruBase ),
                                                True )

      self.setL1CardPowerGeneration( fruBase )

      t0( "intfStatusDir is ", intfStatusDir,
          "; ethIntfCapConfig is ", ethIntfCapConfig,
          "; ethIntfCapConfig.gen ", ethIntfCapConfig.generation,
          "; ethPhyAnConfig is ", ethPhyAnConfig,
          "; ethPhyAnConfig.gen ", ethPhyAnConfig.generation,
          "; ethPhyIntfDescDir is", ethPhyIntfDescDir,
          "; descStatusDir.gen is ", ethPhyIntfDescDir.generation )

      # Check if the interface already exists in the instantiating
      # collection. For why we should check the instantiating collection
      # only and not one of the *all* collections, see BUG94023
      if intfName not in intfStatusDir.intfStatus:
         dsicn = self.defaultIntfConfigName( port )
         d = ethIntfDefaultConfigDir.defaultIntfConfig.get( dsicn )
         t0( 'Creating new intfStatus for', intfName )
         intf = Fru.Dep( intfStatusDir.intfStatus, port ).newMember(
            intfName, d, Fru.powerGenerationId( port ), port.macAddr,
            port.relativeIfindex )

         intf.mtu = EthIntfFactory.getIntfCfgMtu( intfName )
         t0( "Setting MTU for %s to %d" % ( intfName, intf.mtu ) )
         intf.maxMtu = MAX_SUPPORTED_MTU
      else:
         t0( 'intfStatus for', intfName, 'already exists' )
         intf = intfStatusDir.intfStatus[ intfName ]

      # Create the EthPhyIntfDesc entity in the corresponding slice
      # mount point
      if intfName not in ethPhyIntfDescDir.ethPhyIntfDesc:
         dscin = EthIntfFactory.defaultSwitchedIntfConfigName(
            port.defaultLinkMode ) if port.role == "Switched" else \
                                          ( 'DefaultEth%sPort' % port.role )
         desc = Fru.Dep( ethPhyIntfDescDir.ethPhyIntfDesc, port ).newMember(
                                    intfName,
                                    ethPhyIntfDescDir.generation,
                                    port.macAddr, port.relativeIfindex,
                                    dscin,
                                    self.getDescType( port ) )
      else:
         t0( 'EthPhyIntfDesc for %s already exists' % intfName )
         desc = ethPhyIntfDescDir.ethPhyIntfDesc[ intfName ]

      desc.mtu = EthIntfFactory.getIntfCfgMtu( intfName )
      desc.maxMtu = MAX_SUPPORTED_MTU

      # If we stop creating an EthPhyIntfStatus in interface/status/eth/phy,
      # we should return desc here (uncomment the next line)

      # return desc

      return intf

   @staticmethod
   def setExtraDefaultIntfConfig( deic ):
      """Set additional attributes for the Interface::EthPhyIntfDefaultConfig
      entities associated with the port role(s) managed by this class. Some
      attributes are set from self.maybeAddNewDefaultIntfConfig().
      """

   def handlePortRole( self, portRole ):
      # For each port role that we know about, create a default eth intf config.
      # XXX This assumes, perhaps incorrectly, that roles cannot be removed from
      # the list.

      assert portRole is not None 
      if portRole in EthIntfFactory.managedPortRoles:
         if portRole == "Switched":
            for linkMode in ethLinkModes.attributes:
               dsicn = EthIntfFactory.defaultSwitchedIntfConfigName( linkMode )
               EthIntfFactory.maybeAddNewDefaultIntfConfig( dsicn,
                                                            linkMode=linkMode )
         else:
            EthIntfFactory.maybeAddNewDefaultIntfConfig(
               'DefaultEth%sPort' % portRole )
      else:
         # We don't manage this port role. Ignore.
         pass

class EthPortDirDriver( L1PortDriver.L1PortDriver ):
   managedTypeName = "Inventory::EthPortDir"
   managedApiRe = "$"

   def __init__( self, invPortDir, fruEntMib, parentDriver, driverCtx ):

      super().__init__( invPortDir,
                        fruEntMib,
                        parentDriver,
                        driverCtx )
      self.initL1Ports( list( invPortDir.port.values() ),
                        fruEntMib,
                        parentDriver,
                        driverCtx,
                        markBootstrapPorts=True,
                        applyL1Profiles=True )

   def portNamePrefix( self, port ):
      if port.role in EthIntfFactory.managedPortRoles:
         if port.role == "Internal":
            return "Internal"
         elif port.isUnconnected:
            return "UnconnectedEthernet"
         else:
            return "Ethernet"
      elif port.role in [ "FrontPanelFabric", "Fabric" ]:
         return "Fabric"
      else:
         raise Exception( "Unknown port role %s" % port.role )

class EthPortMacUpdateDriver( Fru.FruDriver ):
   managedTypeName = "Inventory::EthPortMacUpdate"

   def __init__( self, inv, parentMib, parentDriver, ctx ):
      t0( "Creating an EthPortMacUpdateDriver Fru driver for dir", inv.name )
      Fru.FruDriver.__init__( self, inv, parentMib, parentDriver, ctx )

      intfStatusDir = ethIntfStatusSliceDir.get( inv.sliceId )
      ethPhyIntfDescDir = ethPhyIntfDescSliceDir.get( inv.sliceId )
      for port, addr in inv.macAddr.items():
         intfStatusDir.intfStatus[ port ].burnedInAddr = addr
         ethPhyIntfDescDir.ethPhyIntfDesc[ port ].burnedInAddr = addr

   def __del__( self ):
      inv = self.invEntity_
      t0( "Deleting an EthPortMacUpdateDriver Fru driver for dir", inv.name )

      intfStatusDir = ethIntfStatusSliceDir.get( inv.sliceId )
      ethPhyIntfDescDir = ethPhyIntfDescSliceDir.get( inv.sliceId )
      for port in inv.macAddr:
         emptyAddr = '00:00:00:00:00:00'
         intfStatusDir.intfStatus[ port ].burnedInAddr = emptyAddr
         ethPhyIntfDescDir.ethPhyIntfDesc[ port ].burnedInAddr = emptyAddr

      Fru.FruDriver.__del__( self )

class EthIntfSliceDefaultConfigDriver( Fru.FruDriver ):
   managedTypeName = "Inventory::EthIntfSliceDefaultConfig"

   def __init__( self, inv, parentMib, parentDriver, ctx ):
      t0( "Creating an EthIntfSliceDefaultConfig Driver Fru driver", inv.name )
      Fru.FruDriver.__init__( self, inv, parentMib, parentDriver, ctx )

      defaultConfig = ethIntfSliceDefaultConfigDir.defaultConfig. \
            newMember( inv.name )
      if inv.qsfpDefaultMode == Tac.Type( "Inventory::XcvrMode" ).xcvrMode4x10G:
         defaultConfig.qsfpDefaultMode = \
               Tac.Type( "Interface::XcvrMode" ).xcvrMode4x10G
      else:
         defaultConfig.qsfpDefaultMode = Tac.Type( "Interface::XcvrMode" ).noXcvrMode
      t0( "Creating an EthIntfSliceDefaultConfig Driver Fru driver", inv.sliceId )

   def __del__( self ):
      inv = self.invEntity_
      t0( "Deleting an EthIntfSliceDefaultConfig Driver Fru driver", inv.name )

      del ethIntfSliceDefaultConfigDir.defaultConfig[ inv.name ]
      Fru.FruDriver.__del__( self )

_ethIntfFactory = {}
def Plugin( context ):
   context.registerDriver( EthPortDirDriver )
   context.registerDriver( EthPortMacUpdateDriver )
   context.registerDriver( EthIntfSliceDefaultConfigDriver )

   em = context.entityManager
   mountGroup = em.mountGroup()
   
   # Have Fru mount EthIntf state if it hasn't done so already
   hardwareInventory = mountGroup.mount( 'hardware/inventory', 'Tac::Dir', 'wi' )

   mountGroup.mount( 'interface/config/eth/phy', 'Tac::Dir', 'wi' )
   mountGroup.mount( 'interface/status/eth/phy', 'Tac::Dir', 'wi' )

   global ethIntfConfigSliceDir, ethIntfStatusSliceDir
   global ethPhyIntfDescSliceDir
   global ethIntfDefaultConfigDir, sysName
   global ethIntfSliceDefaultConfigDir
   global ethIntfCapabilitiesConfigSliceDir
   global ethPhyAutonegConfigSliceDir
   ethIntfConfigSliceDir = mountGroup.mount( 'interface/config/eth/phy/slice',
                                             'Tac::Dir', 'w' )
   ethIntfStatusSliceDir = mountGroup.mount( 'interface/status/eth/phy/slice',
                                             'Tac::Dir', 'w' )
   ethPhyIntfDescSliceDir = mountGroup.mount( 'interface/archer/status/init/eth/'
                                              'phy/slice',
                                              'Tac::Dir', 'wi' )
   ethIntfDefaultConfigDir = mountGroup.mount(
                                  'interface/config/eth/phy/default',
                                  'Interface::EthPhyIntfDefaultConfigDir', 'w' )
   ethIntfSliceDefaultConfigDir = mountGroup.mount(
                                  'interface/archer/config/eth/phy/sliceDefault',
                                  'Interface::EthIntfSliceDefaultConfigDir', 'w' )

   # The above will go away once we finish single-writer work.
   # Create the "config" which has one purpose - to hold the generationId
   # used to qualify the capabilities input provided by other agents.
   ethIntfCapabilitiesConfigSliceDir = mountGroup.mount(
         'interface/archer/config/eth/capabilities/slice',
         'Tac::Dir', 'wi' )
   ethPhyAutonegConfigSliceDir = mountGroup.mount(
         'interface/archer/config/eth/phy/autoneg/slice',
         'Tac::Dir', 'wi' )
   sysName = context.sysname

   def _finished():
      if context.sysname not in _ethIntfFactory:
         _ethIntfFactory[ context.sysname ] = EthIntfFactory(
               hardwareInventory, None, None )

      # Register PortIntfFactory for ethernet switch ports
      context.registerPortIntfFactory( _ethIntfFactory[ context.sysname ] )


   mountGroup.close( _finished )
