#!/usr/bin/env python3
# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Cell
import DesiredTracing
from FabricIntfLib import Constants
import Fru
from FruPlugin import L1IntfFactory
from FruPlugin.EthPort import EthIntfFactory
import Tac
import Tracing
from TypeFuture import TacLazyType

__defaultTraceHandle__ = Tracing.Handle( "Fru.FabricIntfFactory" )
DesiredTracing.desiredTracingIs( 'Fru.FabricIntfFactory/0' )

t0 = Tracing.trace0

DescType = TacLazyType( "Interface::DescType" )
ethLinkModes = Tac.Type( "Interface::EthLinkMode" )

fabIntfDescSlotDir = None
fabIntfDefaultConfigDir = None

class FabricIntfFactory( L1IntfFactory.L1IntfFactory ):
   managedTypeName = "Inventory::EthPort"
   managedPortRoles = [ "FrontPanelFabric", "Fabric" ]

   @staticmethod
   def defaultIntfConfigName( linkMode ):
      return EthIntfFactory.EthTypesApi.linkModeToDefaultSwitchedIntfConfig(
            linkMode ).replace( "Switched", "Fabric" )

   @staticmethod
   def getEthIntfDefaultConfigDir():
      return fabIntfDefaultConfigDir.defaultIntfConfig

   @staticmethod
   def setExtraDefaultIntfConfig( deic ):
      """Set additional attributes for the Interface::EthPhyIntfDefaultConfig
      entities associated with the port role(s) managed by this class. Some
      attributes are set from self.maybeAddNewDefaultIntfConfig().
      """

   def handlePortRole( self, portRole ):
      assert portRole is not None
      if portRole in FabricIntfFactory.managedPortRoles:
         assert portRole in [ "FrontPanelFabric", "Fabric" ]
         for linkMode in ethLinkModes.attributes:
            FabricIntfFactory.maybeAddNewDefaultIntfConfig(
               FabricIntfFactory.defaultIntfConfigName( linkMode ),
               linkMode )
      else:
         # We do not manage this role.
         pass

   # NOTE: This does NOT create any EthPhyIntfStatus object. The method must be
   # named this way to to the PortIntfFactory interface but this will
   # only create an EthPhyIntfDesc object and add an IntfDefaultPortConfig object.
   def createIntfStatus( self, fruBase, port, intfName ):
      super( FabricIntfFactory, self ).createIntfStatus(
            fruBase, port, intfName )

      t0( 'Create EthPhyIntfDesc for ', intfName )
      cardSlotId = fruBase.sliceId
      if not cardSlotId:
         assert Cell.cellType() == "fixed"
         cardSlotId = "FixedSystem"

      self.setL1CardPowerGeneration( fruBase )

      # Create the slot dir for EthPhyIntfDesc.
      ethPhyIntfDescDir = fabIntfDescSlotDir.get( cardSlotId )
      if not ethPhyIntfDescDir:
         ethPhyIntfDescDir = Fru.Dep( fabIntfDescSlotDir, fruBase ).newEntity(
               "Interface::EthPhyIntfDescDir", cardSlotId )
         t0( 'Created desc slot dir for ', cardSlotId )
         ethPhyIntfDescDir.generation = Tac.Value(
            "Ark::Generation", Fru.powerGenerationId( fruBase ), True )
         t0( 'ethPhyIntfDesc dir is ', ethPhyIntfDescDir,
             '; ethPhyIntfDescDir generation is ', ethPhyIntfDescDir.generation )

      # Create the EthPhyIntfDesc for the interface.
      dicn = FabricIntfFactory.defaultIntfConfigName( port.defaultLinkMode )
      epiDesc = Fru.Dep( ethPhyIntfDescDir.ethPhyIntfDesc, port ).newMember(
                  intfName,
                  ethPhyIntfDescDir.generation,
                  port.macAddr,
                  port.relativeIfindex,
                  dicn,
                  self.getDescType( port ) )
      epiDesc.deviceName = cardSlotId
      t0( 'Created EthPhyIntfDesc for ', intfName, ' ; defaultIntfConfig is ',
          dicn, ' ; linkMode is ', port.defaultLinkMode )

_fabricIntfFactory = {}
def Plugin( context ):
   em = context.entityManager
   mountGroup = em.mountGroup()

   hardwareInventory = mountGroup.mount( "hardware/inventory",
                                         "Tac::Dir", 'wi' )

   global fabIntfDescSlotDir, fabIntfDefaultConfigDir
   fabIntfDescSlotDir = mountGroup.mount( Constants.intfDescRootPath(),
                                          'Tac::Dir', 'wi' )
   fabIntfDefaultConfigDir = mountGroup.mount(
         Constants.intfDefaultConfigPath(),
         'Interface::FabricIntfDefaultConfigDir', 'wi' )

   def _finished():
      if context.sysname not in _fabricIntfFactory:
         _fabricIntfFactory[ context.sysname ] = FabricIntfFactory(
            hardwareInventory, None, None )

      context.registerPortIntfFactory( _fabricIntfFactory[ context.sysname ] )

   mountGroup.close( _finished )
