# Copyright (c) 2006-2010, 2011 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

# This plugin provides a FruFactory that is used when the scd cannot
# be read.  With no scd, we assume we are running on a generic
# PC, and synthesize FDL (or at least the effect of running FDL) based
# on what we can read out of the Linux kernel. We assume that we're on a
# fixed system. Interfaces that begin with 'ma' are assumed to be
# Management Ethernet Interfaces. All other interfaces are ignored.

import ctypes
import glob
import os
import random
import re

import Ark
import ArPyUtils
import Cell
import FirmwareRev
import Fru
import Tac
import Tracing
import VeosHypervisor

from Fru.FruBaseVeosDriver import parseVeosConfig
from FruPlugin.Health import registerHealthSource
from SysConstants.if_arp_h import ARPHRD_ETHER

# Force a dependency on the RPM that provides the Sysdb paths
# for routing/hardware/status and routing6/hardware/status
# pkgdeps: rpmwith %{_libdir}/preinit/SysdbIra

__defaultTraceHandle__ = Tracing.Handle( "Fru.GenericPc" )
t0 = Tracing.trace0
t2 = Tracing.trace2
t9 = Tracing.trace9

# Normally renameKernelDevices renames all the kernel devices, but this makes it
# impossible to run in a breadh test since it will cannibalize all the network
# devices on the server. In these cases we will just supply the specific interfaces
# we indent to rename.
devNamesForTest = None

# Similarly, for testing, we can point to a test directory hierarchy
sysClassNet = "/sys/class/net"

def genericPcFactory( parent, fdl, idInParent ):
   t2( "Eos/FruPlugin/GenericPc.genericPcFactory: creating Eos::FruGenericPc" )
   assert idInParent is None
   genericPc = parent.newEntity( "Eos::GenericPcFru", "genericPc" )
   genericPc.component = ( "component", )

   # Set the API as the platform, to allow us to perform other actions on
   # platforms such as vEOS
   genericPc.api = Ark.getPlatform() or ""
   # managingCellId must be consistent with the cell id identifyCell assigns
   # in the GenericPc case
   genericPc.managingCellId = Cell.cellId()
   genericPc.valid = True
   genericPc.generationId = 1
   return genericPc

def doGetDriver( devName ):
   try:
      output = Tac.run( [ "ethtool", "-i", devName ], stdout=Tac.CAPTURE )
   except Tac.SystemCommandError as e:
      t0( "ethtool failed:", e )
      return None
   m = re.search( "driver: (.*)", output )
   if m:
      return m.group( 1 )
   return None

def isMellanoxDriver( devName ):
   return doGetDriver( devName ) in [ "mlx4_en", "mlx5_core" ]

def getRealDevName( devName ):
   loPath = os.path.join( "/sys/class/net", devName, "lower_*" )
   paths = glob.glob( loPath )
   if not paths:
      return None
   return paths[ 0 ].split( "lower_" )[ 1 ]

def _renameInterfacePhase1( oldDevName, newDevName ):
   if newDevName == oldDevName:
      return
   # Clean IP configuration we may have from pre-EOS DHCP
   Tac.run( [ "ip", "addr", "flush", "dev", oldDevName ] )

   # Rename interface
   Tac.run( [ "ip", "link", "set", "dev", oldDevName, "down" ] )
   Tac.run( [ "ip", "link", "set", "dev", oldDevName,
              "name", oldDevName + "tmp" ] )
   t0( "renameInterfacePhase1: Old = %r, New = %r" % ( oldDevName, newDevName ) )

def _renameInterfacePhase2( oldDevName, newDevName ):
   if newDevName == oldDevName:
      return
   # rename interface to their final name and bring them back up
   Tac.run( [ "ip", "link", "set", "dev", oldDevName + "tmp",
              "name", newDevName ] )
   Tac.run( [ "ip", "link", "set", "dev", newDevName, "up" ] )
   t0( "renameInterfacePhase2: Old = %r, New = %r" % ( oldDevName, newDevName ) )

def sortKernelDevices( devNames ):
   # Sort device names by acpi_index, to use the information that
   # vmware gives us, then by pci address and as last resort by
   # the original kernel names.  This may be different
   # from Aboot-veos, since that does not use acpi_index.
   address = {}
   acpi_index = {}
   for devName in devNames:
      path = os.path.realpath(
         os.path.join( sysClassNet, devName, "device" ) )
      addr = re.match( "/sys/devices/(.*)", path ).group( 1 )
      # On Azure the device paths contain random UUID-like parts which make
      # them not usable for sorting so overwrite them with fixed value to
      # skip sorting on this field.
      if VeosHypervisor.platformAzure():
         addr = 'x'
      # The PCI path can be pci*/pci-bridge/device.  If it's a pci device,
      # we remove the bridge ID from the middle; the device portion itself
      # will have a bus ID which distinguishes it, and having the bridge
      # ID in the middle foils our attempt at sorting.  It can also include
      # a virtio suffix.
      if addr.startswith( 'pci' ):
         # We ignore the virtio suffix completely, by keeping only the path
         # segments that contain a colon
         pciPath = [ seg for seg in addr.split( '/' ) if ':' in seg ]
         addr = '/'.join( ( pciPath[ 0 ], pciPath[ -1 ] ) )
      address[ devName ] = addr
      try:
         with open( os.path.join( path, "acpi_index" ) ) as f:
            acpi_index[ devName ] = int( f.read() )
      except OSError:
         # no acpi_index file
         pass
   t9( 'Address map:', address )
   t9( 'ACPI indices:', acpi_index )

   # NO_ACPI_INDEX is an attempt to make a device with no ACPI index
   # supplied sort after any that have valid ACPI index values.
   NO_ACPI_INDEX = 2147483647
   devNames.sort( key=lambda d: ( acpi_index.get( d, NO_ACPI_INDEX ),
                                  address[ d ], d ) )

   t9( "VEosDriver: sorted interfaces:", devNames )

   return devNames

def renameKernelDevices( devNames, veosMode ):
   if devNamesForTest is not None:
      devNames = devNamesForTest
   elif ( os.environ.get( "P4USER" ) or
        os.environ.get( "ABUILD" ) or
        os.environ.get( "A4_CHROOT" ) ):
      # Yikes - we seem to be on a build server, don't do anything
      t9( "VEosDriver - interface rename aborted since we seem "
          "to be on a build server" )
      return devNames

   if veosMode not in [ "test", "linux" ]:
      t0( "Unknown vEOS mode. Skipping interface rename" )
      return devNames

   # Remove mlx devices from list. They will be handled as "lower"
   # of their non-mlx counterparts later.
   t0( "VEosDriver: Filtering out Mellanox devices" )
   devNames = [ d for d in devNames if not isMellanoxDriver( d ) ]

   t0( "VEosDriver: Renaming Interfaces for mode", veosMode )

   devNames = sortKernelDevices( devNames )

   # Rename interfaces based on the configured vEOS mode.
   if veosMode == "test":
      frontPanelPrefix = "vmnicet"
   else:
      frontPanelPrefix = "et"
   oldToNewDeviceName = {}
   realDevs = {}
   portId = 0
   for devName in devNames:
      realDevs[ devName ] = getRealDevName( devName )
      if portId == 0:
         oldToNewDeviceName[ devName ] = "ma1"
      else:
         oldToNewDeviceName[ devName ] = "%s%d" % ( frontPanelPrefix, portId )
      portId += 1
   # First bring down all devices to be renamed and name them
   # something temporary to avoid conflicts
   for oldDevName, newDevName in oldToNewDeviceName.items():
      _renameInterfacePhase1( oldDevName, newDevName )
      # Handle lower_ethX mlx interface
      if realDevs[ oldDevName ]:
         _renameInterfacePhase1( realDevs[ oldDevName ], "mlx_" + newDevName )
   # Now rename interfaces to their final name and bring them
   # back up
   for oldDevName, newDevName in oldToNewDeviceName.items():
      # Handle lower_ethX mlx interface before the main one to avoid EBUSY
      if realDevs[ oldDevName ]:
         _renameInterfacePhase2( realDevs[ oldDevName ], "mlx_" + newDevName )
      _renameInterfacePhase2( oldDevName, newDevName )
   # Return the update device names
   return sorted( oldToNewDeviceName.values() )

def generateSystemMacAddr( lowestMacAddr ):
   # Set the system mac address. We have no choice but to choose
   # this randomly - if we choose the mac address of any of our
   # existing interfaces, then we end up with weird behavior as
   # traffic sent on that interface will then be treated as if it
   # were sent to the system mac address. This can cause weird
   # issues (double responses to ping packets, for
   # example). Unfortunately we can't just choose a locally
   # administered address, as MLAG assumes that the system mac
   # address is not locally administered.
   #
   # To randomly generate an address, we start with the first 3
   # bytes of the mac address, which differs based on the
   # hypervisor being used. We then seed random with the lowest mac
   # address of the system. This ensures the same behavior every
   # time we run (assuming no interfaces were removed / changed, at
   # which point we consider this to be a 'new' machine), and makes
   # it much more unlikely that two systems will come up with the
   # same 3 random bytes (which might happen if two VMs booted with
   # the exact same system time)
   #
   # We allow the mac address to be overwritten by creating the
   # file /mnt/flash/system_mac_address. This gives us a hook in
   # case the automatically generated mac address is not
   # sufficient. Care must be taken when cloning systems with
   # a /mnt/flash/system_mac_address file.
   systemMacAddr = lowestMacAddr[ : 8 ]

   # Py3's random module produces diffeent values with the same, so Py2's
   # behavior must be preserved generate the same mac across reboots
   # and python upgrades.

   # replicate Py2 seed algorithm: hash(string) then cast to uint32
   seed = ctypes.c_uint32( ArPyUtils.py2StringHash( lowestMacAddr ) ).value
   r = random.Random( seed )
   for _ in range( 3 ):
      # py3 changed the random.randint() impl, so use py2 equivalent
      systemMacAddr += ":%02x" % ( int( r.random() * 256 ) )
   return systemMacAddr


class GenericPcDriver( Fru.FruDriver ):
   managedTypeName = "Eos::GenericPcFru"
   managedApiRe = ".*"
   # Set driver priority to 1 so that another driver can
   # override this (with priority 2)
   driverPriority = 1

   def __init__( self, genericPc, parentMibEntity, parentDriver, driverCtx ):
      t2( "Eos/FruPlugin/GenericPc.GenericPcDriver: Populating the Entity Mib" )
      assert ( parentMibEntity is None ) # pylint: disable=superfluous-parens
      entityMibStatus = driverCtx.entity( "hardware/entmib" )
      entityMibStatus.fixedSystem = ( 1, 0, "FixedSystem" )
      self.genericPcMib_ = entityMibStatus.fixedSystem
      if genericPc.api == "veos":
         # We're running as a VM with the vEOS Aboot image,
         # but EosInit-veos is not installed. This means we're
         # running as CVX (or at least, we don't have any
         # other use cases for this today).
         self.genericPcMib_.modelName = "vEOS"
         self.genericPcMib_.description = "CloudVision eXchange"
         self.genericPcMib_.swappability = "notSwappable"
      # pylint: disable-next=consider-using-max-builtin
      if entityMibStatus.nextPhysicalIndex < 2:
         entityMibStatus.nextPhysicalIndex = 2

      # -------------
      # Emulate the FixedConfig FruPlugin, i.e.
      #
      #    o create per-cell state ("sys/config/cell/<cellId>",
      #      "hardware/cell/<cellId>")
      #
      #    o map the appropriate roles to our cell
      #
      #    o Set the default port roles we support

      myCellId = Cell.cellId()
      roles = [ "AllCells", "AllSupervisors", "ActiveSupervisor" ]
      Fru.createAndConfigureCell( driverCtx.sysdbRoot,
                                  genericPc,
                                  myCellId,
                                  roles )

      for portRole in [ "Management", "Switched" ]:
         genericPc.portRole[ portRole ] = portRole

      # BUG77642.eithan: Enabling hardware routing in Sysdb by
      # setting maxEcmp and routingSupported in /routing/hardware/status/
      routingHwStatus = driverCtx.entity( "routing/hardware/status" )
      routingHwStatus.routingSupported = True
      routingHwStatus.maxLogicalProtocolEcmp = 1
      routing6HwStatus = driverCtx.entity( "routing6/hardware/status" )
      routing6HwStatus.routingSupported = True
      routing6HwStatus.maxLogicalProtocolEcmp = 1
      routingHwStatus.staticTunIntfPlatformCapabilityDeclared = True

      # -------------
      # Populate the rest of the inventory tree based on the
      # interfaces we find in the kernel

      t2( "GenericPc.GenericPcDriver: Getting Interfaces" )
      # Get all ethernet interface device names
      devNames = []
      for devName in os.listdir( sysClassNet ):
         if not os.path.isdir( os.path.join( sysClassNet, devName ) ):
            # Possibly a file, like "bonding_masters".
            continue

         with open( os.path.join( sysClassNet, devName, "type" ) ) as f:
            devType = int( f.read() )
         t9( devName, devType )
         # This type apparently corresponds to include/linux/if_arp.h
         # ARPHRD_* values.
         if devType != ARPHRD_ETHER:
            continue

         devNames.append( devName )

      # If we're on a vEOS platform, meaning that we're running as CVX,
      # rename vmnicetN interfaces to be etN.
      # BUG147936 - this will go away in favor of Sfa once the production
      # vEOS support merges
      if genericPc.api == "veos":
         devNames = renameKernelDevices( devNames, "linux" )

      ethPortDir = genericPc.component.newEntity(
         "Inventory::EthPortDir", "ethPortDir" )
      pciPortDir = genericPc.component.newEntity(
         "Inventory::PciPortDir", "pciPortDir" )
      phyEthtoolPhyDir = genericPc.component.newEntity(
         "Inventory::Phy::EthtoolDir", "phy" )

      t2( "GenericPc.GenericPcDriver: Creating Interfaces" )
      lowestMacAddr = None
      for devName in devNames:
         # We support 2 types of interfaces:
         #    - maN, which we assume are management interfaces named
         #      by the boot loader
         #    - etN, which we assume are front-panel interfaces
         #      created as part of vEOS which will be managed by
         #      PhyEthtool
         # All other interfaces are ignored
         m = re.match( r"(ma|et)(\d+)", devName )
         if not m:
            continue

         ( intftype, index ) = m.groups()
         label = int( index )
         if intftype == "ma":
            portId = label + 100
         else:
            portId = label

         with open( os.path.join( sysClassNet, devName, "address" ) )as f:
            mac = f.read().strip()

         # Track the lowest mac address in the system. This is
         # used below to generate the system mac address
         if not lowestMacAddr or ( mac < lowestMacAddr ):
            lowestMacAddr = mac

         if intftype == "et":
            # There's a front-panel interface that we're going to manage
            # with PhyEthtool.
            # Hack - change the default switched interface config
            # to be linkModeForced1GbpsFull instead of
            # linkModeForced10GbpsFull, as otherwise the PhyEthtool
            # agent will fail trying to manage front-panel ports.
            intfConfigDir = driverCtx.entity( 'interface/config/eth/phy/default' )
            defaultSwitchedEthIntfConfig = intfConfigDir.defaultIntfConfig[
               "DefaultEthSwitchedPort" ]
            defaultSwitchedEthIntfConfig.linkModeLocal = "linkModeForced1GbpsFull"

         # Create an Inventory::EthPort
         if intftype == "ma":
            port = pciPortDir.newPort( portId )
            port.description = "Management Ethernet Port %d" % label
            port.role = "Management"
            # Don't set a pci address, the EthPciPort fru driver will assign
            # things correctly based on the device name

            # Create an Inventory::PhyEthtoolPhy
            phy = phyEthtoolPhyDir.newPhy( "Phy%s%d" %( intftype, label ) )
            phy.port = port

         elif intftype == "et":
            port = ethPortDir.newPort( portId )
            port.description = "Front-Panel Port %d" % label
            port.role = "Switched"
            port.macAddr = mac

            # Create an Inventory::PhyEthtoolPhy
            phy = phyEthtoolPhyDir.newPhy( "Phy%s%d" %( intftype, label ) )
            phy.port = port

         else:
            assert False, "Unknown port %s" % intftype

         port.label = label

         if ( devName == 'ma1' ) and ( genericPc.api != "veos" ):
            # Set the bridgeMacAddr from the ma1 interface
            # In the non-veos case it's not actually clear
            # if we even need to set this, but keeping it like
            # this for now to avoid any unexpected fallout.
            # Note that this change (and the related one below
            # for the vEOS case) will all be undone once these
            # changes meet the Sfa project.
            entityMibStatus.systemMacAddr = mac

         t0( "device %s, mac %s" %( devName, mac ) )

      Fru.FruDriver.__init__( self, genericPc, parentMibEntity,
                              parentDriver, driverCtx )
      registerHealthSource( genericPc, "vRoot" )
      # Now that we've built up the full inventory tree, run
      # drivers across it
      self.instantiateChildDrivers( genericPc, self.genericPcMib_ )

      if genericPc.api == "veos":
         # Set the system mac address. We have no choice but to choose
         # this randomly - see the comment in generateSystemMac for details.
         #
         # We allow the mac address to be overwritten by creating the
         # file /mnt/flash/system_mac_address. This gives us a hook in
         # case the automatically generated mac address is not
         # sufficient. Care must be taken when cloning systems with
         # a /mnt/flash/system_mac_address file.
         veosConfig = parseVeosConfig()
         systemMacAddr = veosConfig[ 'SYSTEMMACADDR' ]
         if lowestMacAddr and not systemMacAddr:
            systemMacAddr = generateSystemMacAddr( lowestMacAddr )
         if systemMacAddr:
            entityMibStatus.systemMacAddr = systemMacAddr
      else:
         # Keep the original strategy of just using ma1's mac address,
         # which is done above.
         pass

      self.genericPcMib_.firmwareRev = FirmwareRev.abootFirmwareRev()

      import EosVersion # pylint: disable=import-outside-toplevel
      vi = EosVersion.VersionInfo( sysdbRoot=None )
      if vi.version() == None: # pylint: disable=singleton-comparison
         self.genericPcMib_.softwareRev = ""
      else:
         self.genericPcMib_.softwareRev = vi.version()

      # For all front-panel interfaces managed by PhyEthtool,
      # we have to set the deviceName on the EthIntfStatus
      # ourselves (just like we do in Fru for the management
      # interfaces).
      phyEthtoolPhyDir = genericPc.component.get( "phy" )
      if phyEthtoolPhyDir:
         for invPhy in phyEthtoolPhyDir.phy.values():
            intfStatus = invPhy.port.intfStatus
            if intfStatus and not intfStatus.deviceName:
               intfStatus.deviceName = invPhy.port.intfId.replace( "Ethernet", "et" )

      # Declare FruReady
      hwCellDir = driverCtx.sysdbRoot[ 'hardware' ][ 'cell' ][ '%d' % myCellId ]
      assert hwCellDir
      hwCellDir.newEntity( 'Tac::Dir', 'FruReady' )

      # Declare success at the end
      self.genericPcMib_.initStatus = "ok"

def Plugin( context ):
   context.registerDriver( GenericPcDriver )
   context.registerFruFactory( genericPcFactory, Fru.genericPcFactoryId )

   mg = context.entityManager.mountGroup()
   mg.mount( 'routing/hardware/status', 'Routing::Hardware::Status', 'w' )
   mg.mount( 'routing6/hardware/status', 'Routing6::Hardware::Status', 'w' )
   mg.close( None )
