# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''A module providing common L1 interface creation infrastructure

Interface creation is the proces of translating each inventory port object into an
interface descriptor and default configuration strucutres.

The rest of EOS then reacts on this desciptor to produce an interface configuration
and status.

Note:
   Due to historic reasons the word port is synonymous with interface. However, this
   does not generally hold as these days port has come to refer to the entire
   interface slot.
'''

import abc

from ArPyUtils.Types import ArException
import Cell
import DesiredTracing
import Fru
from Fru.Port import IntfFactory
import Tracing
from TypeFuture import TacLazyType

AutonegMode = TacLazyType( 'Interface::AutonegMode' )
DescType = TacLazyType( 'Interface::DescType' )
EthFecEncoding = TacLazyType( 'Interface::EthFecEncoding' )
EthLinkModeSet = TacLazyType( 'Interface::EthLinkModeSet' )
EthTypesApi = TacLazyType( 'Interface::EthTypesApi' )
Generation = TacLazyType( 'Ark::Generation' )
GenerationConfig = TacLazyType( 'Interface::GenerationConfig' )
HwCapRestrictionsDir = TacLazyType( 'Hardware::Capabilities::HwCapRestrictionsDir' )
L1MountConstants = TacLazyType( 'L1::MountConstants' )

__defaultTraceHandle__ = Tracing.Handle( 'L1Port' )
DesiredTracing.desiredTracingIs( 'L1Port/123' )

TERROR = Tracing.trace1
TWARN = Tracing.trace2
TNOTE = Tracing.trace3
TINFO2 = Tracing.trace5
TINFO3 = Tracing.trace6

capabilityRestrictionDir = None
fabricL1CardPowerGenerationDir = None
l1CardPowerGenerationDir = None
l1PortConfig = None

class L1IntfFactory( IntfFactory.PortIntfFactory ):
   '''A base class for all L1 interface factories.

   Subclasses should override the `createIntfStatus` method so that it can create
   the required interface state entities ( e.g. interface descriptor ). The
   overridden implementation must call super.
   '''

   def getDescType( self, port ):
      '''By default all L1 interfaces are data plane ports.'''
      return DescType.dataPlanePort
   
   @staticmethod
   @abc.abstractmethod
   def getEthIntfDefaultConfigDir():
      '''Overridable method that returns the location of the
      EthIntfDefaultConfigs
      '''
   
   @staticmethod
   def setExtraDefaultIntfConfig( deic ):
      """Set additional attributes for the Interface::EthPhyIntfDefaultConfig
      entities associated with the port role(s) managed by this class. Some
      attributes are set from self.maybeAddNewDefaultIntfConfig().
      """

   @classmethod
   def maybeAddNewDefaultIntfConfig( cls,
                                     defaultIntfConfigName,
                                     linkMode="linkModeUnknown" ):
      '''Adds a new DefaultIntfConfig if one does not already exist with
      the given defaultIntfConfigName in the defaultIntfConfigs dir.
      '''
      defaultIntfConfigs = cls.getEthIntfDefaultConfigDir()
      if ( defaultIntfConfigName and
           defaultIntfConfigName not in defaultIntfConfigs ):
         deic = defaultIntfConfigs.newMember( defaultIntfConfigName )
         deic.adminEnabledStateLocal = "enabled"
         deic.linkModeLocal = "linkModeForced10GbpsFull" \
               if linkMode == "linkModeUnknown" else linkMode
         deic.rxFlowcontrolLocal = "flowControlConfigOff"
         deic.txFlowcontrolLocal = "flowControlConfigOff"
         cls.setExtraDefaultIntfConfig( deic )
         return deic
      return None

   def setL1CardPowerGeneration( self, fruBase ):
      '''Sets the latched card power generation which is used by various card agents
      to tag sysdb entities and check for stale state.

      Note:
         The fabric version of the generation was an unfortunate divergence that we
         used to have. The entity and path still track the same card power generation
         but various fabric only subsystems expect it to be populated.

      Returns:
         The L1 card power generation.
      '''
      cardSlotId = fruBase.sliceId
      if not cardSlotId:
         # TODO: The use of 'generic' is a hack while tests are configured to
         #       correctly declare Cell type.
         assert Cell.cellType() in [ 'fixed', 'generic' ]
         cardSlotId = "FixedSystem"

      l1CardPowerGeneration = l1CardPowerGenerationDir.get( cardSlotId )
      if not l1CardPowerGeneration:
         cardPowerGeneration = Generation( Fru.powerGenerationId( fruBase ), True )

         l1CardPowerGeneration = Fru.Dep(
            l1CardPowerGenerationDir, fruBase ).newEntity(
               GenerationConfig.tacType.fullTypeName, cardSlotId )
         l1CardPowerGeneration.generation = cardPowerGeneration

         TWARN( f'L1 card power generation set to {cardPowerGeneration}' )

         # Long term the L1 card power generation should be the only generation that
         # exists. For now, we assert that all other card power generation tracking
         # entities follow its lifespan.
         if cardSlotId in fabricL1CardPowerGenerationDir:
            raise ArException(
               f'Unexpected fabric L1 card power generation for {cardSlotId}' )

         fabricL1CardPowerGeneration = Fru.Dep(
            fabricL1CardPowerGenerationDir, fruBase ).newEntity(
               GenerationConfig.tacType.fullTypeName, cardSlotId )
         fabricL1CardPowerGeneration.generation = cardPowerGeneration

      return l1CardPowerGeneration.generation

   def createIntfCapabilityRestrictions( self, fruBase, port, intfName ):
      # This is an ugly hack needed to work around the fact that btests don't load
      # this plugin explicitly. As a result the mounts never execute and method
      # asserts due to the directories being null.
      #
      # This hack has to be here until we develop a way to cleanly chain plugin
      # loading in btests.
      if not l1PortConfig or not capabilityRestrictionDir:
         TERROR( 'ERROR: Mounts not present! Interface capabilities will not be '
                 'restricted.' )
         return

      cardSlotId = fruBase.sliceId
      if Cell.cellType() == 'fixed':
         cardSlotId = 'FixedSystem'

      # Ports without registered UIDs do not support extra FDL / L1 profile
      # parameters.
      if not port.uid:
         TINFO3( 'Skipping L1 interface capability restriction due to missing UID:',
                 port.intfId )
         return

      cardL1PortConfigDir = l1PortConfig.get( cardSlotId )
      if not cardL1PortConfigDir:
         TINFO3( 'Skipping L1 interface capability restriction due to missing port '
                 'configs' )
         return

      portConfig = cardL1PortConfigDir.l1PortConfig.get( port.uid )
      if not portConfig:
         TINFO3( 'Skipping interface caps restriction due to missing port config:',
                 port.intfId )
         return

      if not portConfig.bound:
         TINFO3( 'Skipping interface caps restriction for an unbound port:',
                 port.uid )
         return

      if not portConfig.enforcedCapabilities:
         return

      TINFO2( 'Creating L1 interface capability restrictions for', intfName )

      cardCapabilityRestrictionDir = Fru.Dep(
         capabilityRestrictionDir, fruBase ).newEntity( 'Tac::Dir', cardSlotId )

      fdlCardCapabilityRestrictionDir = Fru.Dep(
         cardCapabilityRestrictionDir, fruBase ).newEntity(
            HwCapRestrictionsDir.tacType.fullTypeName, 'fdlCaps' )

      capRestrictions = EthLinkModeSet()
      for cap in portConfig.enforcedCapabilities.capabilities:
         # The capability restriction system does not presently support FEC or
         # auto negotiation.
         if ( ( cap.anegMode != AutonegMode.anegModeUnknown ) or
              ( cap.fec != EthFecEncoding.fecEncodingUnknown ) ):
            raise ArException( 'Detected unsupported capability restriction',
                               intfId=intfName,
                               cap=cap )

         TWARN( 'Restricting interface capability', intfName, ':', cap )
         capEthLinkModeSet = EthTypesApi.ethSpeedLaneToLinkModeSet(
            cap.speed, cap.laneCount, cap.duplex )

         capRestrictions |= capEthLinkModeSet

      fdlCardCapabilityRestrictionDir.newIntfCapEnforced(
         intfName, capRestrictions, portConfig.enforcedCapabilities.action )

   def createIntfStatus( self, fruBase, port, intfName ):
      '''Creates common L1 interface state entities.'''
      TINFO2( 'Creating common L1 interface state for', intfName, 'with UID',
              port.uid )
      self.createIntfCapabilityRestrictions( fruBase, port, intfName )

   @abc.abstractmethod
   def handlePortRole( self, portRole ):
      pass

def Plugin( context ):
   mg = context.entityManager.mountGroup()

   global l1PortConfig
   l1PortConfig = mg.mount( 'hardware/l1/profile/config/port',
                            'Tac::Dir',
                            'ri' )

   global capabilityRestrictionDir
   capabilityRestrictionDir = mg.mount( 'hardware/l1/fru/capabilities/slice',
                                        'Tac::Dir',
                                        'wi' )

   global l1CardPowerGenerationDir
   l1CardPowerGenerationDir = mg.mount(
      L1MountConstants.l1CardPowerGenerationRootDirPath(), 'Tac::Dir', 'wi' )

   global fabricL1CardPowerGenerationDir
   fabricL1CardPowerGenerationDir = mg.mount(
      'hardware/fabric/generation/slot', 'Tac::Dir', 'wi' )

   mg.close( None )
