# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''A module providing common L1 port driver / interface factory logic.

Note:
   The ports infrastructure is not strictly L1 infrastructure, As such, certain
   concepts need to be translated with care:

   Ports, in this context refer to the L1 interfaces, NOT the interface slots / XCVR
   slots.

   A port's label is equivalent to the interface slot ID ( e.g. The 2 in
   Ethernet1/2/3 ).

   A port's sublabel is the equivalent to the interface's lane ID for multi lane
   interface slots ( e.g. the /3 in Ethernet1/2/3 ).
'''

from TypeFuture import TacLazyType
import Cell
import DesiredTracing
import Fru
from Fru.Port import Driver as PortDriver
import Tracing

EthLinkMode = TacLazyType( 'Interface::EthLinkMode' )
EthTypesApi = TacLazyType( 'Interface::EthTypesApi' )
L1PortConfigDir = TacLazyType( 'L1Profile::L1PortConfigDir' )
Port = TacLazyType( 'Inventory::Port' )

__defaultTraceHandle__ = Tracing.Handle( 'L1Port' )
DesiredTracing.desiredTracingIs( 'L1Port/123' )

TERROR = Tracing.trace1
TWARN = Tracing.trace2
TNOTE = Tracing.trace3
TINFO2 = Tracing.trace5
TINFO3 = Tracing.trace6

class L1PortDriver( PortDriver.PortDirDriver ):
   '''A base class for port drivers managing L1 ports.

   This class provides common methods for identifying bootstrap ports on the system
   and applying L1 profiles.

   Note:
      Any subclass should call this base class's constructor prior to calling
      initL1Ports.
   '''

   # pylint: disable=abstract-method

   requires = [ 'L1Profile::L1PortConfig' ]

   def __init__( self, invPortDir, fruEntMib, parentDriver, driverCtx ):
      super().__init__( invPortDir,
                        fruEntMib,
                        parentDriver,
                        driverCtx )

      self.l1PortConfigRootDir = driverCtx.entity(
         'hardware/l1/profile/config/port' )

      self.cardSlotId = Fru.fruBase( invPortDir ).sliceId
      if Cell.cellType() == 'fixed':
         self.cardSlotId = 'FixedSystem'

   def _portStr( self, ports, displayDefaultLinkMode=True ):
      '''Produces a string representation of L1 inventory ports.

      Args:
         ports ( iterable( Inventory::EthPort ) ): The ports to generate a string
                                                   representation of.
      Returns:
         A string representation of the passed in ports where each port's
         representation is on a new line.
      '''
      return '\n\t'.join(
         # pylint: disable-next=consider-using-f-string
         '{} {}{}{}'.format( self.portNamePrefix( port ),
                              port.label,
                              ( '' if port.subLabel == Port.invalidSubLabel else
                                f'/{port.subLabel}' ),
                              ( f': {port.defaultLinkMode}'
                                if displayDefaultLinkMode else '' ) )
         for port in ports
      )

   def _identifyBootstrapPorts( self, invPorts ):
      '''Identifies bootstrap ports on a system.

      Bootstrap ports are used to denote ports which will remain operational in the
      event that an invalid / expired EOS licence is detected. For more details
      please see AID4634.

      Returns:
         A set of bootstrap Inventory::EthPort objects.
      '''
      bootstrapPorts = set()

      def getPortKey( port ):
         '''When identifying bootstrap ports on the system, generally only the
         first interface on every interface slot ( /1 ) matters.

         As such, a scheme is devised where each port can be mapped to a key such
         that all the /1 ports appear before the /2+ ports when sorted.

         Note:
            Ports without a sub label ( e.g. SFPs or other single lane interface
            slots ) will be treated the same as a /1.
         '''
         subLabel = port.subLabel if port.subLabel != Port.invalidSubLabel else 1
         return ( subLabel * 1000 ) + port.label

      bootstrapEligiblePorts = [ port for port in invPorts
                                 if ( ( port.role == 'Switched' ) and
                                      ( not port.isUnconnected ) ) ]

      # The first two interfaces are always considered bootstrap ports no matter
      # the speed.
      bootstrapPorts.update(
         sorted( bootstrapEligiblePorts, key=getPortKey )[ 0 : 2 ] )
      TINFO2( 'Identified first two ports as bootstrap ports:\n' +
              self._portStr( bootstrapPorts ) )

      # The two interfaces with the fastest known default speed ( if available )
      # are also considered bootstrap ports.
      bootstrapEligiblePortsWithKnownSpeeds = [
         port for port in bootstrapEligiblePorts
         if port.defaultLinkMode not in [ EthLinkMode.linkModeUnknown,
                                          EthLinkMode.linkModeAutoneg ] ]

      if len( bootstrapEligiblePortsWithKnownSpeeds ) < 2:
         return bootstrapPorts

      bootstrapEligiblePortsWithKnownSpeeds = sorted(
         bootstrapEligiblePortsWithKnownSpeeds,
         key=getPortKey )

      bootstrapEligiblePortsWithKnownSpeeds = sorted(
         bootstrapEligiblePortsWithKnownSpeeds,
         key=lambda x: EthTypesApi.linkModeToSpeedMbps( x.defaultLinkMode ),
         reverse=True )

      fastestBootstrapEligiblePortSpeed = EthTypesApi.linkModeToSpeedMbps(
         bootstrapEligiblePortsWithKnownSpeeds[ 0 ].defaultLinkMode )

      fastestBootstrapEligiblePorts = [
         port for port in bootstrapEligiblePortsWithKnownSpeeds
         if EthTypesApi.linkModeToSpeedMbps( port.defaultLinkMode ) ==
         fastestBootstrapEligiblePortSpeed ]

      fastestBootstrapPorts = fastestBootstrapEligiblePorts[
         0 : min( 2, len( fastestBootstrapEligiblePorts ) + 1 ) ]
      TINFO2( 'Identified fastest two ports as bootstrap ports:\n' +
              self._portStr( fastestBootstrapPorts ) )

      bootstrapPorts.update( fastestBootstrapPorts )

      return bootstrapPorts

   def _applyL1Profiles( self, invPorts ):
      '''Applies all L1 profiles specified by the L1 profiles port config ( if
      available ) on the L1 port inventories.

      Note:
         This is a 'destructive' call in that it modifies the passed in inventory
         objects.
      '''
      l1PortConfigDir = self.l1PortConfigRootDir.get( self.cardSlotId )
      if not l1PortConfigDir:
         TINFO2( 'No L1 profile port configuration detected, skipping applying L1 '
                 'profiles...' )
         return

      for port in invPorts:
         l1PortConfig = l1PortConfigDir.l1PortConfig.get( port.uid )
         if l1PortConfig is None:
            TINFO3( 'No L1 port configuration for:', port.uid )
            continue

         port.bound = l1PortConfig.bound
         if not port.bound:
            TINFO3( 'L1 port', port.uid, 'unbound from', self._portStr( [ port ] ) )
            continue

         oldPortStr = self._portStr( [ port ] )

         changes = False
         if port.subLabel != l1PortConfig.intfLane:
            port.subLabel = l1PortConfig.intfLane
            changes = True

         # pylint: disable-next=consider-using-in
         if ( ( l1PortConfig.defaultLinkMode != EthLinkMode.linkModeUnknown ) and
              ( port.defaultLinkMode != l1PortConfig.defaultLinkMode ) ):
            port.defaultLinkMode = l1PortConfig.defaultLinkMode
            changes = True

         if changes:
            TINFO3( 'L1 Port', port.uid, 'rebound from', oldPortStr, 'to',
                     self._portStr( [ port ] ) )
         else:
            TINFO3( 'L1 port', port.uid, 'unmodified', self._portStr( [ port ] ) )

   def initL1Ports( self,
                    invPorts,
                    fruEntMib,
                    parentDriver,
                    driverCtx,
                    markBootstrapPorts=False,
                    applyL1Profiles=False ):
      '''Initializes all L1 ports.

      This method is responsible for applying identifying bootstrap ports, applying
      L1 profiles, registering the ports with the entity MIB, creating the interface
      state ( e.g. interface desc ), and deciding the port's interface ID.

      Args:
         ports ( iterable( Inventory::EthPort ) ): An iterable containing all the L1
                                                   inventory ports managed by
                                                   the port driver.
         fruEntityMib ( EntityMib::Card ): The parent card entity MIB entity which
                                           the ports belong to.
         parentDriver ( Fru.Driver ): The parent FRU Driver.
         driverCtx ( Fru.DriverContext ): The FRU driver context.
         markBootstrapPorts ( bool ): If set, the bootstrap ports will automatically
                                      be identified and their port roles set
                                      accordingly. See AID4634 for more details.
         applyL1Profiles ( bool ): If set, L1 profile card configurations are taken
                                   into account and will influence interface
                                   creation, interface naming, and default port
                                   parameters ( e.g. default link mode ).
      '''
      # Bootstrap ports have to be computed before L1 profiles are applied. This
      # is because L1 profiles might change the default link modes for ports which
      # would impact the bootstrap port determination algorithm.
      bootstrapPorts = set()
      if markBootstrapPorts:
         TNOTE( 'Identifying bootstrap ports...' )
         bootstrapPorts = self._identifyBootstrapPorts( invPorts )
         TNOTE( 'Bootstrap ports identified:\n' +
                self._portStr( bootstrapPorts ) )

      if applyL1Profiles:
         TNOTE( 'Applying L1 profiles...' )
         self._applyL1Profiles( invPorts )
         TNOTE( 'L1 profiles applied' )

      for port in sorted( invPorts, key=lambda p: p.id ):
         if not port.bound:
            continue

         TNOTE( 'Initializing L1 port', self._portStr( [ port ] ) )
         self.initPort(
            port,
            fruEntMib,
            parentDriver,
            driverCtx,
            isBSP=( port in bootstrapPorts ) )

def Plugin( context ):
   mg = context.entityManager.mountGroup()

   mg.mount( 'hardware/l1/profile/config/port',
             'Tac::Dir',
             'ri' )

   mg.close( None )
