# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This module defines the FRU driver for loading L1 Builtin Profiles from the
yaml files installed within the system via SKU rpms.

The timing at which this driver run depends on two things:
   - This driver requires knowledge of the SKU basename, therefore it needs to run
     after FDL completes.
   - The driver needs to store Card profiles before the L1 profile resolver runs and
     resolves all profiles, therefore it has to run before the resolver.
'''

import DesiredTracing
from Fru.Driver import FruDriver
from L1ProfileBuiltinsLibrary.Card.Loader import CardProfileLoader
from L1ProfileBuiltinsLibrary.IntfSlot.Loader import IntfSlotProfileLoader
import Tracing
from TypeFuture import TacLazyType
import traceback

__defaultTraceHandle__ = Tracing.Handle( 'L1Profile::L1ProfileBuiltinLoaderDriver' )
DesiredTracing.desiredTracingIs( 'L1Profile::L1ProfileBuiltinLoaderDriver/013' )
DesiredTracing.desiredTracingIs( 'L1ProfileBuiltinsLibrary*/012346' )

TERROR = Tracing.trace0
TWARN = Tracing.trace1
TNOTE = Tracing.trace3

FileSystemConstants = TacLazyType( 'L1Profile::FileSystemConstants' )
InvProductConfig = TacLazyType( 'Inventory::L1Profile::ProductConfig' )
MountConstants = TacLazyType( 'L1Profile::MountConstants' )

class L1ProfileBuiltinLoaderDriver( FruDriver ):
   '''A FruDriver that has the functionality to load both builtin Intf Slot and
      Card profiles.

      Any exceptions raised in loading the builtin L1 Profiles will result in
      FRU dying. Since this logic exists very early on in FRU's initialization
      process, any exceptions will severely limit the switch's functionality,
      as even the management interfaces may become inoperable. As such, we opt
      to not throw any exceptions, and hamstring the L1 Profiles feature and
      proceed without it than potentially have an inoperable switch.
   '''
   managedApiRe = ".*$"
   managedTypeName = InvProductConfig.tacType.fullTypeName

   primaryDriver = False

   def __init__( self, managedObj, parentMib, parentDriver, driverCtx ):
      super().__init__( managedObj, parentMib, parentDriver, driverCtx )
      self.ctx = driverCtx
      self.cardProfileDir = self.ctx.entity(
         MountConstants.cardProfileLibraryBuiltinDirPath() )
      self.intfSlotProfileDir = self.ctx.entity(
         MountConstants.intfSlotProfileLibraryBuiltinDirPath() )

      if driverCtx.sysdbRoot[ 'hardware' ][ 'entmib' ].chassis:
         TNOTE( 'Running L1 Profile Builtin Loader on a Modular System' )
         self.loadBuiltinL1Profiles()
      else:
         TNOTE( 'Running L1 Profile Builtin Loader on a Fixed System' )
         self.loadBuiltinL1Profiles( parentMib.modelName )

   def loadBuiltinL1Profiles( self, baseSkuName='' ):
      # TODO BUG698951: Add a way to communicate errors encountered while loading
      #                 builtin profiles with the rest of the system.
      #                 The lack of Builtin L1 Profiles is enough to "disable" L1
      #                 profiles. However, there should be a more robust status
      #                 reporting mechanism.
      if not self.loadBuiltinIntfSlotProfiles():
         return

      self.loadBuiltinCardProfiles( baseSkuName )

   def loadBuiltinIntfSlotProfiles( self ):
      TNOTE( 'Loading builtin interface slot profile library' )
      try:
         loader = IntfSlotProfileLoader( self.intfSlotProfileDir )
         loader.load( FileSystemConstants.intfSlotProfileLibraryBuiltinDirPath() )
      except Exception as e: # pylint: disable=broad-except
         TERROR( 'Critical Failure: Builtin interface slot profile library '
                 'loading failed. Proceeding without L1 Profiles.\n'
                 f'{e}\n\n---Traceback---\n\n{traceback.format_exc()}' )
         self.intfSlotProfileDir.intfSlotProfile.clear()
         return False

      TNOTE( 'Finished loading builtin interface slot profile library' )
      return True

   def loadBuiltinCardProfiles( self, skuBaseName='' ):
      TNOTE( 'Loading builtin card profile library' )
      try:
         loader = CardProfileLoader( self.cardProfileDir,
                                     self.intfSlotProfileDir,
                                     skuBaseName )
         loader.load( FileSystemConstants.cardProfileLibraryBuiltinDirPath() )
      except Exception as e: # pylint: disable=broad-except
         TERROR( 'Critical Failure: Builtin card profile library '
                 'loading failed. Proceeding without L1 Profiles.\n'
                 f'{e}\n\n---Traceback---\n\n{traceback.format_exc()}' )
         self.cardProfileDir.cardProfile.clear()
         return False

      TNOTE( 'Finished loading builtin card profile library' )
      return True

def Plugin( context ):
   context.registerDriver( L1ProfileBuiltinLoaderDriver )

   mg = context.entityManager.mountGroup()

   mg.mount( MountConstants.cardProfileLibraryBuiltinDirPath(),
             'L1Profile::CardProfileDir',
             'wi' )

   mg.mount( MountConstants.intfSlotProfileLibraryBuiltinDirPath(),
             'L1Profile::InterfaceSlotProfileDir',
             'wi' )

   mg.close( None )
