# Copyright (c) 2022 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

'''This module defines the FRU plugin responsible for generating the L1 profiles
card configuration entities.

This driver acts as a central location in which all the information related to a
card profile is collated, resolved, validated, and stored for easy access by the port
drivers. This process  occurs centrally as the dependant port drivers are
distributed.

The output "card config" is then consumed by the port drivers to influence the
creation of the interfaces.

Card configs are generated based on the CLI configuration and the available profile
definitions in the various profile libraries. This makes card config generation
very sensitive to the current state of the system at the time generation takes place.

In order to prevent card config changes from occurring unexpectedly ( i.e. outside of
card power cycle ), the card configs are only written to Sysdb if the FRU plugin
is processing the card for the first time at the current power generation. All other
driver run conditions ( e.g. SSO ) are short circuit.

ASU is a special case since there has to be seperate external mechanisms to either:
   - Restore the card config in it's previous state
   - Guarantee that the system's state pre and post ASU are identical to ensure
     the same card config get's generated.

Both of the above tasks are not the responsibility of this plugin.
'''

# pkgdeps: rpm Arsys-lib
# pkgdeps: rpm EthIntf

from ArPyUtils.Types import ArException
import Cell
import DesiredTracing
import Fru
from Fru.Driver import FruDriver
import Logging
from L1Profile.Syslog import (
   L1PROFILE_MODULE_APPLIED,
   L1PROFILE_MODULE_ERROR,
   PROFILE_DNE,
   MODULE_INCOMPAT,
)
import Tac
import Tracing
from TypeFuture import TacLazyType

__defaultTraceHandle__ = Tracing.Handle( 'L1Profile::ConfigDriver' )
DesiredTracing.desiredTracingIs( 'L1Profile::ConfigDriver/0123' )

TERROR = Tracing.trace0
TWARN = Tracing.trace1
TNOTE = Tracing.trace3
TINFO3 = Tracing.trace6

CardConfigDir = TacLazyType( 'L1Profile::CardConfigDir' )
CardConfigGenerator = TacLazyType( 'L1Profile::CardConfigGenerator' )
CardDefaultDir = TacLazyType( 'L1Profile::CardDefaultDir' )
CardProfileReader = TacLazyType( 'L1Profile::CardProfileReader' )
CardProfileSource = TacLazyType( 'L1Profile::CardProfileSource::CardProfileSource' )
CliConfig = TacLazyType( 'L1Profile::CliConfig' )
EnforcedCapabilities = TacLazyType( 'Hardware::Capabilities::EnforcedCapabilities' )
EnforcedCapabilitiesAction = TacLazyType(
   'Hardware::Capabilities::EnforcedCapabilitiesAction' )
EthLinkMode = TacLazyType( 'Interface::EthLinkMode' )
EntityMibStatus = TacLazyType( 'EntityMib::Status' )
InterfaceSlotDescriptor = TacLazyType( 'L1Profile::InterfaceSlotDescriptor',
                                       returnValueConst=True )
IntfSlotGroupId = TacLazyType( 'L1::IntfSlotTypes::IntfSlotGroupId' )
L1InterfaceSlotDir = TacLazyType( 'Inventory::L1InterfaceSlotDir' )
L1PortConfig = TacLazyType( 'L1Profile::L1PortConfig' )
L1PortConfigDir = TacLazyType( 'L1Profile::L1PortConfigDir' )
MountConstants = TacLazyType( 'L1Profile::MountConstants' )
ProductConfig = TacLazyType( 'L1Profile::ProductConfig' )
RedundancyMode = TacLazyType( 'Redundancy::RedundancyMode' )

class L1ProfileCardConfigDriver( FruDriver ):
   managedTypeName = L1InterfaceSlotDir.tacType.fullTypeName
   managedApiRe = '.*$'

   requires = [ 'L1Profile::ProductConfig', 'L1Profile::CardDefault' ]
   provides = [ 'L1Profile::L1PortConfig' ]

   def _generateCardConfig( self, cardSlotId, invL1IntfSlotDir ):
      '''Generates the L1 profile card configuration based on either the CLI
      configuration or the default card parameters and the installed L1 profile
      models in the various libraries.

      Args:
         cardSlotId ( string ): The slot ID of the card whose config should be
                                generated.
         invL1IntfSlotDir ( Inventory::L1InterfaceSlotDir ): The inventory containing
                                                             all L1 inventory ports
                                                             on the SKU.

      Returns:
         The generated valid L1Profile::CardConfig if the generation was successful
         else None.
      '''

      def _generateValidCardConfig( desc ):
         '''An anonymous helper which generates and validates card configurations.
            Args:
               desc ( CardProfileDescriptor ): The descriptor of the card profile
                                               to generate the configuration off of.

            Returns:
               The generated valid L1Profile::CardConfig if the generation was
               successful else None.
         '''
         cardConfig = CardConfigGenerator.generateConfig(
            self.cardProfileLibraryRootDir,
            self.intfSlotProfileLibraryRootDir,
            cardSlotId,
            self.modelName,
            desc )
         if not cardConfig:
            TINFO3( 'Could not generate card configuration for:', cardSlotId, '-',
                    desc )
            cardProfileReader = CardProfileReader( self.cardProfileLibraryRootDir )
            profileExists = cardProfileReader.isProfilePresent( desc.name )

            if profileExists:
               return ( MODULE_INCOMPAT % desc.name, None )
            else:
               return ( PROFILE_DNE % desc.name, None )

         error = self._validateCardConfig( invL1IntfSlotDir, cardConfig )
         if error is not None:
            TINFO3( 'Could not validate card configuration for:', cardSlotId, '-',
                    desc )
            return ( error, None )

         return ( None, cardConfig )

      # First, try and generate a valid card configuration based on the CLI input
      cliCardProfileDescriptor = self.cliConfig.cardProfile.get( cardSlotId )
      if cliCardProfileDescriptor:
         TINFO3( 'Attempting to generate CLI specified card configuration for:',
                 cardSlotId, '-', cliCardProfileDescriptor )
         err, cliCardConfig = _generateValidCardConfig( cliCardProfileDescriptor )
         if cliCardConfig:
            TNOTE( 'Generated valid CLI specified card configuration for:',
                    cardSlotId, '-', cliCardProfileDescriptor )
            Logging.log( L1PROFILE_MODULE_APPLIED, cliCardProfileDescriptor.name,
                                                   self.loggingCardSlotId )
            return cliCardConfig

         if err:
            Logging.log( L1PROFILE_MODULE_ERROR, self.loggingCardSlotId, err )

      TNOTE( 'Could not generate valid CLI specified card configuration for:',
             cardSlotId )

      # If no CLI card profile is specified then attempt to determine the card's
      # default profile.
      cardDefault = self.cardDefaultDir.cardDefault.get( cardSlotId )
      if ( not cardDefault ) or ( not cardDefault.cardProfile ):
         TNOTE( 'No default L1 card profile specified for:', cardSlotId )
         return None

      TINFO3( 'Attempting to generate default card configuration for:',
              cardSlotId, '-', cliCardProfileDescriptor )

      # If generation of the default card profile has failed then something has gone
      # terribly wrong and the system is presumed to be unstable. This should never
      # be encountered in the field.
      err, defaultCardConfig = _generateValidCardConfig( cardDefault.cardProfile )
      if not defaultCardConfig:
         raise ArException(
            'Failed to generate valid default card configuration.',
            cardSlotId=cardSlotId,
            defaultCardProfile=cardDefault.cardProfile,
            error=err )

      TNOTE( 'Generated valid default card configuration for:', cardSlotId, '-',
             cliCardProfileDescriptor )
      return defaultCardConfig

   def _validateCardConfig( self, invL1IntfSlotDir, cardConfig ):
      '''Validates that a particular card configuration is actually compatible with
      the inserted card's SKU.

      Args:
         invL1IntfSlotDir ( Inventory::L1InterfaceSlotDir ): The inventory containing
                                                             all L1 inventory ports
                                                             on the SKU.
         cardConfig ( L1Profile::CardConfig ): The L1 profile card config to
                                               validate.

      Returns:
         An error string if the card config is incompatible with the SKU else None.
         If the error is meant to be "silent" then an empty string will be returned.
      '''

      cardProfile = cardConfig.cardProfile

      # Validate that the card profile is applicable to the given card
      # TODO BUG738920: For now we skip the filter if we are working with profiles
      #                 coming from the CLI
      if cardProfile.source != CardProfileSource.cli:
         if self.modelName not in cardProfile.applicability:
            TWARN( 'Invalid L1 card profile detected, card profile is not '
                    'applicable on this card:',
                    self.modelName,
                    cardProfile.applicability.keys() )
            return MODULE_INCOMPAT % cardProfile.descriptor.name


      l1PortsByInterfaceSlot = {}
      for l1IntfSlot in invL1IntfSlotDir.l1InterfaceSlot.values():
         # Skip any interface slots which do not use the same group ID as the card's
         # group ID.
         #
         # TODO BUG736820: Allow this divergence when there is a concrete use case
         #                 that can be adequately tested
         if self.slotGroupId != l1IntfSlot.descriptor.slotGroupId:
            continue

         relativeIntfSlotDesc = InterfaceSlotDescriptor(
            l1IntfSlot.descriptor.slotPrefix,
            l1IntfSlot.descriptor.intfSlotId )

         l1PortsByInterfaceSlot[ relativeIntfSlotDesc ] = l1IntfSlot

      # Validate that the card profile does not reference slots that do not exist.
      nonExistantIntfSlotProfiles = ( set( cardProfile.intfSlotProfile ) -
                                      set( l1PortsByInterfaceSlot ) )
      if nonExistantIntfSlotProfiles:
         TWARN( 'Invalid L1 card profile detected, card profile references non '
                 'existent interface slots:',
                 cardProfile.descriptor,
                 nonExistantIntfSlotProfiles )
         return ''

      # Validate that all the interface slot profiles actually point to valid ports.
      #
      # Note: Not all interface slots have interface slot profiles applied, for such
      #       cases, the FDL definitions are left as is.
      for intfSlotDesc, intfSlotProfileDesc in (
            cardProfile.intfSlotProfile.items() ):
         intfSlotProfile = cardConfig.intfSlotProfile.get( intfSlotProfileDesc )
         if not intfSlotProfile:
            TWARN( 'Invalid L1 card profile detected, card profile references non '
                    'existent interface slot profile:', intfSlotProfileDesc )
            return ''

         intfSlot = l1PortsByInterfaceSlot[ intfSlotDesc ]

         # At present, single root interface slots do not support L1 profiles.
         #
         # TODO BUG736821: Enable support once there is a concrete need.
         if len( intfSlot.rootToPort ) == 1:
            TWARN( 'Invalid L1 card profile detected, card profile references '
                    'unsupported interface slot:',
                    cardProfile.descriptor,
                    intfSlot )
            return ''

         # Any interface slot profile referring to a root that does not exist on the
         # switch should be treated like an error and cause the profile to fail to
         # apply.
         for intfLaneSpec in intfSlotProfile.intfLaneSpec.values():
            if intfLaneSpec.root not in intfSlot.rootToPort:
               TWARN( 'Invalid L1 card profile detected, interface slot profile '
                       'references non existent root:',
                       cardProfile.descriptor,
                       intfSlot,
                       intfSlotProfile.descriptor,
                       intfLaneSpec.root )
               return ''

      return None

   def _generateL1PortConfig( self, invL1IntfSlotDir, cardConfig ):
      '''Generate the L1 inventory port configuration required to satisfy the L1
      profile card config.

      Args:
         invL1IntfSlotDir ( Inventory::L1InterfaceSlotDir ): The inventory containing
                                                             all L1 inventory ports
                                                             on the SKU.
         cardConfig ( L1Profile::CardConfig ): The L1 profile card config to
                                               use in the generation process.
      '''
      l1PortConfigDir = Fru.Dep(
         self.l1PortConfigRootDir, Fru.fruBase( invL1IntfSlotDir ) ).newEntity(
            L1PortConfigDir.tacType.fullTypeName, self.cardSlotId )

      for l1IntfSlot in invL1IntfSlotDir.l1InterfaceSlot.values():
         # Skip any interface slots which do not use the same group ID as the card's
         # group ID.
         #
         # TODO BUG736820: Allow this divergence when there is a concrete use case
         #                 that can be adequately tested
         if self.slotGroupId != l1IntfSlot.descriptor.slotGroupId:
            continue

         relativeIntfSlotDesc = InterfaceSlotDescriptor(
            l1IntfSlot.descriptor.slotPrefix,
            l1IntfSlot.descriptor.intfSlotId )

         # Check if the interface slot has a profile associated with it.
         intfSlotProfileDesc = cardConfig.cardProfile.intfSlotProfile.get(
            relativeIntfSlotDesc )
         if not intfSlotProfileDesc:
            continue

         intfSlotProfile = cardConfig.intfSlotProfile[ intfSlotProfileDesc ]

         intfLaneSpecByRoot = { intfLaneSpec.root: intfLaneSpec
                                for intfLaneSpec in
                                intfSlotProfile.intfLaneSpec.values() }

         for root, port in l1IntfSlot.rootToPort.items():
            intfLaneSpec = intfLaneSpecByRoot.get( root )
            if not intfLaneSpec:
               l1PortConfigDir.l1PortConfig[ port.uid ] = L1PortConfig(
                  False,
                  0,
                  EthLinkMode.linkModeUnknown )
               TINFO3( 'Creating unbound port config for port with UID:', port.uid )
               continue

            portConfig = L1PortConfig( True,
                                       intfLaneSpec.intfLane,
                                       intfLaneSpec.defaultLinkMode )
            enforcedCaps = EnforcedCapabilities()
            for cap in intfLaneSpec.enforcedCapabilities.capabilities:
               enforcedCaps.capabilities.add( cap )
            enforcedCaps.action = intfLaneSpec.enforcedCapabilities.action

            portConfig.enforcedCapabilities = enforcedCaps

            l1PortConfigDir.l1PortConfig[ port.uid ] = portConfig

            TINFO3( 'Creating port config for port with UID:', port.uid,
                    l1PortConfigDir.l1PortConfig[ port.uid ] )

   def __init__( self, invL1IntfSlotDir, parentMib, parentDriver, driverCtx ):
      super().__init__( invL1IntfSlotDir, parentMib, parentDriver, driverCtx )

      self.modelName = parentMib.modelName

      entityMibStatus = driverCtx.entity( 'hardware/entmib' )
      redundancyStatus = driverCtx.entity( Cell.path( 'redundancy/status' ) )

      self.productConfig = driverCtx.entity(
          MountConstants.productConfigPath() )
      self.cardProfileLibraryRootDir = driverCtx.entity(
         MountConstants.cardProfileLibraryRootDirPath() )
      self.intfSlotProfileLibraryRootDir = driverCtx.entity(
         MountConstants.intfSlotProfileLibraryRootDirPath() )
      self.l1PortConfigRootDir = driverCtx.entity(
         MountConstants.l1PortConfigRootDir() )

      self.cardDefaultDir = driverCtx.entity( MountConstants.cardDefaultDirPath() )
      self.cliConfig = driverCtx.entity( MountConstants.cliConfigPath() )

      cardConfigDir = driverCtx.entity( MountConstants.cardConfigPath() )

      if not self.productConfig.enabled:
         TWARN( 'L1 profiles disabled, skipping L1 profile card config creation' )
         return

      self.cardSlotId = Fru.fruBase( invL1IntfSlotDir ).sliceId
      self.loggingCardSlotId = self.cardSlotId
      if not entityMibStatus.chassis:
         self.cardSlotId = 'FixedSystem'
         self.loggingCardSlotId = 'Switch'

      # By default, any interface slot's group ID should match the card's slot number
      # on the chassis ( e.g. 4 in the case of Linecard4 or 19 in the case of
      # Fabric1 ).
      #
      # The cardSlotId ( e.g. Fabric1 ) to card slot number ( e.g. 19 ) mapping is
      # usually specified by the chassis FDL.
      self.slotGroupId = IntfSlotGroupId.unknownIntfSlotGroupId
      if entityMibStatus.chassis:
         self.slotGroupId = Fru.fruBase( invL1IntfSlotDir ).slot

      # The first step of generating the card configuration is detecting if the
      # driver is executing in a context where card config changes should not occur
      # and short circuiting accordingly:
      #  - SSO
      #  - Fru agent restart

      # During SSO, the L1 card profile configuration cannot change as the
      # system is supposed to switch over without any difference in configuration.
      if redundancyStatus.mode != RedundancyMode.active:
         TNOTE( 'System not starting in active mode (', redundancyStatus.mode, ' ), '
                'skipping L1 profile config generation:', self.cardSlotId )
         return

      # If there is a card configuration already present then the most likely case is
      # that FRU has restarted. In such a case, the existing card configuration
      # should be preserved.
      if self.cardSlotId in cardConfigDir.cardConfig:
         TNOTE( 'L1 profile card configuration already exists, skipping L1 profile '
                'config generation:', self.cardSlotId )
         return

      # A card config is always created for a card that reaches this state, even if
      # the CLI did not specify any card profile to apply to it. This empty config
      # serves 2 purposes:
      #  1. It acts as a sentinel which prevents the card from being reprocessed in
      #     the event of SSO / FRU agent restart.
      #  2. It indicates to the port drivers that they should use the SKU defaults.
      cardConfig = Fru.Dep(
         cardConfigDir.cardConfig, Fru.fruBase( invL1IntfSlotDir ) ).newMember(
            self.cardSlotId )

      # With the empty card config created, the remaining steps involve populating
      # the config with the L1 profile definitions. At any step of the way, if errors
      # are encountered, or if no L1 profiles are specified, then the config is left
      # empty to signal to the port drivers that they should use the SKU information
      # as is.
      generatedCardConfig = self._generateCardConfig( self.cardSlotId,
                                                      invL1IntfSlotDir )
      if not generatedCardConfig:
         TERROR( 'Could not generate L1 card profile configuration for:',
                 self.cardSlotId )
         return

      # Once the card configuration has been validated, the L1 inventory port configs
      # need to be created so that the port drivers can apply the L1 Profiles.
      self._generateL1PortConfig( invL1IntfSlotDir, generatedCardConfig )

      # Finally, the generated card configuration is latched so that it may be used
      # in future hitless restarts / SSO / status reporting.
      cardConfig.copyFrom( generatedCardConfig )

      TWARN( 'L1 card profile configuration applied for:', self.cardSlotId, '-',
             generatedCardConfig.cardProfile.descriptor )

def Plugin( context ):
   context.registerDriver( L1ProfileCardConfigDriver )

   mg = context.entityManager.mountGroup()

   mg.mount( 'hardware/entmib',
             EntityMibStatus.tacType.fullTypeName,
             'r' )

   mg.mount( MountConstants.productConfigPath(),
             ProductConfig.tacType.fullTypeName,
             'r' )

   mg.mount( MountConstants.cardProfileLibraryRootDirPath(),
             'Tac::Dir',
             'ri' )

   mg.mount( MountConstants.intfSlotProfileLibraryRootDirPath(),
             'Tac::Dir',
             'ri' )

   mg.mount( MountConstants.cardDefaultDirPath(),
             CardDefaultDir.tacType.fullTypeName,
             'r' )

   mg.mount( MountConstants.cliConfigPath(),
             CliConfig.tacType.fullTypeName,
             'r' )

   mg.mount( MountConstants.cardConfigPath(),
             CardConfigDir.tacType.fullTypeName,
             'w' )

   mg.mount( MountConstants.l1PortConfigRootDir(),
             'Tac::Dir',
             'w' )

   mg.close( None )
