# Copyright (c) 2023 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import re
import weakref

from ArPyUtils.Types import ArException
import Cell
import DesiredTracing
import Fdl
import Fru
from L1Topology import ModuleLib
import Tac
import Tracing
from TypeFuture import TacLazyType

# We follow the tracing standard defined in /src/PhyTrace/README.md
th = Tracing.Handle( 'Fru.L1TopologyModuleLoader' )
DesiredTracing.desiredTracingIs( 'Fru.L1TopologyModuleLoader/0' )

t0 = th.trace0
t2 = th.trace2

L1TOPOLOGY_MODULE_FACTORY_ID = 'L1TopologyModule'

L1TopoSysdbPaths = TacLazyType( 'Hardware::L1Topology::SysdbPathConstants' )
ModuleRequestDir = TacLazyType( 'Hardware::L1Topology::ModuleRequestDir' )
InvModuleDir = TacLazyType( 'Inventory::L1Topology::ModuleDir' )
InvModuleFru = TacLazyType( 'Inventory::L1Topology::ModuleFru' )

class ModuleFdl( Fdl.Rfc822Fdl ):
   '''Mimics what a fdl does for a module defintion in the ModuleLib.'''
   def __init__( self, moduleDef, moduleName ):
      self.moduleDef = moduleDef
      self.moduleName = moduleName
      self.fdlHeaderDowncase_ = {}
      self.fdlHeaderPreservecase_ = {}
      self[ 'FruFactoryIds' ] = L1TOPOLOGY_MODULE_FACTORY_ID

   def fdlBytes( self ):
      return bytes( self.moduleName )

   # We ignore the printTraceback parameter as we'll have the crash traceback
   # directly in the current frame
   def execFdl( self, fruVarname, fruValue, printTraceback=True, extraEnv=None ):
      # Make sure to store the moduleName that this FDL represents so we can know
      # about it on reconcile
      fruValue.moduleName = self.moduleName
      env = { fruVarname: fruValue }
      env.update( self.fdlHeaderPreservecase_ )
      if extraEnv:
         env.update( extraEnv )
      self.moduleDef.buildTopology( fruValue, env )

def l1TopoModuleFactory( inv, fdl, idInParent ):
   '''Create the L1Topology module within the given module dir inventory.'''
   assert idInParent, 'A local id is required for inserted modules'

   t2( 'Constructing ModuleFru with id', idInParent )
   moduleFru = inv.newModuleFru( idInParent )
   nds = Fru.newDependentSet( moduleFru )
   # If we've already seen this moduleFru before skip reinitializing it
   if moduleFru.dependentSetKey == nds.key:
      t2( 'Skipping ModuleFru with already-seen dependent set key', nds.key )
      return moduleFru

   fruBase = Fru.fruBase( inv )
   moduleFru.component = ( L1TOPOLOGY_MODULE_FACTORY_ID, )
   moduleFru.managingCellId = fruBase.managingCellId
   moduleFru.sliceId = fruBase.sliceId + '-' + idInParent
   # Make sure to handle first init case with get
   genId = inv.moduleGenerationId.get( idInParent, 0 ) + 1
   inv.moduleGenerationId[ idInParent ] = genId
   moduleFru.generationId = genId

   env = {
      'parentFru': fruBase,
      'localSlotId': idInParent,
      'moduleSlotId': moduleFru.sliceId,
   }
   fdl.execFdl( 'module', moduleFru, extraEnv=env )
   t2( 'Module in', moduleFru.sliceId, 'executed with generation ID:',
       moduleFru.generationId )

   # Set the key last so we only no-op after the FDL has executed at least once
   moduleFru.dependentSetKey = nds.key
   return moduleFru

class L1TopologyModuleRequestReactor( Tac.Notifiee ):
   '''
   Reacts to a Hardware::L1Topology::ModuleRequestDir object, waiting for requests
   on the system. When a request comes in, we load the correct module from the
   library, create its Fru, and execute the root driver for the Fru. When a request
   is removed, we teardown the correct set of inventory models.
   '''

   notifierTypeName = ModuleRequestDir.tacType.fullTypeName

   def __init__( self, inv, parentMib, parentDriver, ctx, moduleRequestDir ):
      self.inv = inv
      self.parentMib = parentMib
      self.parentDriver = parentDriver
      self.ctx = ctx
      self.moduleDriver = {}
      self.moduleRequestDir = moduleRequestDir
      super().__init__( moduleRequestDir )

      # Reconcile the requests that we have with the modules that are already
      # inserted in the inventory models.
      self.reconcileModuleRequest()

   def reconcileModuleRequest( self ):
      # Preinitialize the modules to reconcile to be the list of all of the modules
      modulesToReconcile = set( self.moduleRequestDir.module )

      # Go through all of the existing inventory models in sorted order to make
      # things more consistent and obvious to those reading logs
      for moduleFru in sorted( self.inv.moduleFru.values(),
                               key=lambda x: x.sliceId ):
         # Allow the child drivers to run for any existing inventory models before we
         # try and clean things up. This allows any FruPlugins that depend on these
         # modules to reinitialize their Fru.Deps. It also ensures that we reconcile
         # any nested modules with the current requests before we process the ones in
         # the parent and potentially remove modules from the system.
         self.setupDriver( moduleFru )

         moduleReqName = self.moduleRequestDir.module.get( moduleFru.sliceId )
         if not moduleReqName:
            # This module has been removed from the new requests, so we need to add
            # it to our reconcile list so that we can remove it from the system.
            modulesToReconcile.add( moduleFru.sliceId )
         elif moduleReqName == moduleFru.moduleName:
            # This module hasn't changed from the last time Fru was running, so
            # theres nothing to reconcile for it.
            modulesToReconcile.remove( moduleFru.sliceId )

      # Once we've reinit everything we have inventory models for, we need to
      # reprocess any of modules that changed their requests
      if modulesToReconcile:
         t0( 'Reconciling existing module insertions with current requests:',
             modulesToReconcile )
      # Sort the processing order to make it easier to read in the logs
      for path in sorted( modulesToReconcile ):
         self.handleModuleRequest( path )

   @Tac.handler( 'module' )
   def handleModuleRequest( self, path ):
      '''
      Processes a new module insertion at the given path. Always removes the
      existing module before processing the new one.
      '''
      # We always need to clear any existing models since any removal or insertion
      # must result in a new set of models.
      localSlotId = path.rsplit( '-', 1 )[ -1 ]
      if localSlotId in self.moduleDriver:
         del self.moduleDriver[ localSlotId ]

      # We must remove the FruReady for a module before we remove its dependant
      # models so that users have a chance to perform any neccessary cleanup.
      hwSliceDir = self.ctx.sysdbRoot[ 'hardware' ][ 'slice' ]
      if path in hwSliceDir:
         hwSliceDir[ path ].deleteEntity( 'FruReady' )

      # We need to recurse through the module's inventory and clean up any models
      # that are dependant on it existing.
      oldModuleName = None
      if localSlotId in self.inv.moduleFru:
         oldModuleName = self.inv.moduleFru[ localSlotId ].moduleName
         Fru.deleteDependents( self.inv.moduleFru[ localSlotId ] )
         del self.inv.moduleFru[ localSlotId ]

      # Get the current request for the module to insert
      moduleName = self.moduleRequestDir.module.get( path )

      # Print the old module being removed if there was one
      if oldModuleName and moduleName != oldModuleName:
         t0( f'Removed "{oldModuleName}" from', path )

      # On removal we're done after clearing the models
      if not moduleName:
         return

      try:
         moduleDef = ModuleLib.getModule( moduleName )()
      except ArException:
         t0( f'No module matching "{moduleName}" found for', path )
         return

      try:
         fdl = ModuleFdl( moduleDef, moduleName )
         moduleFru = self.ctx.fruFactoryRegistry.newFru( self.inv, fdl,
                                                         idInParent=localSlotId )
      except Fru.FruFactoryError:
         t0( f'Module "{moduleName}" failed to execute FDL logic' )
         # We need to ensure that we cleanup any partially initialized state that the
         # FDL might've setup and registered in the FruDep infrastructure.
         Fru.deleteDependents( self.inv.moduleFru[ localSlotId ] )
         del self.inv.moduleFru[ localSlotId ]
         return

      t0( f'Inserted "{moduleName}" into', path )
      self.setupDriver( moduleFru )

   def setupDriver( self, moduleFru ):
      '''
      Sets up the root driver for the given module. This results in us processing all
      of the child drivers for the module.
      '''
      self.moduleDriver[ moduleFru.name ] = self.ctx.driverRegistry.newDriver(
            moduleFru, self.parentMib, self.parentDriver, self.ctx )

class L1TopologyModuleRequestAggregator( Tac.Notifiee ):
   '''
   Waits for a Hardware::L1Topology::ModuleRequestDir object to be created for an
   agent, and then aggregates the requests that are for the managed sliceId into the
   resolved ModuleRequestDir. Also handles any necessary conflict resolution between
   agent requests if and when they happen.
   '''

   notifierTypeName = 'Tac::Dir'

   class AgentReactor( Tac.Notifiee ):
      '''
      Reacts to the Hardware::L1Topology::ModuleRequest object for an agent, waiting
      for any requests to be added/removed by that agent. When any changes occur, we
      forward the change to the parent aggregator to handle.
      '''

      notifierTypeName = ModuleRequestDir.tacType.fullTypeName

      def __init__( self, aggregator, agentModuleRequest ):
         self.aggregator = aggregator
         self.agentName = agentModuleRequest.name
         self.agentModuleRequest = agentModuleRequest
         super().__init__( agentModuleRequest )

         # Handle all of the pre-existing module requests that we have.
         for path in self.agentModuleRequest.module:
            self.handleModuleRequest( path )

      @Tac.handler( 'module' )
      def handleModuleRequest( self, path ):
         moduleName = self.agentModuleRequest.module.get( path )
         self.aggregator.forwardModuleRequest( self.agentName, path, moduleName )

      def clearModuleRequests( self ):
         # When an agent stops requesting anything, we need to make sure to clear up
         # all of the requests modules that it had made previously.
         # Since we maintain a pointer to the request we should still have access to
         # the agent's old requests even after they've been removed.
         for path in self.agentModuleRequest.module:
            self.aggregator.forwardModuleRequest( self.agentName, path, None )

   def __init__( self, sliceId, rootModuleRequestDir, resolvedModuleRequestDir ):
      self.agentReactors = {}
      self.sliceId = sliceId
      self.pathRegex = re.compile( self.sliceId + r'-([^-/]+)$' )
      self.resolvedModuleRequestDir = resolvedModuleRequestDir
      self.rootModuleRequestDir = rootModuleRequestDir
      super().__init__( rootModuleRequestDir )

      # Handle all of the pre-existing agent requests that we have.
      for agentName in rootModuleRequestDir:
         self.handleAgentModuleRequest( agentName )

   @Tac.handler( 'entityPtr' )
   def handleAgentModuleRequest( self, agentName ):
      t0( 'Handling module requests for agent', agentName, 'on', self.sliceId )
      # This case shouldn't ever occur, but just to be on the safe side, skip it
      if not agentName:
         return

      # A new agent is requesting modules, so create a sub reactor for it
      agentModuleRequest = self.rootModuleRequestDir.get( agentName )
      if agentModuleRequest:
         # Pass a weak backpointer to this aggregator to let the reactor call back
         self.agentReactors[ agentName ] = \
               L1TopologyModuleRequestAggregator.AgentReactor( weakref.proxy( self ),
                                                               agentModuleRequest )

      # An agent has removed all of its requests, so cleanup its requests and reactor
      elif agentName in self.agentReactors:
         self.agentReactors[ agentName ].clearModuleRequests()
         del self.agentReactors[ agentName ]

   # Private helper to let the tracing match between the add codepaths
   def _addModuleRequest( self, agentName, path, moduleName ):
      # Log either a change or an add if we need to make a change
      oldModuleName = self.resolvedModuleRequestDir.module.get( path )
      if oldModuleName and oldModuleName != moduleName:
         t2( 'Changed request for', path, 'from', oldModuleName, 'to', moduleName,
             'for', agentName )
      elif not oldModuleName:
         t2( 'Added request for', path, 'to use', moduleName, 'from', agentName )
      self.resolvedModuleRequestDir.module[ path ] = moduleName

   # Private helper to let the tracing match between the remove codepaths
   def _delModuleRequest( self, agentName, path ):
      # Only log a removal if there was an existing request
      if path in self.resolvedModuleRequestDir.module:
         t2( 'Removed request for', path, 'from', agentName )
         del self.resolvedModuleRequestDir.module[ path ]

   def forwardModuleRequest( self, agentName, path, moduleName ):
      # Only handle requests for this aggregator.
      if not self.pathRegex.match( path ):
         return

      agentRequests = set( agentName for agentName in self.rootModuleRequestDir
                           if path in self.rootModuleRequestDir[ agentName ].module )

      # If we don't have any agent requests, we can just blindly try to remove the
      # forwarded request
      if not agentRequests:
         self._delModuleRequest( agentName, path )
         return

      # If we have more than one agent request, then we have a conflict of some kind
      # and we should remove any forwarded request for the path that might exist
      if len( agentRequests ) > 1:
         if moduleName:
            t0( 'Adding a conflicting request for', path, 'from', agentName,
                'to existing requests from', ', '.join( agentRequests ) )
         else:
            t0( 'Removed conflicting request for', path, 'from', agentName,
                'but still have conflicting requests from:',
                ', '.join( agentRequests ) )
         self._delModuleRequest( agentName, path )
         return

      # At this point we only have a single agent request, so we can safely forward
      # the request on

      # If we're handling removing a request then this single request must be a
      # resolved conflict from another agent
      if not moduleName:
         agentName = agentRequests.pop()
         moduleName = self.rootModuleRequestDir[ agentName ].module[ path ]
         t0( 'Conflict resolved for', path, '; using request from', agentName )

      self._addModuleRequest( agentName, path, moduleName )

class L1TopologyModuleRequestAsuReactor( Tac.Notifiee ):
   '''
   Reacts to the Stage::AsuCompletionDetector's resolved values for ASU. When the
   reactor detects that ASU has been completed, it triggers the FruDriver to
   reconcile the pstored module requests with the ones requested from the agents.
   '''

   notifierTypeName = 'Stage::AsuCompletionDetector'

   def __init__( self, asuStageCompletion, moduleFruDriver ):
      self.asuStageCompletion = asuStageCompletion
      self.moduleFruDriver = moduleFruDriver
      super().__init__( asuStageCompletion )
      t0( 'Waiting for ASU to complete to reconcile module requests' )

   @Tac.handler( 'asuBootCompleted' )
   def handleAsuComplete( self ):
      # We only ever expect this reactor to fire when asuBootComplete is set to true,
      # so just ignore any other value in case of spurious triggers.
      if not self.asuStageCompletion.asuBootCompleted:
         t2( 'Preemptive signal for ASU boot completed' )
         return
      t0( 'ASU boot completed, starting reconcile of module requests' )

      # Simply call into the parent driver to complete the initialization.
      self.moduleFruDriver.initializeRequests()

      # Clear out the pointer to ourselves as we have served our purpose.
      del self.moduleFruDriver.asuReactor

class L1TopologyModuleRequestDriver( Fru.FruDriver ):
   '''
   Manages the L1TopologyModuleDir object for a slice. Creates an aggregator to find
   all of the module insertion requests for this slice from all of the various
   agent's requests. Also creates a reactor to actually process the aggregated
   requests and create the models for any inserted modules.
   '''

   managedTypeName = InvModuleDir.tacType.fullTypeName

   provides = [ 'L1TopologyModuleRequest' ]
   requires = []

   def __init__( self, inv, parentMib, parentDriver, ctx ):
      super().__init__( inv, parentMib, parentDriver, ctx )
      self.sliceId = Fru.fruBase( inv ).sliceId
      t0( 'Creating a L1TopologyModuleRequest driver for ModuleDir on',
          self.sliceId )

      # Need to look at the ASU status to determine how to initialize the requests
      asuStatus = ctx.entity( 'asu/hardware/status' )
      bootCompletionStatus = ctx.entity( Cell.path( 'stage/boot/completionstatus' ) )
      redundancyStatus = ctx.entity( Cell.path( 'redundancy/status' ) )
      self.asuCompletionDetector = Tac.newInstance( "Stage::AsuCompletionDetector",
                                                    asuStatus,
                                                    bootCompletionStatus,
                                                    redundancyStatus )

      # If we're not in ASU we can just directly initialize the requests without
      # waiting at all
      if not self.asuCompletionDetector.isHitlessReloadInProgress():
         self.initializeRequests()
         return

      # When in ASU, we have to pre-populate the requests based on pstored values
      # rather than the agents' requests, as the agents wont have had a chance to
      # boot up yet. However, since we restore all of the requests for the entire
      # slice tree, we only want to do this on the root-most card. Once ASU completes
      # we will re-initialize all of the Drivers again to get the proper Driver
      # structure/ownership setup.
      if '-' not in self.sliceId:
         self.initializeAsu()

   def initializeAsu( self ):
      t0( 'Bootstrapping module requests with data from PStore on', self.sliceId )

      # Create the AsuPStoreRestorer to load in the stored requests to a local dir
      asuModuleRequestDir = ModuleRequestDir( 'asuModuleRequestDir' )
      asuHelper = Tac.newInstance( 'Hardware::L1Topology::ModuleAsuPStoreHelper',
                                   self.sliceId, asuModuleRequestDir )
      Tac.newInstance( 'Asu::AsuPStoreRestorer', asuHelper )

      # Instantiate the module request reactor once to allow it to initialize the
      # modules in Sysdb. We discard the reactor afterwards as we don't want to
      # hold onto any of the ASU state when we come back to re-initialize the
      # driver again with the actual agent requests.
      L1TopologyModuleRequestReactor(
         self.invEntity_, self.parentMibEntity_, weakref.proxy( self ),
         self.driverCtx_, asuModuleRequestDir )

      # Initialize a reactor so that we can reconcile the inserted modules on
      # ASU completion.
      self.asuReactor = L1TopologyModuleRequestAsuReactor(
         self.asuCompletionDetector, weakref.proxy( self ) )

   def initializeRequests( self ):
      t0( 'Initializing agent module request resolution logic on', self.sliceId )

      # Create a local request dir for the aggregator to output to
      localModuleRequestDir = ModuleRequestDir( 'localModuleRequestDir' )

      # Create the aggregator that monitors and handles requests from the various
      # agents to insert and remove modules and any neccessary conflict resolution.
      # Note: This must be created before the requestReactor so that it can aggregate
      #       all of the current requests for the system before the reactor attempts
      #       to reconcile those with the modules inserted in Sysdb.
      #       This is neccessary because while Fru is down the agents might change
      #       their requests or disappear entirely.
      rootModuleRequestDir = \
            self.driverCtx_.sysdbRoot.entity[ L1TopoSysdbPaths.moduleRequestDirPath ]
      self.requestAggregator = L1TopologyModuleRequestAggregator(
            self.sliceId, rootModuleRequestDir, localModuleRequestDir )

      # Create the reactor that handles actually inserting and removing modules.
      # We only pass a weakref to ourselves for the parentDriver to use for any
      # inserted modules to ensure that the nested drivers get cleaned up properly
      # when the parent containing this driver is removed. Without this the reactor
      # maintains a back pointer to this driver preventing it from being cleaned up.
      self.requestReactor = L1TopologyModuleRequestReactor(
            self.invEntity_, self.parentMibEntity_, weakref.proxy( self ),
            self.driverCtx_, localModuleRequestDir )

class L1TopologyModuleFruDriver( Fru.FruDriver ):
   '''
   Manages the L1Topology::ModuleFru object that is the root of any inserted module.
   Handles the instantiation of all of the child drivers required to process all of
   the inventory models created in the ModuleFru.
   '''

   driverPriority = 0

   managedTypeName = InvModuleFru.tacType.fullTypeName
   managedApiRe = '.*$'

   provides = [ 'L1TopologyModuleFru' ]
   requires = []

   def __init__( self, inv, parentMib, parentDriver, ctx ):
      super().__init__( inv, parentMib, parentDriver, ctx )
      t0( 'Creating a L1TopologyModuleFru driver on', inv.sliceId )

      # Create the dirs for L1Topology
      l1TopoSliceDir = ctx.entity( L1TopoSysdbPaths.topologyDirPath )[ 'slice' ]
      Fru.Dep( l1TopoSliceDir, inv ).newEntity( 'Tac::Dir', inv.sliceId )
      fruModelSliceDir = ctx.entity( L1TopoSysdbPaths.fruModelDirPath )[ 'slice' ]
      Fru.Dep( fruModelSliceDir, inv ).newEntity( 'Tac::Dir', inv.sliceId )

      # Recurse through all of the inventory models in the fru to instantiate the
      # neccessary FruDrivers for any of the models.
      self.instantiateChildDrivers( inv.component, parentMib )

      # Once we've handled all child drivers we can mark this module as FruReady
      hwSliceDir = ctx.sysdbRoot[ 'hardware' ][ 'slice' ]
      Fru.Dep( hwSliceDir, inv ).newEntity( 'Tac::Dir', inv.sliceId )
      hwSliceDir[ inv.sliceId ].newEntity( 'Tac::Dir', 'FruReady' )

   def __del__( self ):
      t2( 'Deleting L1TopologyModuleFru driver on', self.invEntity_.name )
      super().__del__()

def Plugin( ctx ):
   ctx.registerDriver( L1TopologyModuleRequestDriver )
   ctx.registerDriver( L1TopologyModuleFruDriver )
   ctx.registerFruFactory( l1TopoModuleFactory, L1TOPOLOGY_MODULE_FACTORY_ID )

   mg = ctx.entityManager.mountGroup()
   mg.mountPath( L1TopoSysdbPaths.moduleRequestDirPath )
   mg.mountPath( 'hardware/slice' )
   mg.mountPath( Cell.path( 'redundancy/status' ) )
   mg.mountPath( 'asu/hardware/status' )
   mg.mountPath( Cell.path( 'stage/boot/completionstatus' ) )
   mg.close( None )
