#!/usr/bin/env python3
# Copyright (c) 2014 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Tac, Fru
import Tracing
import Logging
import Fdl
from EntityMib import IndexAllocator
import sys
import Cell

FRU_DEVICE_INSERTED = Logging.LogHandle( "FRU_DEVICE_INSERTED",
              severity=Logging.logInfo,
              fmt="Device in slot %s has been inserted",
              explanation="Device insertion detected",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

FRU_DEVICE_REMOVED = Logging.LogHandle( "FRU_DEVICE_REMOVED",
              severity=Logging.logInfo,
              fmt="Device in slot %s has been removed",
              explanation="Device removal detected",
              recommendedAction=Logging.NO_ACTION_REQUIRED )

FRU_DEVICE_UNSUPPORTED = Logging.LogHandle( "FRU_DEVICE_UNSUPPORTED",
              severity=Logging.logError,
              fmt="Device in slot %s is not supported",
              explanation=( "The device inserted in the above slot is "
                            "unknown and not supported by the software" ),
              recommendedAction=( "Replace the device with a "
                                  "one supported by the version of your "
                                  "software." ) )

FRU_DEVICE_DRIVER = Logging.LogHandle( "FRU_DEVICE_DRIVER",
              severity=Logging.logError,
              fmt="Fatal software failure for the device in slot %s",
              explanation="A fatal error occurred when attempting to instantiate"
              " management software for the device.",
              recommendedAction="Upgrade software to the current release. Contact "
              " support if the problem persists following an upgrade. " )

FRU_FAN_UNSUPPORTED = Logging.LogHandle( "FRU_FAN_UNSUPPORTED",
              severity=Logging.logError,
              fmt="Fan in slot %s is unsupported",
              explanation="Software detected unsupported fan in system.",
              recommendedAction=( "Replace unsupported fan to improve"
              " system performance." ) )

__defaultTraceHandle__ = Tracing.Handle( "Fru.PLSlot" )
t0 = Tracing.trace0

defaultSliceId = "system"

class PLSlotStatusReactor( Tac.Notifiee ):
   notifierTypeName = "Hardware::PLSlot::SlotStatus"

   def __init__( self, hwSlotStatus, hwConfig, invPLSlotDir, parentEntMib,
                 parentDriver, ctx ):
      self.hwSlotConfig_ = hwConfig.slotConfig[ hwSlotStatus.name ]
      self.slotDir_ = invPLSlotDir
      self.slotInv_ = invPLSlotDir.slot[ hwSlotStatus.name ]
      self.parentEntMib_ = parentEntMib
      self.parentDriver_ = parentDriver
      self.driverCtx_ = ctx
      self.deviceDriver_ = None
      self.slotMib_ = None
      if self.slotInv_.slotType == "PowerSupply":
         self.slotMib_ = self.parentEntMib_.powerSupplySlot[ self.slotInv_.slotId ]
      elif self.slotInv_.slotType == "FanTray":
         self.slotMib_ = self.parentEntMib_.fanTraySlot[ self.slotInv_.slotId ]
      Tac.Notifiee.__init__( self, hwSlotStatus )
      self.handleFdl()

   def _setLastChangeTime( self ):
      if self.slotMib_:
         self.slotMib_.lastChangeTime = Tac.now()

   def _handleDeviceRemoved( self ):
      slot = self.notifier_
      self.hwSlotConfig_.fdlHash = ""
      Logging.log( FRU_DEVICE_REMOVED, slot.name )
      self.deviceDriver_ = None
      Fru.deleteDependents( self.slotInv_.device )
      Tac.makeOrphan( self.slotInv_.device )

   @Tac.handler( "state" )
   def handleState( self ):
      slot = self.notifier_
      if slot.state == "unknown":
         return

      if slot.state == "notPresent" and self.slotMib_.lastChangeTime == 0:
         self._setLastChangeTime()

   @Tac.handler( "compressedFdl" )
   def handleFdl( self ):
      slot = self.notifier_
      if slot.state == "unknown" and not self.slotInv_.fixed:
         return

      fdl = slot.fdl
      fdlHash = Fdl.fdlHash( fdl )

      deviceExists = ( self.slotInv_.device is not None )
      deviceShouldExist = ( fdl != b"" or self.slotInv_.fixed )

      if not deviceShouldExist:
         if deviceExists:
            self._handleDeviceRemoved()
            self._setLastChangeTime()
      else:
         if ( not deviceExists or
              ( deviceExists and fdlHash != self.hwSlotConfig_.fdlHash ) or
              self.slotInv_.fixed ):
            if deviceExists and not self.slotInv_.fixed:
               self._handleDeviceRemoved()

            if not self.slotInv_.fixed:
               # Slot has been inserted
               Logging.log( FRU_DEVICE_INSERTED, slot.name )

            if not fdl and not self.slotInv_.device:
               Logging.log( FRU_DEVICE_UNSUPPORTED, slot.name )

            if self.slotDir_.devicesVerificationNeeded:
               for modelName in self.slotDir_.unsupportedDevices.values():
                  if modelName.encode() in fdl:
                     Logging.log( FRU_FAN_UNSUPPORTED, self.slotInv_.slotId )
                     break

            # Fixed devices may be defined in the system fdl.
            # They may also be defined already if Fru restarts.
            device = self.slotInv_.device
            if not device:
               # Slot has been identified
               fdl = Fdl.Rfc822Fdl( fdl )
               try:
                  device = self.driverCtx_.fruFactoryRegistry.newFru(
                     self.slotInv_, fdl )
               except Fru.FruFactoryError:
                  raise Tac.InternalException # pylint: disable=raise-missing-from
               self.hwSlotConfig_.fdlHash = fdlHash

            device.fixed = self.slotInv_.fixed
            if not device.sliceId:
               device.sliceId = defaultSliceId

            Fru.newDependentSet( device )

            # Instantiate the associated driver class
            try:
               self.deviceDriver_ = self.driverCtx_.driverRegistry.newDriver(
                  device, self.parentEntMib_,
                  self.parentDriver_, self.driverCtx_ )
            except NotImplementedError:
               sys.excepthook( *sys.exc_info() )
               Logging.log( FRU_DEVICE_DRIVER, slot.name )

            self._setLastChangeTime()

class PLSlotDirDriver( Fru.FruDriver ):
   managedTypeName = "Inventory::PLSlotDir"
   requires = [ Fru.FruDriver.environmentInit ]

   def __init__( self, invPLSlotDir, parentEntMib, parentDriver, ctx ):
      Fru.FruDriver.__init__( self, invPLSlotDir,
                              parentEntMib, parentDriver, ctx )
      self.plSlotStatusReactor_ = {}
      config = ctx.sysdbRoot.entity[ 'hardware/platform/slot/config' ]
      status = ctx.sysdbRoot.entity[ 'hardware/platform/slot/status' ]
      envCoolingDomainDir = ctx.sysdbRoot.entity[
         'environment/thermostat/coolingDomain' ]

      self.slotStatusReactor_ = Tac.collectionChangeReactor(
         status.slotStatus,
         PLSlotStatusReactor,
         reactorArgs=(
            config,
            invPLSlotDir,
            parentEntMib,
            parentDriver,
            ctx,
         ),
      )

      for ( slotName, slotInv ) in invPLSlotDir.slot.items():
         t0( "PLSlotDriver looking for driver for PLSlot", slotName )
         try:
            if slotInv.slotType == "PowerSupply":
               slotMib = parentEntMib.powerSupplySlot.get( slotInv.slotId )
               if slotMib is None:
                  physicalIndex = IndexAllocator.collectionItemPhysicalIndex(
                     parentEntMib.powerSupplySlot, slotInv.slotId )
                  slotMib = parentEntMib.newPowerSupplySlot(
                     physicalIndex,
                     slotInv.slotId,
                     slotInv.slotType )
               # pylint: disable-next=consider-using-f-string
               slotMib.description = "Power Supply Slot %d" % slotInv.slotId
               slotMib.label = str( slotInv.slotId )
               slotMib.orientation = "horizontal"
            elif slotInv.slotType == "FanTray":
               slotMib = parentEntMib.fanTraySlot.get( slotInv.slotId )
               if slotMib is None:
                  physicalIndex = IndexAllocator.collectionItemPhysicalIndex(
                     parentEntMib.fanTraySlot, slotInv.slotId )
                  slotMib = parentEntMib.newFanTraySlot(
                     physicalIndex,
                     slotInv.slotId,
                     slotInv.slotType )
               # pylint: disable-next=consider-using-f-string
               slotMib.description = "Fan Tray Slot %d" % slotInv.slotId
               slotMib.label = str( slotInv.slotId )
               slotMib.orientation = "horizontal"

            coolingDomain = envCoolingDomainDir.coolingDomain.get(
               slotInv.coolingDomain )
            if coolingDomain:
               envCoolingDomainDir.slotToCoolingDomain[
                  # pylint: disable-next=consider-using-f-string
                  "%s%d" % ( slotInv.slotType, slotInv.slotId ) ] = coolingDomain
            else:
               # pylint: disable-next=consider-using-f-string
               t0( "Warning: Cooling domain \'%s\' was not found for %s%d" %
                   ( slotInv.coolingDomain, slotInv.slotType, slotInv.slotId ) )

            Fru.Dep( config.slotConfig, invPLSlotDir ).newMember(
               slotName,
               slotInv.fixed,
               slotInv.libPath,
               slotInv.devArgs,
            )
         except NotImplementedError:
            sys.excepthook( *sys.exc_info() )
            raise

      parentEntMib.initStatus = "ok"
      envCoolingDomainDir.ready = True

def plDeviceFactory( plSlot, fdl, idInParent ):
   """ Create the slot device within the given slot. This factory is
   valid for all system types. """
   assert idInParent is None

   plSlot.device = ( plSlot.name, )
   plSlot.device.component = ( "component", )
   plSlot.device.managingCellId = Cell.cellId() 
   if not plSlot.device.generationId:
      # New device insertion (as opposed to a Fru restart).
      slotDir = plSlot.parent
      slotDir.generationIdMarker += 1
      plSlot.device.generationId = slotDir.generationIdMarker

   # Create a new DependentSet for the device
   Fru.newDependentSet( plSlot.device )

   fdl.execFdl( "device", plSlot.device )

   return plSlot.device

def Plugin( ctx ):
   ctx.registerDriver( PLSlotDirDriver )

   ctx.registerFruFactory( plDeviceFactory, "PLDevice" )

   mg = ctx.entityManager.mountGroup()
   mg.mount( 'hardware/platform/slot/config', 'Hardware::PLSlot::Config', 'w' )
   mg.mount( 'hardware/platform/slot/status', 'Hardware::PLSlot::Status', 'r' )
   mg.mount( 'environment/thermostat/coolingDomain',
             'Environment::Thermostat::CoolingDomainDir', 'w' )
   mg.close( None )
