# Copyright (c) 2016 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

import Fru
import OnieVersionLib
import Tracing
from EntityMib import ChipType, IndexAllocator
import Tac

__defaultTraceHandle__ = Tracing.Handle( "Fru.PLSystem" )
t0 = Tracing.trace0

class ChipInfoReactor( Tac.Notifiee ):
   """propagates chip info from hardware to entmib"""
   notifierTypeName = "Hardware::PLSystem::Status"

   def __init__( self, status, chipMib ):
      Tac.Notifiee.__init__( self, status )
      self.chipMib = chipMib
      self.updateFirmwareRev()

   @Tac.handler( 'firmwareVersion' )
   def updateFirmwareRev( self ):
      version = self.notifier_.firmwareVersion
      t0( f"Changing firmware info from '{self.chipMib.firmwareRev}' to "
          f"'{version}'" )
      self.chipMib.firmwareRev = version

class PLSystemDriver( Fru.FruDriver ):
   managedTypeName = "Inventory::PLSystem"

   plSystemStatusReactor = None
   def __init__( self, inv, parentEntityMib, parentDriver, driverCtx ):
      Fru.FruDriver.__init__( self, inv, parentEntityMib, parentDriver, driverCtx )

      sysdbRoot = driverCtx.sysdbRoot
      hwSystemConfig = sysdbRoot[ 'hardware' ][ 'pluto' ][ 'system' ][ 'config' ]
      hwSystemConfig.libPath = inv.libPath
      hwSystemConfig.devArgs = inv.devArgs

      for i, key in enumerate( [ "FixedSystem" ] ):
         hwSystemConfig.powerOffKeys[ i ] = key
         hwSystemConfig.powerCycleKeys[ i ] = key

      self.loadOnieInstallerVersion( onieStatus=sysdbRoot[ 'onie' ][ 'status' ] )

      # Create the entMib only when cpldPresent is set
      if inv.cpldPresent:
         self.cpldChipInfo( parentEntityMib, driverCtx )

      hwSystemConfig.initialized = True

   def loadOnieInstallerVersion( self, onieStatus=None ):
      t0( "loading onie-installer into onie/status" )
      if onieStatus and not onieStatus.version:
         onieStatus.version = OnieVersionLib.onieVersion()

   def cpldChipInfo( self, parentEntityMib, driverCtx ):
      chipId = IndexAllocator.getChipId( ChipType.i2cCpld, 0 )
      chipModel = "CPLD"
      cpldChipMib = parentEntityMib.chip.get( chipId )
      if cpldChipMib is None:
         physicalIndex = IndexAllocator.collectionItemPhysicalIndex(
            parentEntityMib.chip, chipId )
         cpldChipMib = parentEntityMib.newChip( physicalIndex,
                                                chipId,
                                                chipModel + "-" +
                                                str( chipId % 20 ) )

      cpldChipMib.label = str( chipId )
      status = driverCtx.sysdbRoot.entity[ "hardware/pluto/system/status" ]
      self.plSystemStatusReactor = ChipInfoReactor( status, cpldChipMib )

def Plugin( ctx ):
   ctx.registerDriver( PLSystemDriver )

   mg = ctx.entityManager.mountGroup()
   mg.mount( 'hardware/pluto/system/config', 'Hardware::PLSystem::Config', 'w' )
   mg.mount( 'hardware/pluto/system/status', 'Hardware::PLSystem::Status', 'r' )
   mg.mount( 'onie/status', 'Onie::Status', 'w' )
   mg.close( None )
