#!/usr/bin/env python3
# Copyright (c) 2012 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

"""This module contains a plugin that manages Broadcom phys managed by the 
Strata SDK."""

import Tac, Tracing, Fru, EntityMib
# pylint: disable-next=consider-using-from-import
import FruPlugin.TempSensorFru as TempSensorFru
from EntityMib import IndexAllocator

th = Tracing.Handle( "PhyBcmFru" )
t9 = th.trace9
t0 = th.trace0

xcvrInit = "xcvr"
class PhyBcm10GBaseTDriver( Fru.FruDriver ):
   requires = [ Fru.FruDriver.systemInit, Fru.FruDriver.interfaceInit, xcvrInit ]
   provides = [ ]

   # programming interface version 1.x (c.f. AID214).
   managedTypeName = "Inventory::PhyBcm::PhyBcm10GBaseTDir"

   def __init__( self, inv, parentMib, parentDriver, driverCtx ):
      t0( "Creating a PhyBcm Fru driver for dir", inv.name )
      Fru.FruDriver.__init__( self, inv, parentMib, parentDriver, driverCtx )

      hwDir = driverCtx.sysdbRoot[ 'hardware' ]
      phyConfigDir = driverCtx.entity( 'hardware/phy/config' )

      fruHelper = Tac.newInstance( 'PhyBcm::FruPluginHelper', '' )

      # PhyBcm::Bcm10GBaseT Hw Tac Model is used only by Sand Platforms. Any merging
      # of Sand and Strata Tac Models should pay attention to Sand Specific
      # attributes in this Tac Model
      # The following directories hold references to the phy config

      fruHelper.config10GBaseTDir = \
         Fru.newStyleHardwareDir( phyConfigDir[ 'sandBcm10GBaseT' ],
                                 'Hardware::PhyBcm::Bcm10GBaseTConfigDir', inv )
      fruHelper.hwXcvrConfigDir =  driverCtx.entity( 'hardware/xcvr/config/all' )
      fruHelper.dependentSet = Fru.Dep( fruHelper, inv ).dependentSet()
      fruHelper.sliceId = Fru.fruBase( inv ).sliceId
      fruHelper.cellId = Fru.fruBase( inv ).managingCellId

      # Get the phy topology directory
      fruHelper.phyTopologyDir = hwDir[ 'phy' ][ 'topology' ][ 'allPhys' ]

      environmentDir = driverCtx.sysdbRoot[ 'environment' ]
      
      for phy in inv.phy.values():
         fruHelper.inv10GBaseT = phy
         fruHelper.phyName = phy.name
         t9( 'Phy name is %s' % phy.name ) # pylint: disable=consider-using-f-string
         fruHelper.doBcm10GBaseTPopulateHardware()
         phyConfig = fruHelper.config10GBaseTDir.phyConfig[ phy.name ]

         # BUG: There is only one sensor per phy.
         # PhyBcm10GBaseT inventory should have a non-collection
         # tempSensor attribute.
         tempSensorInv = next( iter( phy.tempSensor.values() ) )
         assert tempSensorInv
         tempSensorId = tempSensorInv.sensorId # BUG1731
         tempSensorName = EntityMib.componentNameFromParent( parentMib,
                                                             "TempSensor",
                                                             tempSensorId )
         tempSensorEnvConfig = TempSensorFru.createSensorEnvConfig(
            environmentDir[ 'temperature' ],
            tempSensorName, tempSensorInv )
         phyConfig.tempSensorConfig = tempSensorEnvConfig

         # Create the sensor EntityMib object
         sensorMib = parentMib.sensor.get( tempSensorId )
         if sensorMib is None:
            physicalIndex = IndexAllocator.collectionItemPhysicalIndex \
                            ( parentMib.sensor, tempSensorId )
            sensorMib = Fru.Dep(
               parentMib.sensor, tempSensorInv ).newMember(
                  physicalIndex, tempSensorId, "TempSensor" )

            sensorMib.description = tempSensorInv.description
            sensorMib.label = str( tempSensorId )
            sensorMib.groupName = tempSensorInv.groupName
            # Declare success for the sensor. This MUST be done after the
            # TempSensorEnvConfig have been created
            sensorMib.initStatus = "ok"

         phyConfig.configReady = True

      generation = Tac.Value( "Ark::Generation", Fru.powerGenerationId( inv ), True )
      for phyConfig in fruHelper.config10GBaseTDir.phyConfig.values():
         phyConfig.generation = generation
      fruHelper.config10GBaseTDir.generation = generation

class PhyBcm84328Driver( Fru.FruDriver ):
   requires = [ Fru.FruDriver.systemInit, Fru.FruDriver.interfaceInit, xcvrInit ]
   provides = [ ]

   # programming interface version 1.x (c.f. AID214).
   managedTypeName = "Inventory::PhyBcm::PhyBcm84328Dir"

   def __init__( self, inv, parentMib, parentDriver, driverCtx ):
      t0( "Creating a PhyBcm Fru driver for dir", inv.name )
      Fru.FruDriver.__init__( self, inv, parentMib, parentDriver, driverCtx )

      hwDir = driverCtx.sysdbRoot[ 'hardware' ]
      phyConfigDir = hwDir[ 'phy' ][ 'config' ]

      fruHelper = Tac.newInstance( 'PhyBcm::FruPluginHelper', '' )

      # The following directories hold references to the phy config

      fruHelper.configDir = \
         Fru.newStyleHardwareDir( phyConfigDir[ 'bcm84328' ],
                                  'Hardware::PhyBcm::Bcm84328ConfigDir', inv )
      fruHelper.hwXcvrConfigDir = driverCtx.entity( 'hardware/xcvr/config/all' )
      fruHelper.dependentSet = Fru.Dep( fruHelper, inv ).dependentSet()
      fruHelper.sliceId = Fru.fruBase( inv ).sliceId

      for phy in inv.phy.values():
         fruHelper.inv = phy
         fruHelper.phyName = phy.name
         t9( 'Phy name is %s' % phy.name ) # pylint: disable=consider-using-f-string
         fruHelper.doPopulateHardware()

      generation = Tac.Value( "Ark::Generation", Fru.powerGenerationId( inv ), True )
      for phyConfig in fruHelper.configDir.phyConfig.values():
         phyConfig.generation = generation
      fruHelper.configDir.generation = generation

def Plugin( context ):

   t9( "PhyBcm plugin registering with Fru" )
   context.registerDriver( PhyBcm84328Driver )
   context.registerDriver( PhyBcm10GBaseTDriver )

   t9( "PhyBcmFru mounting directories" )
   mg = context.entityManager.mountGroup()
   mg.mount( 'hardware/phy/config/sandBcm10GBaseT', 'Tac::Dir', 'wi' )
   mg.mount( 'hardware/phy/config/bcm84328', 'Tac::Dir', 'wi' )
   mg.mount( 'hardware/phy/topology/allPhys',
             'Hardware::PhyTopology::AllPhyIntfStatuses', 'w' )

   # NOTE - we don't need to mount any other subdirectories of
   # hardware/phy, as we're doing an immediate mount above. This has
   # to be mount-immediate to ensure that Fru mounts everything that
   # it has previously created in the agent restart case.

   mg.close( None )

   t9( "PhyBcmFru Plugin registration done." )
