#!/usr/bin/env python3
# Copyright (c) 2018 Arista Networks, Inc.  All rights reserved.
# Arista Networks, Inc. Confidential and Proprietary.

# pylint: disable=consider-using-f-string

from itertools import chain

import Tac
import Fru
import Tracing
import Cell
from TypeFuture import TacLazyType

from L1Topology.Constants import (
   UNKNOWN_SERDES_GROUP_ID,
   PHY_SCOPE_LINE,
   PHY_SCOPE_SYSTEM,
   POLARITY_DEFAULT,
)
from HwL1TopologyComponentLib import Components
from HwL1TopologyComponentLib.Serdes import (
   SerdesPair,
   WILDCARD
)
from HwL1TopologyComponentLib.Errors import SerdesLogicalMappingConsistencyError
from HwL1TopologyComponentLib.Tuning import NoTuning, MEDIUM_ANY

__defaultTraceHandle__ = Tracing.Handle( 'PhyTopoFru' )

t0 = Tracing.trace0
t1 = Tracing.trace1
t5 = Tracing.trace5

ChannelMode = TacLazyType( "Hardware::Phy::ChannelMode" )
EthDuplex = TacLazyType( "Interface::EthDuplex" )
ManagementScope = TacLazyType(
   'Hardware::L1Topology::ManagementScope::ManagementScope' )
MidplanePinDesc = TacLazyType( "PhyEee::MidplanePinDescriptor" )
PhyScope = TacLazyType( 'PhyEee::PhyScope' )
PhyScopeDesc = TacLazyType( 'PhyEee::PhyScopeDescriptor' )
PllSpeedGroupConfiguration = TacLazyType( 'Interface::PllSpeedGroupConfiguration' )
TuningInfo = TacLazyType( "Inventory::L1TuningInfo" )
XcvrLaneDesc = TacLazyType( "Hardware::L1Topology::XcvrLaneDescriptor" )
XcvrPadPinDesc = TacLazyType( "Hardware::L1Topology::XcvrPadPinDescriptor" )

class PhyComponentTopoDriver( Fru.FruDriver ):

   managedTypeName = 'Inventory::L1ComponentTopoDir'
   managedApiRe = '$'

   provides =  [ 'L1Topology' ]
   requires = [ "L1Topology.InterfaceSlots", "PhyTopoConfig", "PhyTraceTopo",
                "MidplanePositions", "MidplaneConnections", "StaticSerdesMap",
                'L1Capabilities' ]

   def __init__( self, l1ComponentTopo, fruEntMib, parentDriver, driverCtx ):
      Fru.FruDriver.__init__( self, l1ComponentTopo, fruEntMib,
                              parentDriver, driverCtx )

      sliceId = Fru.fruBase( l1ComponentTopo ).sliceId
      if not sliceId:
         assert Cell.cellType() == "fixed"
         sliceId = "FixedSystem"

      t0( 'Handling Inventory::L1ComponentTopoDir for %s' % sliceId )

      policyDir = driverCtx.entity( 'hardware/l1/policy' )
      topoDir = driverCtx.entity( 'hardware/l1/topology' )
      tuningDir = driverCtx.entity( 'hardware/l1/tuning' )
      mappingDir = driverCtx.entity( 'hardware/l1/mapping' )
      fruModelDir = driverCtx.entity( 'hardware/l1/fru/topology' )
      phyFruHelper = Tac.newInstance(
            'Hardware::L1Topology::PhyFruPluginHelper', topoDir )
      fruModelWriter = Tac.newInstance(
            'Hardware::L1Topology::FruModelWriter', fruModelDir )
      self.midplaneFruHelper = Tac.newInstance(
            'Hardware::L1Topology::MidplaneFruPluginHelper', topoDir )
      xcvrFruHelper = Tac.newInstance(
            'Hardware::L1Topology::XcvrFruPluginHelper', topoDir )
      tuningFruHelper = Tac.newInstance(
            'Hardware::L1Topology::TuningFruPluginHelper', tuningDir )
      self.traceFruHelper = Tac.newInstance(
            'Hardware::L1Topology::TraceFruPluginHelper', topoDir )
      self.serdesMappingHelper = Tac.newInstance(
            'Hardware::L1Topology::SerdesMappingFruHelper', mappingDir )
      self.fabricInterfaceHelper = Tac.newInstance(
            'Hardware::L1Topology::FabricInterfaceFruPluginHelper', topoDir )
      capabilitiesWriter = Tac.newInstance(
            'Hardware::L1Topology::CapabilitiesWriter', topoDir )

      # We need to mark that the topology exists on this system before we mark this
      # card as FruReady, so we do the initial bump to ( 1, false ) if necessary.
      # Note: This entity is preinit so its safe to blindly key in here.
      topoGen = topoDir[ "TopologyGeneration" ]
      if not topoGen.generation:
         topoGen.generation = Tac.Value( "Ark::Generation", 1, False )

      # TODO: See BUG574157; We need to initialize the slot generation before any
      #       changes to the topology are made, but can't preinitialize it like we do
      #       for the TOPO_GEN. Until the slice dir is made into an entity, we
      #       instead initialize it here.
      slotDir = phyFruHelper.sliceFruPluginHelper.sliceDir( sliceId )
      slotDir.newEntity( "Ark::GenerationEntity", "SlotGeneration" )

      # Always make sure that the fabric interface topology directory is present
      # before we create the topology models.
      self.fabricInterfaceHelper.addFabricInterfaceTopologyDir( sliceId )

      # Set L1 Infra API to 1
      l1InfraApi = policyDir.newEntity( "Ark::GenerationEntity", "l1InfraApi" )
      l1InfraApi.generation = Tac.Value( "Ark::Generation", 1, True )

      # add a lane for xcvr slots
      def addLane( xcvrFruHelper, sliceId, xcvrId, laneId, tx, traceId, subdomain ):
         xcvrLaneDesc = XcvrLaneDesc( sliceId, xcvrId, laneId, tx )
         xcvrLane = xcvrFruHelper.addXcvrLaneTopology( xcvrLaneDesc, subdomain )
         self.traceFruHelper.addComponentNodeToTrace( xcvrLane, traceId )
         serdesStr = 'tx' if tx else 'rx'
         t1( 'added xcvr lane topology for physical %s lane: %d,'
             'xcvr slot: %d' % ( serdesStr, laneId, xcvrId ) )

      # add a pin for xcvr pads
      def addXcvrPadPin( xcvrFruHelper, sliceId, pinId, tx, traceId ):
         xcvrPadPinDesc = XcvrPadPinDesc( sliceId, pinId, tx )
         xcvrPadPin = xcvrFruHelper.addXcvrPadPin( xcvrPadPinDesc )
         self.traceFruHelper.addComponentNodeToTrace( xcvrPadPin, traceId )
         serdesStr = 'tx' if tx else 'rx'
         t1( 'added xcvr pad pin for physical %s pin: %d' % ( serdesStr, pinId ) )

      # add a serdes for normal l1 components ( not xcvr slots )
      def addSerdes( phyFruHelper, sliceId, serdesId, componentDef,
                     phyScope, tx, chipType, chipId, traceId ):
         # TODO: See BUG489342; We assume that all of the components are of the same
         #       component type to calculate the offset.
         ( coreId, physicalLaneId ) = componentDef.getSerdes( serdesId )
         phyCoreId = chipId * componentDef.getComponentCoreIdOffset() + coreId
         phyType = componentDef.getComponentType( coreId )
         serdesPtr = phyFruHelper.addSerdesTopology(
            sliceId, chipType, chipId, phyType, phyCoreId, phyScope, physicalLaneId,
            tx )
         self.traceFruHelper.addComponentNodeToTrace( serdesPtr, traceId )

         pctt = phyFruHelper.phyChipTypeTopology( sliceId, chipType )
         pctt.chipCategory = componentDef.getChipCategory()

         # TODO BUG486097 adding componentLocalCoreId directly in PhyTopology.py only
         # instead of as a c'tor parameter carried through addSerdesTopology
         # in order to reduce impact of this new attribute which is only needed
         # by a limited set of L1Topo users (namely Tundra). Longterm
         # we would like to work on a cleaner design for platforms that benefit
         # from having the notion of cores scoped to an L1 component.
         pct = phyFruHelper.phyCoreTopology(
                  sliceId, chipType, chipId, phyType, phyCoreId )
         pct.componentLocalCoreId = coreId

         # Assign isPmaCore to the CoreTopology, which is used to identify
         # if a serdes should use PMA mappings ( PmaSerdesMappingChipConfig )
         # or PCS mappings ( SerdesmappingChipConfig ) in SerdesMappingHelper
         pct.isPmaCore = componentDef.isPmaCore( coreId )

         # Add other side serdes for phys unable to do lane remapping
         # See /src/PhyEee/HwL1Topology.tac for more information
         # If the phy does not have an "other side" this is a no-op
         mapping = componentDef.getPhysicalSerdesMappings( coreId )

         # Currently other side SerDes on Blanco are co-managed ( read: multi
         # writer ) by both this FruPlugin as well as the XCVR FRU Plugin.
         #
         # This plugin is only aware of the default mappings which may have been
         # overridden by the XCVR plugin. As such, we need to check for the existence
         # of other side SerDes before trying to re-add them. If we don't we might
         # end up adding the default mappings on top of the existing Blanco mapping,
         # this will cause the topology resolver to crash since Blanco should only
         # have one other side SerDes.
         if mapping and ( phyType != "BlancoMux" or not serdesPtr.otherSideSerdes ):
            physicalLane = serdesPtr.physicalLane
            otherLanes = []
            if phyScope == PhyScope.phyScopeLine:
               otherLanes = [ sysLanes for sysLanes, lineLanes in mapping.items()
                              if physicalLane in lineLanes ] 
            elif phyScope == PhyScope.phyScopeSystem:
               otherLanes = [ lineLanes for sysLanes, lineLanes in mapping.items()
                              if physicalLane in sysLanes ] 
            else:
               assert False, "Invalid phy scope"
            # Each physical lane should be included in at most one mapping
            if otherLanes:
               assert len( otherLanes ) == 1
               for otherLane in otherLanes[ 0 ]:
                  serdesPtr.otherSideSerdes[ otherLane ] = True

         serdesStr = 'tx' if tx else 'rx'
         t1( 'added serdes topology for physical %s serdes lane id: %d,'
             'phy core id: %d phy scope: %s phy type: %s' %
             ( serdesStr, physicalLaneId, phyCoreId, phyScope, phyType ) )
         return serdesPtr

      # For cores that are not lane/polarity-remap capable, we add the appropriate
      # static mappings.
      def addCoreStaticSerdesMap( fruModelWriter, componentDef, phyCoreTopo ):
         localCoreId = phyCoreTopo.componentLocalCoreId

         for scope, scopeTopo in phyCoreTopo.scopeTopology.items():
            coreIsLaneRemappable = componentDef.laneRemapCapable( localCoreId )
            for phyLane, serdesTopo in chain( scopeTopo.txSerdes.items(),
                                              scopeTopo.rxSerdes.items() ):
               serdesIsPolarityRemappable = componentDef.polarityRemapCapable(
                  localCoreId, scope, serdesTopo.tx )

               if coreIsLaneRemappable and serdesIsPolarityRemappable:
                  continue

               # Default P=L lane mapping
               if not coreIsLaneRemappable:
                  fruModelWriter.addStaticLaneMap( serdesTopo.descriptor, phyLane )

               # Default polarity mapping
               if not serdesIsPolarityRemappable:
                  fruModelWriter.addStaticPolarityMap( serdesTopo.descriptor,
                                                       POLARITY_DEFAULT )

      def addExpectedNoTuning( l1Component, componentDef ):
         serdesIdBase = 0
         for coreId in componentDef.getCoreIds():
            numSerdesPerCore = componentDef.getNumSerdesPerCore( coreId )
            for speed, tapGroup in componentDef.getTapGroups( coreId ).items():
               if not issubclass( tapGroup, NoTuning ):
                  continue
               tuningInfo = TuningInfo( speed, MEDIUM_ANY, "defaultTuning" )
               for serdesId in range( serdesIdBase,
                                      serdesIdBase + numSerdesPerCore ):
                  lineTuning = l1Component.newLineSerdesTuning( serdesId )
                  systemTuning = l1Component.newSystemSerdesTuning( serdesId )
                  # All we need to do is instantiate an empty L1TuningData, as
                  # we expect that for the NoTuning case ( initial/no data )
                  lineTuning.newTuningData( tuningInfo )
                  systemTuning.newTuningData( tuningInfo )
            serdesIdBase += numSerdesPerCore

      # add tuning information for normal l1 component serdes ( not xcvr slots )
      def addTuning( tuningFruHelper, sliceId, serdesId, componentDef,
                     phyScope, chipId, serdesTuning ):
         # TODO: See BUG489342; We assume that all of the components are of the same
         #       component type to calculate the offset.
         ( coreId, physicalLaneId ) = componentDef.getSerdes( serdesId )
         phyCoreId = chipId * componentDef.getComponentCoreIdOffset() + coreId
         phyType = componentDef.getComponentType( coreId )
         serdesDesc = Tac.Value( "PhyEee::SerdesDescriptor",
               sliceId, componentDef.name, chipId,
               phyType, phyCoreId, phyScope, physicalLaneId, True ) # tx=True
         supportedTapGroups = componentDef.getTapGroups( coreId )
         assert supportedTapGroups, "Cannot tune %s phy." % phyType

         # TODO: See BUG489469; Consider if we want to assert that we exactly match
         #       the speeds required by the component
         for tuningInfo, tuningData in serdesTuning.tuningData.items():
            # tuningInfo describes when we should apply tuning, while the tuningData
            # describes what actual tuning values to apply.
            tapGroup = supportedTapGroups.get( tuningInfo.speed )
            assert tapGroup, \
                   "%s does not support tuning at %s" % ( phyType, tuningInfo.speed )
            tuningDesc = Tac.Value( "Hardware::Phy::SerdesTuningInfo",
                                    tuningInfo.tuningType, tuningInfo.speed,
                                    tuningInfo.medium )

            tuningParams = tuningData.tuningParams
            if tuningParams:
               # Convert the collection of "index to value" into a tuple in order
               tuningParams = [ v for _, v in sorted( tuningParams.items() ) ]
               tapGroupLen = len( tapGroup.parameters._fields )
               assert len( tuningParams ) == tapGroupLen, \
                      "Invalid tuning params (%s), expected %d entries" % \
                      ( tuningParams, tapGroupLen )
            else:
               # If we have no tuningParams, then we must be in the NoTuning case, so
               # override the tapGroup to handle it
               tapGroup = NoTuning

            # Create the actual model and add the actual tuning values to it.
            serdesTuningData = \
               tuningFruHelper.addSerdesTuningData( serdesDesc,
                                                    tuningDesc )

            tapGroup.addTuning( serdesTuningData,
                                tapGroup.parameters( *tuningParams ) )

         t1( 'added serdes tuning for serdes lane id: %d, phy core id: %d, '
             'phy scope: %s, phy type: %s' %
             ( physicalLaneId, phyCoreId, phyScope, phyType ) )

      # TODO: See BUG512914; We should investigate converting xcvrs into a
      #       first-class component similar to midplanes to allow for subclass
      #       checking
      chipType = l1ComponentTopo.chipType
      subdomain = l1ComponentTopo.subdomain
      if chipType == "XcvrSlot":
         # for the xcvr slot we do not need to get component class
         for slotId, l1Component in l1ComponentTopo.l1ComponentTopo.items():
            t5( "processing %s slot %s" % ( l1ComponentTopo.name, slotId ) )
            for laneId, traceId in l1Component.systemRxSerdesId.items():
               t5( "  processing rx trace id %s lane %s" % ( traceId, laneId ) )
               addLane( xcvrFruHelper, sliceId, slotId, laneId, False, traceId,
                        subdomain or "" )
            for laneId, traceId in l1Component.systemTxSerdesId.items():
               t5( "  processing tx trace id %s lane %s" % ( traceId, laneId ) )
               addLane( xcvrFruHelper, sliceId, slotId, laneId, True, traceId,
                        subdomain or "" )
            for laneId, traceId in l1Component.lineRxSerdesId.items():
               t5( "  processing rx trace id %s lane %s" % ( traceId, laneId ) )
               addLane( xcvrFruHelper, sliceId, slotId, laneId, False, traceId,
                        subdomain or "" )
            for laneId, traceId in l1Component.lineTxSerdesId.items():
               t5( "  processing tx trace id %s lane %s" % ( traceId, laneId ) )
               addLane( xcvrFruHelper, sliceId, slotId, laneId, True, traceId,
                        subdomain or "" )
            # no need to check line side since this is the xcvr slot and
            # nothing is on the line side
            assert ( not l1Component.systemSerdesTuning and
                     not l1Component.lineSerdesTuning ), \
                   "Shouldn't have tuning information for a xcvr in L1Topo"
         return

      elif chipType == "XcvrPad":
         assert len( l1ComponentTopo.l1ComponentTopo ) == 1, \
                "There can only be 1 xcvr pad on a slot"
         # for the xcvr pad we do not need to get component class
         xcvrPad = l1ComponentTopo.l1ComponentTopo.values()[ 0 ]
         t5( "processing %s pad" % ( l1ComponentTopo.name ) )
         for pinId, traceId in xcvrPad.systemRxSerdesId.items():
            t5( "  processing rx trace id %s pin %s" % ( traceId, pinId ) )
            addXcvrPadPin( xcvrFruHelper, sliceId, pinId, False, traceId )
         for pinId, traceId in xcvrPad.systemTxSerdesId.items():
            t5( "  processing tx trace id %s pin %s" % ( traceId, pinId ) )
            addXcvrPadPin( xcvrFruHelper, sliceId, pinId, True, traceId )
         for pinId, traceId in xcvrPad.lineRxSerdesId.items():
            t5( "  processing rx trace id %s pin %s" % ( traceId, pinId ) )
            addXcvrPadPin( xcvrFruHelper, sliceId, pinId, False, traceId )
         for pinId, traceId in xcvrPad.lineTxSerdesId.items():
            t5( "  processing tx trace id %s pin %s" % ( traceId, pinId ) )
            addXcvrPadPin( xcvrFruHelper, sliceId, pinId, True, traceId )
         # no need to check line side since this is the xcvr pad and
         # nothing is on the line side
         assert not ( xcvrPad.systemSerdesTuning or xcvrPad.lineSerdesTuning ), \
                "Shouldn't have tuning information for a xcvr pad in L1Topo"
         return

      linePhyScope = PhyScope.phyScopeLine
      systemPhyScope = PhyScope.phyScopeSystem
      componentClass = Components.getComponent( chipType )
      if issubclass( componentClass, Components.Midplane ):
         # Have no mode for midplanes components yet, and they are modeled as having
         # a single core, so just use default mode and 0 here
         connType = componentClass().getComponentType( 0 )
         for connId, l1Component in l1ComponentTopo.l1ComponentTopo.items():
            rxPins = l1Component.lineRxSerdesId
            txPins = l1Component.lineTxSerdesId
            if rxPins or txPins:
               assert not l1Component.systemRxSerdesId, \
                      "Both system and line RX serdes specified."
               assert not l1Component.systemTxSerdesId, \
                      "Both system and line TX serdes specified."
            else:
               rxPins = l1Component.systemRxSerdesId
               txPins = l1Component.systemTxSerdesId

            # Midplane pin ids are not scoped by rx vs tx; we assume this in the
            # later layers so assert it here
            assert not ( set( rxPins ) & set( txPins ) ), \
                   "Conflicting pin ids across RX and TX for midplane."
            assert ( not l1Component.systemSerdesTuning and
                     not l1Component.lineSerdesTuning ), \
                   "Shouldn't have tuning information for a midplane in L1Topo"

            for pinId, traceId in chain( rxPins.items(),
                                         txPins.items() ):
               self.addMidplane( sliceId, connType, connId, pinId, traceId )
      else:
         # For most components we fetch the component class and use it to get extra
         # information about the serdes
         # Change the coreModes to specify None rather than '' since going through
         # the inventory models changes everything to strings
         coreModes = { k: ( v or None ) for k,v in
                       l1ComponentTopo.coreMode.items() }
         componentDef = componentClass( mode=l1ComponentTopo.mode or None,
                                        coreModes=coreModes or None )
         phyCores = set()
         for chipId, l1Component in l1ComponentTopo.l1ComponentTopo.items():
            t5( "processing %s chip %s" % ( l1ComponentTopo.name, chipId ) )
            # Create the line/system serdes for the component
            for serdesId, traceId in l1Component.lineRxSerdesId.items():
               serdes = addSerdes( phyFruHelper, sliceId, serdesId, componentDef,
                          linePhyScope, False, chipType, chipId, traceId )
               phyCores.add( serdes.phyCoreTopologyNonConst() )

            for serdesId, traceId in l1Component.lineTxSerdesId.items():
               serdes = addSerdes( phyFruHelper, sliceId, serdesId, componentDef,
                          linePhyScope, True, chipType, chipId, traceId )
               phyCores.add( serdes.phyCoreTopologyNonConst() )

            for serdesId, traceId in l1Component.systemRxSerdesId.items():
               serdes = addSerdes( phyFruHelper, sliceId, serdesId, componentDef,
                          systemPhyScope, False, chipType, chipId, traceId )
               phyCores.add( serdes.phyCoreTopologyNonConst() )

            for serdesId, traceId in l1Component.systemTxSerdesId.items():
               serdes = addSerdes( phyFruHelper, sliceId, serdesId, componentDef,
                          systemPhyScope, True, chipType, chipId, traceId )
               phyCores.add( serdes.phyCoreTopologyNonConst() )

            # Add the tuning for the tx serdes on the component
            addExpectedNoTuning( l1Component, componentDef )
            for serdesId, serdesTuning in l1Component.systemSerdesTuning.items():
               addTuning( tuningFruHelper, sliceId, serdesId, componentDef,
                          systemPhyScope, chipId, serdesTuning )
            for serdesId, serdesTuning in l1Component.lineSerdesTuning.items():
               addTuning( tuningFruHelper, sliceId, serdesId, componentDef,
                          linePhyScope, chipId, serdesTuning )

         for phyCore in phyCores:
            # Create static mappings for non-lane/polarity remap-capabable cores
            addCoreStaticSerdesMap( fruModelWriter, componentDef, phyCore )

            phyCore.fabricInterfaceAllocationPolicyName = (
               l1Component.fabricInterfaceAllocationPolicyName )

            componentSubdomain = componentDef.getSubdomain(
               coreId=phyCore.componentLocalCoreId )

            # The default for all subdomains is still currently the empty string.
            coreSubdomain = ""

            if subdomain:
               coreSubdomain = subdomain
            elif componentSubdomain is not None:
               coreSubdomain = componentSubdomain

            phyCore.subdomain = coreSubdomain.format( chipId=phyCore.chipId(),
                                                      # Mako FPGA IDs are 1-based
                                                      fpgaId=phyCore.chipId()+1 )


         self.addSerdesMapping( sliceId, componentDef )
         self.addAutonegCapabilities( componentDef )
         self.addChipCapabilities( capabilitiesWriter, componentDef, chipType )

   def addChipCapabilities( self, capsWriter, component, chipType ):
      pllCounts = component.getPllCounts()
      lineRateToPllRates = component.getLineRateToPllRates()
      logicalPortPoolInfo = component.getLogicalPortPoolInfo()
      hasPllCaps = pllCounts and lineRateToPllRates
      if not hasPllCaps and not logicalPortPoolInfo:
         return
      chipCaps = capsWriter.addChipModeCapabilities( chipType )
      t0( "Populating chip mode capabilities for", chipType )
      if hasPllCaps:
         self.addPllCapabilities( chipCaps, component,
                                  pllCounts, lineRateToPllRates )
      if logicalPortPoolInfo:
         self.addLogicalPortPoolInfo( chipCaps, logicalPortPoolInfo )

   def addPllCapabilities( self, chipCaps, component, pllCounts,
                           lineRateToPllRates ):
      for coreId, pllCount in pllCounts.items():
         phyType = component.getComponentType( coreId )
         t0( "coreId:", coreId, "phyType:", phyType, "pllCount:", pllCount )
         corePll = Tac.Value( "Hardware::L1Topology::CorePllSettings", coreId )
         corePll.pllCount = pllCount
         for lineRate, pllRates in lineRateToPllRates[ coreId ].items():
            assert isinstance( pllRates, list )
            pllRateList = PllSpeedGroupConfiguration()
            for setting in pllRates:
               pllRateList.setting.add( setting )
            corePll.lineRateToPllRateList[ lineRate ] = pllRateList
         chipCaps.corePll.addMember( corePll )

   def addLogicalPortPoolInfo( self, chipCaps, logicalPortPoolInfo ):
      for poolId, poolInfo in logicalPortPoolInfo.items():
         pool = Tac.Value( "Hardware::L1Topology::LogicalPortPoolDesc",
                           poolId, poolInfo.size )
         for coreId in poolInfo.cores:
            pool.memberCoreIds.add( coreId )
            chipCaps.coreLogicalPortPoolId[ coreId ] = poolId
         chipCaps.logicalPortPool.addMember( pool )
         t0( "poolId:", poolId, "poolSize:", pool.size,
             "memberCoreIds:", pool.memberCoreIds )

   def addAutonegCapabilities( self, component ):
      '''Given a HW L1 Topology Component, this function parses the statically
      defined autonegotiation capabilities from the class and creates the HW models
      in SysDb

      Args:
         component ( object ): The HW L1 Component
      '''
      for coreId in component.getCoreIds():
         autonegCaps = component.getSupportedAutoneg( coreId )
         forcedCaps = component.getSupportedForcedModes( coreId )
         autonegCaps.extend( forcedCaps )
         phyType = component.getComponentType( coreId )
         coreModeCaps = self.serdesMappingHelper.coreModeCapabilities( phyType )
         if ( len( coreModeCaps.lineSideAutoneg ) > 0 or
              len( coreModeCaps.systemSideAutoneg ) > 0 ):
            t0( "Autoneg capabilities of", phyType, "were already populated" )
            continue
         t0( "Populating autoneg capabilities for", phyType, autonegCaps )
         for idx, aneg in enumerate( autonegCaps ):
            if aneg.lineSideAutoneg and aneg.lineSideSpeed:
               self.serdesMappingHelper.addLineSideAutonegCap(
                  phyType, idx, aneg.lineSideAutoneg, aneg.lineSideSpeed )
            if aneg.systemSideAutoneg and aneg.systemSideSpeed:
               self.serdesMappingHelper.addSystemSideAutonegCap(
                  phyType, idx, aneg.systemSideAutoneg, aneg.systemSideSpeed )

   def addSerdesMapping( self, sliceId, component ):
      '''Given a HW L1 Topology Component, this function parses the statically
      defined logical SerDes mappings from the class, validates them, and creates the
      HW models in SysDb.

      Args:
         component ( object ): The HW L1 Component ( defined in the HW L1 Component
                                                     Lib )
      '''
      # TODO: BUG463069, investigate using tx/rx
      for coreId in component.getCoreIds():
         phyType = component.getComponentType( coreId )
         if not component.isPmaCore( coreId ):
            serdesMappings = component.getLogicalSerdesMappings( coreId )
            for desc in serdesMappings:
               self.addBidirSerdesMapping( desc, phyType, coreId,
                                           sliceId, component )

         if component.isPmaCore( coreId ):
            serdesPairs = component.getLogicalSerdesPairs( coreId )
            for serdesPair in serdesPairs:
               self.addSerdesPair( serdesPair, phyType, coreId,
                                   sliceId, component, False )

            possibleSerdesPairs = component.getPossibleSerdesPairs( coreId )
            if possibleSerdesPairs == WILDCARD:
               def generateWildCardSerdesPairs( coreId ):
                  for idx in range( component.getNumSerdesPerCore( coreId ) ):
                     for side in ( PHY_SCOPE_SYSTEM, PHY_SCOPE_LINE ):
                        for isTx in ( True, False ):
                           yield ( SerdesPair( idx, side, None, None )
                                   if isTx else
                                   SerdesPair( None, None, idx, side ) )
               possibleSerdesPairs = generateWildCardSerdesPairs( coreId )
               self.serdesMappingHelper \
                  .addPossiblePmaSerdesMappingChipConfig( phyType ) \
                  .hasWildcardMappings = True

            for serdesPair in possibleSerdesPairs:
               self.addSerdesPair( serdesPair, phyType, coreId,
                                   sliceId, component, True )

   def addBidirSerdesMapping( self, mappingDesc, phyType,
                              coreId, sliceId, component ):
      # Mutiple channels are only allowed on one side of a phy, i.e., 
      # no channel remaps
      assert( len( mappingDesc.sysChannels ) == 1 or
              len( mappingDesc.lineChannels ) == 1 )

      for sysChannelId, sysChannelSpeed, sysDuplex in mappingDesc.sysChannels:
         for lineChannelId, lineChannelSpeed, lineDuplex in mappingDesc.lineChannels:
            systemGroupId = UNKNOWN_SERDES_GROUP_ID
            lineGroupId = UNKNOWN_SERDES_GROUP_ID
            systemPhyScope = PhyScope.phyScopeUnknown
            linePhyScope = PhyScope.phyScopeUnknown
            
            sysChannelMode = ChannelMode( sysChannelId, sysChannelSpeed,
                                          sysDuplex )
            lineChannelMode = ChannelMode( lineChannelId, lineChannelSpeed,
                                           lineDuplex )
            descSysChannelMode = ChannelMode( mappingDesc.sysChannels[ 0 ][ 0 ],
                                              mappingDesc.sysChannels[ 0 ][ 1 ],
                                              sysDuplex )
            descLineChannelMode = ChannelMode( mappingDesc.lineChannels[ 0 ][ 0 ],
                                               mappingDesc.lineChannels[ 0 ][ 1 ],
                                               lineDuplex )
            
            # A mix of valid, any, unsupported channels are not allowed
            assert( descSysChannelMode.channelId == sysChannelMode.channelId or
                    ( descSysChannelMode.isValid() and
                      sysChannelMode.isValid() ) )
            assert( descLineChannelMode.channelId == lineChannelMode.channelId or
                    ( descLineChannelMode.isValid() and
                      lineChannelMode.isValid() ) )
            
            # Because there are two decoupled fields: Mode and Serdes, they need to
            # be kept in sync. e.g if Mode is None we expect Serdes to be an empty
            # list or also to be None.
            if bool( mappingDesc.sysMode ) is not bool( mappingDesc.sysSerdes ):
               raise SerdesLogicalMappingConsistencyError(
                  component, coreId, 'system', serdesMode=mappingDesc.sysMode,
                  serdes=mappingDesc.sysSerdes )

            if bool( mappingDesc.lineMode ) is not bool( mappingDesc.lineSerdes ):
               raise SerdesLogicalMappingConsistencyError(
                     component, coreId, 'line', serdesMode=mappingDesc.lineMode,
                     serdes=mappingDesc.lineSerdes )

            if mappingDesc.sysMode:
               systemPhyScope = PhyScope.phyScopeSystem
               phyScopeDesc = PhyScopeDesc(
                     sliceId, component.name, 0, phyType, coreId, systemPhyScope )

               systemGroup = ( self.serdesMappingHelper.serdesMappingHelper.\
                  thisSideSerdesGroup(
                     phyScopeDesc, mappingDesc.sysMode, sysChannelMode,
                     mappingDesc.sysSerdes[ 0 ] ) )
               if not systemGroup:
                  systemGroup = self.serdesMappingHelper.addSerdesGroup(
                     phyType, PhyScope.phyScopeSystem )
                  systemGroup.mode = mappingDesc.sysMode
                  systemGroup.channelMode = sysChannelMode
                  for serdes in mappingDesc.sysSerdes:
                     systemGroup.serdes.add( serdes )

               systemGroupId = systemGroup.id

            if mappingDesc.lineMode:
               linePhyScope = PhyScope.phyScopeLine
               phyScopeDesc = PhyScopeDesc(
                  sliceId, component.name, 0, phyType, coreId,
                  linePhyScope )

               lineGroup = ( self.serdesMappingHelper.serdesMappingHelper.\
                  thisSideSerdesGroup(
                     phyScopeDesc, mappingDesc.lineMode, lineChannelMode,
                     mappingDesc.lineSerdes[ 0 ] ) )
               if not lineGroup:
                  lineGroup = self.serdesMappingHelper.addSerdesGroup(
                     phyType, PhyScope.phyScopeLine )
                  lineGroup.mode = mappingDesc.lineMode
                  lineGroup.channelMode = lineChannelMode
                  for serdes in mappingDesc.lineSerdes:
                     lineGroup.serdes.add( serdes )
                  
               lineGroupId = lineGroup.id

            if mappingDesc.lineMode and mappingDesc.sysMode:
               self.serdesMappingHelper.addBidirSerdesMapping(
                  phyType,
                  Tac.newInstance( "Hardware::L1Topology::SerdesGroupDescriptor",
                     systemGroupId, systemPhyScope ),
                  Tac.newInstance( "Hardware::L1Topology::SerdesGroupDescriptor",
                     lineGroupId, linePhyScope ) )

   def addSerdesPair( self, serdesPair, phyType,
                      coreId, sliceId, component, possible ):
      if possible:
         addBidirSerdesMapping = (
            self.serdesMappingHelper.addBidirPossibleSerdesMapping )
      else:
         addBidirSerdesMapping = self.serdesMappingHelper.addBidirSerdesMapping
      thisSideGlobalPmaSerdes = ( 
            self.serdesMappingHelper.thisSideGlobalPmaSerdes )
      addPmaSerdes = self.serdesMappingHelper.addPmaSerdes
      
      def _addPmaSerdes( phyScope, serdesId, phyType, coreId,
                        sliceId, thisSideGlobalPmaSerdes, addPmaSerdes ):
         # If we specify None for a serdes, then we don't want this side of the pair
         # so just return None rather than doing anything
         if phyScope is None or serdesId is None:
            return None

         # The phyScopeDesc is used here for convenience, only phytype and phyScope
         # information is really needed. Therefore, we mock out the chipType and id
         phyScopeDesc = PhyScopeDesc( sliceId, '', 0, phyType, coreId, phyScope )
         serdesSpeeds = component.getSupportedSpeeds( coreId, serdesId, phyScope )
         # We expect every serdes to have at least one supported speed
         assert serdesSpeeds
 
         pmaSerdes = thisSideGlobalPmaSerdes( phyScopeDesc, serdesId )
         if not pmaSerdes:
            pmaSerdes = addPmaSerdes( phyType, phyScope )
            pmaSerdes.serdesId = serdesId
         for speed in serdesSpeeds:
            pmaSerdes.serdesSpeed.add( speed )

         return pmaSerdes

      firstPmaSerdes = _addPmaSerdes( serdesPair.firstSerdesSide,
                                      serdesPair.firstSerdesId, phyType, coreId,
                                      sliceId, thisSideGlobalPmaSerdes,
                                      addPmaSerdes )
      secondPmaSerdes = _addPmaSerdes( serdesPair.secondSerdesSide,
                                       serdesPair.secondSerdesId, phyType, coreId,
                                       sliceId, thisSideGlobalPmaSerdes,
                                       addPmaSerdes )

      # Only want to add a mapping if both sides exist
      if firstPmaSerdes and secondPmaSerdes:
         addBidirSerdesMapping(
            phyType,
            Tac.newInstance( 'Hardware::L1Topology::SerdesGroupDescriptor',
                             firstPmaSerdes.id, serdesPair.firstSerdesSide ),
            Tac.newInstance( 'Hardware::L1Topology::SerdesGroupDescriptor',
                             secondPmaSerdes.id, serdesPair.secondSerdesSide ) )


   def addMidplane( self, sliceId, connType, connId, pinId, traceId ):
      t1( "adding midplane pin %d in %s connector %s to trace %s" % 
          ( pinId, connType, connId, traceId ) )

      pinDesc = MidplanePinDesc( sliceId, connType, connId, pinId )
      pin = self.midplaneFruHelper.addMidplanePin( pinDesc )
      self.traceFruHelper.addComponentNodeToTrace( pin, traceId )

def Plugin( context ):
   '''Register the plugins.'''
   context.registerDriver( PhyComponentTopoDriver )

   mg = context.entityManager.mountGroup()
   mg.mount( 'hardware/l1/policy', 'Tac::Dir', 'wi' )
   mg.mount( 'hardware/l1/mapping', 'Tac::Dir', 'wi' )
   mg.mount( 'hardware/l1/topology', 'Tac::Dir', 'wi' )
   mg.mount( "hardware/l1/fru/topology", "Tac::Dir", "wi" )
   mg.mount( 'hardware/l1/tuning', 'Tac::Dir', 'wi' )
   mg.close( None )

